/*
 * Copyright (c) 2008 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Artem Verkhovets - initial API and implementation
 */
package org.eclipse.higgins.rpps.core.impl;

import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Date;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.AuthenticationException;
import org.eclipse.higgins.icard.AuthenticationRequiredException;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.ICard;
import org.eclipse.higgins.icard.ICardTemplate;
import org.eclipse.higgins.icard.IClaim;
import org.eclipse.higgins.icard.IClaimType;
import org.eclipse.higgins.icard.IComplexClaim;
import org.eclipse.higgins.icard.IComplexClaimType;
import org.eclipse.higgins.icard.ISimpleClaim;
import org.eclipse.higgins.icard.ISimpleClaimType;
import org.eclipse.higgins.icard.ITemplateContainer;
import org.eclipse.higgins.icard.ITemplateElement;
import org.eclipse.higgins.icard.ITemplateValue;
import org.eclipse.higgins.icard.InvalidTypeException;
import org.eclipse.higgins.icard.auth.ICredential;
import org.eclipse.higgins.icard.auth.ICredentialDescriptor;
import org.eclipse.higgins.icard.auth.IPinCodeCredential;
import org.eclipse.higgins.icard.auth.ITSKerberosV5Credential;
import org.eclipse.higgins.icard.auth.ITSSelfIssuedCredential;
import org.eclipse.higgins.icard.auth.ITSUsernamePasswordCredential;
import org.eclipse.higgins.icard.auth.ITSX509V3Credential;
import org.eclipse.higgins.icard.auth.ITokenServiceCredential;
import org.eclipse.higgins.icard.common.auth.PinCodeCredential;
import org.eclipse.higgins.icard.userprofile.CardCredential;
import org.eclipse.higgins.icard.userprofile.CardInformation;
import org.eclipse.higgins.icard.userprofile.CardUsage;
import org.eclipse.higgins.icard.userprofile.Category;
import org.eclipse.higgins.icard.userprofile.UserProfile;
import org.eclipse.higgins.rpps.core.IRppsService;
import org.eclipse.higgins.rpps.core.tobj.CardInformationTO;
import org.eclipse.higgins.rpps.core.tobj.CardUsageTO;
import org.eclipse.higgins.rpps.core.tobj.CategoryTO;
import org.eclipse.higgins.rpps.core.tobj.ICardTemplateTO;
import org.eclipse.higgins.rpps.core.tobj.ITemplateElementTO;
import org.eclipse.higgins.rpps.core.tobj.PropertyTO;
import org.eclipse.higgins.rpps.core.tobj.TokenResponseTO;
import org.eclipse.higgins.rpps.core.tobj.UITokenServiceCredentialTO;
import org.eclipse.higgins.rpps.core.tobj.UserProfileTO;

public class ConvertHelper {
	private static Log log = LogFactory.getLog(ConvertHelper.class);

	/**
	 * Converts UITokenServiceCredential transfer object to ICredential object
	 * 
	 * @param card ICard object
	 * @param siteCredential UITokenServiceCredentialTO which is need converting
	 * @return ICredential object
	 */
	public static ICredential convertUITokenServiceCredentialTO_to_ICredentials(ICard card, UITokenServiceCredentialTO siteCredential) {
		ICredential credential = null;

		ICredentialDescriptor[] userCredential = card.getRequiredCredentials();

		if ((userCredential != null) && (siteCredential != null)) {
			for (int index = 0; index < userCredential.length; index++) {
				credential = userCredential[index].getCredential();

				if ((credential instanceof IPinCodeCredential) && (IRppsService.IPinCodeCredential.equals(siteCredential.getType()))) {
					if ((siteCredential.getPinCode() != null) && (siteCredential.getPinCode().length() > 0)) {
						try {
							((IPinCodeCredential) credential).setPinCode(siteCredential.getPinCode().getBytes("UTF-8"));
						} catch (UnsupportedEncodingException exc) {
							((IPinCodeCredential) credential).setPinCode(siteCredential.getPinCode().getBytes());
						}
					}
					break;
				}

				if (!(credential instanceof ITokenServiceCredential))
					continue;

				if ((!((ITokenServiceCredential) credential).getAddress().toString().equals(siteCredential.getAddress())) || (!((ITokenServiceCredential) credential).getMetadataAddress().toString().equals(siteCredential.getMetadataAddress())))
					continue;

				if ((IRppsService.ITSUsernamePasswordCredential.equals(siteCredential.getType())) && (credential instanceof ITSUsernamePasswordCredential)) {
					if (!((ITSUsernamePasswordCredential) credential).isUsernamePredefined()) {
						((ITSUsernamePasswordCredential) credential).setUsername(siteCredential.getUsername());
					}
					((ITSUsernamePasswordCredential) credential).setPassword(siteCredential.getPassword());
					break;
				} else if ((IRppsService.ITSSelfIssuedCredential.equals(siteCredential.getType())) && (credential instanceof ITSSelfIssuedCredential)) {
					if ((siteCredential.getPinCode() != null) && (siteCredential.getPinCode().length() > 0)) {
						PinCodeCredential subCredentional = new PinCodeCredential();
						try {
							subCredentional.setPinCode(siteCredential.getPinCode().getBytes("UTF-8"));
						} catch (UnsupportedEncodingException exc) {
							subCredentional.setPinCode(siteCredential.getPinCode().getBytes());
						}
						((ITSSelfIssuedCredential) credential).setPinCodeCredential(subCredentional);
					}
					break;
				} else if ((IRppsService.ITSKerberosV5Credential.equals(siteCredential.getType())) && (credential instanceof ITSKerberosV5Credential)) {
					break;
				} else if ((IRppsService.ITSX509V3Credential.equals(siteCredential.getType())) && (credential instanceof ITSX509V3Credential)) {
					break;
				}
			}
		} else {
			if (userCredential != null) {
				for (int index = 0; index < userCredential.length; index++) {
					credential = userCredential[index].getCredential();

					if ((credential instanceof ITSSelfIssuedCredential)) {
						break;
					}
				}
			}
		}
		return credential;
	}

	public static ICredential convertCardCredential_to_ICredential(ICard card, CardCredential siteCredential) {

		UITokenServiceCredentialTO credential = null;
		if (siteCredential != null) {
			credential = new UITokenServiceCredentialTO();
			credential.setType((siteCredential.getCredentialType() != null) ? siteCredential.getCredentialType() : "");
			credential.setAddress((siteCredential.getTsAddress() != null) ? siteCredential.getTsAddress() : "");
			credential.setMetadataAddress((siteCredential.getTsMetadataAddress() != null) ? siteCredential.getTsMetadataAddress() : "");
			credential.setPassword((siteCredential.getPassword() != null) ? siteCredential.getPassword() : "");
			credential.setUsername((siteCredential.getUsername() != null) ? siteCredential.getUsername() : "");
			try {
				credential.setPinCode((siteCredential.getPinCode() != null) ? new String(siteCredential.getPinCode(), "UTF-8") : "");
			} catch (Exception exc) {
				credential.setPinCode((siteCredential.getPinCode() != null) ? new String(siteCredential.getPinCode()) : "");
			}
		}

		return convertUITokenServiceCredentialTO_to_ICredentials(card, credential);
	}

	public static CardCredential convertICredential_to_CardCredential(ICredential credential) {
		CardCredential siteCredential = null;
		if ((credential != null)) {
			siteCredential = new CardCredential();
			if (credential instanceof IPinCodeCredential) {
				byte[] pinCode = ((IPinCodeCredential) credential).getPinCode();
				siteCredential.setPinCode(pinCode);
				siteCredential.setCredentialType(IRppsService.IPinCodeCredential);
			} else {
				if ((credential instanceof ITokenServiceCredential)) {
					siteCredential.setTsAddress(((ITokenServiceCredential) credential).getAddress().toString());
					siteCredential.setTsMetadataAddress(((ITokenServiceCredential) credential).getMetadataAddress().toString());
				}
				if (credential instanceof ITSUsernamePasswordCredential) {
					siteCredential.setCredentialType(IRppsService.ITSUsernamePasswordCredential);
					siteCredential.setUsername(((ITSUsernamePasswordCredential) credential).getUsername());
					siteCredential.setPassword(((ITSUsernamePasswordCredential) credential).getPassword());
				} else if (credential instanceof ITSSelfIssuedCredential) {
					siteCredential.setCredentialType(IRppsService.ITSSelfIssuedCredential);
					if (((ITSSelfIssuedCredential) credential).getPinCodeCredential() != null) {
						byte[] pinCode = ((ITSSelfIssuedCredential) credential).getPinCodeCredential().getPinCode();
						siteCredential.setPinCode(pinCode);
					}
				} else if (credential instanceof ITSKerberosV5Credential) {
					siteCredential.setCredentialType(IRppsService.ITSKerberosV5Credential);
				} else if (credential instanceof ITSX509V3Credential) {
					siteCredential.setCredentialType(IRppsService.ITSX509V3Credential);
				}
			}
		}
		return siteCredential;
	}

	/**
	 * Converts arrays with credential's fields values from HBX to ICredential object
	 * 
	 * @param handler the callback handler to be used to request user's identity
	 * @param cuid the i-card cuid
	 * @param typeofCredential type of credential which is need convert
	 * @param table table which contains arrays values
	 * @param url url of site
	 * @return ICredential object
	 * @throws CardException
	 * @throws UnsupportedEncodingException
	 */
	public static ICredential convertHashtable_to_Credential(ICard card, String typeofCredential, Hashtable table) throws CardException, UnsupportedEncodingException {

		UITokenServiceCredentialTO credential = null;
		if ((typeofCredential != null) && (typeofCredential.trim().length() > 0)) {
			credential = new UITokenServiceCredentialTO();
			credential.setType(typeofCredential);
			credential.setAddress((table.get("address") != null) ? table.get("address").toString() : "");
			credential.setMetadataAddress((table.get("metadataAddress") != null) ? table.get("metadataAddress").toString() : "");
			credential.setPassword((table.get("password") != null) ? table.get("password").toString() : "");
			credential.setUsername((table.get("username") != null) ? table.get("username").toString() : "");
			credential.setPinCode((table.get("pin") != null) ? table.get("pin").toString() : "");
		}

		return convertUITokenServiceCredentialTO_to_ICredentials(card, credential);

	}

	/**
	 * Converts the card's claim to the property transfer object
	 * 
	 * @param card
	 * @param isCatchAuthentication
	 * @return array of PropertyTO
	 * @throws AuthenticationRequiredException
	 * @throws CardException
	 */
	public static PropertyTO[] convertClaims_to_PropertyTO(ICard card, boolean isCatchAuthentication, Iterator claimIterator) throws AuthenticationRequiredException, CardException {
		List props = new ArrayList();
		if (card != null) {

			for (Iterator it = claimIterator; it != null && it.hasNext();) {
				Object claimTypeObject = it.next();
				String claimTypeString;
				IClaimType claimType = null;
				if (claimTypeObject instanceof IClaimType)
				{
					claimTypeString = ((IClaimType)claimTypeObject).getType();
					claimType = (IClaimType)claimTypeObject;
				}
				else
					claimTypeString = claimTypeObject.toString();
								
				IClaim claim = null;
				try {
					claim = card.getClaim(claimTypeString);
					claimType = claim.getType();
				} catch (InvalidTypeException e) {
					log.error(e);
				} catch (AuthenticationRequiredException e) {
					if (isCatchAuthentication)
						throw e;
				} catch (AuthenticationException e) {
					if (isCatchAuthentication)
						throw e;
				} catch (CardException e) {
					log.error(e);
					break;
				}
				if (claimType!=null)
				{
					if (claimType.isSimple()) {
						// this claim is simple
						props.add(new PropertyTO(claimType.getDisplayName(), claimType.getType(), ((ISimpleClaimType) claimType).getDataType().getName(), getSimpleValues(claim, ((ISimpleClaimType) claimType).getMaxCardinality())));
					} else {
						// this claim is complex
						props.add(new PropertyTO(claimType.getDisplayName(), claimType.getType(), getComplexValues((IComplexClaimType) claimType, (IComplexClaim) claim, isCatchAuthentication, ((IComplexClaimType) claimType).getSupportedClaimTypes())));
					}
				}
			}
		}
		PropertyTO[] properties = new PropertyTO[props.size()];
		props.toArray(properties);

		return properties;
	}

	/**
	 * Gets the simple values from the claim object.
	 * 
	 * @param claim the claim object.
	 * @param size the size of the value array.
	 * @return the string array of values.
	 */
	public static String[] getSimpleValues(IClaim claim, int size) {
		String[] values = new String[size];
		if (claim != null && claim instanceof ISimpleClaim) {
			List claimValues = ((ISimpleClaim) claim).getValues();
			for (int i = 0; i < claimValues.size(); i++) {
				values[i] = claimValues.get(i).toString();
			}
		}
		return values;
	}

	/**
	 * Gets the complex values from the claim object.
	 * 
	 * @param claim the claim object.
	 * @param isCatchAuthentication equals "true" if needs to generete AuthenticationRequiredException or AuthenticationException
	 * @return the array of properties.
	 */
	public static PropertyTO[] getComplexValues(IComplexClaimType rootClaimType, IComplexClaim rootClaim, boolean isCatchAuthentication, Iterator claimIterator) throws CardException, InvalidTypeException {

		List claimValues = new ArrayList();
		for (Iterator it = claimIterator; it != null && it.hasNext();) {
			IClaimType claimType = (IClaimType) it.next();
			IClaim claim = null;
			try {
				claim = rootClaim.getClaim(claimType.getType());
			} catch (AuthenticationRequiredException e) {
				if (isCatchAuthentication)
					throw e;
			} catch (AuthenticationException e) {
				if (isCatchAuthentication)
					throw e;
			}

			if (claimType.isSimple()) {
				// this claim is simple
				claimValues.add(new PropertyTO(claimType.getDisplayName(), claimType.getType(), ((ISimpleClaimType) claimType).getDataType().getName(), getSimpleValues(claim, ((ISimpleClaimType) claimType).getMaxCardinality())));
			} else {
				// this claim is complex
				claimValues.add(new PropertyTO(claimType.getDisplayName(), claimType.getType(), getComplexValues((IComplexClaimType) claimType, (IComplexClaim) claim, isCatchAuthentication, ((IComplexClaimType) claimType).getSupportedClaimTypes())));
			}
		}
		PropertyTO[] values = new PropertyTO[claimValues.size()];
		claimValues.toArray(values);

		return values;
	}

	/**
	 * Convert UserProfile to UserProfile transfer object
	 * 
	 * @param userProfile
	 * @return UserProfile transfer object
	 */
	public static UserProfileTO convertUserProfile_to_UserProfileTO(UserProfile userProfile) {
		UserProfileTO userProfileTO = null;
		if (userProfile != null) {
			userProfileTO = new UserProfileTO();
			userProfileTO.setCreated(userProfile.getCreated());
			userProfileTO.setEmail(userProfile.getEmail());
			userProfileTO.setFirstName(userProfile.getFirstName());
			userProfileTO.setLastName(userProfile.getLastName());
			userProfileTO.setLoginName(userProfile.getLoginName());
			userProfileTO.setModified(userProfile.getModified());
			userProfileTO.setStatus(userProfile.getStatus());
			userProfileTO.setUsedFrom(userProfile.getUsedFrom());
			userProfileTO.setPassword(String.valueOf(userProfile.getPassword()));
		}
		return userProfileTO;
	}

	/**
	 * Convert UserProfile transfer object to UserProfile object
	 * 
	 * @param userProfile transfer object
	 * @return instance of UserProfile object
	 */
	public static UserProfile convertUserProfileTO_to_UserProfile(UserProfileTO userProfileTO) {
		UserProfile userProfile = null;
		if (userProfileTO != null) {
			userProfile = new UserProfile();
			userProfile.setCreated(userProfileTO.getCreated());
			userProfile.setEmail(userProfileTO.getEmail());
			userProfile.setFirstName(userProfileTO.getFirstName());
			userProfile.setLastName(userProfileTO.getLastName());
			userProfile.setLoginName(userProfileTO.getLoginName());
			userProfile.setModified(userProfileTO.getModified());
			userProfile.setStatus(userProfileTO.getStatus());
			userProfile.setUsedFrom(userProfileTO.getUsedFrom());
			userProfile.setPassword(userProfileTO.getPassword());
		}
		return userProfile;
	}

	/**
	 * Convert Category object to Category transfer object
	 * 
	 * @param category
	 * @return category transfer object
	 */
	public static CategoryTO convertCategory_to_CategoryTO(Category category) {
		CategoryTO categoryTO = null;
		if (category != null) {
			categoryTO = new CategoryTO();
			if (category.getIcon() != null)
				categoryTO.setIcon(category.getIcon());
			if (category.getId() != null)
				categoryTO.setId(category.getId());
			if (category.getName() != null)
				categoryTO.setName(category.getName());
			if (category.getParentId() != null)
				categoryTO.setParentId(category.getParentId());
			if (category.getSelectedIcon() != null)
				categoryTO.setSelectedIcon(category.getSelectedIcon());
			if (category.getCardList() != null) {
				String[] cuids = new String[category.getCardList().size()];
				category.getCardList().toArray(cuids);
				categoryTO.setCardList(cuids);
			}
			categoryTO.setIdx(category.getIdx());

		}
		return categoryTO;
	}

	/**
	 * Convert Category transfer object to Category object
	 * 
	 * @param categoryTO transfer object
	 * @return instance of Category object
	 */
	public static Category convertCategoryTO_to_Category(CategoryTO categoryTO) {
		Category category = null;
		if (categoryTO != null) {
			category = new Category();
			if (categoryTO.getIcon() != null)
				category.setIcon(categoryTO.getIcon());
			if (categoryTO.getId() != null)
				category.setId(categoryTO.getId());
			if (categoryTO.getName() != null)
				category.setName(categoryTO.getName());
			if (categoryTO.getParentId() != null)
				category.setParentId(categoryTO.getParentId());
			if (categoryTO.getSelectedIcon() != null)
				category.setSelectedIcon(categoryTO.getSelectedIcon());
			if (categoryTO.getCardList() != null) {
				List list = new ArrayList(categoryTO.getCardList().length);
				for (int i = 0; i < categoryTO.getCardList().length; i++)
					list.add(categoryTO.getCardList()[i]);
				category.setCardList(list);
			}
			category.setIdx(categoryTO.getIdx());
		}
		return category;
	}

	/**
	 * Gets CardInformation transfer object
	 * 
	 * @param cardInformation
	 * @param cardHistory
	 * @return CardInformation transfer object
	 */
	public static CardInformationTO createCardInformationTO(CardInformation cardInformation, List cardHistory) {
		CardInformationTO cardInformationTO = null;
		if (cardInformation != null) {
			cardInformationTO = new CardInformationTO();

			if (cardInformation.getCardCredential() != null)
				cardInformationTO.setSaveCredential(true);
			else
				cardInformationTO.setSaveCredential(false);

			if ((cardInformation.getUrls() != null) && (cardInformation.getUrls().length > 0)) {
				String[] urls = new String[cardInformation.getUrls().length];
				for (int i = 0; i < cardInformation.getUrls().length; i++) {
					String url = cardInformation.getUrls()[i].toString();
					urls[i] = (url.indexOf("?") > -1) ? url.substring(0, url.indexOf("?")) : url;
				}
				cardInformationTO.setUrls(urls);
			}

		}
		if (cardHistory != null) {
			if (cardInformationTO == null)
				cardInformationTO = new CardInformationTO();

			int size = cardHistory.size();

			Hashtable table = new Hashtable();
			for (int i = 0; i < size; i++) {
				CardUsage cardUsage = (CardUsage) cardHistory.get(i);
				String url = cardUsage.getSite().toString();
				url = (url.indexOf("?") > -1) ? url.substring(0, url.indexOf("?")) : url;
				if (table.get(url) != null) {
					Date date = (Date) table.get(url);
					if (date.compareTo(cardUsage.getDate()) < 0)
						table.put(url, cardUsage.getDate());
				} else {
					table.put(url, cardUsage.getDate());
				}
			}

			size = table.keySet().size();
			String[] keys = new String[size];
			table.keySet().toArray(keys);
			CardUsageTO[] cardHistoryTO = new CardUsageTO[size];
			for (int i = 0; i < size; i++) {
				cardHistoryTO[i] = new CardUsageTO(keys[i], (Date) table.get(keys[i]));
			}

			cardInformationTO.setCardHistory(cardHistoryTO);
		}

		return cardInformationTO;
	}

	/**
	 * Loads the properties from the string array.
	 * 
	 * @param properties the string array of properties.
	 * @param templeteTO
	 * @return the properties.
	 */
	public static ICardTemplate loadTemplate(ICardTemplate[] templates, ICardTemplateTO templeteTO) {
		ICardTemplate template = null;

		if (templates != null) {
			for (int index = 0; index < templates.length; index++) {
				// template.
				if (templates[index].getCardType().getName().equalsIgnoreCase(templeteTO.getCardType())) {
					template = templates[index];
					template.setDescription(templeteTO.getDescription());
					template.setDisplayName(templeteTO.getName());
					int size = templeteTO.getElements().length;
					for (int i = 0; i < size; i++) {
						ITemplateElement element = template.getTemplateElementByID(templeteTO.getElements()[i].getId());
						if (element != null) {
							if (element instanceof ITemplateValue) {
								setITemplateValue((ITemplateValue) element, templeteTO.getElements()[i].getValue());
							} else {
								setITemplateContainer((ITemplateContainer) element, templeteTO.getElements()[i].getContainerValue());
							}
						}
					}
				}
			}
		}
		return template;
	}

	/**
	 * 
	 * @param element
	 * @param elementTOValue
	 */
	public static void setITemplateValue(ITemplateValue element, String elementTOValue) {
		if (element != null) {
			element.setValue(elementTOValue);
		}
	}

	/**
	 * 
	 * @param root
	 * @param elementsTO
	 */
	public static void setITemplateContainer(ITemplateContainer root, ITemplateElementTO[] elementsTO) {
		if (root != null) {
			int size = elementsTO.length;
			for (int i = 0; i < size; i++) {
				ITemplateElement element = root.getTemplateElementByID(elementsTO[i].getId());
				if (element != null) {
					if (element instanceof ITemplateValue) {
						setITemplateValue((ITemplateValue) element, elementsTO[i].getValue());
					} else {
						setITemplateContainer((ITemplateContainer) element, elementsTO[i].getContainerValue());
					}
				}
			}
		}
	}

	/**
	 * Gets userProfile trasfer object
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @return user profile trasfer object
	 * @throws Exception
	 */
	public static TokenResponseTO createTokenResponse(String token) {
		TokenResponseTO trto = new TokenResponseTO();
		if (token != null) {
			trto.setTkn(token);
		}
		return trto;
	}

	/**
	 * 
	 * @param template
	 * @return
	 */
	public static ICardTemplateTO convertICardTemplate_to_ICardTemplateTO(ICardTemplate template) {
		ICardTemplateTO templateTO = new ICardTemplateTO();
		templateTO.setName(template.getDisplayName());
		templateTO.setDescription(template.getDescription());
		templateTO.setId(template.getID());
		templateTO.setProviderDescription(template.getProviderDescription());
		templateTO.setProviderID(template.getProviderID());
		templateTO.setProviderName(template.getProviderName());
		templateTO.setCardType(template.getCardType().getName());

		List elementValues = new ArrayList();

		for (Iterator iter = template.getTemplateElements(); iter != null && iter.hasNext();) {
			ITemplateElement elementValue = (ITemplateElement) iter.next();
			ITemplateElementTO elementTOValue = null;
			if (elementValue != null) {
				if (elementValue instanceof ITemplateValue) {
					elementTOValue = getITemplateValue(elementValue);
				} else {
					elementTOValue = getITemplateContainer(elementValue);
				}
			}
			if (elementTOValue != null) {
				elementValues.add(elementTOValue);
			}
		}
		ITemplateElementTO[] values = new ITemplateElementTO[elementValues.size()];
		elementValues.toArray(values);

		templateTO.setElements(values);

		return templateTO;
	}

	/**
	 * 
	 * @param element
	 * @return
	 */
	private static ITemplateElementTO getITemplateValue(ITemplateElement element) {
		if (element != null && element instanceof ITemplateValue) {
			ITemplateElementTO elementTO = new ITemplateElementTO();
			ITemplateValue tempElement = (ITemplateValue) element;
			if (tempElement.getDisplayName() != null)
				elementTO.setName(tempElement.getDisplayName());
			if (tempElement.getDescription() != null)
				elementTO.setDescription(tempElement.getDescription());
			if (tempElement.getID() != null)
				elementTO.setId(tempElement.getID());
			if (tempElement.getUIDescriptor() != null) {
				if (tempElement.getUIDescriptor().getInputMask() != null)
					elementTO.setInputMask(tempElement.getUIDescriptor().getInputMask());
				if (tempElement.getUIDescriptor().getOptionalValues() != null) {
					String[] values = new String[tempElement.getUIDescriptor().getOptionalValues().size()];
					tempElement.getUIDescriptor().getOptionalValues().toArray(values);
					elementTO.setOptionalValues(values);
				}
				if (tempElement.getUIDescriptor().getPattern() != null)
					elementTO.setPattern(tempElement.getUIDescriptor().getPattern());

				elementTO.setType(tempElement.getUIDescriptor().getType());

			}

			if (tempElement.getValue() != null)
				elementTO.setValue(tempElement.getValue());

			return elementTO;
		}
		return null;
	}

	/**
	 * 
	 * @param element
	 * @return
	 */
	private static ITemplateElementTO getITemplateContainer(ITemplateElement element) {
		ITemplateElementTO container = null;
		if (element != null && element instanceof ITemplateContainer) {
			container = new ITemplateElementTO();
			container.setName(element.getDisplayName());
			container.setDescription(element.getDescription());
			container.setId(element.getID());

			List elementValues = new ArrayList();

			for (Iterator iter = ((ITemplateContainer) element).getTemplateElements(); iter != null && iter.hasNext();) {
				ITemplateElement elementValue = (ITemplateElement) iter.next();
				ITemplateElementTO elementTOValue = null;
				if (elementValue != null) {
					if (elementValue instanceof ITemplateValue) {
						elementTOValue = getITemplateValue(elementValue);
					} else {
						elementTOValue = getITemplateContainer(elementValue);
					}
				}
				if (elementTOValue != null) {
					elementValues.add(elementTOValue);
				}
			}
			ITemplateElementTO[] values = new ITemplateElementTO[elementValues.size()];
			elementValues.toArray(values);

			container.setContainerValue(values);
		}
		return container;
	}

	public static URL convertString_to_URL(String url) throws MalformedURLException {
		String url_ = (url.indexOf("?") > -1) ? url.substring(0, url.indexOf("?")) : url;
		return new URL(url_);
	}
}
