/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergei Yakovlev - initial API and implementation
 *     Artem Verkhovets - initial API and implementation
 */

package org.eclipse.higgins.rpps.core.impl;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.LineNumberReader;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.rmi.RemoteException;
import java.security.AccessController;
import java.security.InvalidAlgorithmParameterException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.PrivilegedAction;
import java.security.Security;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Properties;
import java.util.Random;
import java.util.Vector;

import javax.net.ssl.ManagerFactoryParameters;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactorySpi;
import javax.net.ssl.X509TrustManager;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.UnsupportedCallbackException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.xml.security.utils.Base64;
import org.eclipse.higgins.icard.CUID;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.ICard;
import org.eclipse.higgins.icard.ICardProvider;
import org.eclipse.higgins.icard.ICardTemplate;
import org.eclipse.higgins.icard.IClaim;
import org.eclipse.higgins.icard.IComplexClaim;
import org.eclipse.higgins.icard.IPersonalInformationCard;
import org.eclipse.higgins.icard.IPolicy;
import org.eclipse.higgins.icard.ISimpleClaim;
import org.eclipse.higgins.icard.ISimpleClaimType;
import org.eclipse.higgins.icard.ITemplateContainer;
import org.eclipse.higgins.icard.ITemplateElement;
import org.eclipse.higgins.icard.ITemplateValue;
import org.eclipse.higgins.icard.IURICard;
import org.eclipse.higgins.icard.InvalidStateException;
import org.eclipse.higgins.icard.InvalidTypeException;
import org.eclipse.higgins.icard.InvalidValueCountException;
import org.eclipse.higgins.icard.InvalidValueException;
import org.eclipse.higgins.icard.ReadOnlyObjectException;
import org.eclipse.higgins.icard.auth.ICredential;
import org.eclipse.higgins.icard.auth.ICredentialDescriptor;
import org.eclipse.higgins.icard.auth.IPinCodeCredential;
import org.eclipse.higgins.icard.auth.ITSKerberosV5Credential;
import org.eclipse.higgins.icard.auth.ITSSelfIssuedCredential;
import org.eclipse.higgins.icard.auth.ITSUsernamePasswordCredential;
import org.eclipse.higgins.icard.auth.ITSX509V3Credential;
import org.eclipse.higgins.icard.auth.ITokenServiceCredential;
import org.eclipse.higgins.icard.common.auth.PinCodeCredential;
import org.eclipse.higgins.icard.common.auth.callback.PinCodeCallback;
import org.eclipse.higgins.icard.policy.ICardSpacePolicy;
import org.eclipse.higgins.icard.provider.cardspace.common.PPIDCardSpacePolicy;
import org.eclipse.higgins.icard.userprofile.CardInformation;
import org.eclipse.higgins.icard.userprofile.CardUsage;
import org.eclipse.higgins.icard.userprofile.Category;
import org.eclipse.higgins.icard.registry.ICardRegistry;
import org.eclipse.higgins.icard.userprofile.CardCredential;
import org.eclipse.higgins.icard.userprofile.UserProfile;
import org.eclipse.higgins.icard.userprofile.UserProfileException;
import org.eclipse.higgins.iss.CredentialContainer;
import org.eclipse.higgins.iss.ICardSelectorService;
import org.eclipse.higgins.iss.IICardSelector;
import org.eclipse.higgins.iss.IIdentityToken;
import org.eclipse.higgins.iss.SelectionANDofORs;
import org.eclipse.higgins.iss.SelectionANDofORsElm;
import org.eclipse.higgins.iss.UCTelm;
import org.eclipse.higgins.iss.UserChoiceTree;
import org.eclipse.higgins.iss.UserChoiceTree_ANDofORs;
import org.eclipse.higgins.iss.cardspace.IdentityToken;
import org.eclipse.higgins.iss.cardspace.util.UserProfileTokenHelper;
import org.eclipse.higgins.rpps.core.IRppsService;
import org.eclipse.higgins.rpps.core.RppsException;
import org.eclipse.higgins.rpps.core.UserInterfaceResponse;
import org.eclipse.higgins.rpps.core.rss.RssServlet;
import org.eclipse.higgins.rpps.core.tobj.CardInformationTO;
import org.eclipse.higgins.rpps.core.tobj.CategoryTO;
import org.eclipse.higgins.rpps.core.tobj.ExtraTokenResponseTO;
import org.eclipse.higgins.rpps.core.tobj.ICardTO;
import org.eclipse.higgins.rpps.core.tobj.ICardTemplateTO;
import org.eclipse.higgins.rpps.core.tobj.ICardsAndCategoryTO;
import org.eclipse.higgins.rpps.core.tobj.PropertyTO;
import org.eclipse.higgins.rpps.core.tobj.TokenResponseTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseAndTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseCardTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseClauseTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseOrTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseRPTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseTO;
import org.eclipse.higgins.rpps.core.tobj.UIResponseUsableCardsTO;
import org.eclipse.higgins.rpps.core.tobj.UITokenServiceCredentialTO;
import org.eclipse.higgins.rpps.core.tobj.UserProfileTO;
import org.eclipse.higgins.sts.api.ISTSResponse;

/**
 * The object used for executing Higgins API and returning the results it produces.
 * 
 * @author Sergei Yakovlev
 * @author Artem Verkhovets
 * 
 */
public class RppsService implements IRppsService {

	private Log log = LogFactory.getLog(RppsService.class);

	protected static RppsService instance = null;

	/**
	 * Creates an RppsService object.
	 * 
	 * @return the RppsService object
	 */
	public synchronized static IRppsService getInstance() {
		/*
		 * if (instance == null) { instance = new RppsService(); }
		 */
		return new RppsService();
	}

	/**
	 * Gets URI card by context ref and CUID
	 * 
	 * @param handler the callback handler to be used to request user's identity
	 * @param cref the Context Ref as a String
	 * @param cuid the CUID
	 * @return the i-card object
	 */
	private ICard getURIICard(CallbackHandler handler, String cref, String cuid) {
		// gets all registered providers
		for (Iterator pit = ICardRegistry.getInstance().getICardProviders(); pit.hasNext();) {
			ICardProvider provider = (ICardProvider) pit.next();
			try {
				for (Iterator icards = provider.getICards(handler, null); icards.hasNext();) {
					ICard icard = (ICard) icards.next();
					if (icard instanceof IURICard && ((IURICard) icard).getContextRef().equals(cref) && ((IURICard) icard).getCuid().equals(cuid)) {
						return icard;
					}
				}
			} catch (CardException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}
		return null;
	}

	/**
	 * Acquires the i-card from RP site.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param subscribeURL the URL of the site from where i-card will be acquired
	 * @return a new card id
	 */
	public String acquireICard(String userId, String password, String subscribeURL) throws MalformedURLException {
		FeedProperties feedProperties = null;
		URL subscribeURL_ = new URL(subscribeURL);
		String rssDir = RssServlet.getRssDir();
		CallbackHandler handler = RppsHelper.getCallbackHandler(userId, password);
		try {
			NameCallback name = new NameCallback("name");
			handler.handle(new Callback[] { name });
			feedProperties = RssService.getInstance().loadFeedProperties(subscribeURL_, rssDir + "/" + name.getName());
		} catch (IOException e) {
			throw new RppsException(e);
		} catch (UnsupportedCallbackException e) {
			throw new RppsException(e);
		}
		return feedProperties != null ? createICard(handler, feedProperties.getId(), feedProperties.getProperties()).getCUID().toString() : null;
	}

	/**
	 * Locate the URIICard-type I-Card whose URI matches 'cref'. If no such card exists, create it. Find the Node whose unique identifier is 'cuid'. On the uniqueIdentifier attribute value itself: (1) if 'success' is true, add/update to "now" the the lastVerifiedFromSource metadata property's timestamp and (2) add/update the lastVerifiedAttempt metadata property's timestamp to "now".
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param host the host name of the RP site where user is verified
	 * @param username the user name to login to the RP site
	 * @param success the verifying result
	 */
	public void connectionNotify(String userId, String password, String host, String username, boolean success) throws MalformedURLException {

		CallbackHandler handler = RppsHelper.getCallbackHandler(userId, password);
		URL host_ = new URL(host);
		ICard icard = getURIICard(handler, host_.toString(), username);
		if (icard != null) {
			IClaim lastConnectionAttemptClaim;
			try {
				lastConnectionAttemptClaim = icard.getClaimByLocalName("lastConnectionAttempt");
			} catch (CardException e) {
				throw new RppsException(e);
			}
			List values = new ArrayList();
			values.add(new Date());
			if (lastConnectionAttemptClaim != null) {
				// sets current date
				try {
					((ISimpleClaim) lastConnectionAttemptClaim).setValues(values);
				} catch (InvalidValueException e) {
					throw new RppsException(e.getMessage());
				} catch (InvalidStateException e) {
					throw new RppsException(e.getMessage());
				} catch (ReadOnlyObjectException e) {
					throw new RppsException(e.getMessage());
				} catch (InvalidValueCountException e) {
					throw new RppsException(e.getMessage());
				}
			} else {
				throw new RppsException("There is not found a claim: lastConnectionAttempt");
			}
			IClaim lastConnectionClaim;
			try {
				lastConnectionClaim = icard.getClaimByLocalName("lastConnection");
			} catch (CardException e) {
				throw new RppsException(e);
			}
			if (lastConnectionClaim != null) {
				// sets current date if verification is succeed
				if (success) {
					try {
						((ISimpleClaim) lastConnectionClaim).setValues(values);
					} catch (InvalidValueException e) {
						throw new RppsException(e.getMessage());
					} catch (InvalidStateException e) {
						throw new RppsException(e.getMessage());
					} catch (ReadOnlyObjectException e) {
						throw new RppsException(e.getMessage());
					} catch (InvalidValueCountException e) {
						throw new RppsException(e.getMessage());
					}
					/*
					 * NameCallback name = new NameCallback("name"); try { handler.handle(new Callback[] {name}); } catch (IOException e) { throw new RppsException(e); } catch (UnsupportedCallbackException e) { throw new RppsException(e); } setValidated(handler, host, username);
					 */
				}
			} else {
				throw new RppsException("There is not found a claim: lastConnection");
			}
		} else {
			throw new RppsException("There is not found an i-card for host: " + host_ + ", username: " + username);
		}
	}

	/**
	 * The RP site invokes this method to verify that user really has an account on the specified site.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param host The verified site url.
	 * @param username The username to login onto verified site.
	 * @return The result of verification process.
	 */
	public boolean connectionVerify(String userId, String password, String host, String username) throws MalformedURLException {
		CallbackHandler handler = RppsHelper.getCallbackHandler(userId, password);
		URL host_ = new URL(host);
		ICard icard = getURIICard(handler, host_.toString(), username);
		String lastConnectionAttempt = null;
		String lastConnection = null;
		if (icard != null) {
			IClaim lastConnectionAttemptClaim;
			try {
				lastConnectionAttemptClaim = icard.getClaimByLocalName("lastConnectionAttempt");
			} catch (CardException e) {
				throw new RppsException(e);
			}
			if (lastConnectionAttemptClaim != null) {
				lastConnectionAttempt = ((ISimpleClaim) lastConnectionAttemptClaim).getValues().get(0).toString();
			} else {
				throw new RppsException("There is not found a claim: lastConnectionAttempt");
			}
			IClaim lastConnectionClaim;
			try {
				lastConnectionClaim = icard.getClaimByLocalName("lastConnection");
			} catch (CardException e) {
				throw new RppsException(e);
			}
			if (lastConnectionClaim != null) {
				lastConnection = ((ISimpleClaim) lastConnectionClaim).getValues().get(0).toString();
			} else {
				throw new RppsException("There is not found a claim: lastConnection");
			}
		} else {
			throw new RppsException("There is not found an i-card for host: " + host_ + ", username: " + username);
		}
		return lastConnectionAttempt != null && lastConnection != null && lastConnectionAttempt.equals(lastConnection);
	}

	/**
	 * Gets a form filling script for the specified site&user
	 * 
	 * @param page the page url where user lands on
	 * @param username the user name
	 * @param form the HTML form name
	 * @return a form filling script
	 */
	public String getFormFillingScript(URL page, String username, String form) {
		// TODO Auto-generated method stub
		return null;
	}

	/**
	 * Gets an HTML scraping script for the specified site
	 * 
	 * @param page the page url
	 * @return an HTML scraping script
	 */
	public String getHtmlScrapingScript(String page) throws MalformedURLException, RppsException {
		String script = "";
		URL page_ = new URL(page);
		try {
			BufferedReader br = FileService.getInstance().getBufferedReader(FileService.SCRIPTS_DIR + FileService.convertURLToPath(page_) + FileService.SCRIPTS_EXT);
			String line;
			while ((line = br.readLine()) != null) {
				script += line;
			}
		} catch (IOException e) {
			log.error(e);
			throw new RppsException(e);
		}
		return script;
	}

	/**
	 * Sets an HTML scraping script for the specified site
	 * 
	 * @param page the page url
	 * @param script the script body
	 */
	public void setHtmlScrapingScript(String page, String script) throws MalformedURLException, RppsException {
		BufferedWriter bw = null;
		URL page_ = new URL(page);
		try {
			bw = FileService.getInstance().getBufferedWriter(FileService.SCRIPTS_DIR + FileService.convertURLToPath(page_) + FileService.SCRIPTS_EXT);
			bw.write(script);
		} catch (IOException e) {
			log.error(e);
			throw new RppsException(e);
		} finally {
			try {
				if (bw != null) {
					bw.close();
				}
			} catch (IOException e) {
				log.error(e);
				throw new RppsException(e);
			}
		}
	}

	/* Method of icard */

	/**
	 * Creates a new card
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param id the i-card id
	 * @param properties the i-card properties
	 * @return a new card id
	 */
	public ICardTO createICard(String userId, String password, String id, String[] properties) throws RppsException {
		ICard card = null;
		try {
			card = createICard(RppsHelper.getCallbackHandler(userId, password), URI.create(id), RppsHelper.loadProperties(properties));
			return convertICard_To_ICardTO(RppsHelper.getCallbackHandler(userId, password), card, false);

		} catch (Exception exc) {
			log.error(exc);
			throw new RppsException(exc);
		}
	}

	protected ICard createICard(CallbackHandler handler, URI id, Properties properties) throws RppsException {
		// gets all registered providers
		for (Iterator it = ICardRegistry.getInstance().getICardProviders(); it.hasNext();) {
			ICardProvider provider = (ICardProvider) it.next();
			// check if current provider supports needed protocol
			if (provider.canCreateCard(handler, id.toString(), properties)) {
				try {
					return provider.createCard(handler, id.toString(), properties);
				} catch (Exception ex) {
					throw new RppsException(ex);
				}
			}
		}
		throw new RppsException("There is not a registered provider for this type of i-card.");
	}

	/**
	 * Creates a new card by template
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param templete template
	 * @return a new card id
	 */
	public ICardTO createICard(String userId, String password, ICardTemplateTO template) throws RppsException {
		ICard card = null;
		try {
			ICardTemplate[] templates = getICardCreationTemplateByProvider(RppsHelper.getCallbackHandler(userId, password), template.getProviderID());
			card = createICard(RppsHelper.getCallbackHandler(userId, password), RppsHelper.fillICardTemplate_from_ICardTemplateTO(templates, template));
			return convertICard_To_ICardTO(RppsHelper.getCallbackHandler(userId, password), card, false);
		} catch (Exception exc) {
			log.error(exc);
			throw new RppsException(exc);
		} finally {

		}
	}

	protected ICard createICard(CallbackHandler handler, ICardTemplate template) throws RppsException {
		try {
			if (template != null) {
				// gets all registered providers
				String extID = template.getProviderID();
				ICardProvider provider = ICardRegistry.getInstance().getICardProvider(extID);
				if (provider != null) {
					// check if current provider supports needed protocol
					if (provider.getCardCreationTemplates(handler) != null) {
						try {
							return provider.createCard(handler, template);
						} catch (Exception ex) {
							throw new RppsException(ex);
						}
					}
				}
				throw new RppsException("There is not a registered provider for this id.");
			}
			throw new RppsException("Paramenter <<template>> is null");
		} catch (RppsException exc) {
			log.error(exc);
			throw exc;
		}
	}

	/**
	 * Creates the duplicate card with new card's name
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cardName new card's name
	 * @param cuid cuid of old card
	 * @param pinCode pincode if it needs
	 * @return card trasfer object
	 * @throws RppsException
	 */
	public ICardTO createDuplicateICard(String userId, String password, String cardName, String cuid, String pinCode) throws RppsException {
		ICard card = null;
		try {
			card = createDuplicateICard(RppsHelper.getCallbackHandler(userId, password), cardName, new CUID(cuid), pinCode);
			return convertICard_To_ICardTO(RppsHelper.getCallbackHandler(userId, password), card, false);
		} catch (Exception exc) {
			log.error(exc);
			throw new RppsException(exc);
		} finally {

		}
	}

	protected ICard createDuplicateICard(CallbackHandler handler, String cardName, CUID cuid, String pinCode) throws RppsException {
		try {
			ICard card = ICardRegistry.getInstance().getICardByCUID(handler, cuid);

			if (!(card instanceof IPersonalInformationCard)) {
				throw new RppsException("You can not create dublicate this card.");
			}
			if ((pinCode != null) && (pinCode.trim().length() > 0)) {
				PinCodeCredential credential = new PinCodeCredential();
				try {
					credential.setPinCode(pinCode.getBytes("UTF-8"));
				} catch (UnsupportedEncodingException exc) {
					credential.setPinCode(pinCode.getBytes());
				}
				card = card.getProvider().getICardByCUID(handler, cuid, credential);
			}
			ICardTemplate[] cardTemplates = card.getProvider().getCardCreationTemplates(handler);
			ICardTemplate cardTemplate = null;
			for (int i = 0; i < cardTemplates.length; i++) {

				if (!cardTemplates[i].getCardType().isInstance(card))
					continue;

				cardTemplate = cardTemplates[i];

				if (cardTemplate.getTemplateElementByID("cardName") != null)
					((ITemplateValue) cardTemplate.getTemplateElementByID("cardName")).setValue(cardName);
				if ((cardTemplate.getTemplateElementByID("cardPicture") != null) && (card.getImage() != null)) {
					((ITemplateValue) cardTemplate.getTemplateElementByID("cardPicture")).setValue(Base64.encode(card.getImage()));
				}
				if (cardTemplate.getTemplateElementByID("claimList") != null) {
					for (Iterator iter = ((ITemplateContainer) cardTemplate.getTemplateElementByID("claimList")).getTemplateElements(); iter.hasNext();) {
						ITemplateElement element = (ITemplateElement) iter.next();
						IClaim claim = card.getClaim(element.getID());
						if (claim != null) {
							if (((ISimpleClaim) claim).getValues().size() > 0)
								((ITemplateValue) element).setValue((String) ((ISimpleClaim) claim).getValues().get(0));
						}
					}
				}
				break;
			}
			if (cardTemplate != null)
				return createICard(handler, cardTemplate);
		} catch (CardException e) {
			// TODO Auto-generated catch block
			log.error(e);
			throw new RppsException(e);
		}

		return null;
	}

	/**
	 * Creates new card
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cardname name of card
	 * @param surname user's last name
	 * @param givenname user's first name
	 * @param emailaddress user's email
	 * @return cuid of new card
	 * @throws RppsException
	 */
	public String createICardFromHBX(String userId, String password, String cardname) throws RppsException {
		String ret = null;
		try {
			ICardTemplate template = getPersonalICardCreationTemplate(RppsHelper.getCallbackHandler(userId, password));
			template = RppsHelper.fillPersonalCardTemplate(template, cardname, "", "", "", "", "", "", "", "", "", "", "", "", "", "");
			ret = createICard(RppsHelper.getCallbackHandler(userId, password), template).getCUID().toString();
		} finally {
		}
		return ret;
	}

	/**
	 * Gets a card by CUID.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cuid the card CUID.
	 * @return the card transfer object.
	 */
	public ICardTO getICardByCUID(String userId, String password, String cuid) throws RppsException {
		try {
			ICard card = getICardByCUID(RppsHelper.getCallbackHandler(userId, password), new CUID(cuid));
			return convertICard_To_ICardTO(RppsHelper.getCallbackHandler(userId, password), card, false);
		} catch (Exception exc) {
			log.error(exc);
			throw new RppsException(exc);
		} finally {
		}
	}

	protected ICard getICardByCUID(CallbackHandler handler, CUID cuid) throws RppsException {
		try {
			return ICardRegistry.getInstance().getICardByCUID(handler, cuid);
		} catch (CardException e) {
			// TODO Auto-generated catch block
			log.error(e);
			throw new RppsException(e);
		}
	}

	/**
	 * Gets a card by CUID and Credential.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cuid the card CUID.
	 * @param tokenCredential the tokenCredential transfer object
	 * @return the card transfer object.
	 */
	public ICardTO getICardByCUIDAndTokenServiceCredential(String userId, String password, String cuid, UITokenServiceCredentialTO tokenCredential) throws RppsException {
		try {
			ICard card = getICardByCUID(RppsHelper.getCallbackHandler(userId, password), new CUID(cuid));

			ICredential credential = ConvertHelper.convertUITokenServiceCredentialTO_to_ICredentials(card, tokenCredential);

			if (credential != null)
				card = card.getProvider().getICardByCUID(RppsHelper.getCallbackHandler(userId, password), card.getCUID(), credential);

			return convertICard_To_ICardTO(RppsHelper.getCallbackHandler(userId, password), card, true);
		} catch (Exception exc) {
			log.error(exc);
			throw new RppsException(exc);
		} finally {
		}

	}

	/**
	 * Gets the card list of the specified user.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @return the card list
	 */
	public ICardTO[] getICardsByUser(String userId, String password) throws RppsException {
		try {
			List cards = null;
			cards = getICardsByUser(RppsHelper.getCallbackHandler(userId, password));
			ICardTO[] icards = new ICardTO[0];
			if (cards != null) {
				icards = new ICardTO[cards.size()];
				for (int i = 0; i < cards.size(); i++) {
					icards[i] = convertICard_To_ICardTO(RppsHelper.getCallbackHandler(userId, password), (ICard) cards.get(i), false);
				}
			}
			return icards;
		} catch (Exception exc) {
			log.error(exc);
			throw new RppsException(exc);
		} finally {
		}
	}

	protected List getICardsByUser(CallbackHandler handler) throws RppsException {
		List cards = new ArrayList();
		// gets all registered providers
		for (Iterator pit = ICardRegistry.getInstance().getICardProviders(); pit.hasNext();) {
			ICardProvider provider = (ICardProvider) pit.next();
			try {
				for (Iterator icards = provider.getICards(handler, null); icards.hasNext();) {
					cards.add(icards.next());
				}
			} catch (CardException e) {
				// TODO Auto-generated catch block
				log.error(e);
				throw new RppsException(e);
			}
		}
		return cards;
	}

	/**
	 * Gets the card and category lists of the specified user.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @return the card list
	 */
	public ICardsAndCategoryTO getICardsAndCategoriesByUser(String userId, String password) throws RppsException {
		try {
			ICardsAndCategoryTO cardsAndCategoryTO = new ICardsAndCategoryTO();
			try {
				cardsAndCategoryTO.setICardList(getICardsByUser(userId, password));
			} catch (Exception exc) {
				log.error(exc);
			}
			try {
				cardsAndCategoryTO.setCategoryList(getCategory(userId, password));
			} catch (Exception exc) {
				log.error(exc);
			}
			return cardsAndCategoryTO;

		} finally {
		}
	}

	/**
	 * Update card.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param icardTO the card transfer object.
	 * @return the card transfer object.
	 */
	public ICardTO updateICard(String userId, String password, ICardTO icardTO) throws RppsException {

		ICard card = null;
		try {
			CallbackHandler handler = RppsHelper.getCallbackHandler(userId, password);
			card = getICardByCUID(handler, new CUID(icardTO.getCuid()));

			if (((icardTO.getTokenServiceCredentials() != null) && (icardTO.getTokenServiceCredentials().length != 0))) {
				ICredential credential = ConvertHelper.convertUITokenServiceCredentialTO_to_ICredentials(card, icardTO.getTokenServiceCredentials()[0]);
				if (credential != null)
					card = card.getProvider().getICardByCUID(handler, card.getCUID(), credential);
			}

			card.beginUpdates();
			if (!RppsHelper.equalsObject(card.getName(), icardTO.getName()))
				card.setName(icardTO.getName());
			if (!RppsHelper.equalsObject(card.getIssuerName(), icardTO.getIssuerName()))
				card.setIssuerName(icardTO.getIssuerName());

			if ((!RppsHelper.equalsObject(card.getImageType(), icardTO.getImageType())) || (!RppsHelper.equalsObject(card.getImage(), icardTO.getImage())))
				card.setImage(icardTO.getImage(), icardTO.getImageType());

			if (card instanceof IPersonalInformationCard) {
				if (!RppsHelper.equalsObject(card.getTimeExpires(), icardTO.getExpiredTime()))
					card.setTimeExpires(icardTO.getExpiredTime());

				if (icardTO.getProperties() != null) {
					int size = icardTO.getProperties().length;
					for (int index = 0; index < size; index++) {
						PropertyTO propTO = icardTO.getProperties()[index];
						IClaim claim = card.getClaim(propTO.getClaimType());
						if (claim == null)
							claim = card.createClaim(propTO.getClaimType());
						if (propTO.getComplexValue() == null)
							setSimpleValues(claim, propTO);
						else
							setComplexValues(claim, propTO);
					}
				}
			}

			card.applyUpdates();

			card = getICardByCUID(handler, new CUID(icardTO.getCuid()));
			return convertICard_To_ICardTO(handler, card, false);

		} catch (Exception exc) {
			log.error(exc);
			try {
				if (card != null && card.isEditMode())
					card.cancelUpdates();
			} catch (Throwable tr) {
			}
			throw new RppsException(exc);
		} finally {
		}
	}

	/**
	 * 
	 * @param claim
	 * @param value
	 * @throws Exception
	 */
	private void setSimpleValues(IClaim claim, PropertyTO value) throws Exception {
		if (claim != null && claim instanceof ISimpleClaim) {
			if (((ISimpleClaimType) claim.getType()).getDataType().getName().equals(value.getValueType())) {
				List newValues = new ArrayList();
				for (int index = 0; index < value.getValue().length; index++) {
					newValues.add(value.getValue()[index]);
				}
				((ISimpleClaim) claim).setValues(newValues);
			}
		}
	}

	/**
	 * 
	 * @param claim
	 * @param value
	 * @throws RppsException
	 */
	private void setComplexValues(IClaim claim, PropertyTO value) throws Exception {
		if (claim != null && claim instanceof IComplexClaim) {
			int size = value.getComplexValue().length;
			for (int index = 0; index < size; index++) {
				PropertyTO propTO = value.getComplexValue()[index];
				IClaim claimValue = ((IComplexClaim) claim).getClaim(propTO.getClaimType());
				if (claimValue == null)
					claimValue = ((IComplexClaim) claim).createClaim(propTO.getClaimType());
				if (propTO.getValueType() != null)
					setSimpleValues(claimValue, propTO);
				else
					setComplexValues(claimValue, propTO);
			}
		}
	}

	/**
	 * Imports card from a file.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param crd the card as a byte array.
	 */
	public ICardTO[] importICards(String userId, String password, byte[] crd) throws RppsException {
		try {
			List cards = null;
			cards = importICards(RppsHelper.getCallbackHandler(userId, password), new ByteArrayInputStream(crd));
			ICardTO[] icards = new ICardTO[0];
			if (cards != null) {
				icards = new ICardTO[cards.size()];
				for (int i = 0; i < cards.size(); i++) {
					icards[i] = convertICard_To_ICardTO(RppsHelper.getCallbackHandler(userId, password), (ICard) cards.get(i), false);
				}
			}
			return icards;
		} catch (Exception exc) {
			log.error(exc);
			throw new RppsException(exc);
		}
	}

	protected List importICards(CallbackHandler handler, InputStream in) throws RppsException {
		try {
			List result = new ArrayList();
			for (Iterator iter = ICardRegistry.getInstance().importICards(handler, in, null, null); iter.hasNext();) {
				result.add(iter.next());
			}
			return result;
		} catch (CardException e) {
			// TODO Auto-generated catch block
			log.error(e);
			throw new RppsException(e);
		}
	}

	/**
	 * Deletes specified card.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param cuid the i-card CUID
	 */
	public void deleteICard(String userId, String password, String cuid) throws RppsException {
		try {
			CallbackHandler handler = RppsHelper.getCallbackHandler(userId, password);
			URL[] urls = new URL[0];
			ICardRegistry.getInstance().getUserProfileService().setUrls(handler, new CUID(cuid), urls);
			ICardRegistry.getInstance().getUserProfileService().clearCardCredential(handler, new CUID(cuid));
			ICardRegistry.getInstance().getUserProfileService().clearCardHistory(handler, new CUID(cuid));
			ICardRegistry.getInstance().deleteICard(handler, new CUID(cuid));
		} catch (CardException e) {
			// TODO Auto-generated catch block
			log.error(e);
			throw new RppsException(e);
		} catch (UserProfileException e) {
			// TODO Auto-generated catch block
			log.error(e);
			throw new RppsException(e);
		}
	}

	/**
	 * Deletes all user's card.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * 
	 */
	public void deleteAllICard(String userId, String password) throws RppsException {
		List icards = getICardsByUser(RppsHelper.getCallbackHandler(userId, password));

		for (Iterator iter = icards.iterator(); iter.hasNext();) {
			deleteICard(userId, password, ((ICard) iter.next()).getCUID().toString());
		}
	}

	/* Methods of HBX */

	/**
	 * This method takes as input a policy---for example, one that has been retrieved from the Relying Party Agent (called here the 'Requester')---as well as a description of the Requester. The UserInterfaceResponse object should contain all information necessary to display both (a) a user interface which shares the same card metaphor, sequences of experiences and decision points (the ceremony) of CardSpace and (b) other interface paradigms to be developed in the future
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param url the URL to the html page with <object> tag
	 * @param target
	 * @param sslCert the SSL certificate
	 * @param policyType the RP Security Policy type
	 * @param policy the RP Agent's Relying Party Security Policy
	 * @return the UIResponseTO transfer object.
	 * @throws RemoteException
	 */

	public UIResponseTO getUserInterface(String userId, String password, String url, String target, String sslCert, String policytype, String policy) throws RppsException {

		UserInterfaceResponse uir = new UserInterfaceResponse();
		try {
			IICardSelector selector = ICardSelectorService.getInstance().getICardSelector(RppsHelper.getCallbackHandler(userId, password), policy);
			uir.setUCL(selector.getUserChoice());
			uir.setRequestorInformation(RppsHelper.parseSSLCert(sslCert));
		} catch (Exception e) {
			// TODO Auto-generated catch block
			log.error(e);
			throw new RppsException(e);
		}
		UIResponseTO uirto = new UIResponseTO();
		try {
			URL url_ = ConvertHelper.convertString_to_URL(url);
			if (uir != null) {
				uirto.setStatus("ok");
				UIResponseRPTO rp = new UIResponseRPTO();
				rp.setName(uir.getName());
				rp.setAddr1(uir.getAddr1());
				rp.setAddr2(uir.getAddr2());
				rp.setPhone(uir.getPhone());
				rp.setVerifier(uir.getVerifier());
				uirto.setRP(rp);
				uirto.setAnd(getAnds(RppsHelper.getCallbackHandler(userId, password), uir.getUCL(), url_, (ICardSpacePolicy) ICardSelectorService.getInstance().parsePolicy(policy)));
			}
		} catch (Exception e) {
			// TODO Auto-generated catch block
			log.error(e);
			throw new RppsException(e);
		}
		return uirto;
	}

	/* Methods of getting token */

	protected ICredential getICredentialByPPID(CallbackHandler handler, X509Certificate x509cert, String ppid) throws RppsException {

		try {
			byte[] bppid = Base64.decode(ppid);
			PPIDCardSpacePolicy policy = new PPIDCardSpacePolicy(x509cert, bppid);
			IICardSelector selector = ICardSelectorService.getInstance().getICardSelector(handler, policy);

			UserChoiceTree_ANDofORs uct = (UserChoiceTree_ANDofORs) selector.getUserChoice();
			SelectionANDofORs subselection = uct.getDefaultSelection();

			if (subselection.isEmpty()) {
				throw new RppsException("Cannot find the Personal card used to authenticate for this managed card.");
			}
			String cuid = ((SelectionANDofORsElm) subselection.getElements().get(0)).getUUID();
			ICard card = getICardByCUID(handler, new CUID(cuid));
			if (card == null)
				return null;
			ICredentialDescriptor[] der = card.getRequiredCredentials();
			if (der == null)
				return null;
			if (der.length == 0)
				return null;
			return der[0].getCredential();
		} catch (Exception e) {
			log.error(e);
			throw new RppsException(e);
		}
	}

	/**
	 * Web service operation. Makes xmlToken usage claim list.
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param policy the RP Agent's Relying Party Security Policy
	 * @param policytype the RP Security Policy type
	 * @param sslCert the SSL certificate
	 * @param user the username to login to the STS
	 * @param pass the password to login to the STS
	 * @param sels the user-selected subset of the UIDs (handles) contained in the UserInterfaceResponse object returned from an earlier invocation of 'userInterfaceRequest'
	 * @return the security token transfer object
	 * @throws RppsException
	 */
	public TokenResponseTO getTokenByClaims(String sslCert, String[] claimName, String[] claimValue) throws RppsException {
		try {
			Hashtable claimHashtable = new Hashtable();
			for (int i = 0; i < claimName.length; i++) {
				try {
					claimHashtable.put(claimName[i], claimValue[i]);
				} catch (RuntimeException e) {
					e.printStackTrace();
				}
			}

			X509Certificate cert = RppsHelper.parseSSLCert(sslCert);
			ISTSResponse resp = UserProfileTokenHelper.getToken(RppsHelper.parseSSLCert(sslCert), claimHashtable);

			IdentityToken itoken = new IdentityToken(resp, Base64.encode(cert.getEncoded()));

			String tokenAsString = (String) itoken.getAs(String.class);

			return ConvertHelper.createTokenResponse(tokenAsString);
		} catch (Exception e) {
			log.error(e);
			throw new RppsException(e);
		}
	}

	/**
	 * Web service operation. Given the Relying Party Agent (aka 'Requester')'s policy, identity information about the requester, and the set of one or more selections that the user has just made in the ISS Web UI (see userInterfaceRequest), AND presuming that the protocol involved in interacting with the RPA requires a security token, request the token that corresponds to the user's selection(s).
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param policy the RP Agent's Relying Party Security Policy
	 * @param policytype the RP Security Policy type
	 * @param sslCert the SSL certificate
	 * @param cuids the user-selected subset of the UIDs (handles) contained in the UserInterfaceResponse object returned from an earlier invocation of 'userInterfaceRequest'
	 * @param typeofCredential the type of selected credential
	 * @param credentialKey the names of fields of selected credential
	 * @param credentialValue the value of fields of selected credential
	 * @return the security token transfer object
	 * @throws RemoteException
	 */
	public TokenResponseTO getTokenObject(String userId, String password, String policy, String policytype, String sslCert, String[] cuids, String typeofCredential, String[] credentialKey, String[] credentialValue) throws RppsException {
		TokenResponseTO token = null;
		try {
			AllowSelfSignedSSL();
			Hashtable table = new Hashtable();
			boolean saveCard = false;
			boolean saveCredential = false;
			String urlStr = null;
			for (int i = 0; i < credentialKey.length; i++) {
				if ((credentialKey[i] != null) && (credentialValue[i] != null)) {
					table.put(credentialKey[i], credentialValue[i]);
					log.debug("Put in table key='" + credentialKey[i] + "' with value='" + credentialValue[i] + "'");
				}
			}
			if ((table.get("saveCard") != null) && ("true".equalsIgnoreCase((String) table.get("saveCard"))))
				saveCard = true;
			if ((table.get("saveCredential") != null) && ("true".equalsIgnoreCase((String) table.get("saveCredential"))))
				saveCredential = true;
			if ((table.get("url") != null))
				urlStr = (String) table.get("url");
			if (urlStr != null && urlStr.indexOf("?") > -1)
				urlStr = urlStr.substring(0, urlStr.indexOf("?"));

			log.debug("UserId parameter is " + userId);
			log.debug("CUID parameter is " + cuids[0]);
			log.debug("SaveCard parameter is " + Boolean.toString(saveCard));
			log.debug("SaveCredential parameter is " + Boolean.toString(saveCredential));
			log.debug("Url parameter is " + urlStr);

			URL url = null;
			if (urlStr != null && urlStr.length() > 0)
				url = ConvertHelper.convertString_to_URL(urlStr);

			ICredential credential;

			ICard card = getICardByCUID(RppsHelper.getCallbackHandler(userId, password), new CUID(cuids[0]));
			if (card != null) {
				
				CardInformation cardCredentialInfo=null;
				CardInformation getCardInformationByUrl = getCardInformationByUrl(RppsHelper.getCallbackHandler(userId, password), url);
				if (getCardInformationByUrl!=null)
				{
					if (!getCardInformationByUrl.getCuid().equals(card.getCUID()))
					{	
						if (saveCard)
							deleteUrlCardRelation(RppsHelper.getCallbackHandler(userId, password), card.getCUID(), url);
						cardCredentialInfo = getCardInformationByCuid(RppsHelper.getCallbackHandler(userId, password), card.getCUID());
					}
					else				
						cardCredentialInfo = getCardInformationByUrl;
				}
				else
					cardCredentialInfo = getCardInformationByCuid(RppsHelper.getCallbackHandler(userId, password), card.getCUID());
				
				if (!saveCard)
					deleteUrlCardRelation(RppsHelper.getCallbackHandler(userId, password), card.getCUID(), url);

				if ((cardCredentialInfo != null) && (cardCredentialInfo.getCardCredential() != null))
					credential = ConvertHelper.convertCardCredential_to_ICredential(card, cardCredentialInfo.getCardCredential());
				else
					credential = ConvertHelper.convertHashtable_to_Credential(card, typeofCredential, table);
				
				token = getTokenObject(RppsHelper.getCallbackHandler(userId, password), policy, policytype, sslCert, cuids, credential, cardCredentialInfo, url, saveCard, saveCredential);
			} else
				throw new RppsException("Didn't find card with cuid=" + cuids[0]);
		} catch (Exception e) {
			// TODO Auto-generated catch block
			log.error(e);
			throw new RppsException(e);
		} finally {

		}

		return token;
	}

	/**
	 * Web service operation. Given the Relying Party Agent (aka 'Requester')'s policy, identity information about the requester, and the set of one or more selections that the user has just made in the ISS Web UI (see userInterfaceRequest), AND presuming that the protocol involved in interacting with the RPA requires a security token, request the token that corresponds to the user's selection(s).
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param policy the RP Agent's Relying Party Security Policy
	 * @param policytype the RP Security Policy type
	 * @param sslCert the SSL certificate
	 * @param cuids the user-selected subset of the UIDs (handles) contained in the UserInterfaceResponse object returned from an earlier invocation of 'userInterfaceRequest'
	 * @param typeofCredential the type of selected credential
	 * @param credentialKey the names of fields of selected credential
	 * @param credentialValue the value of fields of selected credential
	 * @return the security token transfer object
	 * @throws RemoteException
	 */
	protected TokenResponseTO getTokenObject(CallbackHandler handler, String policy, String policytype, String sslCert, String[] cuids, ICredential credential, CardInformation cardCredentialInfo, URL url, boolean saveCard, boolean saveCredential) throws RppsException {
		TokenResponseTO token = null;
		try {

			CUID cuid = new CUID(cuids[0]);
						
			if (!saveCredential) {
				if ((cardCredentialInfo != null) && (cardCredentialInfo.getCardCredential() != null))
					clearCardCredential(handler, cuid);
			}

			token = getTokenObject(handler, policy, policytype, sslCert, cuids, credential, url);

			if ((token != null) && (token.getTkn() != null) && (token.getTkn().length() > 0)) {
				if (url != null)
					addCardHistory(handler, cuid, url);
				if ((saveCard) && (url != null))
					addUrlCardRelation(handler, cuid, url, cardCredentialInfo);

				if (saveCredential) {
					CardCredential cardCredential = ConvertHelper.convertICredential_to_CardCredential(credential);
					setCardCredential(handler, cuid, cardCredential);
				}
			}
		} catch (Exception e) {
			log.error(e);
			throw new RppsException(e);
		}
		return token;
	}

	/**
	 * Given the Relying Party Agent (aka 'Requester')'s policy, identity information about the requester, and the set of one or more selections that the user has just made in the ISS Web UI (see userInterfaceRequest), AND presuming that the protocol involved in interacting with the RPA requires a security token, request the token that corresponds to the user's selection(s).
	 * 
	 * @param handler
	 * @param policy the RP Agent's Relying Party Security Policy
	 * @param policytype the RP Security Policy type
	 * @param sslCert the SSL certificate
	 * @param cuids the user-selected subset of the UIDs (handles) contained in the UserInterfaceResponse object returned from an earlier invocation of 'userInterfaceRequest'
	 * @param credential
	 * @param cardCredentialInfo
	 * @param url
	 * @return the security token transfer object
	 * @throws RppsException
	 */
	protected TokenResponseTO getTokenObject(CallbackHandler handler, String policy, String policytype, String sslCert, String[] cuids, ICredential credential, URL url) throws RppsException {
		String token = null;

		// get the selections from user's selection
		SelectionANDofORs selection = new SelectionANDofORs();
		selection.idemix_credential_db_uri = "credentials.db";
		selection.cardspace_credential_db_uri = "infocards.properties";
		selection.publickey_db_uri = "pk.db";
		selection.sslCert = RppsHelper.parseSSLCert(sslCert);
		selection.setCredential(credential);
		for (int i = 0; i < cuids.length; i++) {
			String uid = cuids[i];
			// n = (Node)i.next();
			int ui = 0;
			try {
				if (i < (cuids.length - 1)) {
					ui = Integer.parseInt(cuids[++i]);
				}
			} catch (Exception ew) {
				ew.printStackTrace();
			}
			SelectionANDofORsElm se = new SelectionANDofORsElm(uid, ui, 0);
			selection.add(se);
		}
		try {

			IICardSelector selector = ICardSelectorService.getInstance().getICardSelector(handler, policy);
			IIdentityToken iToken = selector.getIdentityToken(selection);
			token = (String) iToken.getAs(String.class);

		} catch (Exception e) {
			log.error(e);
			throw new RppsException(e);
		}
		return ConvertHelper.createTokenResponse(token);
	}

	/**
	 * Web service operation. Makes security token usage saved users information
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param policy the RP Agent's Relying Party Security Policy
	 * @param policytype the RP Security Policy type
	 * @param sslCert the SSL certificate
	 * @param url the url of site which needs token
	 * @return the security token transfer object
	 * @throws RemoteException
	 */
	public TokenResponseTO getTokenByUrl(String userId, String password, String policy, String policytype, String sslCert, String url) throws RppsException {
		TokenResponseTO token = null;
		try {
			URL url_ = ConvertHelper.convertString_to_URL(url);
			CardInformation cardCredentialInfo = getCardInformationByUrl(RppsHelper.getCallbackHandler(userId, password), url_);
			if ((cardCredentialInfo != null)) {
				String[] cuids = new String[] { cardCredentialInfo.getCuid().toString() };

				ICard card = getICardByCUID(RppsHelper.getCallbackHandler(userId, password), cardCredentialInfo.getCuid());
				if (card == null)
					new RppsException("Does not find card with cuid " + cardCredentialInfo.getCuid().toString());

				ICredential credential = null;
				credential = ConvertHelper.convertCardCredential_to_ICredential(card, cardCredentialInfo.getCardCredential());

				token = getTokenObject(RppsHelper.getCallbackHandler(userId, password), policy, policytype, sslCert, cuids, credential, url_);

				if ((token != null) && (token.getTkn() != null) && (token.getTkn().length() > 0)) {
					if (url_ != null)
						addCardHistory(RppsHelper.getCallbackHandler(userId, password), card.getCUID(), url_);
				}
			}
		} catch (Exception e) {
			log.error(e);
		}
		return token;
	}
	
	/**
	 * Web service operation. Makes security token usage saved users information
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param policy the RP Agent's Relying Party Security Policy
	 * @param policytype the RP Security Policy type
	 * @param sslCert the SSL certificate
	 * @param url the url of site which needs token
	 * @return the security token transfer object
	 * @throws RemoteException
	 */
	public ExtraTokenResponseTO getTokenExtraByUrl(String userId, String password, String policy, String policytype, String sslCert, String url) throws RppsException {
		ExtraTokenResponseTO token = null;
		try {
			URL url_ = ConvertHelper.convertString_to_URL(url);
			CardInformation cardCredentialInfo = getCardInformationByUrl(RppsHelper.getCallbackHandler(userId, password), url_);
			if ((cardCredentialInfo != null)) {
				String[] cuids = new String[] { cardCredentialInfo.getCuid().toString() };

				ICard card = getICardByCUID(RppsHelper.getCallbackHandler(userId, password), cardCredentialInfo.getCuid());
				if (card == null)
					new RppsException("Does not find card with cuid " + cardCredentialInfo.getCuid().toString());

				ICredential credential = null;
				credential = ConvertHelper.convertCardCredential_to_ICredential(card, cardCredentialInfo.getCardCredential());

				TokenResponseTO tokenTO = getTokenObject(RppsHelper.getCallbackHandler(userId, password), policy, policytype, sslCert, cuids, credential, url_);

				if ((tokenTO != null) && (tokenTO.getTkn() != null) && (tokenTO.getTkn().length() > 0)) {
					if (url_ != null)
						addCardHistory(RppsHelper.getCallbackHandler(userId, password), card.getCUID(), url_);
				}
				token = new ExtraTokenResponseTO();
				token.setTkn(tokenTO.getTkn());
				token.setCardName(card.getName());
				token.setCardDescription(card.getDescription());				
				token.setCardImage(card.getImage());
				token.setCardImageType(card.getImageType());				
			}
		} catch (Exception e) {
			log.error(e);
		}
		return token;
	}

	/**
	 * Return templates for creating cards
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @return a template list
	 */
	protected List getICardCreationTemplate(CallbackHandler handler) throws RppsException {
		List temps = new Vector();
		for (Iterator it = ICardRegistry.getInstance().getICardProviders(); it.hasNext();) {
			ICardProvider provider = (ICardProvider) it.next();
			try {
				// check if current provider supports creating cards
				if (provider.getCardCreationTemplates(handler) != null) {
					ICardTemplate[] templates = provider.getCardCreationTemplates(handler);
					for (int i = 0; i < templates.length; i++)
						temps.add(templates[i]);
				}
			} catch (Exception exc) {
				exc.printStackTrace();
				throw new RppsException(exc);
			}
		}
		return temps;
	}

	/**
	 * Return templates for creating cards
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @return a template list
	 */
	public ICardTemplateTO[] getICardCreationTemplate(String userId, String password) {
		List templates = null;
		try {
			templates = getICardCreationTemplate(RppsHelper.getCallbackHandler(userId, password));
		} finally {
		}
		ICardTemplateTO[] telmplateList = new ICardTemplateTO[0];
		if (templates != null) {
			telmplateList = new ICardTemplateTO[templates.size()];
			for (int index = 0; index < templates.size(); index++) {
				telmplateList[index] = ConvertHelper.convertICardTemplate_to_ICardTemplateTO((ICardTemplate) templates.get(index));
			}
		}
		return telmplateList;
	}

	public ICardTemplate[] getICardCreationTemplateByProvider(CallbackHandler handler, String extID) throws RppsException {
		ICardProvider provider = ICardRegistry.getInstance().getICardProvider(extID);

		return provider.getCardCreationTemplates(handler);

	}

	/**
	 * Get a icard template for creating personal card
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @return a template
	 */
	public ICardTemplateTO getPersonalICardCreationTemplate(String userId, String password) {
		ICardTemplate template = null;
		try {
			template = getPersonalICardCreationTemplate(RppsHelper.getCallbackHandler(userId, password));
		} finally {
		}
		if (template != null) {
			return ConvertHelper.convertICardTemplate_to_ICardTemplateTO(template);
		}
		return null;
	}

	/**
	 * Get a icard template for creating personal card
	 * 
	 * @param handler the callback handler to be used to request user's identity
	 * @return a template
	 */
	protected ICardTemplate getPersonalICardCreationTemplate(CallbackHandler handler) throws RppsException {
		for (Iterator it = ICardRegistry.getInstance().getICardProviders(); it.hasNext();) {
			ICardProvider provider = (ICardProvider) it.next();
			try {
				// check if current provider supports creating cards
				if (provider.getCardCreationTemplates(handler) != null) {
					ICardTemplate[] templates = provider.getCardCreationTemplates(handler);
					for (int i = 0; i < templates.length; i++) {
						if (templates[i].getCardType().isAssignableFrom(IPersonalInformationCard.class)) {
							return templates[i];
						}
					}
				}
			} catch (Exception exc) {
				log.error(exc);
				throw new RppsException(exc);
			}
		}
		return null;
	}

	public ICardTO setPinCode(String userId, String password, String cuid, String pinCode) throws RppsException {
		ICard card = null;
		try {
			card = setPinCode(RppsHelper.getCallbackHandler(userId, password), new CUID(cuid), pinCode.getBytes("UTF-8"));
			return convertICard_To_ICardTO(RppsHelper.getCallbackHandler(userId, password), card, false);
		} catch (Exception exc) {
			log.error(exc);
			throw new RppsException(exc);
		}
	}

	protected ICard setPinCode(CallbackHandler handler, CUID cuid, byte[] pinCode) throws RppsException {
		try {
			ICard card = ICardRegistry.getInstance().getICardByCUID(handler, cuid);
			if (!(card instanceof IPersonalInformationCard))
				new RppsException("This card is not personal");

			PinCodeCredential pinCodeCredential = new PinCodeCredential();
			try {
				((IPersonalInformationCard) card).beginUpdates();
				pinCodeCredential.setPinCode(pinCode);
				((IPersonalInformationCard) card).setPinCode(pinCodeCredential);
				((IPersonalInformationCard) card).applyUpdates();
			} catch (CardException exc) {
				if (((IPersonalInformationCard) card).isEditMode())
					((IPersonalInformationCard) card).cancelUpdates();
				throw new RppsException(exc);
			}
			pinCodeCredential = new PinCodeCredential();
			pinCodeCredential.setPinCode(pinCode);

			return card.getProvider().getICardByCUID(handler, cuid, pinCodeCredential);
		} catch (Exception exc) {
			log.error(exc);
			throw new RppsException(exc);
		}
	}

	public ICardTO resetPinCode(String userId, String password, String cuid, String oldPinCode) throws RppsException {
		ICard card = null;
		try {
			card = resetPinCode(RppsHelper.getCallbackHandler(userId, password), new CUID(cuid), oldPinCode.getBytes("UTF-8"));
			return convertICard_To_ICardTO(RppsHelper.getCallbackHandler(userId, password), card, false);
		} catch (Exception exc) {
			log.error(exc);
			throw new RppsException(exc);
		}
	}

	protected ICard resetPinCode(CallbackHandler handler, CUID cuid, byte[] oldPinCode) throws RppsException {
		try {
			ICard card = ICardRegistry.getInstance().getICardByCUID(handler, cuid);
			if (!(card instanceof IPersonalInformationCard))
				new RppsException("This card is not personal");

			PinCodeCredential pinCodeCredential = new PinCodeCredential();
			pinCodeCredential.setPinCode(oldPinCode);
			card = (IPersonalInformationCard) card.getProvider().getICardByCUID(handler, cuid, pinCodeCredential);

			try {
				PinCodeCredential pinCodeCredentialNew = new PinCodeCredential();
				pinCodeCredentialNew.setPinCode(null);

				((IPersonalInformationCard) card).beginUpdates();
				((IPersonalInformationCard) card).setPinCode(pinCodeCredentialNew);
				((IPersonalInformationCard) card).applyUpdates();
			} catch (CardException exc) {
				if (((IPersonalInformationCard) card).isEditMode())
					((IPersonalInformationCard) card).cancelUpdates();
				throw new RppsException(exc);
			}
			return ICardRegistry.getInstance().getICardByCUID(handler, cuid);
		} catch (Exception exc) {
			log.error(exc);
			throw new RppsException(exc);
		}
	}

	public ICardTO editPinCode(String userId, String password, String cuid, String oldPinCode, String newPinCode) throws RppsException {
		ICard card = null;
		try {
			resetPinCode(RppsHelper.getCallbackHandler(userId, password), new CUID(cuid), oldPinCode.getBytes("UTF-8"));
			card = setPinCode(RppsHelper.getCallbackHandler(userId, password), new CUID(cuid), newPinCode.getBytes("UTF-8"));
			return convertICard_To_ICardTO(RppsHelper.getCallbackHandler(userId, password), card, false);
		} catch (Exception exc) {
			log.error(exc);
			throw new RppsException(exc);
		}
	}

	/* Methods of card history */

	public void addCardHistory(CallbackHandler handler, CUID cuid, URL site) {
		try {
			ICardRegistry.getInstance().getUserProfileService().addCardHistory(handler, new CardUsage(cuid, site));
		} catch (Exception exc) {
			log.error(exc);
			throw new RppsException(exc);
		}
	}

	public List getCardHistory(CallbackHandler handler, CUID cuid) {
		try {
			return ICardRegistry.getInstance().getUserProfileService().getCardHistory(handler, cuid);
		} catch (Exception exc) {
			log.error(exc);
			throw new RppsException(exc);
		}
	}

	protected void clearCardHistory(CallbackHandler handler, CUID cuid) {
		try {
			ICardRegistry.getInstance().getUserProfileService().clearCardHistory(handler, cuid);
		} catch (Exception exc) {
			log.error(exc);
			throw new RppsException(exc);
		}
	}

	public CardInformationTO clearCardHistory(String userId, String password, String cuid) {
		try {
			ICardRegistry.getInstance().getUserProfileService().clearCardHistory(RppsHelper.getCallbackHandler(userId, password), new CUID(cuid));
			return getCardInformation(userId, password, cuid);
		} catch (Exception exc) {
			log.error(exc);
			throw new RppsException(exc);
		}
	}

	/* Methods of card category */

	/**
	 * 
	 * @param userId
	 * @param password
	 * @return
	 * @throws RppsException
	 */
	public CategoryTO[] getCategory(String userId, String password) throws RppsException {
		try {
			List category = getCategories(RppsHelper.getCallbackHandler(userId, password));
			CategoryTO[] categorysTO = new CategoryTO[0];
			if (category != null) {
				categorysTO = new CategoryTO[category.size()];
				for (int i = 0; i < category.size(); i++) {
					categorysTO[i] = ConvertHelper.convertCategory_to_CategoryTO((Category) category.get(i));
				}
			}
			return categorysTO;
		} catch (RppsException e) {
			log.error(e);
			throw e;
		}
	}

	protected List getCategories(CallbackHandler handler) throws RppsException {
		try {
			List list = ICardRegistry.getInstance().getUserProfileService().getCategories(handler);

			Collections.sort(list, new CategoryComparator());

			return list;
		} catch (UserProfileException exc) {
			log.error(exc);
			throw new RppsException(exc);
		}
	}

	public CategoryTO addCategory(String userId, String password, CategoryTO categoryTO) throws RppsException {
		try {
			Category category = ICardRegistry.getInstance().getUserProfileService().addCategory(RppsHelper.getCallbackHandler(userId, password), ConvertHelper.convertCategoryTO_to_Category(categoryTO));
			return ConvertHelper.convertCategory_to_CategoryTO(category);

		} catch (UserProfileException exc) {
			log.error(exc);
			throw new RppsException(exc);
		}
	}

	public void deleteCategory(String userId, String password, String categoryId) throws RppsException {
		try {
			ICardRegistry.getInstance().getUserProfileService().deleteCategory(RppsHelper.getCallbackHandler(userId, password), categoryId);
		} catch (Exception exc) {
			log.error(exc);
			throw new RppsException(exc);
		}
	}

	public CategoryTO modifyCategory(String userId, String password, CategoryTO categoryTO) throws RppsException {
		try {
			Category category = ICardRegistry.getInstance().getUserProfileService().modifyCategory(RppsHelper.getCallbackHandler(userId, password), ConvertHelper.convertCategoryTO_to_Category(categoryTO));
			return ConvertHelper.convertCategory_to_CategoryTO(category);
		} catch (UserProfileException exc) {
			log.error(exc);
			throw new RppsException(exc);
		}
	}

	/* Methods of user profile */

	public UserProfileTO getUserProfile(String userId, String password) throws RppsException {
		try {
			UserProfile userProfile = ICardRegistry.getInstance().getUserProfileService().getUserProfile(RppsHelper.getCallbackHandler(userId, password));
			return ConvertHelper.convertUserProfile_to_UserProfileTO(userProfile);
		} catch (UserProfileException exc) {
			log.error(exc);
			throw new RppsException(exc);
		}
	}

	/**
	 * Creates new user profile
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param firstName first name of user
	 * @param lastName last name of user
	 * @param email email of user
	 * @throws RppsException
	 */
	public void addUserProfileFromHBX(String userId, String password) throws RppsException {
		UserProfileTO userProfile = new UserProfileTO();
		userProfile.setLoginName(userId);
		userProfile.setPassword(password);

		addUserProfile(userId, password, userProfile);
	}

	/**
	 * Creates new user profile
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param userProfile user profile trasfer object
	 * @throws RppsException
	 */
	public void addUserProfile(String userId, String password, UserProfileTO userProfile) throws RppsException {
		try {
			ICardRegistry.getInstance().getUserProfileService().addUserProfile(RppsHelper.getCallbackHandler(userId, password), ConvertHelper.convertUserProfileTO_to_UserProfile(userProfile));
		} catch (UserProfileException exc) {
			log.error(exc);
			throw new RppsException(exc);
		}
	}

	/**
	 * Deletes user profile
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @throws RppsException
	 */
	public void deleteUserProfile(String userId, String password) throws RppsException {
		try {
			deleteUserProfile(RppsHelper.getCallbackHandler(userId, password));
		} catch (RppsException exc) {
			log.error(exc);
			throw exc;
		}
	}

	/**
	 * Deletes user profile
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @throws RppsException
	 */
	protected void deleteUserProfile(CallbackHandler handler) throws RppsException {
		try {
			ICardRegistry.getInstance().getUserProfileService().deleteUserProfile(handler);
		} catch (UserProfileException exc) {
			log.error(exc);
			throw new RppsException(exc);
		}
	}

	/**
	 * Checks availability of user profile by userIdentifier
	 * 
	 * @param userIdentifier
	 * @return true if user profile exists otherwise false
	 * @throws RppsException
	 */
	public boolean userIdentifierExists(String userIdentifier) throws RppsException {
		try {
			return ICardRegistry.getInstance().getUserProfileService().userIdentifierExists(userIdentifier);
		} catch (UserProfileException exc) {
			log.error(exc);
			throw new RppsException(exc);
		}
	}

	/**
	 * Update user profile
	 * 
	 * @param userId the id to identify the user.
	 * @param password the password to identify the user.
	 * @param userProfile user profile transfer object
	 * @return user profile transfer object
	 * @throws RppsException
	 */
	public UserProfileTO modifyUserProfile(String userId, String password, UserProfileTO userProfile) throws RppsException {
		try {
			ICardRegistry.getInstance().getUserProfileService().modifyUserProfile(RppsHelper.getCallbackHandler(userId, password), ConvertHelper.convertUserProfileTO_to_UserProfile(userProfile));
			return getUserProfile(userId, password);
		} catch (Exception exc) {
			log.error(exc);
			throw new RppsException(exc);
		}
	}

	/* Methods of card information */

	/**
	 * 
	 * @param userId
	 * @param password
	 * @param cuid
	 * @return
	 * @throws RppsException
	 */
	public CardInformationTO getCardInformation(String userId, String password, String cuid) throws RppsException {
		try {
			CardInformation cardInformation = getCardInformationByCuid(RppsHelper.getCallbackHandler(userId, password), new CUID(cuid));
			List cardHistory = getCardHistory(RppsHelper.getCallbackHandler(userId, password), new CUID(cuid));
			return ConvertHelper.createCardInformationTO(cardInformation, cardHistory);
		} catch (Exception exc) {
			log.error(exc);
			throw new RppsException(exc);
		}
	}

	protected CardInformation getCardInformationByUrl(CallbackHandler handler, URL url) throws RppsException {
		try {
			return ICardRegistry.getInstance().getUserProfileService().getCardInformation(handler, url);
		} catch (Exception e) {
			// TODO Auto-generated catch block
			log.error(e);
			throw new RppsException(e);
		}
	}

	protected CardInformation getCardInformationByCuid(CallbackHandler handler, CUID cuid) throws RppsException {
		try {
			return ICardRegistry.getInstance().getUserProfileService().getCardInformation(handler, cuid);
		} catch (Exception e) {
			// TODO Auto-generated catch block
			log.error(e);
			throw new RppsException(e);
		}
	}

	/* Methods of storage card credential */

	public CardInformationTO clearCardCredential(String userId, String password, String cuid) throws RppsException {
		try {
			clearCardCredential(RppsHelper.getCallbackHandler(userId, password), new CUID(cuid));
			return getCardInformation(userId, password, cuid);
		} catch (Exception exc) {
			log.error(exc);
			throw new RppsException(exc);
		}
	}

	protected void clearCardCredential(CallbackHandler handler, CUID cuid) throws RppsException {
		try {
			ICardRegistry.getInstance().getUserProfileService().clearCardCredential(handler, cuid);
		} catch (Exception e) {
			// TODO Auto-generated catch block
			log.error(e);
			throw new RppsException(e);
		}
	}

	protected void setCardCredential(CallbackHandler handler, CUID cuid, CardCredential cardCredential) throws RppsException {
		try {
			ICardRegistry.getInstance().getUserProfileService().setCardCredential(handler, cuid, cardCredential);
		} catch (Exception e) {
			// TODO Auto-generated catch block
			log.error(e);
			throw new RppsException(e);
		}
	}

	/* Methods of storage relation between card and url */

	public CardInformationTO addUrlCardRelation(String userId, String password, String cuid, String url) throws RppsException {
		try {
			if (url != null) {
				CardInformation cardInformation = getCardInformationByUrl(RppsHelper.getCallbackHandler(userId, password), ConvertHelper.convertString_to_URL(url));
				addUrlCardRelation(RppsHelper.getCallbackHandler(userId, password), new CUID(cuid), ConvertHelper.convertString_to_URL(url), cardInformation);
			}
			return getCardInformation(userId, password, cuid);
		} catch (Exception exc) {
			log.error(exc);
			throw new RppsException(exc);
		}
	}

	protected CardInformation addUrlCardRelation(CallbackHandler handler, CUID cuid, URL url, CardInformation cardInformation) throws RppsException {
		try {
			URL url_ = ConvertHelper.convertString_to_URL(url.toString());
			URL[] urls = cardInformation.getUrls();
			if (urls != null) {
				for (int i = 0; i < urls.length; i++) {
					if (urls[i].equals(url))
						return cardInformation;
				}
			}
			return ICardRegistry.getInstance().getUserProfileService().addCardUrlRelation(handler, cuid, url_);
		} catch (Exception e) {
			// TODO Auto-generated catch block
			log.error(e);
			throw new RppsException(e);
		}
	}

	public CardInformationTO deleteUrlCardRelation(String userId, String password, String cuid, String url) throws RppsException {
		try {
			if (url != null) {
				deleteUrlCardRelation(RppsHelper.getCallbackHandler(userId, password), new CUID(cuid), new URL(url));
			}
			return getCardInformation(userId, password, cuid);
		} catch (Exception exc) {
			log.error(exc);
			throw new RppsException(exc);
		}
	}

	protected CardInformation deleteUrlCardRelation(CallbackHandler handler, CUID cuid, URL url) throws RppsException {
		try {
			return ICardRegistry.getInstance().getUserProfileService().deleteCardUrlRelation(handler, cuid, url);
		} catch (Exception e) {
			// TODO Auto-generated catch block
			log.error(e);
			throw new RppsException(e);
		}
	}

	public CardInformationTO deleteUrlsCardRelation(String userId, String password, String cuid, String[] urls) throws RppsException {
		try {
			URL[] urlsArray = new URL[0];
			if (urls != null) {
				URL[] urlArray = new URL[urls.length];
				for (int i = 0; i < urls.length; i++) {
					urlArray[i] = new URL(urls[i]);
				}
			}
			ICardRegistry.getInstance().getUserProfileService().deleteUrls(RppsHelper.getCallbackHandler(userId, password), new CUID(cuid), urlsArray);
			return getCardInformation(userId, password, cuid);
		} catch (Exception e) {
			// TODO Auto-generated catch block
			log.error(e);
			throw new RppsException(e);
		}
	}

	/** ************* Other methods ************** */

	public static void AllowSelfSignedSSL() {
		if (Security.getProvider("Dummy") == null) {
			Security.insertProviderAt(new DummySecurityProvider(), 2);
			Security.setProperty("ssl.TrustManagerFactory.algorithm", DummyTrustManagerFactory.getAlgorithm());
		}
	}

	final static class DummySecurityProvider extends java.security.Provider {

		private static final long serialVersionUID = 594145003249139851L;

		public DummySecurityProvider() {
			super("Dummy", 1.0D, "");
			AccessController.doPrivileged(new PrivilegedAction() {
				public Object run() {
					put("TrustManagerFactory." + DummyTrustManagerFactory.getAlgorithm(), DummyTrustManagerFactory.class.getName());
					return null;
				}
			});
		}
	}

	public final static class DummyTrustManagerFactory extends TrustManagerFactorySpi {
		public DummyTrustManagerFactory() {
		}

		public static String getAlgorithm() {
			return "XTrust509";
		}

		protected void engineInit(KeyStore keystore) throws KeyStoreException {
		}

		protected TrustManager[] engineGetTrustManagers() {
			return new TrustManager[] { new X509TrustManager() {
				public java.security.cert.X509Certificate[] getAcceptedIssuers() {
					return null;
				}

				public void checkClientTrusted(java.security.cert.X509Certificate[] certs, String authType) {
				}

				public void checkServerTrusted(java.security.cert.X509Certificate[] certs, String authType) {
					for (int i = 0; i < certs.length; i++) {
						System.out.println("  >>> Checking certificate " + certs[i]);
					}
				}
			} };
		}

		protected void engineInit(ManagerFactoryParameters arg0) throws InvalidAlgorithmParameterException {
		}
	}

	protected static ArrayList listword = null;

	/**
	 * Gets list of random word
	 * 
	 * @param count
	 * @return list of word
	 */
	public String[] getRandomWordList(int count) throws RppsException {
		try {
			if (listword == null) {
				listword = new ArrayList();
				String dictionaryFile = System.getProperty(WebKeys.HIGGINS_RPPS_WEB_DICTIONARY_PROPERTY) != null ? System.getProperty(WebKeys.HIGGINS_RPPS_WEB_DICTIONARY_PROPERTY) : WebKeys.HIGGINS_RPPS_WEB_DICTIONARY_FILEPATH;

				File file = new File(dictionaryFile);
				if (!file.exists())
					throw new FileNotFoundException("File " + dictionaryFile + " does not find!");

				if (!file.canRead())
					throw new IOException("File " + dictionaryFile + " can not be read!");

				try {
					FileReader fr = new FileReader(file);
					LineNumberReader lnr = new LineNumberReader(fr);
					String s = "";

					while ((s = lnr.readLine()) != null) {
						if (s.trim().length() > 3)
							listword.add(s.trim());
					}
					lnr.close();
					fr.close();
				} catch (IOException e) {
					e.printStackTrace();
					throw new IOException("File " + dictionaryFile + " can not be read!");
				}
			}

			if (listword.size() < count)
				throw new IOException("Dictionary file does not contain enough words");

			ArrayList result = new ArrayList();
			Random random = new Random(new Date().getTime());
			for (int i = 0; i < count; i++) {
				int index = 0;
				do {
					index = random.nextInt(listword.size());
				} while (result.contains(listword.get(index)) || (((String) listword.get(index)).length() < 4));

				result.add(listword.get(index));
			}
			String[] rr = new String[count];
			result.toArray(rr);
			return rr;
		} catch (Exception exc) {
			log.error(exc);
			throw new RppsException(exc);
		}
	}

	/**
	 * Gets the global i-name registered to the Higgins service (e.g. "=parity").
	 * 
	 * @return the global i-name.
	 */
	public String getHigginsServiceGlobalIName() {
		return System.getProperty(WebKeys.HIGGINS_SERVICE_GLOBAL_NAME_PROPERTY, "=parity");
	}

	/**
	 * Creates AND element as well as subelements of the UIResponse transfer object.
	 * 
	 * @param uct the UserChoiceTree object.
	 * @return the array of UIResponseAndTO objects.
	 */
	protected UIResponseAndTO[] getAnds(CallbackHandler handler, UserChoiceTree uct, URL url, ICardSpacePolicy policy) {
		List andList = new ArrayList();
		if (uct != null) {
			for (ListIterator li = uct.getListIterator(); li.hasNext();) {
				Object child = li.next();
				UIResponseAndTO and = new UIResponseAndTO();
				if (child instanceof UserChoiceTree) {
					List orList = new ArrayList();
					for (ListIterator li2 = ((UserChoiceTree) child).getListIterator(); li2.hasNext();) {
						Object child2 = li2.next();
						UIResponseOrTO or = new UIResponseOrTO();
						if (child2 instanceof UserChoiceTree) {
							List andList2 = new ArrayList();
							for (ListIterator li3 = ((UserChoiceTree) child2).getListIterator(); li3.hasNext();) {
								Object child3 = li3.next();
								UIResponseAndTO and2 = new UIResponseAndTO();
								if (child3 instanceof UCTelm) {
									// ???
								}
								andList2.add(and2);
							}
						} else if (child2 instanceof UCTelm) {
							IPolicy atom = ((UCTelm) child2).getAtom();
							UIResponseClauseTO clause = new UIResponseClauseTO();
							clause.setLabel(atom.getSummary());
							clause.setLonglabel(atom.getSummary().toString());

							if (atom instanceof ICardSpacePolicy) {
								// ICardSpacePolicy icp = (ICardSpacePolicy) atom;
								// clause.setOptional(icp.getOptionalClaimSummary());
								// clause.setLongOptional(icp.getOptionalClaimSummary().toString());
							}

							UIResponseUsableCardsTO uc = new UIResponseUsableCardsTO();
							List cardList = new ArrayList();
							for (Iterator it = ((UCTelm) child2).getCredsIterator(); it != null && it.hasNext();) {
								UIResponseCardTO card = new UIResponseCardTO();
								CredentialContainer cc = (CredentialContainer) it.next();
								ICard cred = cc.getCredential();

								CardInformation cardInformation = getCardInformationByCuid(handler, cred.getCUID());
								// sets uid
								card = convertICard_To_UIResponseCardTO(handler, cred, url, policy, cardInformation);

								card.setUid(cc.key); // sets name String nm = cred.getName(); int k; nm = (k = nm.lastIndexOf('/')) > 0 ? nm.substring(k + 1) : nm; card.setName(nm); // sets description card.setDescription(cred.getDescription()); // sets image card.setImage(cred.getImage()); // sets imagetype card.setImagetype(cred.getImageType());

								/*
								 * UITokenServiceCredentialTO[] tokenservices = convertICredentials_to_UITokenServiceCredentialTO(handler, cred, cardInformation.getCardCredential());
								 * 
								 * card.setTokenservices(tokenservices);
								 */

								URL[] urls = cardInformation.getUrls();
								if (urls != null) {
									for (int i = 0; i < urls.length; i++) {
										if (urls[i].toString().equals(url))
											card.setSaveCard(true);
									}
								}

								cardList.add(card);
							}
							UIResponseCardTO[] cards = new UIResponseCardTO[cardList.size()];
							cardList.toArray(cards);
							uc.setCard(cards);
							clause.setUsableCards(uc);
							or.setClause(clause);
						}
						orList.add(or);
					}
					UIResponseOrTO[] ors = new UIResponseOrTO[orList.size()];
					orList.toArray(ors);
					and.setOr(ors);
				} else if (child instanceof UCTelm) {
					// ???
				}
				andList.add(and);
			}
		}
		UIResponseAndTO[] ands = new UIResponseAndTO[andList.size()];
		andList.toArray(ands);
		return ands;
	}

	/**
	 * Converts ICredential object to UITokenServiceCredential transfer object
	 * 
	 * @param handler the callback handler to be used to request user's identity
	 * @param card ICard object
	 * @param cardCredential
	 * @return UITokenServiceCredentialTO object
	 */
	protected UITokenServiceCredentialTO[] convertICredentials_to_UITokenServiceCredentialTO(CallbackHandler handler, ICard card, CardCredential cardCredential) {
		List credList = new ArrayList();
		if (card != null) {
			ICredentialDescriptor[] der = card.getRequiredCredentials();
			if (der != null) {
				for (int i = 0; i < der.length; i++) {
					ICredential credential = der[i].getCredential();
					if ((credential != null) && (credential instanceof ITokenServiceCredential)) {
						UITokenServiceCredentialTO tokenService = new UITokenServiceCredentialTO();
						tokenService.setAddress(((ITokenServiceCredential) credential).getAddress().toString());
						tokenService.setMetadataAddress(((ITokenServiceCredential) credential).getMetadataAddress().toString());
						if (credential instanceof ITSUsernamePasswordCredential) {
							tokenService.setPassword(((ITSUsernamePasswordCredential) credential).getPassword());
							tokenService.setUsername(((ITSUsernamePasswordCredential) credential).getUsername());
							tokenService.setType(RppsService.ITSUsernamePasswordCredential);
							if ((cardCredential != null) && (RppsService.ITSUsernamePasswordCredential.equals(cardCredential.getCredentialType())))
								tokenService.setSaveCredential(true);

							credList.add(tokenService);
						} else if (credential instanceof ITSSelfIssuedCredential) {
							tokenService.setType(RppsService.ITSSelfIssuedCredential);
							try {
								ICredential subCredential = getICredentialByPPID(handler, ((ITSSelfIssuedCredential) credential).getCertificate(), ((ITSSelfIssuedCredential) credential).getPPID());
								if (subCredential != null) {
									for (int index = 0; index < subCredential.getCallbacks().length; index++) {
										if (subCredential.getCallbacks()[index] instanceof PinCodeCallback) {
											if ((cardCredential != null) && (RppsService.ITSSelfIssuedCredential.equals(cardCredential.getCredentialType())))
												tokenService.setSaveCredential(true);
											credList.add(tokenService);
											break;
										}
									}
								}
							} catch (Exception exc) {
								log.error(exc);
							}
						} else if (credential instanceof ITSKerberosV5Credential) {
							tokenService.setType(RppsService.ITSKerberosV5Credential);
							if ((cardCredential != null) && (RppsService.ITSKerberosV5Credential.equals(cardCredential.getCredentialType())))
								tokenService.setSaveCredential(true);
							credList.add(tokenService);

						} else if (credential instanceof ITSX509V3Credential) {
							tokenService.setType(RppsService.ITSX509V3Credential);
							if ((cardCredential != null) && (RppsService.ITSX509V3Credential.equals(cardCredential.getCredentialType())))
								tokenService.setSaveCredential(true);
							credList.add(tokenService);
						}
					} else {
						UITokenServiceCredentialTO tokenService = new UITokenServiceCredentialTO();
						if (credential instanceof IPinCodeCredential) {
							tokenService.setType(RppsService.IPinCodeCredential);
							if ((cardCredential != null) && (RppsService.IPinCodeCredential.equals(cardCredential.getCredentialType())))
								tokenService.setSaveCredential(true);
							credList.add(tokenService);
						}
					}
				}
			}
		}
		UITokenServiceCredentialTO[] array = new UITokenServiceCredentialTO[credList.size()];
		credList.toArray(array);
		return array;
	}

	/**
	 * Converts the card to the transfer object for the CARD element of UserInterfaceResponse
	 * 
	 * @param card the Higgins card.
	 * @param isCatchAuthentication equals "true" if needs to generete AuthenticationRequiredException or AuthenticationException
	 * @return the card transfer object.
	 */
	protected UIResponseCardTO convertICard_To_UIResponseCardTO(CallbackHandler handler, ICard icard, URL url, ICardSpacePolicy policy, CardInformation cardInformation) {
		UIResponseCardTO cardTO = new UIResponseCardTO();
		if (icard != null) {

			// sets uid
			cardTO.setUid(icard.getCUID().toString());
			// sets name
			String nm = icard.getName();
			int k;
			nm = (k = nm.lastIndexOf('/')) > 0 ? nm.substring(k + 1) : nm;
			cardTO.setName(nm);
			// sets description
			cardTO.setDescription(icard.getDescription());
			// sets image
			cardTO.setImage(icard.getImage());
			// sets imagetype
			cardTO.setImagetype(icard.getImageType());

			UITokenServiceCredentialTO[] tokenservices = convertICredentials_to_UITokenServiceCredentialTO(handler, icard, cardInformation.getCardCredential());

			cardTO.setTokenservices(tokenservices);

			URL[] urls = cardInformation.getUrls();
			if (urls != null) {
				for (int i = 0; i < urls.length; i++) {
					if (urls[i].equals(url))
						cardTO.setSaveCard(true);
				}
			}
			if ((icard instanceof IPersonalInformationCard) && ((icard.getRequiredCredentials() == null) || ((icard.getRequiredCredentials() != null) && (icard.getRequiredCredentials().length == 0)))) {
				try {
					cardTO.setProperties(ConvertHelper.convertClaims_to_PropertyTO(icard, false, policy.getRequiredClaims().iterator()));
					cardTO.setOptionalClaims(ConvertHelper.convertClaims_to_PropertyTO(icard, false, policy.getOptionalClaims().iterator()));
				} catch (Exception exc) {
					log.error(exc);
				}

			}
		}
		return cardTO;
	}

	/**
	 * Converts the card to the card transfer object.
	 * 
	 * @param card the Higgins card.
	 * @param isCatchAuthentication equals "true" if needs to generete AuthenticationRequiredException or AuthenticationException
	 * @return the card transfer object.
	 */
	protected ICardTO convertICard_To_ICardTO(CallbackHandler handler, ICard card, boolean isCatchAuthentication) throws CardException, InvalidTypeException {
		ICardTO icard = new ICardTO();
		if (card != null) {
			icard.setId(card.getID());
			icard.setCuid(card.getCUID().toString());
			icard.setName(card.getName());
			icard.setIssuerName(card.getIssuerName());
			icard.setImage(card.getImage());
			icard.setImageType(card.getImageType());
			icard.setSelfIssued(card.isSelfIssued());

			if (card instanceof IPersonalInformationCard)
				icard.setDeleteable(false);
			else
				icard.setDeleteable(true);

			CardCredential cardCredential = null;
			CardInformation cardInformation = getCardInformationByCuid(handler, card.getCUID());
			if (cardInformation != null)
				cardCredential = cardInformation.getCardCredential();

			UITokenServiceCredentialTO[] tokenServiceCredentials = convertICredentials_to_UITokenServiceCredentialTO(handler, card, cardCredential);
			if (tokenServiceCredentials.length > 0) {
				icard.setTokenServiceCredentials(tokenServiceCredentials);
			}

			icard.setProperties(ConvertHelper.convertClaims_to_PropertyTO(card, isCatchAuthentication, card.getSupportedClaimTypes()));
		}
		return icard;
	}

	static {
		AllowSelfSignedSSL();
	}

}
