/*******************************************************************************
 * Copyright (c) 2007 Google
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.saml2idp.server;

import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.spec.KeySpec;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import javax.servlet.ServletContext;
import javax.servlet.ServletException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.configuration.xml.ConfigurationHandler;
import org.eclipse.higgins.idas.api.IContextFactory;
import org.eclipse.higgins.idas.api.IContextId;
import org.eclipse.higgins.idas.registry.IdASRegistry;
import org.eclipse.higgins.idas.registry.contextid.ContextIdFactory;

/**
 * This servlet overrides only the init() method and is used to initialize
 * common objects of our web application. In particular, it loads the
 * Higgins configuration and starts up Higgins-related components.
 * 
 * Static getter methods are provided for retrieving these components.
 */
public class Init extends javax.servlet.http.HttpServlet implements javax.servlet.Servlet {

	private static final long serialVersionUID = -6734653011883038818L;

	private static final Log log = LogFactory.getLog(Init.class);

	// Some constants related to the Higgins configuration.

	private static final String PATH_HIGGINS_CONFIG = "./conf/HigginsConfiguration.xml";
	private static final String HIGGINS_CONFIG_IDASREGISTRY = "IdASRegistry";
	private static final String HIGGINS_CONFIG_CONTEXTID = "samlContext";

	// Application-wide properties.

	private static Properties properties;

	// The private key we use to sign SAML2 messages.

	private static PrivateKey idpPrivateKey;

	// The public key we use for the KeyInfo element in XML Signatures.

	private static X509Certificate idpCertificate;
	private static PublicKey idpPublicKey;

	// The certificates and public keys we use to verify SAML2 messages.

	private static X509Certificate[] rpCertificates;
	private static PublicKey[] rpPublicKeys;

	// The ACS URLs that are acceptable

	private static Set acceptAcsUrls;

	// Various Higgins objects:
	// - The IdASRegistry is a singleton and responsible for looking up
	//   suitable context factories for a context ID.
	// - A context ID is a reference to a context and contains everything 
	//   needed to instantiate the context.
	// - The context type indicates the underlying technology (e.g. LDAP) 
	//   and is used to look up a suitable context factory.
	// - A context factory can instantiate a context based on a context ID.

	private static IdASRegistry higginsRegistry;
	private static IContextId higginsContextId;
	private static String higginsContextType;
	private static IContextFactory higginsContextFactory;

	public Init() {

		super();
	}   	 	  	  	  

	public void init() throws ServletException {

		log.trace("init()");

		super.init();

		ServletContext servletContext = this.getServletContext();

		// Load application-wide properties.

		try {

			File propertiesFile = new File(servletContext.getRealPath("./conf/application.properties"));
			properties = new Properties();
			properties.load(new FileInputStream(propertiesFile));
		} catch (Exception ex) {

			throw new ServletException("Cannot load application properties.", ex);
		}

		log.info("Loaded application properties.");

		// Load the private key we need to sign SAML2 messages.

		try {

			File keyFile = new File(servletContext.getRealPath("./conf/privkey.der"));
			FileInputStream stream = new FileInputStream(keyFile);
			byte[] keyBytes = new byte[stream.available()];
			stream.read(keyBytes);
			stream.close();

			KeySpec keySpec = new PKCS8EncodedKeySpec(keyBytes);
			KeyFactory keyFactory = KeyFactory.getInstance("DSA");
			idpPrivateKey = keyFactory.generatePrivate(keySpec);

			log.info("Loaded IdP private key (algorithm: " + idpPrivateKey.getAlgorithm() + ", format: " + idpPrivateKey.getFormat());
		} catch (Exception ex) {

			idpPrivateKey = null;
			log.warn("No IdP private key loaded. SAML2 messages will not be signed.");
		}

		// Load the certificate and public key we put into the KeyInfo element of XML Signatures.

		try {

			File keyFile = new File(servletContext.getRealPath("./conf/cacert.pem"));
			FileInputStream stream = new FileInputStream(keyFile);

			CertificateFactory certificateFactory = CertificateFactory.getInstance("X.509");
			idpCertificate = (X509Certificate) certificateFactory.generateCertificate(stream);
			idpPublicKey = idpCertificate.getPublicKey();
			stream.close();

			log.info("Loaded IdP certificate from issuer: " + idpCertificate.getIssuerX500Principal().toString());
			log.info("Loaded IdP public key (algorithm: " + idpPublicKey.getAlgorithm() + ", format: " + idpPublicKey.getFormat());
		} catch (Exception ex) {

			idpCertificate = null;
			idpPublicKey = null;
			log.warn("No IdP certificate and public key loaded. SAML2 messages will not be signed.");
		}

		// Load the certificates and public keys of the relying parties we want to allow.

		File rpDirectory = new File(servletContext.getRealPath("./conf/rp"));

		File[] files = rpDirectory.listFiles(new FileFilter() {

			public boolean accept(File file) {
				
				return(file.isFile() && file.getName().toLowerCase().endsWith(".pem"));
			}
		});
		
		log.info("Found " + files.length + " RP certificates.");

		rpCertificates = new X509Certificate[files.length];
		rpPublicKeys = new PublicKey[files.length];

		for (int i=0; i<files.length; i++) {

			File file = files[i];
			X509Certificate rpCertificate;
			PublicKey rpPublicKey;

			try {

				FileInputStream stream = new FileInputStream(file);

				CertificateFactory certificateFactory = CertificateFactory.getInstance("X.509");
				rpCertificate = (X509Certificate) certificateFactory.generateCertificate(stream);
				rpPublicKey = rpCertificate.getPublicKey();
				stream.close();

				if (rpPublicKey == null) throw new NullPointerException();
			} catch (Exception ex) {

				throw new ServletException("Cannot load RP certificate.", ex);
			}

			rpCertificates[i] = rpCertificate;
			rpPublicKeys[i] = rpPublicKey;

			log.info("Loaded RP certificate from issuer: " + rpCertificate.getIssuerX500Principal().toString());
			log.info("Loaded RP public key (algorithm: " + rpPublicKey.getAlgorithm() + ", format: " + rpPublicKey.getFormat() + ")");
		}

		// Load the acceptable ACS URLs

		String acceptAcsUrlsString = properties.getProperty("accept-acs-urls");
		acceptAcsUrls = new HashSet();
		acceptAcsUrls.addAll(Arrays.asList(acceptAcsUrlsString.split(",")));

		log.info("Found " + acceptAcsUrls.size() + " acceptable ACS URLs.");

		for (Iterator i = acceptAcsUrls.iterator(); i.hasNext(); ) {

			String acceptAcsUrl = (String) i.next();

			log.info("ACS URL: " + acceptAcsUrl);
		}

		// Load the Higgins configuration.

		ConfigurationHandler configurationHandler;
		Map configurationSettings;

		try {

			configurationHandler = new ConfigurationHandler();
			configurationHandler.setConfigurationBase("");
			configurationHandler.setFileName(servletContext.getRealPath(PATH_HIGGINS_CONFIG));
			configurationHandler.configure(null);
			configurationSettings = configurationHandler.getSettings();
		} catch (Exception ex) {

			throw new ServletException("Cannot load Higgins configuration.", ex);
		}

		log.info("Loaded Higgins configuration.");

		// Obtain the Higgins IdASRegistry, context ID and context factory.

		try {

			higginsRegistry = (IdASRegistry) configurationSettings.get(HIGGINS_CONFIG_IDASREGISTRY);
			if (higginsRegistry == null) throw new NullPointerException("higginsRegistry is null");

			higginsContextId = ContextIdFactory.fromConfiguration(HIGGINS_CONFIG_CONTEXTID);
			if (higginsContextId == null) throw new NullPointerException("higginsContextId is null");

			higginsContextType = higginsContextId.getTypes()[0];
			if (higginsContextType == null) throw new NullPointerException("higginsContextType is null");

			higginsContextFactory = higginsRegistry.getContextFactory(higginsContextId);
			if (higginsContextFactory == null) throw new NullPointerException("higginsContextFactory is null");
		} catch (Exception ex) {

			throw new ServletException("Cannot initialize Higgins.", ex);
		}

		log.info("IdAS initialized.");

		// Done.

		log.info("Up and running.");
	}

	/**
	 * Returns the application properties.
	 */
	public static Properties getProperties() {

		log.trace("getProperties()");

		return(properties);
	}

	/**
	 * Returns the SAML2 issuer.
	 */
	public static String getSAML2Issuer() {

		log.trace("getSAML2Issuer()");

		return(properties.getProperty("saml2-issuer"));
	}

	/**
	 * Returns the validity period of issued SAML2 assertions. This is used to
	 * construct the value of the NotOnOrAfter attribute on the Conditions element.
	 */
	public static long getSAML2AssertionValidityMillis() {

		log.trace("getSAML2AssertionValidityMillis()");

		return(Long.parseLong(properties.getProperty("saml2-assertion-validity-millis")));
	}

	/**
	 * Returns the name of the parameter from which a username should be extracted.
	 */
	public static String getExtractUsernameParameterName() {

		log.trace("getExtractUsernameParameterName()");

		return(properties.getProperty("extract-username-parameter-name"));
	}

	/**
	 * Returns the name of the header from which a username should be extracted.
	 */
	public static String getExtractUsernameHeaderName() {

		log.trace("getExtractUsernameHeaderName()");

		return(properties.getProperty("extract-username-header-name"));
	}

	/**
	 * Returns the name of the cookie from which a username should be extracted.
	 */
	public static String getExtractUsernameCookieName() {

		log.trace("getExtractUsernameCookieName()");

		return(properties.getProperty("extract-username-cookie-name"));
	}

	/**
	 * Returns the private key used to sign SAML2 messages.
	 */
	public static PrivateKey getIdpPrivateKey() {

		return(idpPrivateKey);
	}

	/**
	 * Returns the certificate used for the KeyInfo element in XML Signatures.
	 */
	public static X509Certificate getIdpCertificate() {

		return(idpCertificate);
	}

	/**
	 * Returns the public key used for the KeyInfo element in XML Signatures.
	 */
	public static PublicKey getIdpPublicKey() {

		return(idpPublicKey);
	}

	/**
	 * Returns the certificates used to validate SAML2 messages.
	 */
	public static X509Certificate[] getRpCertificates() {

		return(rpCertificates);
	}

	/**
	 * Returns the public keys used to validate SAML2 messages.
	 */
	public static PublicKey[] getRpPublicKeys() {

		return(rpPublicKeys);
	}

	/**
	 * Returns the acceptable ACS URLs.
	 */
	public static Set getAcceptAcsUrls() {

		return(acceptAcsUrls);
	}

	/**
	 * Returns the Higgins IdASRegistry.
	 */
	public static IdASRegistry getHigginsRegistry() {

		log.trace("getHigginsRegistry()");

		return(higginsRegistry);
	}

	/**
	 * Returns the Higgins context ID for the context we validate against.
	 */
	public static IContextId getHigginsContextId() {

		log.trace("getHigginsContextId()");

		return(higginsContextId);
	}

	/**
	 * Returns the Higgins context type for the context we validate against.
	 */
	public static String getHigginsContextType() {

		log.trace("getHigginsContextType()");

		return(higginsContextType);
	}

	/**
	 * Returns the Higgins IContextFactory that can instantiate the context we validate against.
	 */
	public static IContextFactory getHigginsContextFactory() {

		log.trace("getHigginsContextFactory()");

		return(higginsContextFactory);
	}
}
