/**
 * Copyright (c) 2008 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.sts.client;

import java.net.URL;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.cert.X509Certificate;
import java.util.LinkedHashMap;

import org.apache.axiom.om.util.Base64;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.keystore.IKeyStoreService;
import org.eclipse.higgins.keystore.registry.KeyStoreRegistry;

public class RelyingPartyIdentifierV15Helper {
	private static Log log = LogFactory.getLog(RelyingPartyIdentifierV15Helper.class);

	/**
	 * Generates RP identifier for PPID
	 * 
	 * @param rpURL
	 *            <URL>URL</URL> of relying party. Used to generate an
	 *            identifier if no if no relying party's certificate present.
	 * @param certificateChain
	 *            Relying party's end-entity certificate or certificate chain.
	 * @return
	 * @throws Exception
	 */
	public static byte[] getRelyingPartyIdentifier(String rpURL, X509Certificate[] certificateChain) throws Exception {
		return getRelyingPartyIdentifier(rpURL, certificateChain, true);
	}

	/**
	 * Generates RP identifier for signing key
	 * 
	 * @param rpURL
	 *            <URL>URL</URL> of relying party. Used to generate an
	 *            identifier if no if no relying party's certificate present.
	 * @param certificateChain
	 *            Relying party's end-entity certificate or certificate chain.
	 * @return
	 * @throws Exception
	 */
	public static byte[] getRelyingPartyIdentifierForSigningKey(String rpURL, X509Certificate[] certificateChain) throws Exception {
		return getRelyingPartyIdentifier(rpURL, certificateChain, false);
	}

	private static byte[] getRelyingPartyIdentifier(String rpURL, X509Certificate[] certificateChain, boolean forPPID) throws Exception {
		if (certificateChain != null && certificateChain.length > 0) {
			return getRelyingPartyIdentifier(certificateChain, forPPID);
		} else if (rpURL != null) {
			return getRelyingPartyIdentifier(rpURL);
		} else
			throw new Exception("Either certificate or URL required to generate relying party identifier.");
	}

	/**
	 * @param certificateChain
	 *            Certificate chain of relying party
	 * @return Certificate-based RP identifier according to CardSpace tech
	 *         reference v 1.5
	 * @param forPPID
	 *            If <code>true</code> then RP id is generated for PPID,
	 *            otherwise for proof key
	 * @throws Exception
	 */
	private static byte[] getRelyingPartyIdentifier(X509Certificate[] certificateChain, boolean forPPID) throws Exception {
		IKeyStoreService securityService = KeyStoreRegistry.getInstance().getSecurityService();
		X509Certificate leaf = securityService.findLeafFromCertificates(certificateChain);
		String subjDN = leaf.getSubjectX500Principal().getName();
		LinkedHashMap subjHash = DNParser.parse(subjDN);
		log.debug("Certificate chain size : " + certificateChain.length);
		if (log.isDebugEnabled()) {
			for (int i = 0, k = certificateChain.length; i < k; i++) {
				log.debug("Certificate " + i + " : /n" + Base64.encode(certificateChain[i].getEncoded()) + " \n"
						+ certificateChain[i].getSubjectDN().getName());
			}
		}
		boolean isCertificateValid = securityService.validateCertificateChain(null, certificateChain, null);
		boolean isEVCertificate = securityService.extendendValidationCertificateChain(null, certificateChain, null);
		String organization = (String) subjHash.get("O");
		String commonName = (String) subjHash.get("CN");
		log.debug("isCertificateValid=" + isCertificateValid + ", isEVCertificate=" + isEVCertificate + ", O=" + organization + ", CN=" + commonName);
		if (isEVCertificate && isCertificateValid) {
			log.debug("Case 1: RP's certificate is EV for organizational identifier attributes and chains to a trusted root certificate authority");
			String orgId = getOrgIdString(subjHash);
			log.debug("orgId : " + orgId);
			byte[] orgIdB = orgId.getBytes("UTF-16LE");
			return getSHA256Digest(orgIdB);
		} else if (isCertificateValid && !isEmpty(organization)) {
			log.debug("Case 2: RP's certificate is not EV for organizational identifier attributes, has a non-empty Organization (O) value, and chains to a trusted root certificate authority");
			String orgId = null;
			if (forPPID) {
				orgId = getOrgIdString(subjHash);
			} else {
				orgId = "|Non-EV" + getOrgIdString(subjHash);
			}
			log.debug("orgId : " + orgId);
			byte[] orgIdB = orgId.getBytes("UTF-16LE");
			return getSHA256Digest(orgIdB);
		} else if (!isCertificateValid || (isEmpty(organization) && isEmpty(commonName))) {
			log.debug("Case 3: RP's certificate has an empty or no Organization (O) value and has an empty or no Common Name (CN) or does not chain to a trusted root certificate authority");
			byte[] pKey = leaf.getPublicKey().getEncoded();
			if (log.isDebugEnabled()) {
				String keyStr = Base64.encode(pKey);
				log.debug("Key : " + keyStr);
			}
			return getSHA256Digest(pKey);
		} else if (isCertificateValid && isEmpty(organization) && !isEmpty(commonName)) {
			log.debug("Case 4: RP's certificate has an empty or no Organization (O) value but has a non-empty Common Name (CN) value and chains to a trusted root certificate authority");
			String cnId = getCnIdString(subjHash);
			log.debug("cnId : " + cnId);
			byte[] cnIdB = cnId.getBytes("UTF-16LE");
			return getSHA256Digest(cnIdB);
		} else {
			log.debug("Impossible case. RPId generation logic is failed.");
			return null;
		}
	}

	/**
	 * @param url
	 *            Relying's party domain name or IP
	 * @return Relying party identifier based on domain name or IP
	 * @throws Exception
	 */
	private static byte[] getRelyingPartyIdentifier(String url) throws Exception {
		URL uri = new URL(url);
		String hostName = uri.getHost();
		byte[] cnIdB = hostName.getBytes("UTF-16LE");
		return getSHA256Digest(cnIdB);
	}

	private static boolean isEmpty(String str) {
		return str == null || str.trim().length() == 0;
	}

	private static String getOrgIdString(LinkedHashMap subjHash) throws Exception {
		StringBuffer buffer = new StringBuffer();
		buffer.append("|");
		appendSubjAttribute(buffer, "O", getSubjAttribute(subjHash, "O"));
		appendSubjAttribute(buffer, "L", getSubjAttribute(subjHash, "L"));
		String state;
		state = getSubjAttribute(subjHash, "S");
		if ("".equals(state))
			state = getSubjAttribute(subjHash, "ST");
		appendSubjAttribute(buffer, "S", state);
		appendSubjAttribute(buffer, "C", getSubjAttribute(subjHash, "C"));
		String orgId = buffer.toString();
		return orgId;
	}

	private static String getCnIdString(LinkedHashMap subjHash) throws Exception {
		StringBuffer buffer = new StringBuffer();
		buffer.append("|");
		appendSubjAttribute(buffer, "CN", getSubjAttribute(subjHash, "CN"));
		return buffer.toString();
	}

	private static String getSubjAttribute(LinkedHashMap subjHash, String attrName) throws Exception {
		return (subjHash.containsKey(attrName)) ? (String) subjHash.get(attrName) : "";
	}

	private static void appendSubjAttribute(StringBuffer buffer, String attrName, String attrValue) throws Exception {
		buffer.append(attrName);
		buffer.append("=");
		if (!attrValue.startsWith("\"")) {
			buffer.append("\"");
		}
		buffer.append(attrValue);
		if (!attrValue.endsWith("\"")) {
			buffer.append("\"");
		}
		buffer.append("|");
	}

	private static byte[] getSHA256Digest(byte[] bytes) throws NoSuchAlgorithmException {
		MessageDigest hash = MessageDigest.getInstance("SHA-256");
		return hash.digest(bytes);
	}

}
