/*******************************************************************************
 * Copyright (c) 2006 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * 	  Michael McIntosh (IBM Corporation) - initial generation from WSDL using WSDL2Java
 *    Jeesmon Jacob (Parity Communications)
 *******************************************************************************/
package org.eclipse.higgins.sts.server.profile.idas;

import java.net.URI;
import java.security.PrivateKey;
import java.security.cert.X509Certificate;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.higgins.configuration.api.ISettingDescriptor;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IValue;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.ISimpleValue;
import org.eclipse.higgins.idas.api.ISingleValuedAttribute;
import org.eclipse.higgins.idas.api.ITypedValue;
import org.eclipse.higgins.idas.common.AuthNAnonymousMaterials;
import org.eclipse.higgins.idas.common.AuthNNamePasswordMaterials;
import org.eclipse.higgins.idas.registry.IdASRegistry;
import org.eclipse.higgins.sts.api.IClaim;
import org.eclipse.higgins.sts.api.IClaimType;
import org.eclipse.higgins.sts.api.IConstants;
import org.eclipse.higgins.sts.api.IProfile;
import org.eclipse.higgins.sts.api.IProfileService;

public class ProfileService implements IProfileService {

	private final org.eclipse.higgins.sts.utilities.LogHelper log = new org.eclipse.higgins.sts.utilities.LogHelper
	(ProfileService.class.getName());
	
	private boolean bInitialized = false;
	
	private Map mapGlobalSettings = null;
	
	protected String strComponentName = null;
	
	private Map mapComponentSettings = null;
	
	private List listSupportedClaims = null;
	
	private Map mapAttributeClaims = null;
	
	private Map mapShortNameAttribute = null;
	
	private Map mapShortNameClaimUri = null;
	
	private String strCardImage = null;
	
	private X509Certificate certSSL = null;
	
	private org.eclipse.higgins.sts.spi.IXMLSecurityExtension xmlSecurityExtension = null;

	private IdASRegistry idasRegistry = null;

	private String contextId = null;

	private URI digitalSubjectType = null;
	
	private boolean allowAnonymousAccess = false;
	
	private URI CARD_KEY_HASH_URI = URI.create(IValue.BASE_OWL_URI + "cardKeyHash");

	private String storeUserName = null;

	private String storePassword = null;

	public ProfileService() {
		log.trace("ProfileService::ProfileService");
	}

	public String createProfile(IConstants constants, String strUserIdentifier,
			String strUserPassword) throws Exception {
		log.trace("ProfileService::createProfile");
		
		IContext context = idasRegistry.createContext(contextId);
		if(this.storeUserName != null && this.storePassword != null) {
			try {
				context.open(new AuthNNamePasswordMaterials(context, this.storeUserName, this.storePassword));
			}
			catch(Exception e) {
				throw new Exception("ERROR Invalid Username / Password");
			}
		}
		else if(this.allowAnonymousAccess) {
			try {
				context.open( new AuthNAnonymousMaterials());
			}
			catch(Exception e) {
				throw new Exception("ERROR Could not open context as anonymous");
			}
		}
		else {
			throw new Exception("Insufficient privileages to open context");
		}
		
		IEntity subj = context.addEntity(digitalSubjectType, strUserIdentifier);
		IAttribute attr = subj.addAttribute(AuthNNamePasswordMaterials.ATTR_MANAGED_URI);
		attr.addSimpleValue(ITypedValue.STRING_TYPE_URI, strUserPassword);
		context.applyUpdates();
		context.close();

		return "SUCCESS";
	}

	public String decryptToken(IConstants constants, String strToken)
			throws Exception {
		log.trace("ProfileService::decryptToken");
		if (null != strToken)
		{
	    	// TODO: Decrypt Token
	    	org.eclipse.higgins.sts.api.IElement elemToken = new org.eclipse.higgins.sts.common.Element();
	    	elemToken.set(strToken);

	    	this.log.trace("Decrypting Security Token");
	    	final org.eclipse.higgins.sts.api.IElement elemDecryptedToken = this.xmlSecurityExtension.DecryptElement
	    		(elemToken,
	    		(PrivateKey)this.mapGlobalSettings.get("IssuerPrivateKey"));
	    	final String strResult = (String)elemDecryptedToken.getAs(String.class);
	    	this.log.trace("Decrypted Security Token: " + strResult);
	    	return strResult;
		}

		return "Success";
	}

	public String deleteProfile(IConstants constants, String strUserIdentifier,
			String strUserPassword) throws Exception {
		this.log.trace("ProfileService::deleteProfile");
		IContext context = null;
		if ((null != strUserIdentifier) && (null != strUserPassword))
		{
			try {
				context = idasRegistry.createContext(contextId);
				context.open( new AuthNNamePasswordMaterials(context, strUserIdentifier, strUserPassword));
			}
			catch(Exception e) {
				throw new Exception("ERROR Invalid Username / Password");
			}
		}
		
		if(context == null) {
			return "ERROR Invalid Username / Password";
		}
		
		IEntity subj = context.getEntity(strUserIdentifier);
		subj.remove();
		context.applyUpdates();
		context.close();
		
		return "SUCCESS";
	}

	public String getManagedCard(IConstants constants, String strCardName,
			String strUsername, String strPassword, String strCredentialType,
			String strRequireAppliesTo, String strToken) throws Exception {
		this.log.trace("ProfileService::getManagedCard");
		
		this.log.trace("CredentialType: " + strCredentialType);
	
		String strPPID = null;
		String strModulus = null;
		String strExponent = null;
		String strThumbprint = null;
		
		boolean bRequireAppliesTo = ((null != strRequireAppliesTo) && (strRequireAppliesTo.matches("true")));
		
		if (null != strToken)
		{
	    	// TODO: Decrypt Token
	    	org.eclipse.higgins.sts.api.IElement elemToken = new org.eclipse.higgins.sts.common.Element();
	    	elemToken.set(strToken);

	    	this.log.trace("Decrypting Security Token");
	    	final org.eclipse.higgins.sts.api.IElement elemDecryptedToken = this.xmlSecurityExtension.DecryptElement
	    		(elemToken,
	    		(PrivateKey)this.mapGlobalSettings.get("IssuerPrivateKey"));
	    	this.log.trace("Decrypted Security Token: " + (String)elemDecryptedToken.getAs(String.class));
	    	this.log.trace("Looking for the PPID");
	    	
	    	org.w3c.dom.Element domDecryptedToken = (org.w3c.dom.Element)elemDecryptedToken.getAs(org.w3c.dom.Element.class);
	    	
	    	final org.w3c.dom.NodeList nlAttributes = domDecryptedToken.getElementsByTagNameNS
	    		(constants.getSAML10AssertionNamespace().toString(),
	    		"Attribute");
			if (null == nlAttributes)
			{
				this.log.trace("No Attribute found (getElementsByTagName returned null)");
			}
			else if (0 == nlAttributes.getLength())
			{
				this.log.trace("No Attribute found (0 == getLength())");
			}
			else
			{
				for (int i = 0; i < nlAttributes.getLength(); ++i)
				{
					this.log.trace("Processing Attribute element");
					final org.w3c.dom.Element elemAttribute = (org.w3c.dom.Element)nlAttributes.item(i);
					final String strAttributeName = elemAttribute.getAttribute("AttributeName");
					this.log.trace("Attribute: " + strAttributeName);
					if (strAttributeName.matches("privatepersonalidentifier"))
					{
				    	final org.w3c.dom.NodeList nlAttributeValues = elemAttribute.getElementsByTagNameNS
				    		(constants.getSAML10AssertionNamespace().toString(), "AttributeValue");
						if (null == nlAttributeValues)
						{
							this.log.trace("No AttributeValue found (getElementsByTagName returned null)");
						}
						else if (0 == nlAttributeValues.getLength())
						{
							this.log.trace("No AttributeValue found (0 == getLength())");
						}
						else if (1 != nlAttributeValues.getLength())
						{
							this.log.trace("More than 1 AttributeValue found");
						}
						else
						{
							final org.w3c.dom.Element elemAttributeValue = (org.w3c.dom.Element)nlAttributeValues.item(0);
							strPPID = org.eclipse.higgins.sts.utilities.XMLHelper.getTextContent(elemAttributeValue);
							this.log.trace("PPID: " + strPPID);
							break;
						}
					}		
				}
			}
	
	    	final org.w3c.dom.NodeList nlModuluses = domDecryptedToken.getElementsByTagNameNS
	    		(constants.getXMLSignatureNamespace().toString(), "Modulus");
			if (null == nlModuluses)
			{
				this.log.trace("No Modulus found (getElementsByTagName returned null)");
			}
			else if (0 == nlModuluses.getLength())
			{
				this.log.trace("No Modulus found (0 == getLength())");
			}
			else
			{
				this.log.trace("Processing Modulus element");
				final org.w3c.dom.Element elemModulus = (org.w3c.dom.Element)nlModuluses.item(0);
				strModulus = org.eclipse.higgins.sts.utilities.XMLHelper.getTextContent(elemModulus);
				this.log.trace("Modulus: " + strModulus);
			}
			final org.w3c.dom.NodeList nlExponents = domDecryptedToken.getElementsByTagNameNS
	    		(constants.getXMLSignatureNamespace().toString(), "Exponent");
			if (null == nlExponents)
			{
				this.log.trace("No Exponent found (getElementsByTagName returned null)");
			}
			else if (0 == nlExponents.getLength())
			{
				this.log.trace("No Exponent found (0 == getLength())");
			}
			else
			{
				this.log.trace("Processing Exponent element");
				final org.w3c.dom.Element elemExponent = (org.w3c.dom.Element)nlExponents.item(0);
				strExponent = org.eclipse.higgins.sts.utilities.XMLHelper.getTextContent(elemExponent);
				this.log.trace("Exponent: " + strExponent);
			}
	
			final String strResult = this.modifyProfileSetCardKeyHash
				(constants,
			    strUsername,
				strPassword,
				strPPID,
				strModulus,
				strExponent);
			if (!strResult.matches("Success"))
			{
				return strResult;
			}
		}

		log.trace("ProfileService::getManagedCard-0");
		
		final String strSSLCertificate = new String
			(org.apache.commons.codec.binary.Base64.encodeBase64
					(this.certSSL.getEncoded()));
		
		log.trace("ProfileService::getManagedCard-1");

		final String strEscapedCardName = strCardName.replaceAll(" ", "-");
		
		log.trace("ProfileService::getManagedCard-2");

	    final String strId = "IC01";
	    
		log.trace("ProfileService::getManagedCard-3");

	    String strInformationCard =
	    	"<InformationCard xmlns=\"http://schemas.xmlsoap.org/ws/2005/05/identity\" xml:lang=\"en-us\">"
	    	+ "<InformationCardReference>"
	    	+ "<CardId>" + ((java.net.URI)this.mapComponentSettings.get("CardId")).toString() + "&amp;cardid=" + strEscapedCardName + "</CardId>"
	    	+ "<CardVersion>1</CardVersion>"
	    	+ "</InformationCardReference>"
	    	+ "<CardName>" + strCardName + "</CardName>"
	    	+ "<CardImage MimeType=\"image/jpeg\">" + this.strCardImage + "</CardImage>"
	    	+ "<Issuer>" + ((java.net.URI)this.mapGlobalSettings.get("TokenServiceIssuerURI")).toString() + "</Issuer>"
	    	+ "<TimeIssued>2006-10-28T00:00:00.999Z</TimeIssued>"
	    	+ "<TimeExpires>9999-12-31T23:59:59.9999999Z</TimeExpires>"
	    	+ "<TokenServiceList>"
	    	+ "<TokenService>"
	    	+ "<EndpointReference xmlns=\"http://www.w3.org/2005/08/addressing\">"
	    	+ "<Address>" + ((java.net.URI)this.mapGlobalSettings.get("TokenServiceTrustURI")).toString() + "</Address>"
	    	+ "<Metadata>"
	    	+ "<Metadata xmlns=\"http://schemas.xmlsoap.org/ws/2004/09/mex\">"
	    	+ "<MetadataSection>"
	    	+ "<MetadataReference>"
	    	+ "<Address xmlns=\"http://www.w3.org/2005/08/addressing\">";
	    
		log.trace("ProfileService::getManagedCard-4");
	    
	    if (strCredentialType.matches("UsernamePassword"))
	    {
	    	this.log.trace("MetadataURI: " + ((java.net.URI)this.mapGlobalSettings.get("UsernameTokenMetadataURI")).toString());
	    	strInformationCard += ((java.net.URI)this.mapGlobalSettings.get("UsernameTokenMetadataURI")).toString();
	    }
	    else if (strCredentialType.matches("SelfSignedSAML"))
	    {
	    	this.log.trace("MetadataURI: " + ((java.net.URI)this.mapGlobalSettings.get("SelfSignedSAMLTokenMetadataURI")).toString());
		    strInformationCard += ((java.net.URI)this.mapGlobalSettings.get("SelfSignedSAMLTokenMetadataURI")).toString();
	    }
	    else if (strCredentialType.startsWith("X509"))
	    {
	    	strThumbprint = strCredentialType.substring("X509".length());
	    	this.log.trace("MetadataURI: " + ((java.net.URI)this.mapGlobalSettings.get("X509TokenMetadataURI")).toString());
		    strInformationCard += ((java.net.URI)this.mapGlobalSettings.get("X509TokenMetadataURI")).toString();
	    }
	    else
	    {
	    	this.log.trace("MetadataURI NOT Added");
	    }
	    
		log.trace("ProfileService::getManagedCard-5");

	    strInformationCard += "</Address>"
	    	+ "</MetadataReference>"
	    	+ "</MetadataSection>"
	    	+ "</Metadata>"
	    	+ "</Metadata>"
	    	+ "<Identity xmlns=\"http://schemas.xmlsoap.org/ws/2006/02/addressingidentity\">"
	    	+ "<KeyInfo xmlns=\"http://www.w3.org/2000/09/xmldsig#\">"
	    	+ "<X509Data>"
	    	+ "<X509Certificate>" + strSSLCertificate + "</X509Certificate>"
	    	+ "</X509Data>"
	    	+ "</KeyInfo>"
	    	+ "</Identity>"
	        + "</EndpointReference>"
	        + "<UserCredential>";
	    
		log.trace("ProfileService::getManagedCard-6");

	    if (strCredentialType.matches("UsernamePassword"))
	    {
		    strInformationCard +=
		        "<DisplayCredentialHint>Enter your username and password</DisplayCredentialHint>"
		        + "<UsernamePasswordCredential>"
		        + "<Username>" + strUsername + "</Username>"
		        + "</UsernamePasswordCredential>";
	    }
	    else if (strCredentialType.matches("SelfSignedSAML"))
	    {
		    strInformationCard +=
		        "<SelfIssuedCredential>"
		    	+ "<PrivatePersonalIdentifier>"
		    	+ strPPID
		    	+ "</PrivatePersonalIdentifier>"
		    	+ "</SelfIssuedCredential>";
	    }
	    else if (strCredentialType.startsWith("X509"))
	    {
	    	// we're going to record the cert in the user's profile
			final String strResult = this.modifyProfileSetX509SHA
			(constants,
		    strUsername,
			strPassword,
			strThumbprint);
			if (!strResult.matches("Success"))
			{
				return strResult;
			}

		    strInformationCard +=
		        "<X509V3Credential>"
		    	+ "<ds:X509Data xmlns:ds=\"http://www.w3.org/2000/09/xmldsig#\">"
		    	+ "<KeyIdentifier xmlns=\"http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd\" ValueType=\"http://docs.oasis-open.org/wss/oasis-wss-soap-message-security-1.1#ThumbprintSHA1\" EncodingType=\"http://docs.oasis-open.org/wss/2004/01/oasis200401-wss-soap-message-security-1.0#Base64Binary\">"
		    	+ strThumbprint
		    	+ "</KeyIdentifier>"
		    	+ "</ds:X509Data>"
		    	+ "</X509V3Credential>"
		    	;
	    }

	    
		log.trace("ProfileService::getManagedCard-7");
	
	    // TODO: Make this list of SupportClaimTypes configurable
	    strInformationCard +=
	    	"</UserCredential>"
	    	+ "</TokenService>"
	        + "</TokenServiceList>"
	        + "<SupportedTokenTypeList>"
	        + "<TokenType xmlns=\"http://schemas.xmlsoap.org/ws/2005/02/trust\">http://docs.oasis-open.org/wss/oasis-wss-saml-token-profile-1.1#SAMLV1.1</TokenType>"
	        + "<TokenType xmlns=\"http://schemas.xmlsoap.org/ws/2005/02/trust\">http://docs.oasis-open.org/wss/oasis-wss-saml-token-profile-1.1#SAMLV2.0</TokenType>"
	        + "<TokenType xmlns=\"http://schemas.xmlsoap.org/ws/2005/02/trust\">urn:oasis:names:tc:SAML:1.0:assertion</TokenType>"
	        + "<TokenType xmlns=\"http://schemas.xmlsoap.org/ws/2005/02/trust\">urn:oasis:names:tc:SAML:2.0:assertion</TokenType>"
	        + "</SupportedTokenTypeList>"
	        + "<SupportedClaimTypeList>";
	    
		for (int i = 0; i < this.listSupportedClaims.size(); ++i)
		{
			final java.net.URI uriAttribute = (java.net.URI)this.listSupportedClaims.get(i);
			log.trace("Attribute: " + uriAttribute.toString());
			final Map mapAttributeClaim = (Map)this.mapAttributeClaims.get(uriAttribute.toString());
			if (null != mapAttributeClaim)
			{
				final String strDisplayName = (String)mapAttributeClaim.get("DisplayName");
				log.trace("DisplayName: " + strDisplayName);
				strInformationCard +=
					"<SupportedClaimType Uri=\"" + uriAttribute.toString() + "\">" 
					+ "<DisplayTag>" + strDisplayName + "</DisplayTag>"
					+ "<Description>" + strDisplayName + "</Description>"
					+ "</SupportedClaimType>";
			}
		}
	        
	    strInformationCard +=	        
	        "</SupportedClaimTypeList>"
	        + (bRequireAppliesTo ? "<RequireAppliesTo/>" : "")
	        + "</InformationCard>";
	    
		log.trace("ProfileService::getManagedCard-8");
	    
	    final org.eclipse.higgins.sts.api.IElement elemInformationCard = new org.eclipse.higgins.sts.common.Element();
	    
		log.trace("ProfileService::getManagedCard-9");

	    elemInformationCard.set(strInformationCard);
	    
		log.trace("ProfileService::getManagedCard-10");

	    final org.eclipse.higgins.sts.api.IElement elemSignedInformationCard = this.xmlSecurityExtension.SignEnveloping
	    	(elemInformationCard,
	    	strId,
	    	(X509Certificate)this.mapGlobalSettings.get("IssuerCertificate"),
	    	(PrivateKey)this.mapGlobalSettings.get("IssuerPrivateKey"));
		
	    log.trace("ProfileService::getManagedCard-11");

		final String strSignedInformationCard = (String)elemSignedInformationCard.getAs(String.class);
		
		log.trace("ProfileService::getManagedCard-12");

	    return strSignedInformationCard.trim();
	}

	public String getProfile(IConstants constants, String strUserIdentifier,
			String strUserPassword, IProfile profile) throws Exception {
		log.trace("ProfileService::getProfile: " + strUserIdentifier);
		
		IContext context = null;
		if ((null != strUserIdentifier) && (null != strUserPassword))
		{
			try {
				context = idasRegistry.createContext(contextId);
				context.open( new AuthNNamePasswordMaterials(context, strUserIdentifier, strUserPassword));
			}
			catch(Exception e) {
				throw new Exception("ERROR Invalid Username / Password");
			}
		}
		
		if(context == null) {
			return "ERROR Invalid Username / Password";
		}
		
		final List listClaims = profile.getClaims();
		
		IEntity subj = context.getEntity(strUserIdentifier);
		
		for (int i = 0; i < this.listSupportedClaims.size(); ++i)
		{
			final java.net.URI uriAttribute = (java.net.URI)this.listSupportedClaims.get(i);
			log.trace("Attribute: " + uriAttribute.toString());
			final Map mapAttributeClaim = (Map)this.mapAttributeClaims.get(uriAttribute.toString());
			if (null != mapAttributeClaim)
			{
				final String strDisplayName = (String)mapAttributeClaim.get("DisplayName");
				log.trace("DisplayName: " + strDisplayName);
				final String strAttributeName = (String)mapAttributeClaim.get("AttributeName");
				log.trace("AttributeName: " + strAttributeName);
				if (null != strAttributeName)
				{
					final String strShortName = (String)mapAttributeClaim.get("ClaimName");
					log.trace("ShortName: " + strShortName);
					final org.eclipse.higgins.sts.api.IClaimType claimType = new org.eclipse.higgins.sts.common.ClaimType();
					claimType.setName(uriAttribute);
					claimType.setDisplayName(strDisplayName);
					claimType.setShortName(strShortName);
					final org.eclipse.higgins.sts.api.IClaim claim = new org.eclipse.higgins.sts.common.Claim();
					claim.setType(claimType);
					ISingleValuedAttribute attr = subj.getSingleValuedAttribute(uriAttribute);
					if(attr != null) {
						ISimpleValue sv = (ISimpleValue) attr.getValue();
						if (sv != null) {
							claim.addValue(sv.getLexical());
						}
					}
					listClaims.add(claim);
				}
			}
			else
			{
				log.error("Attribute Not Found In AttributeClaimsMap");
			}
		}
		
		context.close();
		
		return "Success";
	}

	public String modifyProfile(IConstants constants, String strUserIdentifier,
			String strUserPassword, IProfile profile) throws Exception {
		
		log.trace("ProfileService::modifyProfile");
		
		IContext context = null;
		if ((null != strUserIdentifier) && (null != strUserPassword))
		{
			try {
				context = idasRegistry.createContext(contextId);
				context.open( new AuthNNamePasswordMaterials(context, strUserIdentifier, strUserPassword));
			}
			catch(Exception e) {
				throw new Exception("ERROR Invalid Username / Password");
			}
		}
		
		if(context == null) {
			return "ERROR Invalid Username / Password";
		}
		
		IEntity subj = context.getEntity(strUserIdentifier);
		
		final List listClaims = profile.getClaims();
		int size = listClaims.size();
		log.trace("ProfileService::modifyProfile. Claims count " + size);
		try {
			for (int i = 0; i < size; ++i) {
				final IClaim claim = (IClaim) listClaims.get(i);
				final IClaimType claimType = claim.getType();
				final String strShortName = claimType.getShortName();
				Iterator iterValues = claim.getValues();
				String strValue = null;
				if (iterValues.hasNext())
					strValue = (String) iterValues.next();
				
				if (null != strValue && strValue.trim().length() > 0)
				{
					URI attributeClaimUri = (URI)this.mapShortNameClaimUri.get(strShortName);
					log.trace("ProfileService::modifyProfile: " + attributeClaimUri + "-" + strValue);
					
					IAttribute attr = subj.getSingleValuedAttribute(attributeClaimUri);
					if(attr != null) {
						attr.remove();
					}
					attr = subj.addAttribute(attributeClaimUri);
					attr.addSimpleValue(ITypedValue.STRING_TYPE_URI, strValue);
				}				
			}
			context.applyUpdates();
			context.close();
		} catch (Exception e) {
			log.error(e);
			context.cancelUpdates();
			context.close();
			throw e;
		}
		System.out.println("ProfileService::modifyProfile. Success.");
		
		return "SUCCESS";
	}

	public String modifyProfileSetCardKeyHash(IConstants constants,
			String strUserIdentifier, String strUserPassword, String strPPID,
			String strModulus, String strExponent) throws Exception {
		log.trace("ProfileService::modifyProfileSetCardKeyHash");
		
		IContext context = null;
		if ((null != strUserIdentifier) && (null != strUserPassword))
		{
			context = idasRegistry.createContext(contextId);
			context.open( new AuthNNamePasswordMaterials(context, strUserIdentifier, strUserPassword));
		}
		
		if(context == null) {
			return "ERROR Invalid Username / Password";
		}
		
		final java.io.ByteArrayOutputStream cardKey = new java.io.ByteArrayOutputStream();

		cardKey.write
			(org.apache.commons.codec.binary.Base64.decodeBase64
				(strPPID.getBytes()));	
		cardKey.write
			(org.apache.commons.codec.binary.Base64.decodeBase64
				(strModulus.getBytes()));	
		cardKey.write
			(org.apache.commons.codec.binary.Base64.decodeBase64
				(strExponent.getBytes()));
		final String strCardKeyHash = this.xmlSecurityExtension.SHA1Digest
			(cardKey.toByteArray());
		this.log.trace("CardKeyHash(in): " + strCardKeyHash);
		
		IEntity subj = context.getEntity(strUserIdentifier);
		IAttribute attr = subj.getSingleValuedAttribute(CARD_KEY_HASH_URI);
		if(attr != null) {
			attr.remove();
		}
		attr = subj.addAttribute(CARD_KEY_HASH_URI);
		attr.addSimpleValue(ITypedValue.STRING_TYPE_URI, strCardKeyHash);
		
		context.applyUpdates();
		context.close();
		
		return "Success";
	}

	public String modifyProfileSetX509SHA(IConstants constants,
			String strUserIdentifier, String strUserPassword, String strX509SHA)
			throws Exception {
		log.trace("ProfileService::modifyProfileSetX509SHA");
		
		IContext context = null;
		if ((null != strUserIdentifier) && (null != strUserPassword))
		{
			context = idasRegistry.createContext(contextId);
			context.open( new AuthNNamePasswordMaterials(context, strUserIdentifier, strUserPassword));
		}
		
		if(context == null) {
			return "ERROR Invalid Username / Password";
		}
		
		this.log.trace("X509 SHA(in): " + strX509SHA);
		
		IEntity subj = context.getEntity(strUserIdentifier);
		IAttribute attr = subj.getSingleValuedAttribute(CARD_KEY_HASH_URI);
		if(attr != null) {
			attr.remove();
		}
		attr = subj.addAttribute(CARD_KEY_HASH_URI);
		attr.addSimpleValue(ITypedValue.STRING_TYPE_URI, strX509SHA);
		
		context.applyUpdates();
		context.close();
		
		return "Success";
	}

	public void configure(Map mapGlobalSettings, String strComponentName,
			Map mapComponentSettings, ISettingDescriptor componentDescriptor,
			ISettingDescriptor globalDescriptor) throws Exception {
		
		this.log.trace("ProfileService::configure");
		
		if (this.bInitialized)
		{
			this.log.trace("ProfileService::configure Already Configured.");
			return;
		}
		
		this.mapGlobalSettings = mapGlobalSettings;
		this.strComponentName = strComponentName;
		this.mapComponentSettings = mapComponentSettings;
		
		idasRegistry  = (IdASRegistry)mapGlobalSettings.get("IdentityAttributeService");
		if (idasRegistry == null)
			throw new Exception("Can not get IdentityAttributeService setting");
		this.log.trace("IdentityAttributeService: " + idasRegistry.getClass().getName());
		
		this.storeUserName = (String) mapComponentSettings.get(AuthNNamePasswordMaterials.ATTR_TYPE_USERNAME.toString());
		this.storePassword = (String) mapComponentSettings.get(AuthNNamePasswordMaterials.ATTR_MANAGED_URI.toString());
		
		this.allowAnonymousAccess = mapComponentSettings.containsKey("AllowAnonymousAccess") ? ((Boolean) mapComponentSettings.get("AllowAnonymousAccess")).booleanValue() : false;
		
		java.io.FileInputStream fisCardImage = (java.io.FileInputStream)mapComponentSettings.get("CardImageFile");	
		byte [] byteCardImage = new byte[fisCardImage.available()];
		fisCardImage.read(byteCardImage);
		this.strCardImage = new String(org.apache.commons.codec.binary.Base64.encodeBase64(byteCardImage));
		log.trace("CardImage: " + strCardImage);
		this.certSSL = (X509Certificate)mapGlobalSettings.get("SSLCertificate");
		this.xmlSecurityExtension = (org.eclipse.higgins.sts.spi.IXMLSecurityExtension)mapGlobalSettings.get("XMLSecurityExtension");
		this.listSupportedClaims = (List)mapComponentSettings.get("SupportedClaimList");
		this.mapAttributeClaims = (Map)mapGlobalSettings.get("AttributeClaimMap");
		this.mapShortNameAttribute = new Hashtable();
		this.mapShortNameClaimUri = new Hashtable();
		
		for (int i = 0; i < this.listSupportedClaims.size(); ++i)
		{
			java.net.URI uriSupportedClaim = (java.net.URI)this.listSupportedClaims.get(i);
			Map mapClaim = (Map)this.mapAttributeClaims.get(uriSupportedClaim.toString());
			String strClaimName = (String)mapClaim.get("ClaimName");
			String strAttributeName = (String)mapClaim.get("AttributeName");
			if (null != strAttributeName)
			{
				log.trace("Adding to ShortName Attribute Map: " + uriSupportedClaim.toString() + " " + strClaimName + "/" + strAttributeName);
				this.mapShortNameAttribute.put(strClaimName, strAttributeName);
				this.mapShortNameClaimUri.put(strClaimName, uriSupportedClaim);
			}
		}
		
		if (mapComponentSettings.containsKey("ContextId"))
			contextId = (String)mapComponentSettings.get("ContextId");
		else
			throw new Exception("Cannot get \"ContextId\" from ProfileService setting.");
		
		if (mapComponentSettings.containsKey("DigitalSubjectType"))
			digitalSubjectType = (URI) mapComponentSettings.get("DigitalSubjectType");
		else
			throw new Exception("Cannot get \"DigitalSubjectType\" ProfileService setting.");
		
		this.bInitialized = true;
		
		this.log.trace("ProfileService::configure DONE");
	}

	public ISettingDescriptor getComponentDescriptor() {
		return null;
	}

}
