/** 
 * Copyright (c) 2009 Parity Communications, Inc.  
 * All rights reserved. This program and the accompanying materials  
 * are made available under the terms of the Eclipse Public License v1.0  
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 *  
 * Contributors: 
 * Alexander Yuhimenko - implementation 
 * 
 */

package org.eclipse.higgins.sync.bean;

import java.io.ByteArrayInputStream;
import java.io.UnsupportedEncodingException;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.security.auth.callback.CallbackHandler;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.configuration.xml.ConfigurationHandler;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.ICardProvider;
import org.eclipse.higgins.icard.IInformationCard;
import org.eclipse.higgins.icard.common.auth.PasswordCredential;
import org.eclipse.higgins.icard.registry.ICardRegistry;
import org.eclipse.higgins.sync.auth.IAuthenticateService;
import org.eclipse.higgins.sync.auth.entity.AccessTokenEntity;
import org.eclipse.higgins.sync.auth.exceptions.AuthenticationException;
import org.eclipse.higgins.sync.command.DeleteCardCommand;
import org.eclipse.higgins.sync.command.GetCardCommand;
import org.eclipse.higgins.sync.command.GetUserProfileCommand;
import org.eclipse.higgins.sync.command.IResourceCommand;
import org.eclipse.higgins.sync.command.PersistCardCommand;
import org.eclipse.higgins.sync.command.PersistUserProfileCommand;
import org.eclipse.higgins.sync.meta.IResourceMetaDataService;
import org.eclipse.higgins.sync.meta.entity.ResourceRevisionEntity;
import org.eclipse.higgins.sync.meta.entity.RevisionEntity;
import org.eclipse.higgins.sync.meta.entity.RootRevisionEntity;
import org.eclipse.higgins.sync.meta.exceptions.ResourceMetaDataException;
import org.eclipse.higgins.sync.meta.utilities.ResourceUtil;
import org.eclipse.higgins.sync.to.AccessTokenTO;
import org.eclipse.higgins.sync.to.AuthCredentialTO;
import org.eclipse.higgins.sync.to.BaseTO;
import org.eclipse.higgins.sync.to.CardTO;
import org.eclipse.higgins.sync.to.CmdExecStatusTO;
import org.eclipse.higgins.sync.to.CmdExecStatusesTO;
import org.eclipse.higgins.sync.to.CommandTO;
import org.eclipse.higgins.sync.to.CommandsTO;
import org.eclipse.higgins.sync.to.CrdsFileTO;
import org.eclipse.higgins.sync.to.FileTO;
import org.eclipse.higgins.sync.to.RevisionTO;
import org.eclipse.higgins.sync.to.SelectorAuthCredentialTO;
import org.eclipse.higgins.sync.to.UserProfileTO;
import org.eclipse.higgins.sync.to.UsernamePasswordAuthCredentialTO;
import org.eclipse.higgins.sync.utilities.ConverterHelper;
import org.eclipse.higgins.sync.utilities.SecureUtil;
import org.eclipse.higgins.user.account.Account;
import org.eclipse.higgins.user.account.IUserAccountService;
import org.eclipse.higgins.user.account.UserAccountServiceFactory;
import org.eclipse.higgins.user.account.exception.UserAccountException;
import org.eclipse.higgins.user.account.exception.UserAlreadyExistsException;
import org.eclipse.higgins.user.account.utils.PasswordUtils;
import org.eclipse.higgins.user.login.LoginService;
import org.eclipse.higgins.user.login.UserAccount;
import org.eclipse.higgins.user.login.exception.ConfigurationException;
import org.eclipse.higgins.user.login.exception.LoginException;
import org.eclipse.higgins.user.profile.IUserProfileService;
import org.eclipse.higgins.user.profile.UserProfileServiceFactory;
import org.eclipse.higgins.user.profile.entity.SelectorClient;
import org.eclipse.higgins.user.profile.entity.UserProfile;
import org.eclipse.higgins.user.profile.exception.UserProfileException;

/**
 * CardSync Bean Facade.
 * 
 * @author Alexander Yuhimenko
 * 
 */
public class CardSyncFacadeBean {

	/**
	 * 
	 */
	// private static final String PERSONAL_CP =
	// "org.eclipse.higgins.icard.provider.cardspace.personal.db";

	private static Log LOG = LogFactory.getLog(CardSyncFacadeBean.class);

	private static Integer maxIdleTime = new Integer(60 * 30);// default 30 min
	private static Integer maxLiveTime = new Integer(60 * 60 * 24); // default
	// 24h

	/**
	 * Instance of IUserProfileService .
	 */
	public static final ThreadLocal<IUserProfileService> userProfileService = new ThreadLocal<IUserProfileService>() {
		protected synchronized IUserProfileService initialValue() {
			IUserProfileService userProfileServiceInstance = null;
			try {
				ConfigurationHandler configurationHandler = new ConfigurationHandler();
				configurationHandler.setConfigurationBase((String) SecureUtil.getContext().lookup(
						"HigginsConfigurationBase"));
				configurationHandler.setFileName((String) SecureUtil.getContext().lookup("HigginsConfigurationFile"));

				if (!configurationHandler.configure(null)) {
					throw new ConfigurationException("Could not configure");
				}
				userProfileServiceInstance = UserProfileServiceFactory.getService(configurationHandler.getSettings());
			} catch (Exception e) {
				LOG.error(e, e);
			}
			return userProfileServiceInstance;
		}
	};

	/**
	 * Instance of LoginService..
	 */
	public static final ThreadLocal<LoginService> loginService = new ThreadLocal<LoginService>() {
		protected synchronized LoginService initialValue() {
			LoginService loginServiceInstance = null;
			try {
				ConfigurationHandler configurationHandler = new ConfigurationHandler();
				configurationHandler.setConfigurationBase((String) SecureUtil.getContext().lookup(
						"HigginsConfigurationBase"));
				configurationHandler.setFileName((String) SecureUtil.getContext().lookup("HigginsConfigurationFile"));

				if (!configurationHandler.configure(null)) {
					throw new ConfigurationException("Could not configure");
				}
				loginServiceInstance = LoginService.getInstance(configurationHandler.getSettings());
			} catch (Exception e) {
				LOG.error(e, e);
			}
			return loginServiceInstance;
		}
	};

	public static final ThreadLocal<IResourceMetaDataService> metaDataService = new ThreadLocal<IResourceMetaDataService>() {
		protected synchronized IResourceMetaDataService initialValue() {
			IResourceMetaDataService metaDataServiceInstance = null;
			try {
				metaDataServiceInstance = (IResourceMetaDataService) SecureUtil.getContext().lookup(
						"cardsync/ResourceMetaDataService");
			} catch (Exception e) {
				LOG.error(e, e);
			}
			return metaDataServiceInstance;
		}
	};

	/**
	 * Instance of IUserAccountService.
	 */
	public static final ThreadLocal<IUserAccountService> userAccountService = new ThreadLocal<IUserAccountService>() {
		protected synchronized IUserAccountService initialValue() {
			IUserAccountService userAccountServiceInstance = null;
			try {
				ConfigurationHandler configurationHandler = new ConfigurationHandler();
				configurationHandler.setConfigurationBase((String) SecureUtil.getContext().lookup(
						"HigginsConfigurationBase"));
				configurationHandler.setFileName((String) SecureUtil.getContext().lookup("HigginsConfigurationFile"));

				if (!configurationHandler.configure(null)) {
					throw new ConfigurationException("Could not configure");
				}
				userAccountServiceInstance = UserAccountServiceFactory.getService(configurationHandler.getSettings());
			} catch (Exception e) {
				LOG.error(e, e);
			}
			return userAccountServiceInstance;
		}
	};

	public static final ThreadLocal<IAuthenticateService> authenticateService = new ThreadLocal<IAuthenticateService>() {
		protected synchronized IAuthenticateService initialValue() {
			IAuthenticateService authServiceInstance = null;
			try {
				authServiceInstance = (IAuthenticateService) SecureUtil.getContext().lookup(
						"cardsync/AuthenticateService");
			} catch (Exception e) {
				LOG.error(e, e);
			}
			return authServiceInstance;
		}
	};

	public static final ThreadLocal<Map<String, IResourceCommand>> commandMap = new ThreadLocal<Map<String, IResourceCommand>>() {
		protected synchronized Map<String, IResourceCommand> initialValue() {
			Map<String, IResourceCommand> cmdMap = new HashMap<String, IResourceCommand>();
			try {
				// Card
				cmdMap.put(GetCardCommand.COMMAND_ID, new GetCardCommand(metaDataService.get()));
				cmdMap.put(PersistCardCommand.COMMAND_ID, new PersistCardCommand(metaDataService.get()));
				cmdMap.put(DeleteCardCommand.COMMAND_ID, new DeleteCardCommand(metaDataService.get()));

				// UserProfile
				cmdMap.put(GetUserProfileCommand.COMMAND_ID, new GetUserProfileCommand(metaDataService.get(),
						userProfileService.get(), userAccountService.get()));
				cmdMap.put(PersistUserProfileCommand.COMMAND_ID, new PersistUserProfileCommand(metaDataService.get(),
						userProfileService.get(), userAccountService.get()));

			} catch (Exception e) {
				LOG.error(e, e);
			}
			return cmdMap;
		}
	};

	/**
	 * Create new UserProfile.
	 * 
	 * @param userProfileTO
	 * @param authCredentialTO
	 * @return
	 * @throws ResourceMetaDataException
	 * @throws UserProfileException
	 * @throws UserAccountException
	 * @throws UnsupportedEncodingException
	 * @throws NoSuchAlgorithmException
	 */
	// TODO RppsService has the same method(addUserProfileBase), it should be
	// moved to Utility.
	public UserProfileTO createUserProfile(UserProfileTO userProfileTO) throws ResourceMetaDataException,
			UserProfileException, NoSuchAlgorithmException, UnsupportedEncodingException, UserAccountException {

		if (!((IUserAccountService) userAccountService.get()).existsAccount(userProfileTO.getLoginName())) {
			// register account
			// FIXME shouldn't cast to UsernamePasswordAuthCredentialTO
			Account account = ((IUserAccountService) userAccountService.get()).createAccount(null, new Account()
					.setCreated(new Date()).setLogin(userProfileTO.getLoginName()).setPasswordHash(
							PasswordUtils.getPasswordHash(((UsernamePasswordAuthCredentialTO) userProfileTO
									.getAuthCredentialTO()).getPassword())));
			// add user profile
			UserProfile userProfile = new UserProfile();
			ConverterHelper.userProfileTO2UserProfile(userProfileTO, userProfile);
			userProfile.setUserIdentifier(account.getId());

			UserProfile up = ((IUserProfileService) userProfileService.get()).addUserProfile(account.getId(),
					userProfile);

			// try {
			// new EmailUtils().sendWelcomeEmail(userProfile.getEmail(),
			// userProfile.getLoginName());
			// } catch (Exception e) {
			// LOG.error(e);
			// }

			// update revision
			// persist RootRevisionEntity
			ResourceUtil resourceUtil = new ResourceUtil();
			RootRevisionEntity rootRevisionEntity = resourceUtil.persistRoorRevision(metaDataService.get(), up
					.getUserIdentifier());

			// persistRevisionEntity
			ResourceRevisionEntity revisionEntity = resourceUtil.persistResourceRevision(metaDataService.get(), up
					.getUserIdentifier(), IResourceMetaDataService.USER_PROFILE_RESOURCE_TYPE, (String) up
					.getUserIdentifier(), rootRevisionEntity.getId());

			return ConverterHelper.userProfile2UserProfileTO(up, account, revisionEntity);
		} else {
			throw new UserAlreadyExistsException();
		}
	}

	/**
	 * @param id
	 * @see org.eclipse.higgins.sync.auth.IAuthenticateService#deleteAccessToken(java.lang.String)
	 */
	public void deleteAccessToken(String id) {
		authenticateService.get().deleteAccessToken(id);
	}

	// /**
	// * Authenticate user by Access Token Id.
	// *
	// * @param accessTokenId
	// * @return
	// * @throws ConfigurationException
	// * @throws AuthenticationException
	// */
	// public UserAccount authenticate(final String accessTokenId) throws
	// ConfigurationException,
	// AuthenticationException {
	// return new UserAccount().setUserId(((IAuthenticateService)
	// authenticateService.get()).getAccessToken(
	// accessTokenId).getUserIdentifier());
	// }

	/**
	 * Authenticate user by AuthCredentialTO.
	 * 
	 * @return
	 * @throws ConfigurationException
	 * @throws LoginException
	 * @throws AuthenticationException
	 * @throws UserProfileException
	 */
	// TODO RppsService has the same method, it should be moved to Utility.
	public UserAccount authenticate(AuthCredentialTO authCredentialTO) throws ConfigurationException, LoginException,
			AuthenticationException, UserProfileException {
		if (authCredentialTO instanceof SelectorAuthCredentialTO) {
			SelectorAuthCredentialTO auth = (SelectorAuthCredentialTO) authCredentialTO;
			UserAccount account = ((LoginService) loginService.get()).login(SecureUtil.getCallbackHandler(auth
					.getUsername(), auth.getPassword()));
			// check selector state
			List selectorClients = ((IUserProfileService) userProfileService.get()).getSelectorClients(account
					.getUserId());
			SelectorClient selector = new SelectorClient().setSerialNumber(auth.getSelectorSerialNumber());
			if (null != selectorClients && selectorClients.contains(selector)) {// find
				// selector
				// client
				// by
				// selector
				// serial number
				selector = (SelectorClient) selectorClients.get(selectorClients.indexOf(selector));
				if (SelectorClient.STATE_ACTIVE.equals(selector.getState())) {// check
					// selector
					// state
					return account;
				} else {
					throw new AuthenticationException("Selector was locked.");
				}
			}
			throw new AuthenticationException("Couldn't find selector client.");
		} else
			if (authCredentialTO instanceof UsernamePasswordAuthCredentialTO) {
				UsernamePasswordAuthCredentialTO auth = (UsernamePasswordAuthCredentialTO) authCredentialTO;
				return ((LoginService) loginService.get()).login(SecureUtil.getCallbackHandler(auth.getUsername(), auth
						.getPassword()));
			} else {
				throw new AuthenticationException("Unsupported  AuthCredentialTO type");
			}
	}

	/**
	 * Authenticate user and build AccessTokenEntity.
	 * 
	 * @param authCredentialTO
	 * @return AccessTokenTO
	 * @throws AuthenticationException
	 * @throws UserProfileException
	 * @throws LoginException
	 * @throws ConfigurationException
	 */
	public AccessTokenTO createAccessToken(AuthCredentialTO authCredentialTO) throws AuthenticationException,
			ConfigurationException, LoginException, UserProfileException {
		UserAccount account = authenticate(authCredentialTO);

		AccessTokenEntity accessToken = ((IAuthenticateService) authenticateService.get())
				.persistAccessToken(new AccessTokenEntity().setUserIdentifier(account.getUserId().toString()));

		AccessTokenTO accessTokenTO = new AccessTokenTO();
		accessTokenTO.setId(accessToken.getId());
		accessTokenTO.setIssuedTime(accessToken.getIssuedTime());
		accessTokenTO.setMaxIdleTime(maxIdleTime);
		accessTokenTO.setMaxLiveTime(maxLiveTime);

		// TODO remove it in future :)
		// Persist RootRevisionEntity. it needs for supporting old account.
		RootRevisionEntity rootRevisionEntity = new RootRevisionEntity();
		rootRevisionEntity.setResourceType(IResourceMetaDataService.ROOT_RESOURCE_TYPE).setResourceId(
				accessToken.getUserIdentifier()).setUserId(accessToken.getUserIdentifier());
		try {
			if (null == metaDataService.get().findResourceRevisionEntity(rootRevisionEntity)) {
				metaDataService.get().persistResourceRevisionEntity(rootRevisionEntity);
			}
		} catch (ResourceMetaDataException e) {
			LOG.error(e, e);
		}
		return accessTokenTO;
		//ConverterHelper.accessToken2AccessTokenTO(accessToken);
	}

	/**
	 *Return AccessTokenEntity by Access Token Id.
	 * 
	 * @param accessTokenId
	 * @return
	 * @throws AuthenticationException
	 * @see org.eclipse.higgins.sync.auth.IAuthenticateService#getAccessToken(java.lang.String)
	 */
	public AccessTokenEntity getAccessToken(String accessTokenId) throws AuthenticationException {
		if (null != accessTokenId && accessTokenId.trim().length() > 0
				&& accessTokenId.trim().startsWith(IAuthenticateService.AUTHENTICATE_HEADER_PREFIX)) {
			accessTokenId = accessTokenId.trim().substring(IAuthenticateService.AUTHENTICATE_HEADER_PREFIX.length());
		}

		return authenticateService.get().getAccessToken(accessTokenId);
	}

	/**
	 * Return Root resource revision.
	 * 
	 * @param accessToken
	 * @param resourceTypeList
	 * @return
	 * @throws AuthenticationException
	 * @throws ResourceMetaDataException
	 */
	public RevisionTO getRootRevision(AccessTokenEntity accessToken, List<String> resourceTypeList)
			throws AuthenticationException, ResourceMetaDataException {

		RevisionEntity res = null;
		if (null == resourceTypeList || resourceTypeList.size() == 0) {
			// if isn't specified add ROOT_RESOURCE_TYPE
			resourceTypeList = new ArrayList<String>() {
				{
					add(IResourceMetaDataService.ROOT_RESOURCE_TYPE);
				}
			};
		}

		// find root revision
		RevisionEntity revisionEntity;
		for (String resourceType : resourceTypeList) {

			revisionEntity = metaDataService.get().findMaxRevisionEntity(
					new RevisionEntity().setUserId(accessToken.getUserIdentifier()).setResourceType(resourceType));

			if (null != revisionEntity) {
				if (null == res || res.getNumber() < revisionEntity.getNumber()) {
					res = revisionEntity;
				}
			}
		}

		return ConverterHelper.revisionEntity2RevisionTO(res);
	}

	/**
	 * Return UserProfile by user identifier.
	 * 
	 * @param accessTokenEntity
	 * @return
	 * @throws ResourceMetaDataException
	 * @throws UserProfileException
	 * @throws UserAccountException
	 */
	public UserProfileTO getUserProfile(AccessTokenEntity accessTokenEntity) throws ResourceMetaDataException,
			UserProfileException, UserAccountException {

		final UserProfile userProfile = ((IUserProfileService) userProfileService.get())
				.getUserProfile(accessTokenEntity.getUserIdentifier());

		// load RevisionEntity
		RevisionEntity revisionEntity = new RevisionEntity().setResourceType(
				IResourceMetaDataService.USER_PROFILE_RESOURCE_TYPE).setResourceId(
				accessTokenEntity.getUserIdentifier()).setUserId(accessTokenEntity.getUserIdentifier());
		revisionEntity = metaDataService.get().findResourceRevisionEntity(revisionEntity);

		return ConverterHelper.userProfile2UserProfileTO(userProfile, userAccountService.get().getAccount(
				accessTokenEntity.getUserIdentifier()), revisionEntity);
	}

	/**
	 * Delete UserProfile.
	 * 
	 * @param accessTokenEntity
	 * @return
	 * @throws ResourceMetaDataException
	 * @throws UserProfileException
	 * @throws UserAccountException
	 */
	public void deleteUserProfile(AccessTokenEntity accessTokenEntity) throws ResourceMetaDataException,
			UserProfileException, UserAccountException {

		// deleteAlICard(userAccount);
		for (final Iterator<ICardProvider> iter = ICardRegistry.getInstance().getICardProviders(); iter.hasNext();) {
			ICardProvider provider = iter.next();
			try {
				CallbackHandler callbackHandler = SecureUtil.getCallbackHandler(accessTokenEntity);
				for (final Iterator<IInformationCard> icards = provider.getICards(callbackHandler); icards.hasNext();) {
					try {
						deleteCard(accessTokenEntity, icards.next().getCUID().toString());
					} catch (Exception e) {
						LOG.error(e, e);
					}
				}
			} catch (final Exception e) {
				LOG.error(e, e);
			}
		}

		// delete UserProfile
		((IUserProfileService) userProfileService.get()).deleteUserProfile(accessTokenEntity.getUserIdentifier());
		((IUserAccountService) userAccountService.get()).deleteAccount(accessTokenEntity.getUserIdentifier());

		// delete user profile revision
		RevisionEntity revisionEntity = new RevisionEntity().setResourceType(UserProfileTO.class.getCanonicalName())
				.setResourceId(accessTokenEntity.getUserIdentifier()).setUserId(accessTokenEntity.getUserIdentifier());
		metaDataService.get().deleteResourceRevisionEntity(revisionEntity);

		// delete root revision
		revisionEntity = new RevisionEntity().setResourceType(IResourceMetaDataService.ROOT_RESOURCE_TYPE)
				.setResourceId(accessTokenEntity.getUserIdentifier()).setUserId(accessTokenEntity.getUserIdentifier());
		metaDataService.get().deleteResourceRevisionEntity(revisionEntity);

	}

	/**
	 * Update UserProfile.
	 * 
	 * @param accessTokenEntity
	 * @param userProfileTO
	 * @return
	 * @throws ResourceMetaDataException
	 * @throws UserProfileException
	 * @throws UserAccountException
	 */
	public UserProfileTO updateUserProfile(AccessTokenEntity accessTokenEntity, UserProfileTO userProfileTO)
			throws ResourceMetaDataException, UserProfileException, UserAccountException {
		// TODO add validation revision nummber.

		UserProfile userProfile = new UserProfile().setUserIdentifier(accessTokenEntity.getUserIdentifier());
		ConverterHelper.userProfileTO2UserProfile(userProfileTO, userProfile);
		((IUserProfileService) userProfileService.get()).modifyUserProfile(accessTokenEntity.getUserIdentifier(),
				userProfile);

		// find root (parent) revision entity
		RevisionEntity rootRevisionEntity = new ResourceUtil().findRootRevision(metaDataService.get(),
				accessTokenEntity.getUserIdentifier());

		// update RevisionEntity
		ResourceRevisionEntity revisionEntity = new ResourceRevisionEntity();
		revisionEntity.setResourceType(UserProfileTO.class.getCanonicalName()).setResourceId(
				accessTokenEntity.getUserIdentifier()).setUserId(accessTokenEntity.getUserIdentifier()).setParentId(
				rootRevisionEntity.getId());
		metaDataService.get().persistResourceRevisionEntity(revisionEntity);

		return ConverterHelper.userProfile2UserProfileTO(userProfile, userAccountService.get().getAccount(
				accessTokenEntity.getUserIdentifier()), revisionEntity);
	}

	public List<CardTO> getCards(AccessTokenEntity accessTokenEntity) throws ResourceMetaDataException, CardException {

		final List<CardTO> cards = new ArrayList<CardTO>();

		for (final Iterator<ICardProvider> iter = ICardRegistry.getInstance().getICardProviders(); iter.hasNext();) {
			ICardProvider provider = iter.next();
			// LOG.trace("Processes icard probvider : " + provider.getID());//
			// org.eclipse.higgins.icard.provider.cardspace.personal.db

			RevisionEntity revisionEntity;
			CardTO cardTO;
			try {
				CallbackHandler callbackHandler = SecureUtil.getCallbackHandler(accessTokenEntity);
				for (final Iterator<IInformationCard> icards = provider.getICards(callbackHandler); icards.hasNext();) {
					cardTO = ConverterHelper.card2CardTO(icards.next());

					try {
						revisionEntity = new RevisionEntity().setResourceType(
								IResourceMetaDataService.CARD_RESOURCE_TYPE).setResourceId(cardTO.getId()).setUserId(
								accessTokenEntity.getUserIdentifier());
						cardTO.setRevision(ConverterHelper.revisionEntity2RevisionTO(metaDataService.get()
								.findResourceRevisionEntity(revisionEntity)));
					} catch (Exception e) {
						LOG.error(e, e);
					}

					cards.add(cardTO);
				}

			} catch (final Exception e) {
				LOG.error(e, e);
			}
		}
		// try {
		// CallbackHandler callbackHandler =
		// SecureUtil.getCallbackHandler(accessTokenEntity);
		// ICardProvider provider =
		// ICardRegistry.getInstance().getICardProvider(PERSONAL_CP);
		// for (final Iterator<IInformationCard> icards =
		// provider.getICards(callbackHandler); icards.hasNext();) {
		// CardTO cardTO = ConverterHelper.card2CardTO(icards.next());
		// RevisionEntity revisionEntity = new RevisionEntity().setResourceType(
		// cardTO.getClass().getCanonicalName()).setResourceId(cardTO.getId()).setUserId(
		// accessTokenEntity.getUserIdentifier());
		// cardTO.setRevision(ConverterHelper.revisionEntity2RevisionTO(metaDataService.get()
		// .findResourceRevisionEntity(revisionEntity)));
		// cards.add(cardTO);
		// }
		// } catch (final Exception e) {
		// LOG.error(e, e);
		// }
		return cards;
	}

	/**
	 * Returns Card by Id.
	 * 
	 * @param accessTokenEntity
	 * @param id
	 * @return
	 * @throws ResourceMetaDataException
	 * @throws CardException
	 */
	public CardTO getCard(AccessTokenEntity accessTokenEntity, String id) throws ResourceMetaDataException,
			CardException {
		IResourceCommand cmd = commandMap.get().get(
				IResourceCommand.GET_CMD_NAME + IResourceMetaDataService.CARD_RESOURCE_TYPE);

		if (null == cmd) {
			throw new CardException("Unsupported command : " + IResourceCommand.GET_CMD_NAME
					+ IResourceMetaDataService.CARD_RESOURCE_TYPE);
		}

		CmdExecStatusTO execCmd = cmd.execCmd(accessTokenEntity, new CommandTO().setName(IResourceCommand.GET_CMD_NAME)
				.setResourceId(id).setResourceType(IResourceMetaDataService.CARD_RESOURCE_TYPE));
		// check status command
		if ("0".equals(execCmd.getStatusCode())) {
			return (CardTO) execCmd.getResource();
		} else {
			throw new NullPointerException("Couldn't get card due to :" + execCmd.getStatusMessage());
		}
	}

	/**
	 * Create new P/M Card.
	 * 
	 * @param accessTokenEntity
	 * @param cardTO
	 * @return
	 * @throws ResourceMetaDataException
	 * @throws CardException
	 */
	public CardTO persistCard(AccessTokenEntity accessTokenEntity, CardTO cardTO) throws ResourceMetaDataException,
			CardException {
		IResourceCommand cmd = commandMap.get().get(
				IResourceCommand.PERSIST_CMD_NAME + IResourceMetaDataService.CARD_RESOURCE_TYPE);

		if (null == cmd) {
			throw new CardException("Unsupported command : " + IResourceCommand.PERSIST_CMD_NAME
					+ IResourceMetaDataService.CARD_RESOURCE_TYPE);
		}

		CmdExecStatusTO execCmd = cmd.execCmd(accessTokenEntity, new CommandTO().setName(
				IResourceCommand.PERSIST_CMD_NAME).setResource(cardTO).setResourceType(
				IResourceMetaDataService.CARD_RESOURCE_TYPE));
		// check status command
		if ("0".equals(execCmd.getStatusCode())) {
			return (CardTO) execCmd.getResource();
		} else {
			throw new NullPointerException("Couldn't get card due to :" + execCmd.getStatusMessage());
		}
	}

	/**
	 * @param accessToken
	 * @param id
	 */
	public void deleteCard(AccessTokenEntity accessTokenEntity, String id) {
		try {

			IResourceCommand cmd = commandMap.get().get(
					IResourceCommand.DELETE_CMD_NAME + IResourceMetaDataService.CARD_RESOURCE_TYPE);

			if (null != cmd) {

				CmdExecStatusTO execCmd = cmd.execCmd(accessTokenEntity, new CommandTO().setName(
						IResourceCommand.DELETE_CMD_NAME).setResourceId(id).setResourceType(
						IResourceMetaDataService.CARD_RESOURCE_TYPE));
			}
		} catch (Exception e) {
			LOG.error(e, e);

		}
	}

	// /**
	// * Update ICard.
	// *
	// * @param accessToken
	// * @param cardTO
	// * @return
	// * @throws CardException
	// * @throws ResourceMetaDataException
	// * @throws
	// *
	// *
	// * {@link Deprecated} use {@link this#persistCard(AccessTokenEntity, CardTO) instead}
	// */
	// public CardTO updateCard(AccessTokenEntity accessTokenEntity, CardTO cardTO) throws CardException,
	// ResourceMetaDataException {
	// return persistCard(accessTokenEntity, cardTO);
	// }

	/**
	 * Import crd or crds file.
	 * 
	 * @param accessTokenEntity
	 * @param crd
	 *            - binary file.
	 * @param password
	 *            for crds file.
	 * @return CardsTO
	 * @throws CardException
	 * @throws ResourceMetaDataException
	 */
	public List<CardTO> importFromCrd(AccessTokenEntity accessTokenEntity, FileTO fileTO) throws CardException,
			ResourceMetaDataException {

		final List<CardTO> cards = new ArrayList<CardTO>();

		PasswordCredential credential = null;
		if (fileTO instanceof CrdsFileTO) {
			credential = new PasswordCredential();
			credential.setPassword(((CrdsFileTO) fileTO).getPasswordPhrase());
		}

		// find root (parent) revision entity
		RevisionEntity rootRevisionEntity = new ResourceUtil().findRootRevision(metaDataService.get(),
				accessTokenEntity.getUserIdentifier());

		// define RevisionEntity
		ResourceRevisionEntity revisionEntity = new ResourceRevisionEntity();

		for (Iterator<IInformationCard> icards = ICardRegistry.getInstance().importICards(
				SecureUtil.getCallbackHandler(accessTokenEntity), new ByteArrayInputStream(fileTO.getFileData()), null,
				credential); icards.hasNext();) {

			CardTO cardTO = ConverterHelper.card2CardTO(icards.next());
			// persist RevisionEntity
			revisionEntity.setResourceType(IResourceMetaDataService.CARD_RESOURCE_TYPE).setResourceId(cardTO.getId())
					.setUserId(accessTokenEntity.getUserIdentifier()).setParentId(rootRevisionEntity.getId());

			metaDataService.get().persistResourceRevisionEntity(revisionEntity);

			cardTO.setRevision(ConverterHelper.revisionEntity2RevisionTO(revisionEntity));

			cards.add(cardTO);
		}

		return cards;
	}

	/**
	 * Retrieve changes between client RootRevisionTO and server RootRevisionTO.
	 * 
	 * @param accessTokenEntity
	 * @param clientRootRevisionTO
	 * @param resourceTypeList
	 *            if null return for all resources.
	 * @return
	 * @throws ResourceMetaDataException
	 * @throws AuthenticationException
	 */
	public CommandsTO getCommandLog(AccessTokenEntity accessTokenEntity, RevisionTO clientRootRevisionTO,
			List<String> resourceTypeList) throws ResourceMetaDataException, AuthenticationException {
		ArrayList<CommandTO> res = new ArrayList<CommandTO>();

		if (null == resourceTypeList || resourceTypeList.size() == 0) {

			resourceTypeList = new ArrayList<String>() {
				{
					add(null); // according to IResourceMetaDataService, it means all resources
				}
			};
		}

		List<RevisionEntity> revList;

		for (String resourceType : resourceTypeList) {

			revList = metaDataService.get().getResourceRevisionEntityList(
					ConverterHelper.revisionTO2revisionEntity(clientRootRevisionTO).setUserId(
							accessTokenEntity.getUserIdentifier()), resourceType);

			int i = 0;
			for (RevisionEntity revisionEntity : revList) {
				try {
					BaseTO resource = null;
					// load resource if command is not delete
					if (!IResourceCommand.DELETE_CMD_NAME.equals(revisionEntity.getCommandName())) {

						IResourceCommand cmd = commandMap.get().get(
								IResourceCommand.GET_CMD_NAME + revisionEntity.getResourceType());

						if (null != cmd) {
							CmdExecStatusTO execCmd = cmd.execCmd(accessTokenEntity, new CommandTO().setName(
									IResourceCommand.GET_CMD_NAME).setResourceId(revisionEntity.getResourceId())
									.setResourceType(revisionEntity.getResourceType()));

							// check status command
							if ("0".equals(execCmd.getStatusCode())) {
								resource = execCmd.getResource();
							} else {
								throw new NullPointerException("Couldn't get resource due to :"
										+ execCmd.getStatusMessage());
							}

						} else {
							throw new NullPointerException("Unsupported command : " + IResourceCommand.GET_CMD_NAME
									+ revisionEntity.getResourceType());

						}
					}

					res.add(new CommandTO().setName(revisionEntity.getCommandName()).setResourceId(
							revisionEntity.getResourceId()).setResourceType(revisionEntity.getResourceType()).setId(
							"" + i++).setResource(resource));

				} catch (Exception e) {
					LOG.error(e, e);
				}
			}

		}
		CommandsTO commandTOs = new CommandsTO().setCommandTO(res.toArray(new CommandTO[res.size()]));
		commandTOs.setRevision(getRootRevision(accessTokenEntity, resourceTypeList));

		return commandTOs;
	}

	/**
	 * Retrieve changes between client RootRevisionTO and server RootRevisionTO.
	 * 
	 * @param accessTokenEntity
	 * @param clientRootRevisionTO
	 * @return list of CommandTO
	 * @throws ResourceMetaDataException
	 * @throws AuthenticationException
	 */
	public CmdExecStatusesTO execCommands(AccessTokenEntity accessTokenEntity, CommandsTO commandsTO,
			List<String> resourceTypeList) throws ResourceMetaDataException, AuthenticationException {
		List<CmdExecStatusTO> res = new ArrayList<CmdExecStatusTO>();
		CommandTO[] commandTOs = commandsTO.getCommandTO();
		for (int i = 0; i < commandTOs.length; i++) {
			try {
				IResourceCommand cmd = commandMap.get().get(commandTOs[i].getName() + commandTOs[i].getResourceType());
				if (null != cmd) {
					res.add(cmd.execCmd(accessTokenEntity, commandTOs[i]));
				}
			} catch (Exception e) {
				res.add(new CmdExecStatusTO().setId(commandTOs[i].getId()).setStatusCode("-1").setStatusMessage(
						e.getMessage()));
			}
		}
		CmdExecStatusesTO cmdExecStatusesTO = new CmdExecStatusesTO();
		cmdExecStatusesTO.setCmdExecStatusTO(res.toArray(new CmdExecStatusTO[res.size()]));
		cmdExecStatusesTO.setRevision(getRootRevision(accessTokenEntity, resourceTypeList));

		return cmdExecStatusesTO;
	}
}
