/** 
 * Copyright (c) 2009 Parity Communications, Inc.  
 * All rights reserved. This program and the accompanying materials  
 * are made available under the terms of the Eclipse Public License v1.0  
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 *  
 * Contributors: 
 * Alexander Yuhimenko - implementation 
 * 
 */

package org.eclipse.higgins.sync.utilities;

import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.IClaim;
import org.eclipse.higgins.icard.IClaimType;
import org.eclipse.higgins.icard.IEndpointReference;
import org.eclipse.higgins.icard.IInformationCard;
import org.eclipse.higgins.icard.IInformationCardExtension;
import org.eclipse.higgins.icard.IManagedInformationCard;
import org.eclipse.higgins.icard.IPersonalInformationCard;
import org.eclipse.higgins.icard.IUIDescriptor;
import org.eclipse.higgins.icard.auth.ICredentialDescriptor;
import org.eclipse.higgins.icard.common.ClaimType;
import org.eclipse.higgins.icard.common.io.IOElement;
import org.eclipse.higgins.icard.common.utils.XMLHelper;
import org.eclipse.higgins.icard.policy.ISTSPrivacyPolicy;
import org.eclipse.higgins.icard.provider.cardspace.common.CredentialDescriptor;
import org.eclipse.higgins.icard.provider.cardspace.common.EndpointReference;
import org.eclipse.higgins.icard.provider.cardspace.common.InformationCard;
import org.eclipse.higgins.icard.provider.cardspace.common.InformationCardExtension;
import org.eclipse.higgins.icard.provider.cardspace.common.MCardEntity;
import org.eclipse.higgins.icard.provider.cardspace.common.PCardEntity;
import org.eclipse.higgins.icard.provider.cardspace.common.STSPrivacyPolicy;
import org.eclipse.higgins.icard.provider.cardspace.common.TokenService;
import org.eclipse.higgins.sync.auth.entity.AccessTokenEntity;
import org.eclipse.higgins.sync.meta.entity.RevisionEntity;
import org.eclipse.higgins.sync.to.AccessTokenTO;
import org.eclipse.higgins.sync.to.CardExtensionTO;
import org.eclipse.higgins.sync.to.CardTO;
import org.eclipse.higgins.sync.to.ClaimTO;
import org.eclipse.higgins.sync.to.ClaimTypeTO;
import org.eclipse.higgins.sync.to.ClaimUiDescriptorTO;
import org.eclipse.higgins.sync.to.CredentialDescriptorTO;
import org.eclipse.higgins.sync.to.EndpointReferenceTO;
import org.eclipse.higgins.sync.to.MCardTO;
import org.eclipse.higgins.sync.to.PCardTO;
import org.eclipse.higgins.sync.to.RevisionTO;
import org.eclipse.higgins.sync.to.StsPrivacyPolicyTO;
import org.eclipse.higgins.sync.to.TokenServiceTO;
import org.eclipse.higgins.sync.to.UserProfileTO;
import org.eclipse.higgins.user.account.Account;
import org.eclipse.higgins.user.profile.entity.UserProfile;

/**
 * Convert entities to transfer objects and vice-versa.
 * 
 * @author Alexander Yuhimenko
 * 
 */
public class ConverterHelper {

	private static Log LOG = LogFactory.getLog(ConverterHelper.class);

	/**
	 * Convert AccessTokenEntity to AccessTokenTO
	 * 
	 * @param entity
	 *            AccessTokenEntity
	 * @return AccessTokenTO
	 */
//	public static AccessTokenTO accessToken2AccessTokenTO(AccessTokenEntity entity) {
//		AccessTokenTO to = new AccessTokenTO();
//		try {
//			to.setId(entity.getId());
//			to.setIssuedTime(entity.getIssuedTime());
////			to.setMaxIdleTime(entity.getIssuedTime());
//		} catch (Exception e) {
//			LOG.error(e, e);
//		}
//		return to;
//	}

	/**
	 * Convert Card to CardTO
	 * 
	 * @param cardEntity
	 * @return
	 */
	public static CardTO card2CardTO(final IInformationCard cardEntity) {
		CardTO cardTO = null;
		try {
			if (null == cardEntity) {
				return null;
			}
			if (cardEntity instanceof IPersonalInformationCard) {
				cardTO = new PCardTO();
				card2CardTO(cardTO, cardEntity);
				pCardEntity2PCardTO((IPersonalInformationCard) cardEntity, (PCardTO) cardTO);
			} else
				if (cardEntity instanceof IManagedInformationCard) {
					cardTO = new MCardTO();
					card2CardTO(cardTO, cardEntity);
					mCardEntity2MCardTO((IManagedInformationCard) cardEntity, (MCardTO) cardTO);
				} else {
					cardTO = new CardTO();
					card2CardTO(cardTO, cardEntity);
				}
		} catch (Exception e) {
			LOG.error(e, e);
		}
		return cardTO;
	}

	public static CardExtensionTO cardExtension2CardExtensionTO(final IInformationCardExtension extension) {
		CardExtensionTO extensionTO = new CardExtensionTO();
		extensionTO.setEnabled(extension.isEnabled());
		extensionTO.setExtensionXmlElement(extension.getElement().toString());
		return extensionTO;
	}

	// public static void CardTOToCardEntity(final CardTO cardTO, final InformationCard cardEntity) {

	public static IInformationCardExtension cardExtensionTO2CardExtension(final CardExtensionTO extensionTO) {
		// TODO Ask Sergey
		try {
			IOElement element = new IOElement();
			element.set(extensionTO.getExtensionXmlElement());
			return new InformationCardExtension(element, extensionTO.getEnabled(), null);
		} catch (Exception e) {
			LOG.error(e, e);
		}
		return null;
	}

	/**
	 * Convert Card to CardTO
	 * 
	 * @param card
	 *            CardTO
	 * @param cardTO
	 *            InformationCard
	 * @throws CardException
	 */
	private static void card2CardTO(final CardTO cardTO, final IInformationCard card) throws CardException {
		cardTO.setId(card.getCUID().toString());

		cardTO.setName(card.getName());
		cardTO.setCardId(card.getID().toString());
		cardTO.setIssuerName(card.getIssuerName());
		cardTO.setIssuer(card.getIssuer());
		cardTO.setImage(card.getImage());
		cardTO.setImageType(card.getImageType());
		cardTO.setIssuedTime(card.getTimeIssued());
		cardTO.setExpiredTime(card.getTimeExpires());

		cardTO.setSelfIssued(card.isSelfIssued());

		List<ClaimTypeTO> lst = new ArrayList<ClaimTypeTO>();
		Iterator<ClaimType> itr = card.getSupportedClaimTypes();
		while (itr.hasNext()) {
			lst.add(claimType2ClaimTypeTO(itr.next()));
		}
		cardTO.setSupportedClaimTypes(lst.toArray(new ClaimTypeTO[lst.size()]));

		cardTO.setVersion(card.getVersion());

		List supportedTokenTypes = card.getSupportedTokenTypes();
		String[] tokenTypes = new String[supportedTokenTypes.size()];
		for (int i = 0; i < tokenTypes.length; i++) {
			tokenTypes[i] = supportedTokenTypes.get(i).toString();
		}
		cardTO.setSupportedTokenTypes(tokenTypes);

		cardTO.setLastUpdatedTime(card.getTimeLastUpdated());

		// infocard
		cardTO.setIssuerID(card.getIssuerID());
		cardTO.setHashSalt(card.getHashSalt());
		cardTO.setMasterKey(card.getRawMasterKey());

		IInformationCardExtension[] extensions = card.getExtensions();
		if (extensions != null && extensions.length > 0) {
			CardExtensionTO[] cardExtensionTOs = new CardExtensionTO[extensions.length];
			for (int i = 0; i < extensions.length; i++) {
				cardExtensionTOs[i] = cardExtension2CardExtensionTO(extensions[i]);
			}
			cardTO.setExtensions(cardExtensionTOs);
		}

	}

	/**
	 * Convert CardTO to P/MCardEntity.
	 * 
	 * @param cardTO
	 * @return
	 */
	public static InformationCard cardTO2Card(final CardTO cardTO) {
		InformationCard cardEntity = null;
		if (cardTO instanceof PCardTO) {
			cardEntity = new PCardEntity();
			cardTO2CardEntity(cardTO, cardEntity);
			pCardTO2PCardEntity((PCardTO) cardTO, (PCardEntity) cardEntity);
		} else
			if (cardTO instanceof MCardTO) {
				cardEntity = new MCardEntity();
				cardTO2CardEntity(cardTO, cardEntity);
				mCardTO2MCardEntity((MCardTO) cardTO, (MCardEntity) cardEntity);
			}
		return cardEntity;
	}

	/**
	 * Convert CardTO to InformationCard
	 * 
	 * @param cardTO
	 *            CardTO
	 * @param cardEntity
	 *            InformationCard
	 */
	private static void cardTO2CardEntity(final CardTO cardTO, final InformationCard cardEntity) {
		try {
			cardEntity.setID(new URI(cardTO.getCardId()));

			cardEntity.setName(cardTO.getName());
			cardEntity.setIssuerName(cardTO.getIssuerName());
			cardEntity.setIssuer(cardTO.getIssuer());
			cardEntity.setImage(cardTO.getImage());
			cardEntity.setImageType(cardTO.getImageType());
			cardEntity.setTimeIssued(cardTO.getIssuedTime() != null ? cardTO.getIssuedTime() : new Date());
			cardEntity.setTimeExpires(cardTO.getExpiredTime());
			List<IClaimType> claimTypes = new ArrayList<IClaimType>();
			ClaimTypeTO[] supportedClaimTypes = cardTO.getSupportedClaimTypes();
			for (int i = 0; i < supportedClaimTypes.length; i++) {
				claimTypes.add(claimTypeTO2ClaimType(supportedClaimTypes[i]));
			}

			cardEntity.setSupportedClaimTypes(claimTypes);
			cardEntity.setVersion(cardTO.getVersion());
			cardEntity.setSupportedTokenTypes(Arrays.asList(cardTO.getSupportedTokenTypes()));
			cardEntity.setTimeLastUpdated(cardTO.getLastUpdatedTime() != null ? cardTO.getLastUpdatedTime()
					: new Date());

			// infocard
			cardEntity.setIssuerID(cardTO.getIssuerID());
			cardEntity.setHashSalt(cardTO.getHashSalt());
			cardEntity.setRawMasterKey(cardTO.getMasterKey());

			CardExtensionTO[] extensions = cardTO.getExtensions();
			if (extensions != null && extensions.length > 0) {
				ArrayList<IInformationCardExtension> cardExtensions = new ArrayList<IInformationCardExtension>();
				for (int i = 0; i < extensions.length; i++) {
					cardExtensions.add(cardExtensionTO2CardExtension(extensions[i]));
				}
				cardEntity.setExtensions(cardExtensions);
			}

		} catch (Exception e) {
			LOG.error(e, e);
		}
	}

	/**
	 * Convert
	 * 
	 * @param card
	 *            CardTO
	 * @param cardTO
	 *            InformationCard
	 */
	public static ClaimTypeTO claimType2ClaimTypeTO(final IClaimType claimType) {
		ClaimTypeTO claimTypeTO = new ClaimTypeTO();
		claimTypeTO.setDescription(claimType.getDescription());
		claimTypeTO.setDisplayName(claimType.getDisplayName());
		claimTypeTO.setIsSimple(claimType.isSimple());
		claimTypeTO.setType(claimType.getType());
		return claimTypeTO;
	}

	public static ClaimUiDescriptorTO claimType2ClaimUiDescriptor(final IClaimType claimType) {
		ClaimUiDescriptorTO claimUiDescriptorTO = new ClaimUiDescriptorTO();
		IUIDescriptor uiDescriptor = claimType.getUIDescriptor();
		if (null != uiDescriptor) {
			claimUiDescriptorTO.setInputMask(uiDescriptor.getInputMask());
			if (null != uiDescriptor.getOptionalValues() && uiDescriptor.getOptionalValues().size() > 0) {
				claimUiDescriptorTO.setOptionalValues((String[]) uiDescriptor.getOptionalValues().toArray(
						new String[uiDescriptor.getOptionalValues().size()]));
			}
			claimUiDescriptorTO.setPattern(uiDescriptor.getPattern());
			claimUiDescriptorTO.setType(""+uiDescriptor.getType());
		}
		return claimUiDescriptorTO;
	}

	/**
	 * Convert
	 * 
	 * @param card
	 *            CardTO
	 * @param cardTO
	 *            InformationCard
	 */
	public static IClaimType claimTypeTO2ClaimType(final ClaimTypeTO claimTypeTO) {
		try {
			return new ClaimType(claimTypeTO.getType(), claimTypeTO.getDisplayName(), claimTypeTO.getDescription());
		} catch (CardException e) {
			LOG.error(e, e);
			return null;
		}
	}

	private static void pCardEntity2PCardTO(final IPersonalInformationCard pcardEntity, final PCardTO pcardTO) {
		try {
			pcardTO.setPinDigest(pcardEntity.getPinDigest());
			List<ClaimTO> claimList = new ArrayList<ClaimTO>();
			Map<String, String> claimMap = pcardEntity.getDirtyClaims();
			for (Iterator<IClaimType> iterator = pcardEntity.getSupportedClaimTypes(); iterator.hasNext();) {
				IClaimType claimTypeItem = iterator.next();
				// FIXME add .setClaimUiDescriptor(...)
				claimList.add(new ClaimTO().setClaimType(claimType2ClaimTypeTO(claimTypeItem)).setValues(
						new String[] { claimMap.get(claimTypeItem.getType()) }).setClaimUiDescriptor(
						claimType2ClaimUiDescriptor(claimTypeItem)));
			}
			pcardTO.setClaims(claimList.toArray(new ClaimTO[claimList.size()]));

		} catch (Exception e) {
			LOG.error(e, e);
		}
	}

	private static CredentialDescriptorTO credentialDescriptor2CredentialDescriptorTO(
			final ICredentialDescriptor credentialDescriptor, final CredentialDescriptorTO credentialDescriptorTO)
			throws IOException {
		return credentialDescriptorTO.setType(credentialDescriptor.getType()).setDisplayCredentialHint(
				credentialDescriptor.getDisplayCredentialHint()).setCredentialXmlElement(
				XMLHelper.toString(credentialDescriptor.asXML()));
	}

	private static EndpointReferenceTO endpointReference2EndpointReferenceTO(
			final IEndpointReference endpointReference, final EndpointReferenceTO endpointReferenceTO)
			throws IOException {
		return endpointReferenceTO.setAddress(endpointReference.getAddress()).setIdentityXml(
				XMLHelper.toString(endpointReference.getIdentity())).setMetadataAddress(
				endpointReference.getMetadataAddress()).setMetadataXml(
				XMLHelper.toString(endpointReference.getMetadata()));
	}

	private static TokenServiceTO tokenService2TokenServiceTO(final TokenService tokenService,
			final TokenServiceTO tokenServiceTO) throws IOException {

		tokenServiceTO.setEndpointReference(endpointReference2EndpointReferenceTO(tokenService.getEndpointReference(),
				new EndpointReferenceTO()));
		ICredentialDescriptor userCredential = tokenService.getUserCredential();
		tokenServiceTO.setUserCredential(credentialDescriptor2CredentialDescriptorTO(userCredential,
				new CredentialDescriptorTO()));
		return tokenServiceTO;
	}

	private static MCardTO mCardEntity2MCardTO(final IManagedInformationCard mcardEntity, final MCardTO mcardTO) {
		try {
			mcardTO.setRequireAppliesTo(mcardEntity.getRequireAppliesTo());

			ISTSPrivacyPolicy policy = mcardEntity.getPrivacyNotice();
			if (policy != null) {
				mcardTO.setStsPrivacyPolicyTO(new StsPrivacyPolicyTO().setUrl(policy.getPrivacyUrl()).setVersion(
						policy.getPrivacyVersion()));
			}

			List<TokenService> tslist = mcardEntity.getTokenServices();
			TokenServiceTO[] serviceTOs = new TokenServiceTO[tslist.size()];
			for (int i = 0; i < serviceTOs.length; i++) {
				serviceTOs[i] = tokenService2TokenServiceTO(tslist.get(i), new TokenServiceTO());
			}
			mcardTO.setTokenServices(serviceTOs);

		} catch (Exception e) {
			LOG.error(e, e);
		}
		return mcardTO;
	}

	/**
	 * Convert PCardTO to PCardEntity
	 * 
	 * @param pcardTO
	 * @param pcardEntity
	 */
	private static void pCardTO2PCardEntity(final PCardTO pcardTO, final PCardEntity pcardEntity) {
		try {
			// CardTOToCardEntity(pcardTO, pcardEntity);

			pcardEntity.setPinDigest(pcardTO.getPinDigest());
			pcardEntity.setSelfIssued(true);

			// Set claim values.
			HashMap<String, String> claimMap = new HashMap<String, String>();
			ClaimTO[] claimTOs = pcardTO.getClaims();
			if (null != claimTOs) {
				for (int i = 0; i < claimTOs.length; i++) {
					try {
						claimMap.put(claimTOs[i].getClaimType().getType(), claimTOs[i].getValues()[0]);
					} catch (Exception e) {
						LOG.error(e, e);
					}
				}
			}
			pcardEntity.setDirtyClaims(claimMap);

		} catch (Exception e) {
			LOG.error(e, e);
		}
	}

	private static MCardEntity mCardTO2MCardEntity(final MCardTO mcardTO, final MCardEntity mcardEntity) {
		try {
			mcardEntity.setRequireAppliesTo(mcardTO.getRequireAppliesTo());

			if (null != mcardTO.getStsPrivacyPolicyTO()) {
				STSPrivacyPolicy policy = new STSPrivacyPolicy(mcardTO.getStsPrivacyPolicyTO().getUrl(), mcardTO
						.getStsPrivacyPolicyTO().getVersion());
				mcardEntity.setPrivacyNotice(policy);
			}

			TokenServiceTO[] tokenServices = mcardTO.getTokenServices();
			List<TokenService> tokenServiceList = new ArrayList<TokenService>();
			EndpointReference endpointReference;
			CredentialDescriptor credentialDescriptor;
			for (int i = 0; i < tokenServices.length; i++) {
				endpointReference = new EndpointReference(tokenServices[i].getEndpointReference().getAddress(),
						tokenServices[i].getEndpointReference().getMetadataXml(), tokenServices[i]
								.getEndpointReference().getIdentityXml());
				credentialDescriptor = new CredentialDescriptor(tokenServices[i].getUserCredential()
						.getDisplayCredentialHint(), tokenServices[i].getUserCredential().getCredentialXmlElement());
				tokenServiceList.add(new TokenService(endpointReference, credentialDescriptor));
			}
			mcardEntity.setTokenServices(tokenServiceList);

		} catch (Exception e) {
			LOG.error(e, e);
		}
		return mcardEntity;
	}

	/**
	 * Convert RevisionEntity to RevisionTO
	 * 
	 * @param entity
	 *            RevisionEntity
	 * @return RevisionTO
	 */
	public static RevisionTO revisionEntity2RevisionTO(RevisionEntity entity) {
		RevisionTO to = new RevisionTO();
		try {
			to.setModifiedTime(entity.getModifiedTime());
			to.setNumber(entity.getNumber());
		} catch (Exception e) {
			// LOG.error(e, e);
		}
		return to;
	}

	/**
	 * Convert RevisionTO to RevisionEntity
	 * 
	 * @param to
	 * @return
	 */
	public static RevisionEntity revisionTO2revisionEntity(RevisionTO to) {
		RevisionEntity entity = new RevisionEntity();
		try {
			entity.setModifiedTime(to.getModifiedTime());
			entity.setNumber(to.getNumber());

		} catch (Exception e) {
			LOG.error(e, e);
		}
		return entity;
	}

	/**
	 * Convert UserProfileTO to UserProfile
	 * 
	 * @param userProfileTO
	 * @return
	 */
	public static UserProfile userProfileTO2UserProfile(final UserProfileTO userProfileTO, UserProfile userProfile) {
		try {
			userProfile.setEmail(userProfileTO.getEmail());
			userProfile.setFirstName(userProfileTO.getFirstName());
			userProfile.setLastName(userProfileTO.getLastName());
			// userProfile.setLoginName(userProfileTO.getLoginName());
			userProfile.setSms(userProfileTO.getMobile());
			userProfile.setReferralTracking(userProfileTO.getReferralTracking());
			userProfile.setSendMeNews(userProfileTO.getSendMeNews());
		} catch (Exception e) {
			LOG.error(e, e);
		}
		return userProfile;
	}

	/**
	 * Convert UserProfile to UserProfileTO
	 * 
	 * @param userProfile
	 *            UserProfile entity.
	 * @param revisionEntity
	 *            RevisionEntity
	 * @return
	 */
	public static UserProfileTO userProfile2UserProfileTO(UserProfile userProfile, Account account,
			RevisionEntity revisionEntity) {
		UserProfileTO to = new UserProfileTO();
		try {

			to.setId((String) userProfile.getUserIdentifier());

			to.setCreatedTime(userProfile.getCreated());
			to.setEmail(userProfile.getEmail());
			to.setFirstName(userProfile.getFirstName());
			to.setLastName(userProfile.getLastName());
			to.setLoginName(account.getLogin());
			to.setMobile(userProfile.getSms());
			to.setReferralTracking(userProfile.getReferralTracking());
			to.setSendMeNews(userProfile.isSendMeNews());

			to.setRevision(revisionEntity2RevisionTO(revisionEntity));

		} catch (Exception e) {
			LOG.error(e, e);
		}
		return to;
	}

}
