/** 
 * Copyright (c) 2009 Parity Communications, Inc.  
 * All rights reserved. This program and the accompanying materials  
 * are made available under the terms of the Eclipse Public License v1.0  
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 *  
 * Contributors: 
 * Alexander Yuhimenko - implementation 
 * 
 */

package org.eclipse.higgins.sync.meta;

import java.util.List;

import org.eclipse.higgins.sync.meta.entity.RevisionEntity;
import org.eclipse.higgins.sync.meta.exceptions.ResourceMetaDataException;

/**
 * IResourceMetaDataService represents API for managing resource meta information (revision).
 * 
 * @author Alexander Yuhimenko
 * 
 */
public interface IResourceMetaDataService {

	/**
	 * Define Root resource type.
	 */
	public static final String ROOT_RESOURCE_TYPE = "RootResource";
	/**
	 * Define UserProfile resource type.
	 */
	public static final String USER_PROFILE_RESOURCE_TYPE = "org.eclipse.higgins.sync.to.UserProfileTO";
	/**
	 * Define ICard resource type.
	 */
	public static final String CARD_RESOURCE_TYPE = "org.eclipse.higgins.sync.to.CardTO";
																							    
	// /**
	// * Define MCard resource type.
	// */
	// public static final String CARD_RESOURCE_TYPE = "org.eclipse.higgins.sync.to.MCardTO";

	/**
	 * Define CardHistory resource type.
	 */
	public static final String CARD_HISTORY_RESOURCE_TYPE = "org.eclipse.higgins.sync.to.CardHistoryTO";

	/**
	 * Define CardCategory resource type.
	 */
	public static final String CARD_CATEGORY_RESOURCE_TYPE = "org.eclipse.higgins.sync.to.CardCategoryTO";

	/**
	 * Define CardCategory resource type.
	 */
	public static final String CARD_CREDENTIALS_RESOURCE_TYPE = "org.eclipse.higgins.sync.to.CardCredentialTO";

	/**
	 * Return RevisionEntity by Id
	 * 
	 * @param id
	 *            RevisionEntity Id
	 * @return RevisionEntity
	 * @throws ResourceMetaDataException
	 *             if couldn't find RevisionEntity.
	 */
	RevisionEntity getResourceRevisionEntity(Long id) throws ResourceMetaDataException;

	/**
	 * * Return RevisionEntity which satisfied the following conditions:
	 * <ol>
	 * <li>RevisionEntity.resourceId=revisionEntity.resourceId;</li>
	 * <li>RevisionEntity.resourceType=revisionEntity.resourceType.</li>
	 * <li>RevisionEntity.userId=revisionEntity.userId.</li>
	 * </ol>
	 * 
	 * @param revisionEntity
	 * @return RevisionEntity or Null if couldn't find RevisionEntity.
	 * @throws ResourceMetaDataException
	 * 
	 * @throws NullPointerException
	 *             if revisionEntity or revisionEntity.resourceId or revisionEntity.resourceType are Null.
	 */
	RevisionEntity findResourceRevisionEntity(RevisionEntity revisionEntity) throws ResourceMetaDataException;

	/**
	 * Return RevisionEntity with maximal number which satisfied the following conditions: <li>
	 * RevisionEntity.resourceType=revisionEntity.resourceType.</li> <li>RevisionEntity.userId=revisionEntity.userId.</li>
	 * 
	 * 
	 * @param revisionEntity
	 * @return
	 * @throws ResourceMetaDataException
	 */
	RevisionEntity findMaxRevisionEntity(RevisionEntity revisionEntity) throws ResourceMetaDataException;

	/**
	 * Return list of ResourceRevisionEntity with number more than @clientRootRevision.number.
	 * 
	 * @param clientRootRevision
	 * @param resourceType,
	 *             if resourceType is null or "%", have to return for all resource types.
	 * @return
	 * @throws ResourceMetaDataException
	 */
	List getResourceRevisionEntityList(RevisionEntity clientRootRevision, String resourceType)
			throws ResourceMetaDataException;

	/**
	 * Save or update RevisionEntity.
	 * 
	 * @param revisionEntity
	 * @throws ResourceMetaDataException
	 *             if couldn't persist revisionEntity.
	 * @throws NullPointerException
	 *             if revisionEntity or revisionEntity.resourceId or revisionEntity.resourceType are Null.
	 */
	void persistResourceRevisionEntity(RevisionEntity revisionEntity) throws ResourceMetaDataException;

	/**
	 * Delete RevisionEntity. It finds RevisionEntity by Id or by resourceId and resourceType.
	 * 
	 * @param revisionEntity
	 * 
	 * @deprecated, it shouldn't be deleted due to it needs for sync. Use @see #persistResourceRevisionEntity with
	 *              commandName "Delete"
	 */
	void deleteResourceRevisionEntity(RevisionEntity revisionEntity);

}
