/**
 * Copyright (c) 2007-2008 Novell, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *  	Duane Buss
 *  	Tom Doman
 *		Jim Sermersheim
 */

package org.eclipse.higgins.util.idas.cp.jscript;

import java.net.URI;
import java.util.Iterator;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.util.idas.cp.IAttributePDPs;
import org.eclipse.higgins.util.jscript.JScriptException;
import org.eclipse.higgins.util.jscript.JScriptExec;
import org.eclipse.higgins.util.jscript.JScriptExecHelper;
import org.eclipse.higgins.util.jscript.SingleObjectImmutableIterator;
import org.eclipse.higgins.idas.api.IValue;
import org.eclipse.higgins.idas.api.IdASException;

/**
 * Part of the JSPolicy <a href="http://www.eclipse.org/higgins/">Higgins</a>
 * Context provider. This implementation allows for javascript policy at
 * critical points.
 * <p>
 * This module defines the PDP interfaces that are common to most property based
 * Higgins objects.
 * 
 * @author dbuss@novell.com
 */

public class JScriptCPMetadataPDPs implements IAttributePDPs
{
	/**
	 * Generic name which is used on many of the PDPs to pass a single parameter
	 * from java to the script.
	 */
	public static final String genericParamName = "PARAM";

	/**
	 * Generic name which is used on many of the PDPs to pass the result from
	 * JavaScript back to JAVA PDP code.
	 */
	public static final String genericResultName = "RESULT";

	/**
	 * Generic parameter name for for selected PDPs. See specific PDP
	 * descriptions for details.
	 */
	public static final String consumerTypeParamName = "consumerType";

	/**
	 * Generic parameter name for for selected PDPs. See specific PDP
	 * descriptions for details.
	 */
	public static final String providerTypeParamName = "providerType";

	/**
	 * Generic parameter name for for selected PDPs. See specific PDP
	 * descriptions for details.
	 */
	public static final String consumerIDParamName = "consumerID";

	/**
	 * Generic parameter name for for selected PDPs. See specific PDP
	 * descriptions for details.
	 */
	public static final String providerIDParamName = "providerID";
	
	private JScriptExec _consumerIDToProviderExec;
	private JScriptExec _providerIDToConsumerExec;	
	private JScriptExec _consumerTypeToProviderExec;
	private JScriptExec _providerTypeToConsumerExec;
	private JScriptExec _consumerValueToProviderExec;
	private JScriptExec _providerValueToConsumerExec;
	private Log _log;

	/**
	 * TODO: add logging if mismached configuration
	 * 
	 * @param consumerIDToProviderExec {@link #consumerIDToProvider}
	 * @param providerIDToConsumerExec  {@link #providerIDToConsumer}
	 * @param consumerTypeToProviderExec  {@link #consumerTypeToProvider}
	 * @param providerTypeToConsumerExec   {@link #providerTypeToConsumer}
	 * @param consumerValueToProviderExec  {@link #consumerValueToProvider}
	 * @param providerValueToConsumerExec  {@link #providerValueToConsumer}
	 */
	public JScriptCPMetadataPDPs(
		JScriptExec consumerIDToProviderExec,
		JScriptExec providerIDToConsumerExec,
		JScriptExec consumerTypeToProviderExec,
		JScriptExec providerTypeToConsumerExec,
		JScriptExec consumerValueToProviderExec,
		JScriptExec providerValueToConsumerExec,
		boolean allowAsymmetricPDP)
	{
		// setup logging
		_log = LogFactory.getLog(this.getClass());

		_consumerIDToProviderExec = consumerIDToProviderExec;
		_providerIDToConsumerExec = providerIDToConsumerExec;
		_consumerTypeToProviderExec = consumerTypeToProviderExec;
		_providerTypeToConsumerExec = providerTypeToConsumerExec;
		_consumerValueToProviderExec = consumerValueToProviderExec;
		_providerValueToConsumerExec = providerValueToConsumerExec;

		if (allowAsymmetricPDP == false)
		{
			if (_consumerIDToProviderExec != _providerIDToConsumerExec
				&& (_consumerIDToProviderExec == null || _providerIDToConsumerExec == null))
				_log.error("Asymmetric PDP definitions found, consumerIDToProvider is "
						+ _consumerIDToProviderExec == null ? "not"
						: "" + " set, providerIDToConsumer is " + _providerIDToConsumerExec == null ? "not"
							: "" + " set.");	
			if (_consumerTypeToProviderExec != _providerTypeToConsumerExec
				&& (_consumerTypeToProviderExec == null || _providerTypeToConsumerExec == null))
				_log.error("Asymmetric PDP definitions found, consumerTypeToProvider is "
						+ _consumerTypeToProviderExec == null ? "not"
						: "" + " set, providerTypeToConsumer is " + _providerTypeToConsumerExec == null ? "not"
							: "" + " set.");
			if (_consumerValueToProviderExec != _providerValueToConsumerExec
				&& (_consumerValueToProviderExec == null || _providerValueToConsumerExec == null))
				_log.error("Asymmetric PDP definitions found, consumerValueToProvider is "
						+ _consumerValueToProviderExec == null ? "not"
						: "" + " set, providerValueToConsumer is " + _providerValueToConsumerExec == null ? "not"
							: "" + " set.");
		}
	}

	/**
	 * Helper PDP for converting from stored or child Context Provider to
	 * consumer presented ID.
	 * 
	 * @param providerID
	 *           The actual or provider ID of the metadata
	 * @return The consumer or mapped ID of the metadata, returning null
	 *         indicates that this property should not be returned to the
	 *         consumer.   The providerID may be returned when no PDP has
	 *         been configured.
	 * @throws IdASException
	 * 
	 */
	public URI providerIDToConsumer(
		URI providerID) throws IdASException
	{
			if (_providerIDToConsumerExec != null)
			{
				try
				{
					return JScriptExecHelper.transformURI(_providerIDToConsumerExec, 
						providerIDParamName,providerID);
				}
				catch (JScriptException e)
				{
					throw new IdASException(e);
				}
			}
			else
			{
				return providerID;
			}
	}
	
	/**
	 * PDP interface for converting from passed by consumer (presented) ID to
	 * what is correct for storage or relaying on to another Context Provider. 
	 * 
	 * @param consumerID the consumer or unmapped ID of the metadata
	 * @return The actual or provider basic type of the metadata value, returning null
	 *         indicates that this property should not be returned to the
	 *         consumer.
	 * @throws IdASException
	 */

	public Iterator consumerIDToProviders(
			URI consumerID) throws IdASException
	{
		if (_consumerIDToProviderExec != null)
		{
			try
			{
				return JScriptExecHelper.transformURItoMany(_consumerIDToProviderExec,
					consumerIDParamName, consumerID);
			}
			catch (Exception e)
			{
				throw new IdASException(e);
			}
		}
		else 
			return  new SingleObjectImmutableIterator(consumerID);
	}
	
	/**
	 * TODO document this one
	 */
	public URI consumerIDToProvider(
		URI consumerID) throws IdASException
	{
		if (_consumerIDToProviderExec != null)
		{
			try
			{
				return JScriptExecHelper.transformURI(_consumerIDToProviderExec,
					consumerIDParamName, consumerID);
			}
			catch (Exception e)
			{
				throw new IdASException(e);
			}
		}
		else 
			return consumerID;	
	}
	
	/**
	 * 
	 * @param consumerID the consumer's version of the ID of the metadata, passed on
	 * 				to the javascript pdp as  {@link #consumerIDParamName}
	 * @param type
	 *           The consumer data type, passed on to the javascript pdp as
	 *           {@link #consumerTypeParamName}
	 * @return A single URI if the policy results in multiple URIs being returned
	 *         then the first one is selected and returned.
	 *      
	 * When this PDP is invoked the parameters are stored in an evaluation
	 * specific JavaScript scope.  It is required that either the result of the script 
	 * <b>or</b> a JavaScript variable with the name 
	 * {@link #genericResultName} is the new type.
	 * <p>
	 * To avoid problems it is recomended that the reverse mapping also be
	 * defined using the id {@link #providerTypeToConsumer}
	 * <p>
	 * 
	 * A simple nonsensical sample where the provider Entity Type mappings
	 * are stored in a globally scoped metadata multimap, and are referenced
	 * using JavaScript Associative Arrays.
	 * 
	 * <pre>
	 *  
	 * 		&lt;JSPolicyAction id=&quot;consumerEntityTypeToProvider&quot; evalType=&quot;javascript&quot;&gt;
	 * 			RESULT = multimap.consumer[(PARAM.toString())];
	 * 		&lt;/JSPolicyAction&gt;
	 * 		&lt;JSPolicyAction id=&quot;providerEntityTypeToConsumer&quot; evalType=&quot;javascript&quot;&gt;
	 * 			RESULT = multimap.provider[String(PARAM.toString())];
	 * 		&lt;/JSPolicyAction&gt;
	 * </pre>
	 *      
	 *      
	 */
	public URI consumerTypeToProvider(
		URI consumerID,
		URI type) throws IdASException
	{
		final String[] paramNames =
		{consumerIDParamName, consumerTypeParamName};
		if (_consumerTypeToProviderExec != null)
			try
			{
				URI params[] =  {consumerID, type};
				return JScriptExecHelper.transformURI(_consumerTypeToProviderExec, 
					paramNames, params);
			}
			catch (JScriptException e)
			{
				throw new IdASException(e);
			}
		else
			return type;
	}

	/**
	 * For JavaScript PDPs this passes the type in the JavaScript var PARAM as a
	 * java URI and handles a string or URI back.
	 * 
	 * @param providerID 
	 * @param type
	 *           The consumer type
	 * @return A single URI if the policy results in multiple URIs being returned
	 *         then the first one is selected and returned.
	 * @see org.eclipse.higgins.util.idas.cp.IPropertyPDP#providerTypeToConsumer(URI
	 *      type)
	 */
	public URI providerTypeToConsumer(
		URI providerID,
		URI type) throws IdASException
	{
		final String[] paramNames =
		{providerIDParamName, providerTypeParamName};
		if (_providerTypeToConsumerExec != null)
		{
			try
			{
				URI params[] = {providerID, type};
				return JScriptExecHelper.transformURI(_providerTypeToConsumerExec, 
					paramNames, params);
			}
			catch (JScriptException e)
			{
				throw new IdASException(e);
			}
		}
		else
		{
			return type;
		}
	}

	/**
	 * When this PDP is invoked the consumer type, the provider type and the
	 * Metadata Value are all stored in an evaluation specific JavaScript scope
	 * using the names {@link #consumerTypeParamName},
	 * {@link #providerTypeParamName}, {@link #genericParamName} respectively. It is
	 * consumer that either the result of the script <b>or</b> a JavaScript
	 * variable with the name {@link #genericResultName} is the new value
	 * <p>
	 * To avoid problems it is recommended that the reverse mapping also be
	 * defined.
	 * 
	 */
	public IValue consumerValueToProvider(URI consumerID,
			URI consumerType, URI providerID, URI providerType,
			IValue consumerValue) throws IdASException {
		final String[] paramNamesConsumerValueToProvider =
		{consumerIDParamName, consumerTypeParamName, 
		providerIDParamName, providerTypeParamName, genericParamName};
	
	if (_consumerValueToProviderExec != null)
	{
		Object params[] =
		{consumerID, consumerType, 
			providerID, providerType,
			consumerValue};

		Object result = null;
		try
		{
			result = _consumerValueToProviderExec.evaluate(paramNamesConsumerValueToProvider, params);
		}
		catch (Exception e)
		{
			throw new IdASException(e);
		}
		return (IValue)result;
	}
	return consumerValue;
	}

	/**
	 * 
	 * When this PDP is invoked the consumer type, the provider type and the
	 * Metadata Value are all stored in an evaluation specific JavaScript scope
	 * using the names {@link #consumerTypeParamName},
	 * {@link #providerTypeParamName}, {@link #genericParamName} respectively. It is
	 * consumer that either the result of the script <b>or</b> a JavaScript
	 * variable with the name {@link #genericResultName} is the new value
	 * <p>
	 * To avoid problems it is recommended that the reverse mapping also be
	 * defined
	 * 
	 */
	public IValue providerValueToConsumer(URI providerID,
			URI providerType, URI consumerID, URI consumerType,
			IValue providerValue) throws IdASException {
		final String[] paramNamesProviderValueToConsumer =
		{providerIDParamName, providerTypeParamName, consumerIDParamName,
		consumerTypeParamName, genericParamName};
	
	if (_providerValueToConsumerExec != null)
	{
		Object params[] =	{	providerID, providerType,
			consumerID, consumerType, 
			providerValue};

		Object result;
		try
		{
			result = _providerValueToConsumerExec.evaluate(paramNamesProviderValueToConsumer,
				params);
		}
		catch (Exception e)
		{
			throw new IdASException(e);
		}
		return  (IValue)result;
	}
	return providerValue;
	}
}
