/*******************************************************************************
 * Copyright (c) 2007-2008 Parity Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Valery Kokhan - initial implementation
 *******************************************************************************/

package org.eclipse.higgins.ant.pde;

import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.higgins.ant.service.AbstractBuildFileCreator;
import org.eclipse.higgins.ant.service.EclipseClasspath;
import org.eclipse.higgins.ant.service.ExportUtil;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.pde.internal.core.ifeature.IFeature;
import org.eclipse.pde.internal.core.ifeature.IFeatureModel;
import org.eclipse.swt.widgets.Shell;
import org.w3c.dom.Comment;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;


/**
 * Creates build.xml file.
 */
public class FeatureBuildFileCreator extends AbstractBuildFileCreator {
	protected IFeatureModel model;
	protected IFeature feature;

    public FeatureBuildFileCreator() {
    	super();
    	//System.out.println("FeatureBuildFileCreator()");
    }

    /**
     * Constructor. Please prefer {@link #createBuildFiles(Set, Shell, IProgressMonitor)} if
     * you do not want call the various createXXX() methods yourself.
     * 
     * @param project    create buildfile for this project
     * @param shell      parent instance for dialogs
     */
	public FeatureBuildFileCreator(IFeatureModel model) {
		//super(ExportUtil.getProject(model));
		super(projectManager.getProject(model));
		this.model = model;
		this.feature = this.model.getFeature();
		//System.out.println("FeatureBuildFileCreator()");
	}
	
	protected void Init(IProject project) {
		super.Init(project);
		this.model = ((ProjectManager) projectManager).getFeatureModel(project);
		this.feature = this.model.getFeature();
	}

	private void addSubProperties() throws JavaModelException {
		String location = projectName + ".location";
		String relativePath = ExportUtil.getRelativePath(projectRoot, projectRoot);

		variable2valueMap.put(location, relativePath);
		List list = getSubProjects();
		for (Iterator iterator = list.iterator(); iterator.hasNext(); ) {
			IProject subProject = projectManager.getProject(iterator.next());
			location = projectManager.getProjectName(subProject) + ".location";
			String subProjectRoot = projectManager.getProjectRoot(subProject);
			relativePath = ExportUtil.getRelativePath(subProjectRoot, projectRoot);
			variable2valueMap.put(location, relativePath);
		}
	}

	/**
	 * Find buildfiles in projectroot directory and automatically import them.
	 */
	
	public void createFeature() {
		try {
				Document tdoc = projectManager.getTemplate("higgins2ant.pde", "featureBuilder"); 
				Element troot = tdoc.getDocumentElement();
				NodeList nl = troot.getChildNodes();
				int length = nl.getLength();
				for (int i = 0; i < length; i++) {
					Node n = nl.item(i);
					Node n1 = doc.importNode(n, true);
					if (n1.getNodeType() == Node.ELEMENT_NODE ) {
						Element e = (Element) n1;
						if (e.getNodeName().equals("target") && e.getAttribute("name").equals("-initFeatureSources")) {
							createInitFeatureSources(e);
						} else if (e.getNodeName().equals("target") && e.getAttribute("name").equals("-updateFeatureSources")) {
							createUpdateFeatureSources(e);
						}
					}
					root.appendChild(n1);
				}
			
		} catch (Exception e) {
			e.printStackTrace();
		}
		
	}
	
	private void createUpdateFeatureSources(Element target) throws JavaModelException {
		createCopyProjectSources(target, model, true);
		
		List subProjects = getSubProjects();
		for (Iterator iterator = subProjects.iterator(); iterator.hasNext();) {
			Object o = iterator.next();
			if (o instanceof IFeatureModel) {
				createCopyProjectSources(target, o, true);
			}
		}
	}
	
	private void createInitFeatureSources(Element target) throws JavaModelException {
		createCopyProjectSources(target, model, false);
		
		List subProjects = getSubProjects();
		for (Iterator iterator = subProjects.iterator(); iterator.hasNext();) {
			createCopyProjectSources(target, iterator.next(), false);
		}
	}
	
	private void createCopyProjectSources(Element target, Object project, boolean update) throws JavaModelException {
		if (project instanceof IFeatureModel) {
			createCopyProjectSources(target, (IFeatureModel) project, update);
		} else if (project instanceof IJavaProject) {
			createCopyProjectSources(target, (IJavaProject) project, update);
		} else if (project instanceof IProject) {
			createCopyProjectSources(target, (IProject) project, update);
		}
	}
	
	private void createCopyProjectSources(Element target, IProject project, boolean update) {
		//String projectName = ExportUtil.getProjectName(project);
		String projectName = projectManager.getProjectName(project);
		Element copy = doc.createElement("copy");
		copy.setAttribute("todir", "${temp.dir.plugins}" + "/" + projectName);
		copy.setAttribute("failonerror", "true");
		
		Element fileset = doc.createElement("fileset");
		fileset.setAttribute("dir", "${" + projectName + ".location}");
		fileset.setAttribute("excludes", DEFAULT_EXCLUDES);
		copy.appendChild(fileset);

		target.appendChild(copy);
	}
	
	private void createCopyProjectSources(Element target, IFeatureModel model, boolean update) {
		//String projectName = ExportUtil.getProjectName(model);
		String projectName = projectManager.getProjectName(model);
		Element copy;
		Element fileset;
		
		if (!update) {
			copy = doc.createElement("copy");
			copy.setAttribute("todir", "${temp.dir.features}" + "/" + projectName);
			copy.setAttribute("failonerror", "true");
			copy.setAttribute("overwrite", "true");

			fileset = doc.createElement("fileset");
			fileset.setAttribute("dir", "${" + projectName + ".location}");
			fileset.setAttribute("excludes", DEFAULT_EXCLUDES + ",build.xml,feature.xml");
			copy.appendChild(fileset);

			target.appendChild(copy);
		}

		copy = doc.createElement("copy");
		copy.setAttribute("todir", "${temp.dir.features}" + "/" + projectName);
		copy.setAttribute("failonerror", "true");
		copy.setAttribute("overwrite", "true");
		
		fileset = doc.createElement("fileset");
		fileset.setAttribute("dir", "${" + projectName + ".location}");
		fileset.setAttribute("includes", "feature.xml");
		copy.appendChild(fileset);
		
		if (!update) {
			copy.setAttribute("encoding", "UTF-8");
			
			Element filterchain = doc.createElement("filterchain");
			copy.appendChild(filterchain);
			Element tokenfilter = doc.createElement("tokenfilter");
			filterchain.appendChild(tokenfilter);
			Element replaceregex = doc.createElement("replaceregex");
			tokenfilter.appendChild(replaceregex);
			replaceregex.setAttribute("pattern", "os=\\x22.*\\x22");
			replaceregex.setAttribute("replace", "os=\"*\"");
			
			replaceregex = doc.createElement("replaceregex");
			tokenfilter.appendChild(replaceregex);
			replaceregex.setAttribute("pattern", "ws=\\x22.*\\x22");
			replaceregex.setAttribute("replace", "ws=\"*\"");
			
			replaceregex = doc.createElement("replaceregex");
			tokenfilter.appendChild(replaceregex);
			replaceregex.setAttribute("pattern", "arch=\\x22.*\\x22");
			replaceregex.setAttribute("replace", "arch=\"*\"");
		}
		
		target.appendChild(copy);
	}
	
	private void createCopyProjectSources(Element target, IJavaProject project, boolean update) throws JavaModelException {
		EclipseClasspath classpath = new EclipseClasspath(project);
		//String projectName = ExportUtil.getProjectName(project);
		String projectName = projectManager.getProjectName(project);
		Element copy = doc.createElement("copy");
		copy.setAttribute("todir", "${temp.dir.plugins}" + "/" + projectName);
		copy.setAttribute("failonerror", "true");
		
		Element fileset = doc.createElement("fileset");
		fileset.setAttribute("dir", "${" + projectName + ".location}");
		fileset.setAttribute("excludes", DEFAULT_EXCLUDES);
		copy.appendChild(fileset);
		
		if (classpath.getSrcDirs().size() > 0) {
			int size = classpath.getSrcDirs().size(); 
			for (int i = 0; i < size; i++) {
				String srcDir = (String) classpath.getSrcDirs().get(i);
				if (!EclipseClasspath.isReference(srcDir)) {
					List exclusions = (List) classpath.getExclusionLists().get(i);
					for (Iterator iter = exclusions.iterator(); iter.hasNext();) {
						String exclusion = (String) iter.next();
						Element excludeElement = doc.createElement("exclude");
						excludeElement.setAttribute("name", srcDir + "/" + exclusion);
						fileset.appendChild(excludeElement);
					}
				}
			}
		}
		String target_ = project.getOption(JavaCore.COMPILER_CODEGEN_TARGET_PLATFORM, true);
		String oldTarget_ = (String) variable2valueMap.get("target");
		if (oldTarget_ != null) {
			if (oldTarget_.compareTo(target_) < 0) {
				target_ = oldTarget_;
			}
		}
		variable2valueMap.put("target", target_);
		
		String source = project.getOption(JavaCore.COMPILER_SOURCE, true);
		String oldSource = (String) variable2valueMap.get("source");
		if (oldSource != null) {
			if (oldSource.compareTo(source) < 0) {
				source = oldSource;
			}
		}
		variable2valueMap.put("source", source); //$NON-NLS-1$

		target.appendChild(copy);
	}
	
	protected void createProperty() {
		Map v2p = new LinkedHashMap();

		Comment c = doc.createComment(" ======================================== ");
		v2p.put("c1", c);

		v2p.put("sources.dir", "${basedir}/..");
		v2p.put("build.dir", "${basedir}/build");
		v2p.put("build.dir.bin", "${build.dir}/bin");
		v2p.put("build.dir.lib", "${build.dir}/lib");
		v2p.put("build.dir.doc", "${build.dir}/doc");

		c = doc.createComment(" ======================================== ");
		v2p.put("c1.1", c);

		String jarName = projectName;
		jarName = jarName.replaceAll("org\\.eclipse\\.higgins", "higgins");
		jarName = jarName.replaceAll("\\.", "-");
		v2p.put("build.file.name", jarName);
		boolean first = true;

		Node node;
		try {
			Document tdoc = projectManager.getTemplate("higgins2ant.pde", "featureProperties");
			Element troot = tdoc.getDocumentElement();
			NodeList nl = troot.getChildNodes();
			int length = nl.getLength();
			first = true;
			node = root.getFirstChild();
			for (int i = 0; i < length; i++) {
				Node n = nl.item(i);
				Node n1 = doc.importNode(n, true);
				if (first) {
					first = false;
				} else {
					node = node.getNextSibling();
				}
				node = root.insertBefore(n1, node);
				//root.appendChild(n1);
			}
		} catch (Exception e) {
			e.printStackTrace();
		}

		// <property name="x" value="y"/>
		first = true;
		node = root.getFirstChild();
		for (Iterator iterator = v2p.keySet().iterator(); iterator.hasNext();) {
			Node prop = null;
			String key = (String) iterator.next();
			Object o = v2p.get(key);
			if (o instanceof Node) {
				prop = (Node) o;
			} else if (o instanceof String) {
				String value = (String) o;
				Element e = doc.createElement("property"); //$NON-NLS-1$
				e.setAttribute("name", key); //$NON-NLS-1$
				e.setAttribute("value", value); //$NON-NLS-1$
				prop = e;
			}
			if (prop != null) {
				if (first) {
					first = false;
				} else {
					node = node.getNextSibling();
				}
				node = root.insertBefore(prop, node);
				//root.appendChild(prop);
			}
		}

	}

	/**
	 * Add property tag.
	 */
	protected void createPropertyM() {
		boolean first = true;
		Node node = root.getFirstChild();
		for (Iterator iterator = variable2valueMap.keySet().iterator(); iterator
				.hasNext();) {
			String key = (String) iterator.next();
			String value = (String) variable2valueMap.get(key);
			Element prop = doc.createElement("property"); //$NON-NLS-1$
			prop.setAttribute("name", key); //$NON-NLS-1$
			prop.setAttribute("value", value); //$NON-NLS-1$
			if (first) {
				first = false;
			} else {
				node = node.getNextSibling();
			}
			node = root.insertBefore(prop, node);
		}
		
		IPath home = JavaCore.getClasspathVariable("ECLIPSE_HOME"); //$NON-NLS-1$
		String home_ecl = home.toString();
		Element ecl = doc.createElement("property"); //$NON-NLS-1$
		ecl.setAttribute("name", "ECLIPSE_HOME"); //$NON-NLS-1$ //$NON-NLS-2$
		ecl.setAttribute("value", home_ecl); //$NON-NLS-1$ //$NON-NLS-2$
		root.insertBefore(ecl, root.getFirstChild());
	}

	protected boolean create(List projects, IProgressMonitor pm) throws Exception {
		createDoc();
		createRoot("feature");
		createFeature();
		createFetch();
		addSubProperties();		
		createProperty();
		createPropertyM();
		return true;
	}

}