/*******************************************************************************
 * Copyright (c) 2008 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Valery Kokhan - Initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.ant.pde;

import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.higgins.ant.service.AbstractBuildFileCreator;
import org.eclipse.higgins.ant.service.ExportUtil;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.pde.internal.core.ifeature.IFeatureModel;
import org.w3c.dom.Comment;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class SiteBuildFileCreator extends AbstractBuildFileCreator {

	protected final String PROPERTY = "property";
	protected final String TARGET = "target";
	protected final String NAME = "name";
	protected final String VALUE = "value";
	protected final String MKDIR = "mkdir";
	protected final String DIR = "dir";
	protected final String COPY = "copy";
	protected final String FILESET = "fileset";
	protected final String UNZIP = "unzip";
	protected final String ANT = "ant";
	protected final String ANTCALL = "antcall";
	protected final String DELETE = "delete";

	protected final String TARGET_CLEAN = "-clean";
	protected final String TARGET_SITE = "site";
	protected final String TARGET_UNZIP_FEATURES = "-unzipFeatures";
	protected final String TARGET_BUILD_FEATURES = "-buildFeatures";
	protected final String TARGET_COPY_DEP_PROJECTS = "-copyDepProjects";
	protected final String TARGET_INIT_SITE_BASE = "-initSiteBase";
	protected final String TARGET_MAKE_FOLDERS = "-makeFolders";

	protected IFeatureModel[] featureModels;

	public SiteBuildFileCreator() {
		super();
		System.out.println("SiteBuildFileCreator");
	}

	public SiteBuildFileCreator(IProject project) {
		super(projectManager.getProject(project));
	}

	protected void Init(IProject project) {
		super.Init(project);
		featureModels = getFeatureModels();
	}

	protected IFeatureModel[] getFeatureModels() {
		IFeatureModel[] models = new IFeatureModel[0];
		if (projectManager instanceof ProjectManager) {
			models = ((ProjectManager) projectManager).getFeatureModels(project);
		}
		return models;
	}

	private Element createElement(String name) {
		Element element = doc.createElement(name);
		return element;
	}

	private Element createElement(String eName, String attrName,
			String attrValue) {
		Element element = doc.createElement(eName); //$NON-NLS-1$
		element.setAttribute(attrName, attrValue); //$NON-NLS-1$ //$NON-NLS-2$
		return element;
	}

	private Element createElement(String eName, String attrName1,
			String attrValue1, String attrName2, String attrValue2) {
		Element element = createElement(eName, attrName1, attrValue1);
		element.setAttribute(attrName2, attrValue2); //$NON-NLS-1$ //$NON-NLS-2$
		return element;
	}

	private void setAttribute(Element element, String name, String value) {
		if (value != null && !value.equals("")) {
			element.setAttribute(name, value);
		}
	}

	private Element createTarget(String name, String depends, String unless) {
		Element element = createElement("target", "name", name);
		setAttribute(element, "unless", unless);
		setAttribute(element, "depends", depends);
		return element;
	}

	private Element createAnt(String antfile, String dir, String target,
			String output, String inheritAll, String inheritRefs) {
		Element element = createElement(ANT);
		setAttribute(element, "antfile", antfile);
		setAttribute(element, "dir", dir);
		setAttribute(element, "target", target);
		setAttribute(element, "output", output);
		setAttribute(element, "inheritall", inheritAll);
		setAttribute(element, "inheritrefs", inheritRefs);
		return element;
	}
	
	private Element createAntcall(String target, String inheritAll, String inheritRefs) {
		Element element = createElement(ANTCALL);
		setAttribute(element, "target", target);
		setAttribute(element, "inheritAll", inheritAll);
		setAttribute(element, "inheritRefs", inheritRefs);
		return element;
	}

	private void appendElement(String eName, String attrName, String attrValue) {
		Element element = createElement(eName, attrName, attrValue);
		root.appendChild(element);
	}

	private void appendElement(String eName, String attrName1,
			String attrValue1, String attrName2, String attrValue2) {
		Element element = createElement(eName, attrName1, attrValue1,
				attrName2, attrValue2);
		root.appendChild(element);
	}

	private Element createProperty(String name, String value) {
		Element element = createElement(PROPERTY, NAME, name, VALUE, value);
		return element;
	}

	private Element createCopy(String encoding, String failOnError, String overWrite, String toDir) {
		Element copy = createElement(COPY);
		setAttribute(copy, "encoding", encoding);
		setAttribute(copy, "failonerror", failOnError);
		setAttribute(copy, "overwrite", overWrite);
		setAttribute(copy, "todir", toDir);
		return copy;
	}
	
	private Element createCopy(String file, String tofile, String overwrite) {
		Element copy = createElement(COPY);
		setAttribute(copy, "file", file);
		setAttribute(copy, "tofile", tofile);
		setAttribute(copy, "overwrite", overwrite);
		return copy;
	}

	private Element createFileset(String dir, String excludes) {
		Element fileset = createElement(FILESET);
		setAttribute(fileset, "dir", dir);
		setAttribute(fileset, "excludes", excludes);
		return fileset;
	}

	private void appendProperty(String name, String value) {
		appendElement(PROPERTY, NAME, name, VALUE, value);
	}

	private void appendProperty(Element element, String name, String value) {
		Element property = createProperty(name, value);
		element.appendChild(property);
	}

	private void appendMkdir(Element element, String dir) {
		Element mkdir = createElement(MKDIR, DIR, dir);
		element.appendChild(mkdir);
	}
	
	private void appendUnzip(Element element, String src, String dest, String overwrite) {
		Element unzip = createElement(UNZIP);
		setAttribute(unzip, "src", src);
		setAttribute(unzip, "dest", dest);
		setAttribute(unzip, "overwrite", overwrite);
		element.appendChild(unzip);
	}

	private void appendComment(String data) {
		Comment comment = doc.createComment(data);
		root.appendChild(comment);
	}

	private void appendDoc(String name) {
		Document tdoc = projectManager.getTemplate("higgins2ant.pde", name);
		Element troot = tdoc.getDocumentElement();
		NodeList nl = troot.getChildNodes();
		for (int i = 0; i < nl.getLength(); i++) {
			Node n = nl.item(i);
			Node n1 = doc.importNode(n, true);
			root.appendChild(n1);
		}
	}

	protected void createProperty() {
		IPath home = JavaCore.getClasspathVariable("ECLIPSE_HOME");
		String home_ecl = home.toString();
		Element ecl = doc.createElement("property");
		ecl.setAttribute("name", "ECLIPSE_HOME");
		ecl.setAttribute("value", home_ecl);
		Node node = root.insertBefore(ecl, root.getFirstChild());
		
		String location = projectName + ".location";
		String relativePath = ExportUtil.getRelativePath(projectRoot, projectRoot);

		List list = getSubProjects();
		Element element = createElement(PROPERTY, NAME, location, VALUE, relativePath);
		node = node.getNextSibling();
		node = root.insertBefore(element, node);
		
		for (Iterator iterator = list.iterator(); iterator.hasNext();) {
			IProject subProject = projectManager.getProject(iterator.next());
			location = projectManager.getProjectName(subProject) + ".location";
			String subProjectRoot = projectManager.getProjectRoot(subProject);
			relativePath = ExportUtil.getRelativePath(subProjectRoot, projectRoot);
			// variable2valueMap.put(location, relativePath);
			element = createElement(PROPERTY, NAME, location, VALUE, relativePath);
			node = node.getNextSibling();
			node = root.insertBefore(element, node);
		}
		try {
			Document tdoc = projectManager.getTemplate("higgins2ant.pde", "siteProperties");
			Element troot = tdoc.getDocumentElement();
			NodeList nl = troot.getChildNodes();
			int length = nl.getLength();
			for (int i = 0; i < length; i++) {
				Node n = nl.item(i);
				Node n1 = doc.importNode(n, true);
				node = node.getNextSibling();
				node = root.insertBefore(n1, node);
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		
	}

	private Element createUnzipFeature(IFeatureModel model) {
		String name = projectManager.getProjectName(model);
		
		Element unzip = createElement(UNZIP);
		setAttribute(unzip, "src", "${" + name + ".location}/build/features/" + name + "_" + model.getFeature().getVersion() + ".zip");
		setAttribute(unzip, "dest", "${build.dir}");
		setAttribute(unzip, "overwrite", "true");
		return unzip;
	}

	protected void createUnzipFeatures(Element target) {
		for (int i = 0; i < featureModels.length; i++) {
			Element copy = createUnzipFeature(featureModels[i]);
			target.appendChild(copy);
		}
	}
	
	protected Element createBuildFeature(IFeatureModel model) {
		String name = projectManager.getProjectName(model);
		Element ant = createAnt("build.xml", "${" + name + ".location}/", "feature", "", "", "");
		return ant; 
	}

	protected void createBuildFeatures(Element target) {
		for (int i = 0; i < featureModels.length; i++) {
			Element ant = createBuildFeature(featureModels[i]);
			target.appendChild(ant);
		}
	}
	
	protected void createUnzipFeatures() {
		appendComment(COMMENT_SEPARATOR);
		Element target = createTarget(TARGET_UNZIP_FEATURES,
				TARGET_BUILD_FEATURES, "");
		for (int i = 0; i < featureModels.length; i++) {
			IFeatureModel model = featureModels[i];
			String name = projectManager.getProjectName(model);
			String src = "${temp.dir}/" + name + "/build/features/" + name + "_" + model.getFeature().getVersion() + ".zip";
			String dest = "${build.dir}";
			appendUnzip(target, src, dest, "true");
		}
		root.appendChild(target);
	}

	protected void createSite() {
		try {
				Document tdoc = projectManager.getTemplate("higgins2ant.pde", "siteBuilder"); 
				Element troot = tdoc.getDocumentElement();
				NodeList nl = troot.getChildNodes();
				int length = nl.getLength();
				for (int i = 0; i < length; i++) {
					Node n = nl.item(i);
					Node n1 = doc.importNode(n, true);
					if (n1.getNodeType() == Node.ELEMENT_NODE ) {
						Element e = (Element) n1;
						if (e.getNodeName().equals("target") && e.getAttribute("name").equals("-buildFeatures")) {
							createBuildFeatures(e);
						} else if (e.getNodeName().equals("target") && e.getAttribute("name").equals("-unzipFeatures")) {
							createUnzipFeatures(e);
						}
					}
					root.appendChild(n1);
				}
			
		} catch (Exception e) {
			e.printStackTrace();
		}
		
	}
	
	protected void createClean() {
		appendComment(COMMENT_SEPARATOR);
		Element target = createTarget(TARGET_CLEAN, TARGET_INIT_SITE_BASE, "");
		Element delete = createElement(DELETE, "dir", "${temp.dir}");
		target.appendChild(delete);
		root.appendChild(target);
	}

	protected boolean create(List projects, IProgressMonitor pm) throws Exception {
		createDoc();
		createRoot("site");
		createFetch();
		createProperty();
		createSite();
		return true;
	}

}