/*******************************************************************************
 * Copyright (c) 2007-2008 Parity Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Valery Kokhan - initial implementation
 *******************************************************************************/

package org.eclipse.higgins.ant.core;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.parsers.ParserConfigurationException;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.higgins.ant.service.AbstractBuildFileCreator;
import org.eclipse.higgins.ant.service.BuildCreator;
import org.eclipse.higgins.ant.service.EclipseClasspath;
import org.eclipse.higgins.ant.service.ExportUtil;
import org.eclipse.jdt.core.IClasspathContainer;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.w3c.dom.CDATASection;
import org.w3c.dom.Comment;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.ProcessingInstruction;
import org.xml.sax.SAXException;

/**
 * Creates build.xml file.
 */
public class JavaProjectBuildFileCreator extends AbstractBuildFileCreator {

	private IJavaProject javaProject;

	private EclipseClasspath classpath;

	private Set visited = new TreeSet(); // record used subclasspaths

	private Node classpathNode;
	
	public JavaProjectBuildFileCreator() {
		super();
		//System.out.println("JavaProjectBuildFileCreator() for java project");
	}

	public JavaProjectBuildFileCreator(IJavaProject project) throws Exception {
		super(projectManager.getProject(project));
		this.javaProject = project;
		this.classpath = new EclipseClasspath(project);
		this.variable2valueMap = new LinkedHashMap();
	}
	
	protected void Init(IProject project){
		super.Init(project);
		this.javaProject = projectManager.getJavaProject(project);
		try {
			this.classpath = new EclipseClasspath(this.javaProject);
		} catch (JavaModelException e) {
			e.printStackTrace();
		}
		this.variable2valueMap = new LinkedHashMap();
	}

	/**
	 * Add property tag.
	 * 
	 * @param srcDirs
	 */
	protected void createProperty(EclipseClasspath classpath) {
		Map v2p = new LinkedHashMap();

		List srcDirs = new ArrayList();
		List pkgNames = new ArrayList();
		for (int i = 0; i < classpath.getSrcDirs().size(); i++) {
			String dir = (String) classpath.getSrcDirs().get(i);
			if (!EclipseClasspath.isReference(dir)) {
				srcDirs.add(dir);
				pkgNames.add(getRootPackageName(dir,
						(IClasspathEntry) classpath.getSrcDir2Entry().get(dir)));
			}
		}

		Comment c = doc.createComment(" ======================================== ");
		v2p.put("c1", c);

		if (srcDirs.size() == 1) {
			v2p.put("source.dir", "${basedir}/" + srcDirs.get(0));
		} else {
			for (int i = 0; i < srcDirs.size(); i++) {
				v2p.put("source.dir." + i, "${basedir}/" + srcDirs.get(i));
			}
		}

		v2p.put("sources.dir", "${basedir}/..");
		v2p.put("build.dir", "${basedir}/build");
		v2p.put("build.dir.bin", "${build.dir}/bin");
		v2p.put("build.dir.lib", "${build.dir}/lib");
		v2p.put("build.dir.doc", "${build.dir}/doc");

		c = doc.createComment(" ======================================== ");
		v2p.put("c1.1", c);

		String jarName = projectName;
		jarName = jarName.replaceAll("org\\.eclipse\\.higgins", "higgins");
		jarName = jarName.replaceAll("\\.", "-");
		v2p.put("build.file.name", jarName);

		String javadocPackages = "";
		boolean first = true;
		for (int i = 0; i < pkgNames.size(); i++) {
			if (first) {
				first = false;
			} else {
				javadocPackages += ",";
			}
			javadocPackages += pkgNames.get(i);
		}
		v2p.put("build.doc.packages", javadocPackages);
		
		Node node;
		try {
			Document tdoc = projectManager.getTemplate("higgins2ant.core", "properties");
			Element troot = tdoc.getDocumentElement();
			NodeList nl = troot.getChildNodes();
			int length = nl.getLength();
			first = true;
			node = root.getFirstChild();
			for (int i = 0; i < length; i++) {
				Node n = nl.item(i);
				Node n1 = doc.importNode(n, true);
				if (first) {
					first = false;
				} else {
					node = node.getNextSibling();
				}
				node = root.insertBefore(n1, node);
			}
		} catch (Exception e) {
			e.printStackTrace();
		}

		c = doc.createComment(" ======================================== ");
		v2p.put("c11", c);

		Element path = doc.createElement("path");
		path.setAttribute("id", "project.source");
		if (srcDirs.size() == 1) {
			Element pathelement = doc.createElement("pathelement");
			pathelement.setAttribute("location", "${source.dir}");
			path.appendChild(pathelement);
		} else {
			for (int i = 0; i < srcDirs.size(); i++) {
				Element pathelement = doc.createElement("pathelement");
				pathelement.setAttribute("location", "${source.dir." + i + "}");
				path.appendChild(pathelement);
			}
		}

		v2p.put("path", path);

		// <property name="x" value="y"/>
		first = true;
		node = root.getFirstChild();
		for (Iterator iterator = v2p.keySet().iterator(); iterator.hasNext();) {
			Node prop = null;
			String key = (String) iterator.next();
			Object o = v2p.get(key);
			if (o instanceof Node) {
				prop = (Node) o;
			} else if (o instanceof String) {
				String value = (String) o;
				Element e = doc.createElement("property"); //$NON-NLS-1$
				e.setAttribute("name", key); //$NON-NLS-1$
				e.setAttribute("value", value); //$NON-NLS-1$
				prop = e;
			}
			if (prop != null) {
				if (first) {
					first = false;
				} else {
					node = node.getNextSibling();
				}
				node = root.insertBefore(prop, node);
			}
		}

	}

	/**
	 * Add properties of subprojects to internal properties map.
	 */
	private void addSubProperties(IJavaProject subproject,
			EclipseClasspath classpath) throws JavaModelException {
		String location = subproject.getProject().getName() + ".location";
		String relativePath = ExportUtil.getRelativePath(projectRoot, projectRoot);
		variable2valueMap.put(location, relativePath);
		variable2valueMap.putAll(classpath.getVariable2valueMap());
		for (Iterator iterator = projectManager.getClasspathProjectsRecursive(
				subproject).iterator(); iterator.hasNext();) {
			IJavaProject subProject = (IJavaProject) iterator.next();
			//String location = subProject.getProject().getName() + ".location"; //$NON-NLS-1$
			location = subProject.getProject().getName() + ".location"; //$NON-NLS-1$
			// add subproject properties to variable2valueMap
			//String subProjectRoot = ExportUtil.getProjectRoot(subProject);
			String subProjectRoot = projectManager.getProjectRoot(subProject);
			//String relativePath = ExportUtil.getRelativePath(subProjectRoot, projectRoot);
			relativePath = ExportUtil.getRelativePath(subProjectRoot, projectRoot);
			variable2valueMap.put(location, relativePath);
		}
	}

	/**
	 * Create -version target. Reads project version from plugin's manifest
	 * file. Initialises version properties.
	 */
	protected void createVersion() {
		Comment c = doc.createComment(" Reads project version from plugin's manifest file ");
		root.appendChild(c);
		Element element = doc.createElement("target");
		element.setAttribute("name", "-version");
		element.setAttribute("unless", "project.version");

		Element property = doc.createElement("property");
		property.setAttribute("file", "${project.manifest}");
		property.setAttribute("prefix", "project.mf");
		element.appendChild(property);

		property = doc.createElement("xmlproperty");
		property.setAttribute("file", "${project.xml}");
		property.setAttribute("prefix", "project.xml");
		property.setAttribute("collapseattributes", "true");
		element.appendChild(property);

		Element condition = doc.createElement("condition");
		condition.setAttribute("property", "project.version");
		condition.setAttribute("value", "${project.mf.Bundle-Version}");
		element.appendChild(condition);

		Element isset = doc.createElement("isset");
		isset.setAttribute("property", "project.mf.Bundle-Version");
		condition.appendChild(isset);

		condition = doc.createElement("condition");
		condition.setAttribute("property", "project.version");
		condition.setAttribute("value", "${project.xml.plugin.version}");
		element.appendChild(condition);

		isset = doc.createElement("isset");
		isset.setAttribute("property", "project.xml.plugin.version");
		condition.appendChild(isset);

		property = doc.createElement("property");
		property.setAttribute("name", "project.version");
		property.setAttribute("value", "0.1.0");
		element.appendChild(property);

		condition = doc.createElement("condition");
		condition.setAttribute("property", "project.name");
		condition.setAttribute("value", "${project.mf.Bundle-Name}");
		element.appendChild(condition);

		isset = doc.createElement("isset");
		isset.setAttribute("property", "project.mf.Bundle-Name");
		condition.appendChild(isset);

		condition = doc.createElement("condition");
		condition.setAttribute("property", "project.name");
		condition.setAttribute("value", "${project.xml.plugin.name}");
		element.appendChild(condition);

		isset = doc.createElement("isset");
		isset.setAttribute("property", "project.xml.plugin.name");
		condition.appendChild(isset);

		root.appendChild(element);
	}

	/**
	 * Create -override target. Overrides plugin dependencies classpath.
	 */
	protected void createOverride(EclipseClasspath classpath) {
		Comment c = doc.createComment(" Overrides plug-in dependencies classpath ");
		root.appendChild(c);
		Element element = doc.createElement("target");
		element.setAttribute("name", "-override");
		element.setAttribute("depends", "-override_e,-override_n");

		root.appendChild(element);

		element = doc.createElement("target");
		element.setAttribute("name", "-override_e");
		element.setAttribute("if", "eclipse.available");

		IClasspathContainer container = classpath
				.resolveUserLibraryReference(classpath.getPluginRef());
		if (container != null) {
			String name = ExportUtil.removePrefixAndSuffix(classpath.getPluginRef(),
					"${", "}");
			Element path = doc.createElement("path");
			path.setAttribute("id", name);
			element.appendChild(path);

			Element fileset = doc.createElement("fileset");
			fileset.setAttribute("dir", "${eclipse.home}/plugins");
			IClasspathEntry entries[] = container.getClasspathEntries();
			for (int i = 0; i < entries.length; i++) {
				String jarFile = entries[i].getPath().toString();
				IPath home = JavaCore.getClasspathVariable("ECLIPSE_HOME");
				if (home.isPrefixOf(entries[i].getPath())) {
					jarFile = "${eclipse.home}"
							+ jarFile.substring(home.toString().length());
					// Try to remove version information from plugin
					// dependencies
					// Pattern p =
					// Pattern.compile("(\\$\\{eclipse.home}/plugins/)([^_]*)_(\\S*).jar");
					Pattern p = Pattern
							.compile("(\\$\\{eclipse.home}/plugins/)([^_]*)_([^/]*)(.*).jar");
					Matcher m = p.matcher(jarFile);
					if (m.matches()) {
						String jname = m.replaceAll("$2_*$4.jar");
						Element includeElement = doc.createElement("include");
						includeElement.setAttribute("name", jname);
						fileset.appendChild(includeElement);
					}
				}
			}
			if (fileset.hasChildNodes()) {
				path.appendChild(fileset);
			}
		}

		root.appendChild(element);

		element = doc.createElement("target");
		element.setAttribute("name", "-override_n");
		element.setAttribute("unless", "eclipse.available");

		if (container != null) {
			String name = ExportUtil.removePrefixAndSuffix(classpath.getPluginRef(), "${", "}");
			Element path = doc.createElement("path");
			path.setAttribute("id", name);
			element.appendChild(path);
		}

		root.appendChild(element);
	}

	protected void createCopyJar(List srcDirs) {
		Comment c = doc.createComment("Copies project's JAR file to user defined dir");
		root.appendChild(c);

		Element element = doc.createElement("target");
		element.setAttribute("name", "copy.jar");
		element.setAttribute("depends", "jar");
		element.setAttribute("if", "toDir");

		if (srcDirs.size() > 0) {
			Element copy = doc.createElement("copy");
			copy.setAttribute("file",
			"${build.dir.lib}/${build.file.name}_${project.version}.jar");
			copy.setAttribute("todir", "${toDir}");
			element.appendChild(copy);
		}

		root.appendChild(element);
	}

	protected void createCopyDepJars() throws JavaModelException {
		Comment c = doc.createComment("Copies all dependency JARs to user defined dir");
		root.appendChild(c);

		Element element = doc.createElement("target");
		element.setAttribute("name", "-copy.dep.jars");
		element.setAttribute("if", "toDir");
		element.setAttribute("depends", "-load.build.properties");
		element.setAttribute("unless", "build.properties." + projectName + ".copy.all.jars.done");


		for (Iterator iterator = projectManager.getClasspathProjectsRecursive(
				javaProject).iterator(); iterator.hasNext();) {
			IJavaProject subProject = (IJavaProject) iterator.next();
			Element ant = doc.createElement("ant");
			ant.setAttribute("antfile", "build.xml");
			ant.setAttribute("dir", "${"
					+ subProject.getProject().getName()
					+ ".location}");
			ant.setAttribute("inheritAll", "false");
			ant.setAttribute("target", "copy.all.jars");

			Element proptwo = doc.createElement("propertyset");
			proptwo.setAttribute("refid", "copy.properties");

			ant.appendChild(proptwo);

			element.appendChild(ant);
		}

		root.appendChild(element);
	}

	protected void createCopyAllJars() throws JavaModelException {
		Comment c = doc.createComment("Copies all JARs to user defined dir");
		root.appendChild(c);

		Element element = doc.createElement("target");
		element.setAttribute("name", "copy.all.jars");
		element.setAttribute("if", "toDir");
		element.setAttribute("depends", "copy.jar,-copy.dep.jars");
		element.setAttribute("unless", "build.properties." + projectName + ".copy.all.jars.done");

		Element property = doc.createElement("property");
		property.setAttribute("name", "build.properties." + projectName + ".copy.all.jars.done");
		property.setAttribute("value", "true");
		element.appendChild(property);
		Element antcall = doc.createElement("antcall");
		antcall.setAttribute("target", "-save.build.properties");
		element.appendChild(antcall);
		
		root.appendChild(element);
	}

	protected void createCopyLibraries(EclipseClasspath classpath)
			throws JavaModelException {
		Comment c = doc.createComment("Copies all project required libraries to user defined dir");
		root.appendChild(c);

		Element element = doc.createElement("target");
		element.setAttribute("name", "copy.libs");
		element.setAttribute("if", "toDir");
		Map paths = new HashMap();
		List list = ExportUtil.removeDuplicates(classpath.getRawClassPathEntries());
		//List listOfFolders = new ArrayList(1);
		for (int i = 0; i < list.size(); i++) {
			if (!((String) list.get(i)).endsWith("}")
					&& ((((String) list.get(i)).endsWith(".jar")) 
					|| (((String) list.get(i)).endsWith(".zip")))) {
				String rawPath = (String) list.get(i);
//				String s = ExportUtil.getRelativePath(rawPath, projectRoot);
				//paths.add(rawPath);
				String folderPath;
				String path;
				if (rawPath.startsWith("${")) {
					folderPath = rawPath.substring(0, rawPath.lastIndexOf('}') + 1);
					path = rawPath.substring(rawPath.lastIndexOf('}') + 2);
				} else {
					folderPath = rawPath.substring(0, rawPath.lastIndexOf('/'));
					path = rawPath.substring(rawPath.lastIndexOf('/') + 1);
				}
				List l = (List) paths.get(folderPath);
				if (l == null) {
					l = new ArrayList();
					paths.put(folderPath, l);
				}
				l.add(path);
			}
		}
		//listOfFolders = ExportUtil.removeDuplicates(listOfFolders);

		Iterator itr = paths.keySet().iterator();

		if (itr.hasNext()) {
			Element copy = doc.createElement("copy");
			copy.setAttribute("todir", "${toDir}");
			copy.setAttribute("flatten", "true");

			while (itr.hasNext()) {
				String curFolder = (String) itr.next();
				Element fileset = doc.createElement("fileset");
				List l = (List) paths.get(curFolder);
				for (int k = 0; k < l.size(); k++) {
					String s = (String) l.get(k);
					Element lib = doc.createElement("include");
					lib.setAttribute("name", s);
					fileset.appendChild(lib);
				}
				for (Iterator iterator = projectManager.getClasspathProjectsRecursive(
						javaProject).iterator(); iterator.hasNext();) {
					IJavaProject subProject = (IJavaProject) iterator.next();
					//String subProjectRoot = ExportUtil.getProjectRoot(subProject);
					String subProjectRoot = projectManager.getProjectRoot(subProject);
					int pathIsHere = curFolder.indexOf(subProjectRoot);
					if ((pathIsHere > -1)
							&& (curFolder.charAt(pathIsHere
									+ subProjectRoot.length()) == '/')) {
						curFolder = "${" + subProject.getProject().getName()
						+ ".location}"
						+ curFolder.substring(subProjectRoot.length());
						break;
					}
				}

				if (!curFolder.startsWith("${")) {
					curFolder = "${basedir}/"
						+ ExportUtil.getRelativePath(curFolder, projectRoot);

				}
				fileset.setAttribute("dir", curFolder);
				copy.appendChild(fileset);
			}

			element.appendChild(copy);
		}

		root.appendChild(element);
	}

	/**
	 * Create jar target.
	 */
	protected void createJar(List srcDirs) {
		Comment c = doc.createComment(" Creates jar file ");
		root.appendChild(c);
		Element element = doc.createElement("target");
		element.setAttribute("name", "jar");
		element.setAttribute("depends", "build");
		element.setAttribute("unless", "build.properties." + projectName + ".jar.done");

		if (srcDirs.size() > 0) {
			Element jar = doc.createElement("jar");
			jar.setAttribute("jarfile", "${build.dir.lib}/${build.file.name}_${project.version}.jar");
			jar.setAttribute("basedir", "${build.dir.bin}");
			jar.setAttribute("filesonly", "false");
			element.appendChild(jar);

			Element property = doc.createElement("property");
			property.setAttribute("name", "build.properties." + projectName + ".jar.done");
			property.setAttribute("value", "true");
			Element antcall = doc.createElement("antcall");
			antcall.setAttribute("target", "-save.build.properties");
			element.appendChild(property);
			element.appendChild(antcall);
		}

		root.appendChild(element);
	}

	/**
	 * Create plugin target.
	 */
	protected void createPlugin(EclipseClasspath classpath) {
		try {
			if (projectManager.isPluginProject(javaProject.getProject())) {
				
				Document tdoc = projectManager.getTemplate("higgins2ant.core", "pluginBuilder");;
				Element troot = tdoc.getDocumentElement();
				NodeList nl = troot.getChildNodes();
				int length = nl.getLength();
				for (int i = 0; i < length; i++) {
					Node n = nl.item(i);
					Node n1 = doc.importNode(n, true);
					if (n1.getNodeType() == Node.ELEMENT_NODE ) {
						Element e = (Element) n1;
						if (e.getNodeName().equals("target") && e.getAttribute("name").equals("-initPluginSources")) {
							createInitPluginSources(classpath, e);
						}
					}
					root.appendChild(n1);
				}
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	private void createInitPluginSources(EclipseClasspath classpath, Element target) throws JavaModelException {
//		Element target = doc.createElement("target");
//		target.setAttribute("name", "-initPluginSources");
//		target.setAttribute("depends", "-makeFolders");
		
		Element copy = createCopyProjectSources(javaProject, classpath);
		target.appendChild(copy);
		
		List subProjects = projectManager.getClasspathProjectsRecursive(javaProject);
		for (Iterator iterator = subProjects.iterator(); iterator.hasNext();) {
			IJavaProject subProject = (IJavaProject) iterator.next();
			EclipseClasspath cp = new EclipseClasspath(subProject);
			copy = createCopyProjectSources(subProject, cp);
			target.appendChild(copy);
		}
		
//		root.appendChild(target);
	}
	
	private Element createCopyProjectSources(IJavaProject project, EclipseClasspath classpath) {
		String projectName = project.getProject().getName();
		Element copy = doc.createElement("copy");
		copy.setAttribute("todir", "${temp.dir.plugins}" + "/" + projectName);
		copy.setAttribute("failonerror", "true");
		
		Element fileset = doc.createElement("fileset");
		fileset.setAttribute("dir", "${" + projectName + ".location}");
		fileset.setAttribute("excludes", DEFAULT_EXCLUDES);
		copy.appendChild(fileset);
		
		if (classpath.getSrcDirs().size() > 0) {
			int size = classpath.getSrcDirs().size(); 
			for (int i = 0; i < size; i++) {
				String srcDir = (String) classpath.getSrcDirs().get(i);
				if (!EclipseClasspath.isReference(srcDir)) {
					List exclusions = (List) classpath.getExclusionLists().get(i);
					for (Iterator iter = exclusions.iterator(); iter.hasNext();) {
						String exclusion = (String) iter.next();
						Element excludeElement = doc.createElement("exclude");
						excludeElement.setAttribute("name", srcDir + "/" + exclusion);
						fileset.appendChild(excludeElement);
					}
				}
			}
		}
		return copy;
	}

	/**
	 * Create javadoc target.
	 */
	protected void createJavadoc() {
		Comment c = doc.createComment(" ======================================== ");
		root.appendChild(c);
		Element element = doc.createElement("target");
		element.setAttribute("name", "-prejavadoc");
		element.setAttribute("depends", "-init");

		Element delete = doc.createElement("delete");
		element.appendChild(delete);

		Element fileset = doc.createElement("fileset");
		fileset.setAttribute("dir", "${build.dir.doc}");
		delete.appendChild(fileset);

		Element include = doc.createElement("include");
		include.setAttribute("name", "**/*");
		fileset.appendChild(include);

		Element exclude = doc.createElement("exclude");
		exclude.setAttribute("name", "**/CVS");
		fileset.appendChild(exclude);

		exclude = doc.createElement("exclude");
		exclude.setAttribute("name", "**/Entries");
		fileset.appendChild(exclude);

		exclude = doc.createElement("exclude");
		exclude.setAttribute("name", "**/Repository");
		fileset.appendChild(exclude);

		exclude = doc.createElement("exclude");
		exclude.setAttribute("name", "**/Root");
		fileset.appendChild(exclude);

		root.appendChild(element);

		/*
		 * <target name="javadoc" depends="init, -override"> <javadoc
		 * packagenames="org.eclipse.higgins.icard.*"
		 * excludepackagenames="${build.src.excludes}"
		 * sourcepath="${source.dir}" defaultexcludes="yes"
		 * destdir="${build.dir.doc}" author="true" version="true"
		 * windowtitle="${project.name}" use="true" public="true"
		 * source="${javacSource}"> <classpath
		 * refid="org.eclipse.higgins.icard.classpath"/> <doctitle><![CDATA[<h1>${project.name}</h1>]]></doctitle>
		 * <link href="http://java.sun.com/j2se/1.4.2/docs/api/"/> </javadoc>
		 * </target>
		 */
		c = doc.createComment(" Creates javadoc for the project ");
		root.appendChild(c);
		element = doc.createElement("target");
		element.setAttribute("name", "javadoc");
		element.setAttribute("depends", "-init, -override");

		Element javadoc = doc.createElement("javadoc");
		javadoc.setAttribute("packagenames", "${build.doc.packages}");
		javadoc.setAttribute("excludepackagenames", "${build.src.excludes}");
		javadoc.setAttribute("sourcepathref", "project.source");

		javadoc.setAttribute("defaultexcludes", "yes");
		javadoc.setAttribute("destdir", "${build.dir.doc}");
		javadoc.setAttribute("author", "true");
		javadoc.setAttribute("version", "true");
		javadoc.setAttribute("windowtitle", "${project.name}");
		javadoc.setAttribute("use", "true");
		javadoc.setAttribute("public", "true");
		javadoc.setAttribute("source", "${javacSource}");
		element.appendChild(javadoc);

		Element classpath = doc.createElement("classpath");
		classpath.setAttribute("refid", projectName + ".classpath");
		javadoc.appendChild(classpath);

		Element doctitle = doc.createElement("doctitle");
		javadoc.appendChild(doctitle);

		CDATASection data = doc.createCDATASection("<h1>${project.name}</h1");
		doctitle.appendChild(data);

		Element link = doc.createElement("link");
		link.setAttribute("href", "http://java.sun.com/j2se/1.4.2/docs/api/");

		javadoc.appendChild(link);

		root.appendChild(element);

		// <target name="-postjavadoc">
		// <replaceregexp
		// match="-- Generated by javadoc .* --"
		// replace="-- Generated by javadoc --" flags="g" byline="true">
		// <fileset dir="${build.dir.doc}" includes="**/*.html" />
		// </replaceregexp>
		// <zip destfile="${javadoczip}"
		// basedir="${build.dir.doc}"
		// update="true">
		// </zip>
		// </target>

		c = doc.createComment(" ======================================== ");
		root.appendChild(c);
		element = doc.createElement("target");
		element.setAttribute("name", "-postjavadoc");

		Element zip = doc.createElement("zip");
		zip.setAttribute("destfile",
				"${build.file.name}_${project.version}_javadoc.zip");
		zip.setAttribute("basedir", "${build.dir.doc}");
		zip.setAttribute("update", "true");
		element.appendChild(zip);

		root.appendChild(element);
	}

	/**
	 * Create clean target.
	 */
	protected void createClean() {
		Comment c = doc.createComment(" Cleans build folder of the project ");
		root.appendChild(c);

		Element element = doc.createElement("target");
		element.setAttribute("name", "clean");

		Element deleteElement = doc.createElement("delete");
		deleteElement.setAttribute("failonerror", "false");
		Element filesetElement = doc.createElement("fileset");
		filesetElement.setAttribute("dir", "${build.dir}");
		deleteElement.appendChild(filesetElement);
		deleteElement.setAttribute("includeemptydirs", "true");
		element.appendChild(deleteElement);

		root.appendChild(element);
	}

	/**
	 * Create cleanall target.
	 */
	protected void createCleanAll() throws JavaModelException {
		Comment c = doc.createComment(" Cleans build folders of dependency projects ");
		root.appendChild(c);

		Element element = doc.createElement("target");
		element.setAttribute("name", "cleanall");
		element.setAttribute("depends", "clean");

		List subProjects = projectManager.getClasspathProjectsRecursive(javaProject);
		for (Iterator iterator = subProjects.iterator(); iterator.hasNext();) {
			IJavaProject subProject = (IJavaProject) iterator.next();
			String targetName = "clean";
			Element antElement = doc.createElement("ant");
			antElement.setAttribute("dir", "${"
					+ subProject.getProject().getName() + ".location}"
					);
			antElement.setAttribute("antfile",  BuildCreator.getBUILD_XML());
			antElement.setAttribute("target", targetName);
			antElement.setAttribute("inheritAll", "false");
			element.appendChild(antElement);
		}

		root.appendChild(element);
	}

	/**
	 * Create build target.
	 * 
	 * @param exclusionLists
	 *            TODO
	 */
	protected void createBuild(List srcDirs, List classDirs, List inclusionLists, List exclusionLists) throws JavaModelException {
		Comment c = doc.createComment(" Builds the source code ");
		root.appendChild(c);
		Element element = doc.createElement("target");
		element.setAttribute("name", "build");
		element.setAttribute("depends", "-build-project");
		element.setAttribute("unless", "build.properties." + projectName + ".build.done");
		createCopyResources(srcDirs, classDirs, element);
		Element property = doc.createElement("property");
		property.setAttribute("name", "build.properties." + projectName + ".build.done");
		property.setAttribute("value", "true");
		Element antcall = doc.createElement("antcall");
		antcall.setAttribute("target", "-save.build.properties");
		element.appendChild(property);
		element.appendChild(antcall);
		root.appendChild(element);
	}

	protected void createBuildProject(List srcDirs, List classDirs, List inclusionLists, List exclusionLists) throws JavaModelException {
		Comment c = doc.createComment(" Compiles the source code of the project ");
		root.appendChild(c);
		Element element = doc.createElement("target");
		element.setAttribute("name", "-build-project");
		element.setAttribute("depends",
				"-init, -override, -build-subprojects");
		element.setAttribute("unless", "build.properties." + projectName
				+ ".build.done");

		if (srcDirs.size() > 0) {
			Element javac = doc.createElement("javac");
			javac.setAttribute("destdir", "${build.dir.bin}");
			javac.setAttribute("debug", "${javacDebugInfo}");
			javac.setAttribute("source", "${javacSource}");
			javac.setAttribute("target", "${javacTarget}");
			javac.setAttribute("verbose", "${javacVerbose}");
			javac.setAttribute("excludes", "${build.src.excludes}");

			Element src = doc.createElement("src");
			src.setAttribute("refid", "project.source");
			javac.appendChild(src);

			for (int i = 0; i < srcDirs.size(); i++) {
				String srcDir = (String) srcDirs.get(i);
				if (!EclipseClasspath.isReference(srcDir)) {
//					String classDir = (String) classDirs.get(i);
					List inclusions = (List) inclusionLists.get(i);
					List exclusions = (List) exclusionLists.get(i);
					for (Iterator iter = inclusions.iterator(); iter.hasNext();) {
						String inclusion = (String) iter.next();
						Element includeElement = doc.createElement("include"); //$NON-NLS-1$
						includeElement.setAttribute("name", inclusion); //$NON-NLS-1$
						javac.appendChild(includeElement);
					}
					for (Iterator iter = exclusions.iterator(); iter.hasNext();) {
						String exclusion = (String) iter.next();
						Element excludeElement = doc.createElement("exclude"); //$NON-NLS-1$
						excludeElement.setAttribute("name", exclusion); //$NON-NLS-1$
						javac.appendChild(excludeElement);
					}
				}
			}

			Element classpath = doc.createElement("classpath");
			classpath.setAttribute("refid", projectName + ".classpath");
			javac.appendChild(classpath);

			element.appendChild(javac);
		}

		root.appendChild(element);
	}

	protected void createBuildSubprojects(List srcDirs, List classDirs, List inclusionLists, List exclusionLists) throws JavaModelException {
		Comment c = doc.createComment(" Compiles the source code of the subprojects ");
		root.appendChild(c);
		Element element = doc.createElement("target"); //$NON-NLS-1$
		element.setAttribute("name", "-build-subprojects");
		element.setAttribute("depends", "-init, -override");
		element.setAttribute("unless", "build.properties." + projectName + ".build.done");
		List subProjects = projectManager.getClasspathProjectsRecursive(javaProject);
		for (Iterator iterator = subProjects.iterator(); iterator.hasNext();) {
			IJavaProject subProject = (IJavaProject) iterator.next();
			Element antElement = doc.createElement("ant");
			antElement.setAttribute("dir", "${" + subProject.getProject().getName() + ".location}");
			antElement.setAttribute("antfile",  BuildCreator.getBUILD_XML());
			antElement.setAttribute("target", "build");
			antElement.setAttribute("inheritAll", "false");

			Element propertyset = doc.createElement("propertyset");
			propertyset.setAttribute("refid", "build.properties");
			antElement.appendChild(propertyset);

			element.appendChild(antElement);
		}
		root.appendChild(element);
	}

	/*private String getRootPackageName(String dir, IClasspathEntry entry) {
		String name = "*";
		try {
			IPackageFragmentRoot r[] = javaProject.findPackageFragmentRoots(entry);
			for (int i = 0; i < r.length; i++) {
				IPackageFragment prev = null;
				IPackageFragment root = null;
				IJavaElement e[] = r[i].getChildren();
				for (int j = 0; j < e.length; j++) {
					if (e[j] instanceof IPackageFragment) {
						IPackageFragment pf = (IPackageFragment) e[j];
						if (!pf.isDefaultPackage()) {
							if (pf.containsJavaResources()) {
								if (root == null) {
									root = pf;
								} else {
									IPath p1 = pf.getPath();
									IPath p2 = root.getPath();
									if ((p2.segmentCount() == p1.segmentCount())
											&& (p2.matchingFirstSegments(p1) == (p1
													.segmentCount() - 1))) {
										root = prev;
									}
									break;
								}
							} else {
								if (root == null) {
									prev = pf;
								}
							}
						}
					}
				}
				name = root.getElementName() + ".*";
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		return name;
	}*/
	
	private String getRootPackageName(String dir, IClasspathEntry entry) {
		String name = "*";
		try {
			IPackageFragmentRoot r[] = javaProject.findPackageFragmentRoots(entry);
			for (int i = 0; i < r.length; i++) {
				IPackageFragment prev = null;
				IPackageFragment root = null;
				IJavaElement e[] = r[i].getChildren();
				for (int j = 0; j < e.length; j++) {
					if (e[j] instanceof IPackageFragment) {
						IPackageFragment pf = (IPackageFragment) e[j];
						if (!pf.isDefaultPackage()) {
							if (pf.containsJavaResources()) {
								if (root == null) {
									root = pf;
								} else {
									IPath p1 = pf.getPath();
									IPath p2 = root.getPath();
									if ((p2.segmentCount() == p1.segmentCount())
											&& (p2.matchingFirstSegments(p1) == (p1
													.segmentCount() - 1))) {
										root = prev;
									}
									break;
								}
							} else {
								if (root == null) {
									prev = pf;
								}
							}
						}
					}
				}
				if (root != null)
					name = root.getElementName() + ".*";
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		return name;
	}

	/**
	 * Add property tag.
	 */
	protected void createPropertyM() {
		// read debug options from Eclipse settings
		boolean source = JavaCore.GENERATE.equals(javaProject.getOption(JavaCore.COMPILER_SOURCE_FILE_ATTR, true));
		boolean lines = JavaCore.GENERATE.equals(javaProject.getOption(JavaCore.COMPILER_LINE_NUMBER_ATTR, true));
		boolean vars = JavaCore.GENERATE.equals(javaProject.getOption(JavaCore.COMPILER_LOCAL_VARIABLE_ATTR, true));

		List debuglevel = new ArrayList();
		if (source) {
			debuglevel.add("source"); //$NON-NLS-1$
		}
		if (lines) {
			debuglevel.add("lines"); //$NON-NLS-1$
		}
		if (vars) {
			debuglevel.add("vars"); //$NON-NLS-1$
		}
		if (debuglevel.size() == 0) {
			debuglevel.add("none"); //$NON-NLS-1$
		}
		variable2valueMap.put("debuglevel", ExportUtil.toString(debuglevel, ",")); //$NON-NLS-1$ //$NON-NLS-2$

		// "Generated .class files compatibility"
		String target = javaProject.getOption(JavaCore.COMPILER_CODEGEN_TARGET_PLATFORM, true);
		variable2valueMap.put("target", target); //$NON-NLS-1$

		// "Compiler compliance level"
		// String compliance = project.getOption(JavaCore.COMPILER_COMPLIANCE,
		// true);

		// "Source compatibility"
		String sourceLevel = javaProject.getOption(JavaCore.COMPILER_SOURCE, true);
		variable2valueMap.put("source", sourceLevel); //$NON-NLS-1$

		// <property name="x" value="y"/>
		boolean first = true;
		Node node = root.getFirstChild();
		for (Iterator iterator = variable2valueMap.keySet().iterator(); iterator
				.hasNext();) {
			String key = (String) iterator.next();
			String value = (String) variable2valueMap.get(key);
			Element prop = doc.createElement("property"); //$NON-NLS-1$
			prop.setAttribute("name", key); //$NON-NLS-1$
			prop.setAttribute("value", value); //$NON-NLS-1$
			if (first) {
				first = false;
			} else {
				node = node.getNextSibling();
			}
			node = root.insertBefore(prop, node);
		}

		// <property environment="env"/>
		Element env = doc.createElement("property"); //$NON-NLS-1$
		env.setAttribute("environment", "env"); //$NON-NLS-1$ //$NON-NLS-2$
		root.insertBefore(env, root.getFirstChild());

		IPath home = JavaCore.getClasspathVariable("ECLIPSE_HOME"); //$NON-NLS-1$
		String home_ecl = home.toString();
		// if(home_ecl=="" || home_ecl==null) home_ecl="${basedir}";
		Element ecl = doc.createElement("property"); //$NON-NLS-1$
		ecl.setAttribute("name", "ECLIPSE_HOME"); //$NON-NLS-1$ //$NON-NLS-2$
		ecl.setAttribute("value", home_ecl); //$NON-NLS-1$ //$NON-NLS-2$
		root.insertBefore(ecl, root.getFirstChild());

	}

/*	/**
	 * Create project tag.
	 *
	protected void createRoot() {
		// <project name="hello" default="build" basedir=".">
		root = doc.createElement("project"); //$NON-NLS-1$
		root.setAttribute("name", projectName); //$NON-NLS-1$
		root.setAttribute("default", "build"); //$NON-NLS-1$ //$NON-NLS-2$
		root.setAttribute("basedir", "."); //$NON-NLS-1$ //$NON-NLS-2$
		doc.appendChild(root);

		// <!-- warning -->
		Comment comment = doc.createComment(WARNING + ExportUtil.NEWLINE + NOTE);
		doc.insertBefore(comment, root);
	}*/

	/**
	 * Find buildfiles in projectroot directory and automatically import them.
	 */
	protected void createImports() {
		// <import file="javadoc.xml"/>
		File dir = new File(projectRoot);
		FilenameFilter filter = new FilenameFilter() {
			public boolean accept(File acceptDir, String name) {
				return name.endsWith(".xml") && !name.endsWith(BuildCreator.getBUILD_XML()); //$NON-NLS-1$
			}
		};

		File[] files = dir.listFiles(filter);
		if (files == null) {
			return;
		}
		for (int i = 0; i < files.length; i++) {
			// import file if it is an XML document with marker comment as first
			// child
			File file = files[i];
			Document docCandidate;
			try {
				docCandidate = ExportUtil.parseXmlFile(file);
				Node node = docCandidate.getFirstChild();
				if (node instanceof ProcessingInstruction
						&& IMPORT_BUILDFILE_PROCESSING_TARGET
								.equals(((ProcessingInstruction) node)
										.getTarget().trim())) {
					Element element = doc.createElement("import"); //$NON-NLS-1$
					element.setAttribute("file", file.getName()); //$NON-NLS-1$
					root.appendChild(element);
				}
			} catch (ParserConfigurationException e) {
				Higgins2AntCorePlugin
						.log(
								"invalid XML file not imported: " + file.getAbsolutePath(), e); //$NON-NLS-1$
			} catch (SAXException e) {
				Higgins2AntCorePlugin
						.log(
								"invalid XML file not imported: " + file.getAbsolutePath(), e); //$NON-NLS-1$
			} catch (IOException e) {
				Higgins2AntCorePlugin
						.log(
								"invalid XML file not imported: " + file.getAbsolutePath(), e); //$NON-NLS-1$
			}
		}
	}

	/**
	 * Create classpath tags.
	 */
	private void createClasspaths(EclipseClasspath classpath)
			throws JavaModelException {
		createClasspathsM(null, javaProject, classpath);
	}

	/**
	 * Create classpath tags. Allows to specify ID.
	 * 
	 * @param pathId
	 *            specify id, if null project name is used
	 */
	private void createClasspathsM(String pathId, IJavaProject currentProject,
			EclipseClasspath classpath) throws JavaModelException {
		if (currentProject == null) {
			Higgins2AntCorePlugin.log(
					"project is not loaded in workspace: " + pathId, null); //$NON-NLS-1$ 
			return;
		}
		boolean isMain = false;
		// <path id="hello.classpath">
		// <pathelement location="${hello.location}/classes"/>
		// <pathelement location="${hello.location}/x.jar"/>
		// <path refid="${goodbye.classpath}"/>
		// </path>
		Element element = doc.createElement("path"); //$NON-NLS-1$
		if (pathId == null) {
			pathId = currentProject.getProject().getName() + ".classpath"; //$NON-NLS-1$
			isMain = true;
		}
		element.setAttribute("id", pathId); //$NON-NLS-1$
		visited.add(pathId);
		variable2valueMap.putAll(classpath.getVariable2valueMap());
		for (Iterator iter = ExportUtil.removeDuplicates(
				classpath.getRawClassPathEntries()).iterator(); iter.hasNext();) {
			String entry = (String) iter.next();
			if (EclipseClasspath.isProjectReference(entry)) {
				Element pathElement = doc.createElement("path"); //$NON-NLS-1$
				IJavaProject referencedProject = EclipseClasspath
						.resolveProjectReference(entry);
				if (referencedProject == null) {
					Higgins2AntCorePlugin.log(
									"project is not loaded in workspace: " + pathId, null);
					continue;
				}
				String refPathId = referencedProject.getProject().getName()
						+ ".classpath"; //$NON-NLS-1$
				pathElement.setAttribute("refid", refPathId); //$NON-NLS-1$
				element.appendChild(pathElement);
				if (visited.add(refPathId)) {
					createClasspathsM(refPathId, referencedProject,
							new EclipseClasspath(referencedProject)); // recursion
				}
			} else if (EclipseClasspath.isUserLibraryReference(entry)
					|| EclipseClasspath.isUserSystemLibraryReference(entry)
					|| EclipseClasspath.isLibraryReference(entry)) {
				if (isMain) {
					addUserLibraryM(classpath, element, entry);
				}
			} else {
				// prefix with ${project.location}
				String prefix = ""; //$NON-NLS-1$
				if (!entry.startsWith("${") && // no variable ${var}/classes
												// //$NON-NLS-1$
						!projectName.equals(currentProject.getProject()
								.getName())) // not main project
				{
					//String currentProjectRoot = ExportUtil.getProjectRoot(currentProject);
					String currentProjectRoot = projectManager.getProjectRoot(currentProject);
					entry = ExportUtil.getRelativePath(entry,
							currentProjectRoot);
					if (!new Path(entry).isAbsolute()) {
						prefix = "${" + currentProject.getProject().getName() + ".location}/"; //$NON-NLS-1$ //$NON-NLS-2$
					}
				}
				Element pathElement = doc.createElement("pathelement"); //$NON-NLS-1$
				String path = ExportUtil.getRelativePath(prefix + entry,
						projectRoot);
				pathElement.setAttribute("location", path); //$NON-NLS-1$
				element.appendChild(pathElement);
			}
		}
		addToClasspathBlockM(element);
	}

	private void addUserLibraryM(EclipseClasspath classpath, Element element,
			String entry) {
		// add classpath reference
		Element pathElement = doc.createElement("path"); //$NON-NLS-1$
		IClasspathContainer container = classpath
				.resolveUserLibraryReference(entry);
		String name = ExportUtil.removePrefixAndSuffix(entry, "${", "}"); //$NON-NLS-1$ //$NON-NLS-2$
		pathElement.setAttribute("refid", name); //$NON-NLS-1$
		if (!EclipseClasspath.isUserSystemLibraryReference(entry)) {
			element.appendChild(pathElement);
		}
		// add classpath
		if (visited.add(entry)
				&& !name.equals("Plug-in Dependencies.libraryclasspath")) {
			Element userElement = doc.createElement("path"); //$NON-NLS-1$
			userElement.setAttribute("id", name); //$NON-NLS-1$
			Element filesetElement = null;
			IClasspathEntry entries[] = container.getClasspathEntries();
			for (int i = 0; i < entries.length; i++) {
				String jarFile = entries[i].getPath().toString();
				boolean addPathElement = true;
				// use ECLIPSE_HOME variable for library jars
				if (EclipseClasspath.isLibraryReference(entry)) {
					IPath home = JavaCore.getClasspathVariable("ECLIPSE_HOME"); //$NON-NLS-1$

					if (home.isPrefixOf(entries[i].getPath())) {
						variable2valueMap.put("ECLIPSE_HOME", home.toString()); //$NON-NLS-1$

						if (filesetElement == null) {
							filesetElement = doc.createElement("fileset");
							filesetElement.setAttribute("dir",
									"${ECLIPSE_HOME}/plugins");
							userElement.appendChild(filesetElement);
						}
						jarFile = "${ECLIPSE_HOME}" + jarFile.substring(home.toString().length()); //$NON-NLS-1$
						// Try to remove version information from plugin
						// dependencies
						// Pattern p =
						// Pattern.compile("(\\$\\{ECLIPSE_HOME}/plugins/)([^_]*)_(\\S*).jar");
						Pattern p = Pattern
								.compile("(\\$\\{ECLIPSE_HOME}/plugins/)([^_]*)_([^/]*)(.*).jar");
						Matcher m = p.matcher(jarFile);
						if (m.matches()) {
							// String jname = m.replaceAll("$2_*.jar");
							String jname = m.replaceAll("$2_*$4.jar");
							Element includeElement = doc
									.createElement("include");
							includeElement.setAttribute("name", jname);
							filesetElement.appendChild(includeElement);
							addPathElement = false;
						}
					} else if (!new File(jarFile).exists()
							&& jarFile.startsWith('/' + projectName)
							&& new File(projectRoot, jarFile
									.substring(('/' + projectName).length()))
									.exists()) {
						// workaround that additional jars are stored with
						// leading project root in container object, although
						// they are relative and indeed correctly stored in
						// build.properties (jars.extra.classpath)
						jarFile = jarFile.substring(('/' + projectName)
								.length() + 1);
					} else {
						// Check whether this entry is a reference to local
						// project
						String projectName = ExportUtil.removePrefix(entries[i]
								.getPath().toString(), "/");
						IJavaProject project = projectManager
								.getJavaProjectByName(projectName);
						if (project != null) {
							//jarFile = ExportUtil.getProjectRoot(project);
							jarFile = projectManager.getProjectRoot(project);
						}
					}
				}
				if (addPathElement) {
					jarFile = ExportUtil.getRelativePath(jarFile, projectRoot);
					Element userPathElement = doc.createElement("pathelement"); //$NON-NLS-1$
					userPathElement.setAttribute("location", jarFile); //$NON-NLS-1$
					userElement.appendChild(userPathElement);
				}
			}
			addToClasspathBlockM(userElement);
		}
	}

	private void addToClasspathBlockM(Element element) {
		// remember node to insert all classpaths at same location
		if (classpathNode == null) {
			classpathNode = root.appendChild(element);
		} else {
			classpathNode = classpathNode.getNextSibling();
			classpathNode = root.insertBefore(element, classpathNode);
		}
	}

	protected void createLoadProperties() {
		Comment c = doc.createComment(" Loads build properties ");
		root.appendChild(c);
		Element target = doc.createElement("target");
		target.setAttribute("name", "-load.build.properties");
		target.setAttribute("if", "build.properties.file");

		Element property = doc.createElement("property");
		property.setAttribute("file", "${build.properties.file}");
		target.appendChild(property);

		root.appendChild(target);
	}

	protected void createSaveProperties() {
		Comment c = doc.createComment(" Saves build properties ");
		root.appendChild(c);
		Element target = doc.createElement("target");
		target.setAttribute("name", "-save.build.properties");
		target.setAttribute("if", "build.properties.file");

		Element echoproperties = doc.createElement("echoproperties");
		echoproperties.setAttribute("destfile", "${build.properties.file}");
		echoproperties.setAttribute("prefix", "build.properties");
		target.appendChild(echoproperties);

		root.appendChild(target);
	}

	/**
	 * Create init target. Creates directories and copies resources.
	 * 
	 * @param srcDirs
	 *            source directories to copy resources from
	 * @param classDirs
	 *            classes directories to copy resources to
	 */
	protected void createInit(List srcDirs, List classDirs) {
		// <target name="-init" depends="-version">
		// <mkdir dir="${build.dir}"/>
		// <mkdir dir="${build.dir.bin}"/>
		// <mkdir dir="${build.dir.lib}"/>
		// <mkdir dir="${build.dir.doc}"/>
		// </target>
		Comment c = doc.createComment(" Initializes build directories ");
		root.appendChild(c);
		Element element = doc.createElement("target");
		element.setAttribute("name", "-init");
		element.setAttribute("depends", "-version,-load.build.properties");

		Element mkdir = doc.createElement("mkdir");
		mkdir.setAttribute("dir", "${build.dir}");
		element.appendChild(mkdir);

		mkdir = doc.createElement("mkdir");
		mkdir.setAttribute("dir", "${build.dir.bin}");
		element.appendChild(mkdir);

		mkdir = doc.createElement("mkdir");
		mkdir.setAttribute("dir", "${build.dir.lib}");
		element.appendChild(mkdir);

		mkdir = doc.createElement("mkdir");
		mkdir.setAttribute("dir", "${build.dir.doc}");
		element.appendChild(mkdir);

		/*
		 * List classDirsUnique = ExportUtil.removeDuplicates(classDirs); for
		 * (Iterator iterator = classDirsUnique.iterator(); iterator.hasNext();) {
		 * String classDir = (String) iterator.next(); if (!classDir.equals(".") &&
		 * //$NON-NLS-1$ !EclipseClasspath.isReference(classDir)) { Element
		 * pathElement = doc.createElement("mkdir"); //$NON-NLS-1$
		 * pathElement.setAttribute("dir", classDir); //$NON-NLS-1$
		 * element.appendChild(pathElement); } }
		 */

		root.appendChild(element);

		// createCopyResources(srcDirs, classDirs, element);
	}
	
	private void createCopyResources(List srcDirs, List classDirs,
			Element element) {
		// Check filter for copying resources
		String filter = javaProject.getOption(
				JavaCore.CORE_JAVA_BUILD_RESOURCE_COPY_FILTER, true);
		StringTokenizer tokenizer = new StringTokenizer(filter, ","); //$NON-NLS-1$
		List filters = Collections.list(tokenizer);
		filters.add("*.java"); //$NON-NLS-1$

		// prefix filters with wildcard
		for (int i = 0; i < filters.size(); i++) {
			String item = ((String) filters.get(i)).trim();
			if (item.equals("*")) //$NON-NLS-1$
			{
				// everything is excluded from copying
				return;
			}
			filters.set(i, "**/" + item); //$NON-NLS-1$
		}

		// <copy todir="classes" includeemptydirs="false">
		// <fileset dir="src" excludes="**/*.java"/>
		// </copy>
		for (int i = 0; i < srcDirs.size(); i++) {
			String srcDir = (String) srcDirs.get(i);
			String classDir = (String) classDirs.get(i);
			if (!EclipseClasspath.isReference(classDir)) {
				Element copyElement = doc.createElement("copy"); //$NON-NLS-1$
				copyElement.setAttribute("todir", classDir); //$NON-NLS-1$
				copyElement.setAttribute("includeemptydirs", "false"); //$NON-NLS-1$ //$NON-NLS-2$
				Element filesetElement = doc.createElement("fileset"); //$NON-NLS-1$
				filesetElement.setAttribute("dir", srcDir); //$NON-NLS-1$
				filesetElement.setAttribute(
						"excludes", ExportUtil.toString(filters, ", ")); //$NON-NLS-1$ //$NON-NLS-2$
				copyElement.appendChild(filesetElement);
				element.appendChild(copyElement);
			}
		}
	}

	/*
	protected void createFetch(IProgressMonitor pm) throws JavaModelException {
		Comment c = doc.createComment(" ======================================== ");
		root.appendChild(c);
		c = doc.createComment(" Fetch dependencies ");
		root.appendChild(c);
		Element target = doc.createElement("target");
		target.setAttribute("name", "fetch.dependencies");
		
		List dependencies  = projectManager.getClasspathProjectsRecursive(javaProject);
		
		//sort projects by provider
		Map byProvider = new HashMap(); 
		for(int i = 0; i < dependencies.size(); i++){
			IJavaProject javaProject = (IJavaProject) dependencies.get(i);
			//---------------------------------------
			RepositoryProvider provider = RepositoryProvider.getProvider(javaProject.getProject());
			if(provider != null){
				String id = provider.getID();
				Set list = (Set) byProvider.get(id);
				if(list == null){
					list = new TreeSet(new Comparator(){
						public int compare(Object arg0, Object arg1) {
							// TODO Auto-generated method stub
							return ((IProject) arg0).getName().toLowerCase().compareTo(((IProject) arg1).getName().toLowerCase());
						}
					});
					byProvider.put(id, list);
				}
				list.add(javaProject.getProject());
			}
		}
		Comparator comparator = new Comparator(){
			public int compare(Object arg0, Object arg1) {
				// TODO Auto-generated method stub
				return ((Element) arg0).getAttribute("name").compareTo(((Element) arg1).getAttribute("name"));
			}
		};
		Set fetchTargetList = new TreeSet(comparator);
		Set checkoutTargetList = new TreeSet(comparator);
		Set updateTargetList = new TreeSet(comparator);
		
		boolean taskdefReqired = true;
		boolean cvsTargetsReuired = true;
		boolean svnTargetsReuired = true;
		
		for(Iterator providersIt = byProvider.keySet().iterator(); providersIt.hasNext();){
			String id = (String) providersIt.next();
			Set list = (Set) byProvider.get(id);
			IProject[] projectArray = (IProject[])list.toArray(new IProject[list.size()]);
			
			RepositoryProviderType providerType = RepositoryProviderType.getProviderType(id);
			ProjectSetCapability capability = providerType.getProjectSetCapability();
			ProjectSetCapability.ensureBackwardsCompatible(providerType, capability);
			
			String[] references = null;
			if(capability != null){
				try {
					IProgressMonitor spm = null;
//					if (pm != null) {
//						spm = new SubProgressMonitor(pm, 100);
//					} else {
						spm = new NullProgressMonitor();
//					}
					references = capability.asReference(projectArray, new ProjectSetSerializationContext(), spm);
				} catch (TeamException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
			}
			
			if(references != null){
				if("org.eclipse.team.cvs.core.cvsnature".equalsIgnoreCase(id)){
					if(cvsTargetsReuired){
						Element element = doc.createElement("target");
						element.setAttribute("name", "fetch.from.cvs");
						element.setAttribute("depends", "check.project.existence,checkout.dependency.cvs,update.dependency.cvs");
						Element echo = doc.createElement("echo");
						echo.setAttribute("message", "fetch of ${id} - comlete");
						element.appendChild(echo);
						fetchTargetList.add(element);
						element = doc.createElement("target");
						element.setAttribute("name", "checkout.dependency.cvs");
						element.setAttribute("unless", "${id}-exists");
						Element cvs = doc.createElement("cvs");
						cvs.setAttribute("cvsRoot", "${cvsroot}");
						cvs.setAttribute("package", "${path}");
						cvs.setAttribute("command", "checkout -d ../${id}");
						cvs.setAttribute("tag", "${revision}");
						cvs.setAttribute("taskname", "cvs - checkout ${id}");
						element.appendChild(cvs);
						checkoutTargetList.add(element);
						element = doc.createElement("target");
						element.setAttribute("name", "update.dependency.cvs");
						element.setAttribute("if", "${id}-exists");
						cvs = doc.createElement("cvs");
						cvs.setAttribute("command", "update ../${id}");
						element.appendChild(cvs);
						updateTargetList.add(element);
						cvsTargetsReuired = false;
					}
					for(int i = 0; i < references.length; i++){
						StringTokenizer tokenizer = new StringTokenizer(references[i], ",");
						tokenizer.nextToken();
						Element antcall =doc.createElement("antcall");
						antcall.setAttribute("target", "fetch.from.cvs");
						Element param = doc.createElement("param");
						param.setAttribute("name", "cvsroot");
						param.setAttribute("value", tokenizer.nextToken());
						antcall.appendChild(param);
						param = doc.createElement("param");
						param.setAttribute("name", "path");
						param.setAttribute("value", tokenizer.nextToken());
						antcall.appendChild(param);
						param = doc.createElement("param");
						param.setAttribute("name", "id");
						param.setAttribute("value", tokenizer.nextToken());
						antcall.appendChild(param);
						param = doc.createElement("param");
						param.setAttribute("name", "revision");
						param.setAttribute("value", "HEAD");
						antcall.appendChild(param);
						target.appendChild(antcall);
					}
				} else if("org.eclipse.team.svn.core.svnnature".equalsIgnoreCase(id) || 
						"org.tigris.subversion.subclipse.core.svnnature".equalsIgnoreCase(id) ||
						"org.polarion.team.svn.core.svnnature".equalsIgnoreCase(id)){
					if(taskdefReqired){
						//add taskdef
						//default
						Element defaultSVNTaskDefTarget = doc.createElement("target");
						defaultSVNTaskDefTarget.setAttribute("unless", "svn.lib.path");
						defaultSVNTaskDefTarget.setAttribute("name", "default.svn.task.def");
						Element svnTaskDef = doc.createElement("taskdef");
						svnTaskDef.setAttribute("resource", "org/tigris/subversion/svnant/svnantlib.xml");
						defaultSVNTaskDefTarget.appendChild(svnTaskDef);
						//with path
						Element svnTaskDefTarget = doc.createElement("target");
						svnTaskDefTarget.setAttribute("if", "svn.lib.path");
						svnTaskDefTarget.setAttribute("name", "svn.task.def");
						//classpath 
						Element path = doc.createElement("path");
						path.setAttribute("id", "svn.classpath");
						Element pathelement = doc.createElement("pathelement");
						pathelement.setAttribute("location", "${svn.lib.path}/svnant.jar");
						path.appendChild(pathelement);
						pathelement = doc.createElement("pathelement");
						pathelement.setAttribute("location", "${svn.lib.path}/svnjavahl.jar");
						path.appendChild(pathelement);
						pathelement = doc.createElement("pathelement");
						pathelement.setAttribute("location", "${svn.lib.path}/javasvn.jar");
						path.appendChild(pathelement);
						pathelement = doc.createElement("pathelement");
						pathelement.setAttribute("location", "${svn.lib.path}/svnClientAdapter.jar");
						path.appendChild(pathelement);
						svnTaskDefTarget.appendChild(path);
						
						svnTaskDef = doc.createElement("taskdef");
						svnTaskDef.setAttribute("resource", "org/tigris/subversion/svnant/svnantlib.xml");
						svnTaskDef.setAttribute("classpathref", "svn.classpath");
						svnTaskDefTarget.appendChild(svnTaskDef);
						target.setAttribute("depends", "default.svn.task.def,svn.task.def");
						
						root.appendChild(defaultSVNTaskDefTarget);
						root.appendChild(svnTaskDefTarget);
						taskdefReqired = false;
					}
					if(svnTargetsReuired){
						Element element = doc.createElement("target");
						element.setAttribute("name", "fetch.from.svn");
						element.setAttribute("depends", "check.project.existence,checkout.dependency.svn,update.dependency.svn");
						Element echo = doc.createElement("echo");
						echo.setAttribute("message", "fetch of ${id} - comlete");
						element.appendChild(echo);
						fetchTargetList.add(element);
						element = doc.createElement("target");
						element.setAttribute("name", "checkout.dependency.svn");
						element.setAttribute("unless", "${id}-exists");
						Element svn = doc.createElement("svn");
						svn.setAttribute("taskname", "svn - checkout ${id}");
						Element checkout = doc.createElement("checkout");
						checkout.setAttribute("url", "${url}");
						checkout.setAttribute("destpath", "../${id}");
						checkout.setAttribute("revision", "${revision}");
						svn.appendChild(checkout);
						element.appendChild(svn);
						checkoutTargetList.add(element);
						element = doc.createElement("target");
						element.setAttribute("name", "update.dependency.svn");
						element.setAttribute("if", "${id}-exists");
						svn = doc.createElement("svn");
						svn.setAttribute("taskname", "svn - update ${id}");
						Element update = doc.createElement("update");
						update.setAttribute("dir", "../${id}");
						svn.appendChild(update);
						element.appendChild(svn);
						updateTargetList.add(element);
						svnTargetsReuired = false;
					}
					for(int i = 0; i < references.length; i++){
						StringTokenizer tokenizer = new StringTokenizer(references[i], ",");
						tokenizer.nextToken();
						Element antcall =doc.createElement("antcall");
						antcall.setAttribute("target", "fetch.from.svn");
						Element param = doc.createElement("param");
						param.setAttribute("name", "url");
						param.setAttribute("value", tokenizer.nextToken());
						antcall.appendChild(param);
						param = doc.createElement("param");
						param.setAttribute("name", "id");
						param.setAttribute("value", tokenizer.nextToken());
						antcall.appendChild(param);
						param = doc.createElement("param");
						param.setAttribute("name", "revision");
						param.setAttribute("value", "HEAD");
						antcall.appendChild(param);
						target.appendChild(antcall);
					}
				}
			}
		}
		root.appendChild(target);
		
		//fetching targets
		if(!cvsTargetsReuired || !svnTargetsReuired){
			c = doc.createComment(" fetching ");
			root.appendChild(c);
			Element checkTarget = doc.createElement("target");
			checkTarget.setAttribute("name", "check.project.existence");
			Element available = doc.createElement("available");
			available.setAttribute("property", "${id}-exists");
			available.setAttribute("file", "../${id}");
			checkTarget.appendChild(available);
			root.appendChild(checkTarget);
		
			for(Iterator it = fetchTargetList.iterator(); it.hasNext();){
				root.appendChild((Element) it.next());
			}
			c = doc.createComment(" checkout ");
			root.appendChild(c);
			for(Iterator it = checkoutTargetList.iterator(); it.hasNext();){
				root.appendChild((Element) it.next());
			}
			c = doc.createComment(" update ");
			root.appendChild(c);
			for(Iterator it = updateTargetList.iterator(); it.hasNext();){
				root.appendChild((Element) it.next());
			}
		}
	}
	*/

	protected boolean create(List projects, IProgressMonitor pm) throws Exception {
		createDoc();
		createRoot("build");
		createImports();
		createClasspaths(classpath);
		createVersion();
		createOverride(classpath);
		createInit(classpath.getSrcDirs(), classpath.getClassDirs());
		createLoadProperties();
		createSaveProperties();
		createBuild(classpath.getSrcDirs(), classpath.getClassDirs(), classpath.getInclusionLists(), classpath.getExclusionLists());
		createBuildProject(classpath.getSrcDirs(), classpath.getClassDirs(), classpath.getInclusionLists(), classpath.getExclusionLists());
		createBuildSubprojects(classpath.getSrcDirs(), classpath.getClassDirs(), classpath.getInclusionLists(), classpath.getExclusionLists());

		createJar(classpath.getSrcDirs());

		createJavadoc();

		createClean();

		createCopyLibraries(classpath);

		createCopyJar(classpath.getSrcDirs());

		createCopyDepJars();
		createCopyAllJars();

		createCleanAll();
		
		createFetch();
		
		createPlugin(classpath);

		addSubProperties(javaProject, classpath);
		createProperty(classpath);
		createPropertyM();
		
		return true;
	}

}