/*******************************************************************************
 * Copyright (c) 2007-2008 Parity Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Valery Kokhan - initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.ant.service;

import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.higgins.ant.Higgins2AntPlugin;
import org.eclipse.jdt.core.IClasspathContainer;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaModel;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.launching.JavaRuntime;
import org.osgi.framework.Bundle;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;

public abstract class AbstractProjectManager implements IProjectManager {
	
	protected interface IConstants {
		public static IPath MANIFEST_PATH = new Path("META-INF/MANIFEST.MF");
		public static IPath PLUGIN_PATH = new Path("plugin.xml");
		public static IPath FEATURE_PATH = new Path("feature.xml");
		public static IPath SITE_PATH = new Path("site.xml");
	}
	
	public static final String TEMPLATES_LOCATION = "templates";
	
	public synchronized Comparator getProjectComparator()
    {
        if (projectComparator == null)
        {
        	projectComparator = new ProjectComparator();
        }
        return projectComparator;
    }
    
    private static Comparator projectComparator;
    
    private static class ProjectComparator implements Comparator
    {
        public int compare(Object o1, Object o2)
        {
        	String name1 = instance.getProject(o1).getName();
        	String name2 = instance.getProject(o2).getName();
            return name1.compareTo(name2);
        }
        
        public boolean equals(Object obj)
        {
            if (obj == null)
            {
                return false;
            }
            return compare(this, obj) == 0;
        }
    }
	
	protected static IProjectManager instance = null;
	
	public static IProjectManager getInstance() {
		if (instance == null) {
			instance = ExtensionManager.createProjectManager();
		}
		return instance;
	}
	
	protected void addJavaProjects(List projects) {
		if (projects == null)
			projects = new ArrayList();
		IWorkspaceRoot rootWorkspace = ResourcesPlugin.getWorkspace().getRoot();
        IJavaModel javaModel = JavaCore.create(rootWorkspace);
        IJavaProject[] javaProjects;
        try {
            javaProjects = javaModel.getJavaProjects();
            for(int i = 0; i < javaProjects.length; i++) {
            	IProject p = getProject(javaProjects[i]);
           		projects.add(p);
            }
        }
        catch (Exception e) {
            javaProjects= new IJavaProject[0];
        }
	}
	
	protected void addOtherProjects(List projects) {
		if (projects == null)
			projects = new ArrayList();
		IWorkspaceRoot rootWorkspace = ResourcesPlugin.getWorkspace().getRoot();
        IProject[] otherProjects = rootWorkspace.getProjects();
        if (otherProjects.length == projects.size()) {
        	return;
        }
        for (int i = 0; i < otherProjects.length; i++) {
			if (!projects.contains((IProject)otherProjects[i]))
				projects.add(otherProjects[i]);
		}
	}

	public List getWorkspaceProjects() {
		List projects = new ArrayList();

		addJavaProjects(projects);
		addOtherProjects(projects);

		return projects;
	}
	
	protected boolean addJavaProject(Set projects, IProject project, boolean requiredProjects) throws JavaModelException {
		boolean res = false;
		if (isJavaProject(project)) {
            IJavaProject javaProject = getJavaProject(project);
            if (requiredProjects){
            	projects.addAll(getClasspathProjectsRecursive(javaProject));
            }
            projects.add(javaProject);
            res = true;
    	}
		return res;
	}

	public Set getBuildProjects(Iterator projects, boolean requiredProjects) throws JavaModelException {
		Set res = new TreeSet(getProjectComparator());
		while (projects.hasNext()) {
			IProject project = (IProject) projects.next();
			addJavaProject(res, project, requiredProjects);
		}
		return res;
	}
	
	public abstract IProject getProject(Object o);
	
	public IProject getProject(String projectId) {
		IWorkspaceRoot rootWorkspace = ResourcesPlugin.getWorkspace().getRoot();
		return rootWorkspace.getProject(projectId);
	}
	
	public String getProjectRoot(Object project) {
		return getProject(project).getLocation().toString();
	}
	
	public String getProjectName(Object project) {
		return getProject(project).getName();
	}
	
    public IJavaProject getJavaProject(IProject project) {
    	if (!isJavaProject(project)) {
    		return null;
    	}
        IWorkspaceRoot rootWorkspace = ResourcesPlugin.getWorkspace().getRoot();
        IJavaModel javaModel = JavaCore.create(rootWorkspace);
        IJavaProject javaProject = javaModel.getJavaProject(project.getName());
        if (javaProject != null) {
        	return javaProject;
        }
        try
        {
            IJavaProject[] javaProjects;
            javaProjects = javaModel.getJavaProjects();
            for (int i = 0; i < javaProjects.length; i++)
            {
                IJavaProject jp = javaProjects[i];
                if (project.equals(jp))
                {
                	javaProject = jp;
                    break;
                }
            }      
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
        return javaProject;
    }
    
    public IJavaProject getJavaProject(String root)
    {
        IWorkspaceRoot rootWorkspace = ResourcesPlugin.getWorkspace().getRoot();
        IJavaModel javaModel = JavaCore.create(rootWorkspace);
        IJavaProject[] javaProjects;
        try
        {
            javaProjects = javaModel.getJavaProjects();
        }
        catch (JavaModelException e)
        {
            return null;
        }
        for (int i = 0; i < javaProjects.length; i++)
        {
            IJavaProject javaProject = javaProjects[i];
            if (root.equals(javaProject.getPath().toString()))
            {
                return javaProject;
            }
        }      
        return null;
    }
    
    public IJavaProject getJavaProjectByName(String name)
    {
        IWorkspaceRoot rootWorkspace = ResourcesPlugin.getWorkspace().getRoot();
        IJavaModel javaModel = JavaCore.create(rootWorkspace);
        IJavaProject[] javaProjects;
        try
        {
            javaProjects = javaModel.getJavaProjects();
        }
        catch (JavaModelException e)
        {
            return null;
        }
        for (int i = 0; i < javaProjects.length; i++)
        {
            IJavaProject javaProject = javaProjects[i];
            if (name.equals(javaProject.getProject().getName()))
            {
                return javaProject;
            }
        }      
        return null;
    }
	
	public final List getSubProjects(IProject project) {
		List res = new ArrayList();
		Set set = new TreeSet(getProjectComparator());
		getSubProjects(project, set);
		res.addAll(set);
		return res;
	}
	
	protected boolean getJavaSubProjects(IProject project, Set set) {
		boolean res = false;
		try {
			if (isJavaProject(project)) {
				res = true;
				IJavaProject javaProject = getJavaProject(project);
				if (javaProject != null) {
					if (!set.contains(javaProject)) {
						//set.add(javaProject);
						List l = getClasspathProjectsRecursive(javaProject);
						for (int j = 0; j < l.size(); j++) {
							set.add(getProject(l.get(j)));
						}
					}
				}
			}
		} catch (CoreException e) {
			e.printStackTrace();
		}
		return res;
	}

	public void getSubProjects(IProject project, Set set) {
		getJavaSubProjects(project, set);
	}

	public List getClasspathProjectsRecursive(Object object) throws JavaModelException {
		List classpath = null;
		if (object instanceof IJavaProject)
			classpath = getClasspathProjectsRecursive((IJavaProject)object);
		return classpath;
	}
	
	protected List getClasspathProjectsRecursive(IJavaProject project) throws JavaModelException {
		LinkedList result = new LinkedList();
        getClasspathProjectsRecursive(project, result);
        return sortProjectsUsingBuildOrder(result);		
	}
	
    protected void getClasspathProjectsRecursive(IJavaProject project, LinkedList result) throws JavaModelException
    {
        List projects = getClasspathProjects(project);
        for (Iterator iter = projects.iterator(); iter.hasNext();)
        {
            IJavaProject javaProject = (IJavaProject) iter.next();
            if (!result.contains(javaProject)) {
                result.addFirst(javaProject);
                getClasspathProjectsRecursive(javaProject, result); // recursion
            }
        }
    }
    
    /**
     * Sort projects according to General -&gt; Workspace -&gt; Build Order.
     * @param projects list of IJavaProject objects
     * @return list of IJavaProject objects with new order
     */
    protected List sortProjectsUsingBuildOrder(List javaProjects) {
    	if (javaProjects.isEmpty()) {
    		return javaProjects;
    	}
        List result = new ArrayList(javaProjects.size());
        IWorkspace workspace = ResourcesPlugin.getWorkspace();
        String[] buildOrder= workspace.getDescription().getBuildOrder();
        if (buildOrder == null) {//default build order
        	IProject[] projects= new IProject[javaProjects.size()];
        	int i= 0;
        	for (Iterator iter = javaProjects.iterator(); iter.hasNext(); i++) {
        		IJavaProject javaProject = (IJavaProject) iter.next();
        		projects[i]= javaProject.getProject();             
        	}
        	IWorkspace.ProjectOrder po = ResourcesPlugin.getWorkspace().computeProjectOrder(projects);
        	projects=po.projects;
        	buildOrder= new String[projects.length];
        	for (i = 0; i < projects.length; i++) {
				buildOrder[i]= projects[i].getName();
			}
        }
        
        for (int i = 0; i < buildOrder.length && !javaProjects.isEmpty(); i++) {
        	String projectName = buildOrder[i];            
        	for (Iterator iter = javaProjects.iterator(); iter.hasNext();) {
        		IJavaProject javaProject = (IJavaProject) iter.next();
        		if (javaProject.getProject().getName().equals(projectName)) {
        			result.add(javaProject);
        			iter.remove();
        		}                
        	}
        }
        //add any remaining projects not specified in the build order
        result.addAll(javaProjects);
        return result;
    }
    
    /**
     * Get for given project all directly dependent projects.
     * 
     * @return set of IJavaProject objects
     */
    public List getClasspathProjects(IJavaProject project) throws JavaModelException
    {
        List projects = new ArrayList();
        IClasspathEntry entries[] = project.getRawClasspath();
        addClasspathProjects(project, projects, entries);
        return sortProjectsUsingBuildOrder(projects);
    }

    private static void addClasspathProjects(IJavaProject project, List projects, IClasspathEntry[] entries) {
        for (int i = 0; i < entries.length; i++)
        {
            IClasspathEntry classpathEntry = entries[i];
            if (classpathEntry.getContentKind() == IPackageFragmentRoot.K_SOURCE &&
                classpathEntry.getEntryKind() == IClasspathEntry.CPE_PROJECT) {
                    // found required project on build path
                    String subProjectRoot = classpathEntry.getPath().toString();
                    IJavaProject subProject = instance.getJavaProject(subProjectRoot);
                    // is project available in workspace
                    if (subProject != null)
                    {
                        projects.add(subProject);
                    }
            } else if (classpathEntry.getContentKind() == IPackageFragmentRoot.K_SOURCE &&
            		classpathEntry.getEntryKind() == IClasspathEntry.CPE_CONTAINER) {
                IClasspathContainer container;
				try {
					container = JavaCore.getClasspathContainer(classpathEntry.getPath(), project);
	                if (container != null) {
	                    String jar = container.getPath().toString();
	                    if (!jar.startsWith(JavaRuntime.JRE_CONTAINER) && !jar.startsWith(JavaCore.USER_LIBRARY_CONTAINER_ID))
	                    {
	                    	addClasspathProjects(project, projects, container.getClasspathEntries());
	                    }
	                }
				} catch (Exception e) {
					e.printStackTrace();
				}
            }
        }
    }
    
    public boolean isJavaProject(IProject project) {
		boolean res = false;
		try {
			if (project.getNature("org.eclipse.jdt.core.javanature") != null) {
				res = true;
			}
		} catch (CoreException e) {
			res = false;
		}
		return res;
	}

	public boolean isFeatureProject(IProject project) {
		boolean res = false;
		try {
			if (project.getNature("org.eclipse.pde.FeatureNature") != null) {
				res = true;
			}
		} catch (CoreException e) {
			res = false;
		}
		//return res;
		return res && project.isOpen() && project.exists(IConstants.FEATURE_PATH);
	}
	
	public boolean isPluginProject(IProject project) {
		boolean res = false;
		try {
			if (project.getNature("org.eclipse.pde.PluginNature") != null) {
				res = true;
			}
		} catch (CoreException e) {
			res = false;
		}
		//return res;
		return res && project.isOpen() && (project.exists(IConstants.PLUGIN_PATH) || project.exists(IConstants.MANIFEST_PATH));
	}
	
	public boolean isSiteProject(IProject project) {
		boolean res = false;
		try {
			if (project.getNature("org.eclipse.pde.UpdateSiteNature") != null) {
				res = true;
			}
		} catch (CoreException e) {
			res = false;
		}
		//return res;
		return res && project.isOpen() && project.exists(IConstants.SITE_PATH);
	}

	public Document getTemplate(String bundle, String name) {
		Document doc = null;
		InputStream is = null;
		String resource = TEMPLATES_LOCATION + "/" + name + ".xml";
		try {
			Bundle b = Platform.getBundle(bundle);
			URL url = b.getEntry(resource);
			is = url.openStream();
			DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
			DocumentBuilder db = dbf.newDocumentBuilder();
			doc = db.parse(new InputSource(is));
		} catch (Exception e) {
			Higgins2AntPlugin.log("Can't load template with id=\"" + name + "\"", e);
		} finally {
			if (is != null) {
				try {
					is.close();
				} catch (Exception e) {
				}
			}
		}
		return doc;
	}
}
