/*******************************************************************************
 * Copyright (c) 2007-2008 Parity Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Valery Kokhan - initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.ant.pde;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.higgins.ant.core.JavaProjectBuildFileCreator;
import org.eclipse.higgins.ant.service.ICreator;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.pde.core.plugin.IPluginModelBase;
import org.eclipse.pde.internal.core.FeatureModelManager;
import org.eclipse.pde.internal.core.PDECore;
import org.eclipse.pde.internal.core.PluginModelManager;
import org.eclipse.pde.internal.core.WorkspaceModelManager;
import org.eclipse.pde.internal.core.ifeature.IFeatureChild;
import org.eclipse.pde.internal.core.ifeature.IFeatureModel;
import org.eclipse.pde.internal.core.ifeature.IFeaturePlugin;
import org.eclipse.pde.internal.core.isite.ISiteFeature;
import org.eclipse.pde.internal.core.isite.ISiteModel;
import org.eclipse.pde.internal.core.site.WorkspaceSiteModel;

public class ProjectManager extends org.eclipse.higgins.ant.core.ProjectManager {
	
	private final String SITE_XML = "site.xml";

	public ProjectManager() {
		// System.out.println("ProjectManager() for feature project");
	}

	protected void addFeatureProjects(List projects) {
		if (projects == null)
			projects = new ArrayList();

		FeatureModelManager manager = PDECore.getDefault().getFeatureModelManager();
		IFeatureModel[] features;

		try {
			features = manager.getWorkspaceModels();
			for (int i = 0; i < features.length; i++) {
				projects.add(getProject(features[i]));
			}
		} catch (Exception e) {
			features = new IFeatureModel[0];
		}
	}

	public List getWorkspaceProjects() {
		List projects = new ArrayList();

		addJavaProjects(projects);
		addFeatureProjects(projects);
		addOtherProjects(projects);

		return projects;
	}

	protected boolean addFeatureProject(Set projects, IProject project, boolean requiredProjects) throws JavaModelException {
		boolean res = false;
		if (isFeatureProject(project)) {
			IFeatureModel model = getFeatureModel(project);
			if (requiredProjects) {
				IFeaturePlugin[] plugins = model.getFeature().getPlugins();
				for (int i = 0; i < plugins.length; i++) {
					IJavaProject javaProject = getJavaProject(plugins[i]);
					if (javaProject != null) {
						projects.addAll(getClasspathProjectsRecursive(javaProject));
					}
				}
			}
			projects.add(model);
			res = true;
		}
		return res;
	}

	protected boolean addSiteProject(Set projects, IProject project, boolean requiredProjects) {
		boolean res = false;
		if (isSiteProject(project)) {
			projects.add(project);
		}
		return res;
	}

	public Set getBuildProjects(Iterator projects, boolean requiredProjects) throws JavaModelException {
		Set res = new TreeSet(getProjectComparator());
		while (projects.hasNext()) {
			IProject project = (IProject) projects.next();
			if (addJavaProject(res, project, requiredProjects)) {
			} else if (addFeatureProject(res, project, requiredProjects)) {
			} else if (addSiteProject(res, project, requiredProjects)) {				
			}
		}
		return res;
	}

	public ICreator getBuildFileCreator(Object project) throws Exception {
		ICreator creator = null;
		if (project instanceof IJavaProject) {
			creator = new JavaProjectBuildFileCreator((IJavaProject) project);
		} else if (project instanceof IFeatureModel) {
			creator = new FeatureBuildFileCreator((IFeatureModel) project);
		} else if ((project instanceof IProject)) {
			IProject p = (IProject) project;
			if (isJavaProject(p)) {
				IJavaProject jp = getJavaProject(p);
				creator = getBuildFileCreator(jp);
			} else if (isFeatureProject(p)) {
				IFeatureModel m = getFeatureModel(p);
				creator = getBuildFileCreator(m);
			} else if (isSiteProject(p)) {
				creator = new SiteBuildFileCreator(p);
			}
		}
		return creator;
	}

	private IJavaProject getJavaProject(IFeaturePlugin plugin) {
		IJavaProject javaProject = null;
		IProject project = getProject(plugin);
		if (project != null) {
			javaProject = getJavaProject(project);
		}
		return javaProject;
	}

	public IProject getProject(Object o) {
		IProject p = super.getProject(o);
		if (p == null) {
			if (o instanceof IFeatureModel) {
				p = getProject((IFeatureModel) o);
			} else if (o instanceof IFeaturePlugin) {
				p = getProject((IFeaturePlugin) o);
			}
		}
		return p;
	}

	private static IProject getProject(IJavaProject p) {
		return p.getProject();
	}

	private static IProject getProject(IFeatureModel m) {
		IProject p = null;
		IResource r = m.getUnderlyingResource();
		if (r != null) {
			p = r.getProject();
		}
		return p;
	}

	private static IProject getProject(IFeaturePlugin fp) {
		IProject p = null;
		PluginModelManager manager = PDECore.getDefault().getModelManager();
		IPluginModelBase pm = manager.findModel(fp.getId());
		if (pm != null) {
			IResource r = pm.getUnderlyingResource();
			if (r != null) {
				p = r.getProject();
			}
		}
		return p;
	}

	protected boolean getFeatureSubProjects(IProject project, Set set) {
		boolean res = false;
		try {
			if (isFeatureProject(project)) {
				res = true;
				IFeatureModel model = getFeatureModel(project);
				if (model != null) {
					IFeatureChild[] features = model.getFeature()
							.getIncludedFeatures();
					if (features != null) {
						for (int i = 0; i < features.length; i++) {
							IFeatureModel m = getFeatureModel(features[i]
									.getId());
							if (m != null) {
								if (!set.contains(m)) {
									set.add(m);
									getSubProjects(getProject(m), set);
								}
							}
						}
					}
					IFeaturePlugin[] plugins = model.getFeature().getPlugins();
					if (plugins != null) {
						for (int i = 0; i < plugins.length; i++) {
							IProject p = getProject(plugins[i]);
							if (p != null) {
								IJavaProject javaProject = getJavaProject(plugins[i]);
								if (javaProject == null) {
									set.add(p);
								} else {
									if (!set.contains(javaProject)) {
										set.add(javaProject);
										List l;

										l = getClasspathProjectsRecursive(javaProject);
										for (int j = 0; j < l.size(); j++) {
											set.add(getProject(l.get(j)));
										}
									}
								}
							}
						}
					}
				}
			}
		} catch (JavaModelException e) {
			e.printStackTrace();
		}
		return res;
	}
	
	protected boolean getSiteSubProjects(IProject project, Set set) {
		boolean res = false;
		if (isSiteProject(project)) {
			IFeatureModel[] models = getFeatureModels(project);
			for (int i = 0; i < models.length; i++) {
				IFeatureModel model = models[i];
				if (!set.contains(model)) {
					set.add(model);
					getSubProjects(getProject(model), set);
				}
			}
		}
		return res;
	}

	public void getSubProjects(IProject project, Set set) {
		if (getSiteSubProjects(project, set)) {			
		} else if (getFeatureSubProjects(project, set)) {
		} else if (getJavaSubProjects(project, set)) {
		}
	}

	protected IFeatureModel getFeatureModel(IProject project) {
		IFeatureModel model = null;
		if (WorkspaceModelManager.isFeatureProject(project)) {
			model = getFeatureModel(project.getName());
		}
		return model;
	}

	protected IFeatureModel getFeatureModel(String id) {
		FeatureModelManager manager = PDECore.getDefault().getFeatureModelManager();
		IFeatureModel model = null;
		IFeatureModel models[] = manager.getWorkspaceModels();
		for (int i = 0; i < models.length; i++) {
			if (id.equals(models[i].getFeature().getId())) {
				model = models[i];
				break;
			}
		}
		return model;
	}
	
	private ISiteModel getSiteModel(IProject project) {
		ISiteModel model = null;
		if (isSiteProject(project)) {
			IFile siteXml = project.getFile(SITE_XML);
			model = new WorkspaceSiteModel(siteXml);
			try {
				model.load();
			} catch (CoreException e) {
				model = null;
				// e.printStackTrace();
			}
		}
		return model;
	}
	
	protected IFeatureModel[] getFeatureModels(ISiteModel siteModel) {
		List models = new ArrayList();
		if (siteModel != null) {
			ISiteFeature[] sbFeatures = siteModel.getSite().getFeatures();
			for (int i = 0; i < sbFeatures.length; i++) {
				ISiteFeature siteFeature = sbFeatures[i];
				IFeatureModel model = getFeatureModel(siteFeature.getId());
				if (model != null)
					models.add(model);
			}
		}
		return (IFeatureModel[]) models.toArray(new IFeatureModel[models.size()]);
	}
	
	protected IFeatureModel[] getFeatureModels(IProject project) {
		return getFeatureModels(getSiteModel(project));
	}

}
