/*******************************************************************************
 * Copyright (c) 2007 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Valery Kokhan - Initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.icard.common.auth;

import java.net.URI;
import java.security.cert.X509Certificate;

import javax.security.auth.callback.Callback;

import org.eclipse.higgins.icard.IEndpointReference;
import org.eclipse.higgins.icard.auth.ICredential;
import org.eclipse.higgins.icard.auth.IKerberosV5Credential;
import org.eclipse.higgins.icard.auth.ISelfIssuedCredential;
import org.eclipse.higgins.icard.auth.ITSKerberosV5Credential;
import org.eclipse.higgins.icard.auth.ITSSelfIssuedCredential;
import org.eclipse.higgins.icard.auth.ITSUsernamePasswordCredential;
import org.eclipse.higgins.icard.auth.ITSX509V3Credential;
import org.eclipse.higgins.icard.auth.ITokenServiceCredential;
import org.eclipse.higgins.icard.auth.ITokenServiceCredentialFactory;
import org.eclipse.higgins.icard.auth.IUsernamePasswordCredential;
import org.eclipse.higgins.icard.auth.IX509V3Credential;
import org.eclipse.higgins.icard.common.auth.callback.TokenServiceCallback;

public class TokenServiceCredential implements ITokenServiceCredential, ITokenServiceCredentialFactory {

	Callback[] callbacks = new Callback[1];

	TokenServiceCallback tc;

	public TokenServiceCredential(IEndpointReference er) {
		this(er.getAddress(), er.getCertificate(), er.getMetadataAddress());
	}

	public TokenServiceCredential(URI address, X509Certificate certificate,
			URI metadataAddress) {
		tc = new TokenServiceCallback(address, certificate, metadataAddress);
		callbacks[0] = tc;
	}

	public URI getAddress() {
		return tc.getAddress();
	}

	public X509Certificate getCertificate() {
		return tc.getCertificate();
	}

	public URI getMetadataAddress() {
		return tc.getMetadataAddress();
	}

	public Callback[] getCallbacks() {
		return callbacks;
	}

	public ITSUsernamePasswordCredential create(IUsernamePasswordCredential upc) {
		return new TSUsernamePasswordCredential(this, upc);
	}

	public boolean equals(Object obj) {
		if (obj != null && obj instanceof ITokenServiceCredential) {
			ITokenServiceCredential tsc = (ITokenServiceCredential) obj;
			try {
				if (getAddress().equals(tsc.getAddress())
						&& getCertificate().equals(tsc.getCertificate())
						&& getMetadataAddress().equals(tsc.getMetadataAddress())) {
					return true;
				}
				return false;
			} catch (Throwable e) {
			}
		}
		return super.equals(obj);
	}

	public ITokenServiceCredential createCredential(ICredential credential) {
		if (credential instanceof IUsernamePasswordCredential) {
			return createCredential((IUsernamePasswordCredential) credential); 
		} else if (credential instanceof ISelfIssuedCredential) {
			return createCredential((ISelfIssuedCredential) credential); 
		} else if (credential instanceof IKerberosV5Credential) {
			return createCredential((IKerberosV5Credential) credential); 
		} else if (credential instanceof IX509V3Credential) {
			return createCredential((IX509V3Credential) credential); 
		}
		throw new IllegalArgumentException("Unsupported type of credential: " + credential);
	}
	
	protected ITSUsernamePasswordCredential createCredential(IUsernamePasswordCredential credential) {
		return new TSUsernamePasswordCredential(this, credential);
	}
	
	protected ITSSelfIssuedCredential createCredential(ISelfIssuedCredential credential) {
		return new TSSelfIssuedCredential(this, credential);
	}
	
	protected ITSKerberosV5Credential createCredential(IKerberosV5Credential credential) {
		return new TSKerberosV5Credential(this, credential);
	}
	
	protected ITSX509V3Credential createCredential(IX509V3Credential credential) {
		return new TSX509V3Credential(this, credential);
	}
}
