/*******************************************************************************
 * Copyright (c) 2007 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Valery Kokhan - Initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.icard.common.auth.callback;

import java.io.Serializable;

import javax.security.auth.callback.Callback;

public class PinCodeCallback implements Callback, Serializable {
	private static final long serialVersionUID = -6723778370783650274L;

	private String prompt;

	private boolean echoOn;

	private byte[] inputPin;

	/**
	 * Construct a <code>PinCodeCallback</code> with a prompt and a boolean
	 * specifying whether the PIN code should be displayed as it is being typed.
	 * 
	 * <p>
	 * 
	 * @param prompt
	 *            the prompt used to request the PIN code.
	 *            <p>
	 * 
	 * @param echoOn
	 *            true if the PIN code should be displayed as it is being typed.
	 * 
	 * @exception IllegalArgumentException
	 *                if <code>prompt</code> is null or if <code>prompt</code>
	 *                has a length of 0.
	 */
	public PinCodeCallback(String prompt) {
		if (prompt == null || prompt.length() == 0)
			throw new IllegalArgumentException();

		this.prompt = prompt;
		this.echoOn = false;
	}

	/**
	 * Get the prompt.
	 * 
	 * <p>
	 * 
	 * @return the prompt.
	 */
	public String getPrompt() {
		return prompt;
	}

	/**
	 * Return whether the PIN code should be displayed as it is being typed.
	 * 
	 * <p>
	 * 
	 * @return the whether the PIN code should be displayed as it is being
	 *         typed.
	 */
	public boolean isEchoOn() {
		return echoOn;
	}

	/**
	 * Set the retrieved PIN code.
	 * 
	 * <p>
	 * This method makes a copy of the input <i>pin</i> before storing it.
	 * 
	 * <p>
	 * 
	 * @param password
	 *            the retrieved PIN code, which may be null.
	 * 
	 * @see #getPin
	 */
	public void setPin(byte[] pin) {
		this.inputPin = (pin == null ? null : (byte[]) pin.clone());
	}

	/**
	 * Get the retrieved PIN code.
	 * 
	 * <p>
	 * This method returns a copy of the retrieved PIN code.
	 * 
	 * <p>
	 * 
	 * @return the retrieved PIN code, which may be null.
	 * 
	 * @see #setPin
	 */
	public byte[] getPin() {
		return (inputPin == null ? null : (byte[]) inputPin.clone());
	}

	/**
	 * Clear the retrieved PIN code.
	 */
	public void clearPin() {
		if (inputPin != null) {
			for (int i = 0; i < inputPin.length; i++) {
				inputPin[i] = ' ';
			}
		}
	}
}
