/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.icard.common;

import java.net.URI;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;

import javax.xml.namespace.QName;

import org.eclipse.higgins.icard.AuthenticationRequiredException;
import org.eclipse.higgins.icard.CUID;
import org.eclipse.higgins.icard.ICard;
import org.eclipse.higgins.icard.ICardConstants;
import org.eclipse.higgins.icard.ICardProvider;
import org.eclipse.higgins.icard.IClaim;
import org.eclipse.higgins.icard.IClaimType;
import org.eclipse.higgins.icard.ISimpleClaim;
import org.eclipse.higgins.icard.InvalidTypeException;
import org.eclipse.higgins.icard.common.ClaimValue;
import org.eclipse.higgins.icard.common.utils.CardContext;
import org.eclipse.higgins.icard.io.IElementFormat;
import org.eclipse.higgins.icard.io.IXMLElementFormat;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;


public abstract class Card implements ICard {
	
	private static Log log = LogFactory.getLog(Card.class);
	
	protected String name_ = null;

	protected URI id_ = null;

	protected String version_ = null;

	protected String issuer_ = null;

	protected byte[] issuerID_ = null;

	protected String issuerName_ = null;

	protected byte[] image_ = null;

	protected String imageMimeType_ = null;

	protected Date timeIssued_ = null;

	protected Date timeExpires_ = null;

	protected List claimTypes_ = null;

	protected List supportedTokenTypes_ = null;

	protected String language_ = null;

	protected ICardProvider provider_ = null;

	protected boolean isSelfIssued_ = false;

	protected Date timeLastUpdated_ = null;

	protected String description_ = null;

	protected HashMap claimValues_ = null;

	protected HashMap claimValuesByLocalName_ = null;
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#getName()
	 */
	public String getName() {
		return name_ != null ? name_ : "";
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#getID()
	 */
	public String getID() {
		return id_ != null ? id_.toString() : "";
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#getIssuerName()
	 */
	public String getIssuerName() {
		if (issuerName_ != null)
			return issuerName_;
		else {
			try {
				URI uri =  new URI(issuer_);
				return uri.getHost().toString();
			}
			catch(Exception e) {
				return null;
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#getIssuer()
	 */
	public String getIssuer() {
		return issuer_ != null ? issuer_ : "";
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#getImage()
	 */
	public byte[] getImage() {
		return image_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#getImageType()
	 */
	public String getImageType() {
		return imageMimeType_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#getTimeIssued()
	 */
	public Date getTimeIssued() {
		return timeIssued_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#getTimeExpires()
	 */
	public Date getTimeExpires() {
		return timeExpires_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#getSupportedClaimTypes()
	 */
	public Iterator getSupportedClaimTypes() {
		if (claimTypes_ != null)
			return claimTypes_.iterator();
		else
			return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#getSupportedClaimTypesUris()
	 */
	public List getSupportedClaimTypesUris() {
		List res = new ArrayList();
		if (claimTypes_ != null) {
			for (int i = 0; i < claimTypes_.size(); i++) {
				IClaimType type = (IClaimType) claimTypes_.get(i);
				res.add(type.getType().toString());
			}
		}
		return Collections.unmodifiableList(res);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#getVersion()
	 */
	public String getVersion() {
		return version_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#getSupportedTokenTypes()
	 */
	public List getSupportedTokenTypes() {
		return supportedTokenTypes_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#getProvider()
	 */
	public ICardProvider getProvider() {
		return provider_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#getTimeLastUpdated()
	 */
	public Date getTimeLastUpdated() {
		return timeLastUpdated_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#isSelfIssued()
	 */
	public boolean isSelfIssued() {
		return isSelfIssued_;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.icard.ICard#getCUID()
	 */
	public CUID getCUID() {
		String providerId = provider_ != null ? getProvider().getID() : "";
		return new CUID(providerId, issuer_, getID());
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.icard.ICard#getDescription()
	 */
	public String getDescription() {
		return description_;
	}

	public boolean isClaimsRetrieved() {
		return claimValues_ != null;
	}
	
	/**
	 * Sets a cache of claim values. Unless call to this method any attempts to
	 * access claim values throws the {@link AuthenticationRequiredException}.
	 * 
	 * This is a helper method used either by
	 * {@link ICardProvider#getICardByCUID(javax.security.auth.callback.CallbackHandler, CUID, org.eclipse.higgins.icard.auth.ICredential)}
	 * or card itself when fully loaded card is requested.
	 * 
	 * @param claims
	 * @throws InvalidTypeException
	 */
	public void setClaims(List claims) throws InvalidTypeException {
		// TODO validate that clime type is supported be this card!
		if (claimValues_ == null) {
			claimValues_ = new LinkedHashMap();
			claimValuesByLocalName_ = new HashMap();
		}
		for (int i = 0; i < claims.size(); i++) {
			IClaim c = (IClaim) claims.get(i);
			try {
				if (c instanceof ISimpleClaim) {
					ISimpleClaim sc = (ISimpleClaim) c;
					ClaimValue cv = new ClaimValue(sc.getType(), sc.getValues(), this);
					c = cv;
				}
			} catch (Throwable e) {
				log.error(e,e);
			}
			claimValues_.put(c.getType().getType(), c);
			claimValuesByLocalName_.put(c.getTypeLocalName(), c);
		}
	}

	public boolean isFormatSupported(IElementFormat format) {
		if (format.getFormatID() == IElementFormat.XML_ELEMENT_FORMAT) {
			IXMLElementFormat xmlf = (IXMLElementFormat) format;
			QName[] names = xmlf.getElementNames();
			for (int i = 0; i < names.length; i++) {
				if (names[i].equals(CardContext.QN_IC_ROAMING_INFORMATION_CARD)) {
					return true;
				}
			}
		}
		return false;
	}
}
