/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.icard.provider.cardspace.common;

import java.net.URI;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.higgins.icard.AuthenticationException;
import org.eclipse.higgins.icard.AuthenticationRequiredException;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.ICardProvider;
import org.eclipse.higgins.icard.IClaim;
import org.eclipse.higgins.icard.IClaimType;
import org.eclipse.higgins.icard.IPersonalInformationCard;
import org.eclipse.higgins.icard.InvalidTypeException;
import org.eclipse.higgins.icard.auth.ICredential;
import org.eclipse.higgins.icard.auth.ICredentialDescriptor;
import org.eclipse.higgins.icard.common.ClaimType;
import org.eclipse.higgins.icard.common.ClaimValue;
import org.eclipse.higgins.icard.common.auth.PinCodeCredential;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.CardContext;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.CardCryptography;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.DateConvertor;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.SelfIssuedCardClaims;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.XMLUtils;
import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public abstract class PersonalCard extends InformationCard implements IPersonalInformationCard {
	// can contain pin-protected claim values as it stored within *.crds file.
	// Used ONLY for parsing/creation the card. Should not be used to return/set
	// claim value in methods getClaim(), setClaim() etc. because claims can be
	// encrypted
	protected ArrayList tmpClaimValueList_ = null;

	protected URI claimListContextID_ = null;

	protected String claimListSubjectID_ = null;

	protected byte[] pinDigest_ = null;

	/**
	 * @param card
	 * @throws CardException
	 * @throws DOMException
	 */
	public void initFromXML(Element card) throws CardException, DOMException {
		isSelfIssued_ = true;
		issuer_ = "http://schemas.xmlsoap.org/ws/2005/05/identity/issuer/self";
		issuerName_ = "Self";
		Element informationCardMetaData = XMLUtils.getChildElement(card, CardContext.IC_NS, "InformationCardMetaData");
		if (informationCardMetaData == null)
			throw new CardException("Can not find \"InformationCardMetaData\" element.");
		language_ = null;
		if (informationCardMetaData.hasAttribute("xml:lang"))
			language_ = informationCardMetaData.getAttribute("xml:lang");

		Element informationCardPrivateData = XMLUtils.getChildElement(card, CardContext.IC_NS, "InformationCardPrivateData");
		checkIsSelfIssued(informationCardMetaData);
		parseCardId(informationCardMetaData);
		parseCardVersion(informationCardMetaData);
		parseCardName(informationCardMetaData);
		parseCardImage(informationCardMetaData);
		parseTimeIssued(informationCardMetaData);
		parseTimeExpires(informationCardMetaData);
		parseSupportedTokenTypes(informationCardMetaData);
		parseClaimTypes(informationCardMetaData);
		parseHashSalt(informationCardMetaData);
		parsePinDigest(informationCardMetaData);
		parseTimeLastUpdated(informationCardMetaData);
		parseClaimValues(informationCardPrivateData);
		parseMasterKey(informationCardPrivateData);
	}

	private void checkIsSelfIssued(Element metaData) throws CardException {
		Element isSelfIssued = XMLUtils.getChildElement(metaData, CardContext.IC_NS, "IsSelfIssued");
		String boolValue = (isSelfIssued != null) ? XMLUtils.getTextContent(isSelfIssued) : "";
		if (boolValue == null || "true".equals(boolValue.trim().toLowerCase()) == false)
			throw new CardException("Can not import non-self issued card.");
	}

	/**
	 * @param metaData
	 * @throws CardException
	 */
	private void parseCardId(Element metaData) throws CardException {
		Element informationCardReference = XMLUtils.getChildElement(metaData, CardContext.IC_NS, "InformationCardReference");
		if (informationCardReference != null) {
			Element cardId = XMLUtils.getChildElement(informationCardReference, CardContext.IC_NS, "CardId");
			String id = (cardId != null) ? XMLUtils.getTextContent(cardId) : "";
			if (id == null || id.trim().length() == 0)
				throw new CardException("Couldn't get CardId from InfoCard.");
			id_ = URI.create(id);
		} else
			throw new CardException("Couldn't get \"InformationCardReference\" from InfoCard.");
	}

	/**
	 * @param metaData
	 * @throws CardException
	 */
	private void parseCardVersion(Element metaData) throws CardException {
		Element informationCardReference = XMLUtils.getChildElement(metaData, CardContext.IC_NS, "InformationCardReference");
		if (informationCardReference != null) {
			Element cardVersion = XMLUtils.getChildElement(informationCardReference, CardContext.IC_NS, "CardVersion");
			version_ = (cardVersion != null) ? XMLUtils.getTextContent(cardVersion) : "";
			if (version_ == null || version_.trim().length() == 0)
				throw new CardException("Couldn't get CardVersion from InfoCard with id = " + id_);
		} else
			throw new CardException("Couldn't get \"InformationCardReference\" from InfoCard.");
	}

	/**
	 * @param metaData
	 * @throws CardException
	 */
	private void parseCardName(Element metaData) throws CardException {
		Element cardName = XMLUtils.getChildElement(metaData, CardContext.IC_NS, "CardName");
		name_ = (cardName != null) ? XMLUtils.getTextContent(cardName) : "";
	}

	/**
	 * @param metaData
	 * @throws CardException
	 * @throws DOMException
	 */
	private void parseCardImage(Element metaData) throws CardException, DOMException {
		Element cardImg = XMLUtils.getChildElement(metaData, CardContext.IC_NS, "CardImage");
		if (cardImg == null)
			throw new CardException("Couldn't get CardImage from InfoCard with id = " + id_);
		if (cardImg != null) {
			image_ = CardCryptography.decodeBase64(XMLUtils.getTextContent(cardImg));
			imageMimeType_ = cardImg.getAttribute("MimeType");
		}
	}

	/**
	 * @param metaData
	 * @throws CardException
	 */
	private void parseTimeIssued(Element metaData) throws CardException {
		Element timeIssuedElm = XMLUtils.getChildElement(metaData, CardContext.IC_NS, "TimeIssued");
		String timeIssued = (timeIssuedElm != null) ? XMLUtils.getTextContent(timeIssuedElm) : "";
		if (timeIssued == null || timeIssued.trim().length() == 0)
			throw new CardException("Couldn't get TimeIssued from InfoCard with id = " + id_);
		timeIssued_ = DateConvertor.parse(timeIssued);
	}

	/**
	 * @param metaData
	 * @throws CardException
	 */
	private void parseTimeExpires(Element metaData) throws CardException {
		Element timeExpiresElm = XMLUtils.getChildElement(metaData, CardContext.IC_NS, "TimeExpires");
		String timeExpires = (timeExpiresElm != null) ? XMLUtils.getTextContent(timeExpiresElm) : "";
		if (timeExpires != null && timeExpires.trim().length() == 0)
			timeExpires_ = DateConvertor.parse(timeExpires);
	}

	private void parseSupportedTokenTypes(Element metaData) throws CardException {
		supportedTokenTypes_ = new ArrayList();
		Element supportedTokenTypeList = XMLUtils.getChildElement(metaData, CardContext.IC_NS, "SupportedTokenTypeList");
		if (supportedTokenTypeList == null)
			throw new CardException("Couldn't get \"SupportedTokenTypeList\" element from InfoCard with id = " + id_);
		NodeList tokenTypes = supportedTokenTypeList.getChildNodes();
		int len = tokenTypes.getLength();
		for (int i = 0; i < len; i++) {
			Node nd = tokenTypes.item(i);
			if (nd.getNodeType() == Node.ELEMENT_NODE && "TokenType".equals(nd.getLocalName()) && CardContext.TRUST_NS.equals(nd.getNamespaceURI())) {
				Element elm = (Element)nd;
				supportedTokenTypes_.add(URI.create(XMLUtils.getTextContent(elm)));
			}
		}
		if (supportedTokenTypes_.size() == 0)
			throw new CardException("Couldn't get the list of TokenType elements from InfoCard with id = " + id_);
	}

	/**
	 * @param metaData
	 * @throws CardException
	 */
	private void parseClaimTypes(Element metaData) throws CardException {
		claimTypes_ = new ArrayList();
		Element supportedClaimTypeList = XMLUtils.getChildElement(metaData, CardContext.IC_NS, "SupportedClaimTypeList");
		if (supportedClaimTypeList == null)
			throw new CardException("Couldn't get \"SupportedClaimTypeList\" element from InfoCard with id = " + id_);
		NodeList claimTypes = supportedClaimTypeList.getChildNodes();
		int len = claimTypes.getLength();
		for (int i = 0; i < len; i++) {
			Node node = claimTypes.item(i);
			if (node.getNodeType() == Node.ELEMENT_NODE && "SupportedClaimType".equals(node.getLocalName()) && CardContext.IC_NS.equals(node.getNamespaceURI())) {
				Element elm = (Element) node;
				String uri = elm.getAttribute("Uri");
				if (uri == null || uri.trim().length() == 0)
					throw new CardException("Couldn't get \"uri\" attribute of SupportedClaimType element for InfoCard with id = " + id_);
				Element displayTagElm = XMLUtils.getChildElement(elm, CardContext.IC_NS, "DisplayTag");
				String displayTag = (displayTagElm != null) ? XMLUtils.getTextContent(displayTagElm) : "";
				Element descriptionElm = XMLUtils.getChildElement(elm, CardContext.IC_NS, "Description");
				String description = (descriptionElm != null) ? XMLUtils.getTextContent(descriptionElm) : "";
				claimTypes_.add(new ClaimType(uri, displayTag, description));
			}
		}
	}

	/**
	 * @param privateData
	 * @throws CardException
	 */
	private void parseClaimValues(Element privateData) throws CardException {
		tmpClaimValueList_ = new ArrayList();
		Element claimValueList = XMLUtils.getChildElement(privateData, CardContext.IC_NS, "ClaimValueList");
		if (claimValueList == null)
			throw new CardException("Couldn't get the list of \"ClaimValueList\" elements from InfoCard with id = " + id_);
		NodeList claimValues = claimValueList.getChildNodes();
		int len = claimValues.getLength();
		for (int i = 0; i < len; i++) {
			Node node = claimValues.item(i);
			if (node.getNodeType() == Node.ELEMENT_NODE && "ClaimValue".equals(node.getLocalName()) && CardContext.IC_NS.equals(node.getNamespaceURI())) {
				Element elm = (Element) node;
				String uri = elm.getAttribute("Uri");
				if (uri == null || uri.trim().length() == 0)
					throw new CardException("Couldn't get \"uri\" attribute of ClaimValue element for InfoCard with id = " + id_);
				Element valueElm = XMLUtils.getChildElement(elm, CardContext.IC_NS, "Value");
				String value = (valueElm != null) ? XMLUtils.getTextContent(valueElm) : "";
				if (value != null) {
					ClaimValue claimValue = new ClaimValue(SelfIssuedCardClaims.getClaimType(uri), value);
					tmpClaimValueList_.add(claimValue);
				}
			}
		}
		if (tmpClaimValueList_.size() == 0)
			throw new CardException("Couldn't get list of ClaimValue elements from InfoCard with id = " + id_);
	}

	/**
	 * @param metaData
	 * @throws CardException
	 */
	private void parseHashSalt(Element metaData) throws CardException {
		Element hashSalt = XMLUtils.getChildElement(metaData, CardContext.IC_NS, "HashSalt");
		String str = (hashSalt != null) ? XMLUtils.getTextContent(hashSalt) : "";
		if (str == null || str.trim().length() == 0)
			throw new CardException("Couldn't get HashSalt element from InfoCard with id = " + id_);
		hashSalt_ = CardCryptography.decodeBase64(str);
	}

	/**
	 * @param metaData
	 * @throws CardException
	 */
	private void parsePinDigest(Element metaData) throws CardException {
		pinDigest_ = null;
		Element pinDigest = XMLUtils.getChildElement(metaData, CardContext.IC_NS, "PinDigest");
		String str = (pinDigest != null) ? XMLUtils.getTextContent(pinDigest) : "";
		if (str != null)
			pinDigest_ = CardCryptography.decodeBase64(str);
	}

	/**
	 * @param metaData
	 * @throws CardException
	 */
	private void parseTimeLastUpdated(Element metaData) throws CardException {
		Element timeLastUpdated = XMLUtils.getChildElement(metaData, CardContext.IC_NS, "TimeLastUpdated");
		String str = (timeLastUpdated != null) ? XMLUtils.getTextContent(timeLastUpdated) : "";
		if (str == null || str.trim().length() == 0)
			throw new CardException("Couldn't get TimeLastUpdated element from InfoCard with id = " + id_);
		timeLastUpdated_ = DateConvertor.parse(str);
	}

	/**
	 * @param privateData
	 * @throws CardException
	 */
	private void parseMasterKey(Element privateData) throws CardException {
		Element masterKey = XMLUtils.getChildElement(privateData, CardContext.IC_NS, "MasterKey");
		String str = (masterKey != null) ? XMLUtils.getTextContent(masterKey) : "";
		if (str == null || str.trim().length() == 0)
			throw new CardException("Couldn't get MasterKey element from RoamingInformationCard with id = " + id_);
		masterKey_ = CardCryptography.decodeBase64(str);
	}

	public URI getClaimListContextID() {
		return claimListContextID_;
	}

	public String getClaimListSubjectID() {
		return claimListSubjectID_;
	}

	public Element toXML(Document doc) throws CardException {
		// TODO reimplement this method to return RoaimngInformationCard
		Element root = doc.createElementNS(CardContext.IC_NS, "InformationCard");
		if (language_ != null)
			root.setAttribute("xml:lang", language_);
		doc.appendChild(root);
		addCardReference(doc, root);
		addCardName(doc, root);
		addCardImage(doc, root);
		addIssuer(doc, root);
		addTimeIssued(doc, root);
		addTimeExpires(doc, root);
		addSupportedTokenTypes(doc, root);
		addClaimTypes(doc, root);
		return root;
	}

	/**
	 * @param doc
	 * @return
	 * @throws CardException
	 */
	public Element claimsToXML(Document doc) throws CardException {
		Element root = doc.createElementNS(CardContext.IC_NS, "ClaimValueList");
		doc.appendChild(root);
		Iterator values = tmpClaimValueList_.iterator();
		while (values.hasNext()) {
			ClaimValue claim = (ClaimValue) values.next();
			Element claimValue = doc.createElementNS(CardContext.IC_NS, "ClaimValue");
			claimValue.setAttribute("Uri", claim.getType().toString());
			root.appendChild(claimValue);
			Element value = doc.createElementNS(CardContext.IC_NS, "Value");
			XMLUtils.setTextContent(value, claim.getValue());
			claimValue.appendChild(value);
		}
		return root;
	}

	/**
	 * @param doc
	 * @param root
	 */
	private void addCardReference(Document doc, Element root) {
		Element ref = doc.createElementNS(CardContext.IC_NS, "InformationCardReference");
		root.appendChild(ref);
		Element id = doc.createElementNS(CardContext.IC_NS, "CardId");
		XMLUtils.setTextContent(id, id_.toString());
		ref.appendChild(id);
		Element version = doc.createElementNS(CardContext.IC_NS, "CardVersion");
		XMLUtils.setTextContent(version, version_);
		ref.appendChild(version);
	}

	/**
	 * @param doc
	 * @param root
	 */
	private void addCardName(Document doc, Element root) {
		Element name = doc.createElementNS(CardContext.IC_NS, "CardName");
		XMLUtils.setTextContent(name, name_);
		root.appendChild(name);
	}

	/**
	 * @param doc
	 * @param root
	 */
	private void addCardImage(Document doc, Element root) {
		Element image = doc.createElementNS(CardContext.IC_NS, "CardImage");
		image.setAttribute("MimeType", imageMimeType_);
		XMLUtils.setTextContent(image, new String(CardCryptography.encodeBase64(image_, 0)));
		root.appendChild(image);
	}

	/**
	 * @param doc
	 * @param root
	 */
	private void addIssuer(Document doc, Element root) {
		Element issuer = doc.createElementNS(CardContext.IC_NS, "Issuer");
		XMLUtils.setTextContent(issuer, issuer_);
		root.appendChild(issuer);
	}

	/**
	 * @param doc
	 * @param root
	 */
	private void addTimeIssued(Document doc, Element root) {
		String time = DateConvertor.format(timeIssued_);
		Element timeIssued = doc.createElementNS(CardContext.IC_NS, "TimeIssued");
		XMLUtils.setTextContent(timeIssued, time);
		root.appendChild(timeIssued);
	}

	/**
	 * @param doc
	 * @param root
	 */
	private void addTimeExpires(Document doc, Element root) {
		String time = DateConvertor.format(timeExpires_);
		Element timeExpires = doc.createElementNS(CardContext.IC_NS, "TimeExpires");
		XMLUtils.setTextContent(timeExpires, time);
		root.appendChild(timeExpires);
	}

	/**
	 * @param doc
	 * @param root
	 */
	private void addSupportedTokenTypes(Document doc, Element root) {
		Element tokenList = doc.createElementNS(CardContext.IC_NS, "SupportedTokenTypeList");
		root.appendChild(tokenList);
		Iterator it = supportedTokenTypes_.iterator();
		while (it.hasNext()) {
			String uri = it.next().toString();
			Element tokenType = doc.createElementNS(CardContext.TRUST_NS, "TokenType");
			XMLUtils.setTextContent(tokenType, uri);
			tokenList.appendChild(tokenType);
		}
	}

	/**
	 * @param doc
	 * @param root
	 */
	private void addClaimTypes(Document doc, Element root) {
		Element claimTypeList = doc.createElementNS(CardContext.IC_NS, "SupportedClaimTypeList");
		root.appendChild(claimTypeList);
		Iterator it = claimTypes_.iterator();
		while (it.hasNext()) {
			IClaimType type = (IClaimType) it.next();
			Element claimType = doc.createElementNS(CardContext.IC_NS, "SupportedClaimType");
			claimType.setAttribute("Uri", type.getType().toString());
			claimTypeList.appendChild(claimType);
			Element displayTag = doc.createElementNS(CardContext.IC_NS, "DisplayTag");
			XMLUtils.setTextContent(displayTag, type.getDisplayName());
			claimType.appendChild(displayTag);
			Element description = doc.createElementNS(CardContext.IC_NS, "Description");
			XMLUtils.setTextContent(description, type.getDescription());
			claimType.appendChild(description);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.IPersonalInformationCard#getPinDigest()
	 */
	public byte[] getPinDigest() throws CardException {
		return pinDigest_;
	}

	public IClaim getClaim(String type) throws AuthenticationRequiredException, InvalidTypeException, CardException {
		if (claimValues_ == null) {
			retrieveClaims(null);
		}
		if (claimValues_ == null) {
			throw new AuthenticationRequiredException();
		}
		if (claimValues_.containsKey(type)) {
			return (IClaim) claimValues_.get(type);
		}
		throw new InvalidTypeException("Claim type: " + type + " is not supported.");
	}

	public IClaim getClaimByLocalName(String shortTypeName) throws AuthenticationRequiredException, InvalidTypeException, CardException {
		if (claimValuesByLocalName_ == null) {
			retrieveClaims(null);
		}
		if (claimValuesByLocalName_ == null) {
			throw new AuthenticationRequiredException();
		}
		if (claimValuesByLocalName_.containsKey(shortTypeName)) {
			return (IClaim) claimValuesByLocalName_.get(shortTypeName);
		}
		throw new InvalidTypeException("Claim with local type name: " + shortTypeName + " is not supported.");
	}

	public Iterator getClaims() throws CardException {
		if (claimValues_ == null) {
			retrieveClaims(null);
		}
		if (claimValues_ == null) {
			throw new AuthenticationRequiredException();
		}
		return claimValues_.values().iterator();
	}

	public ICredentialDescriptor[] getRequiredCredentials() {
		ICredentialDescriptor[] requiredCredentials = null; 
		if (pinDigest_ != null) {
			requiredCredentials = new ICredentialDescriptor[1];
			requiredCredentials[0] = new BasicCredentialDescriptor("PinCodeCredential", "Please provide PIN code", new PinCodeCredential());
		}
		return requiredCredentials;
	}

	/**
	 * This method should be used by
	 * {@link ICardProvider#getICardByCUID(javax.security.auth.callback.CallbackHandler, org.eclipse.higgins.icard.CUID, ICredential)}
	 * to retrieve fully loaded card instance.
	 * 
	 * This methid should use {@link InformationCard#setClaims(List)} to set
	 * cached list of claim values.
	 * 
	 * @param credential
	 *            the <code>ICredential</code> object used as an
	 *            authentication materials. Can be <code>null</code> if this
	 *            card is not pin protected.
	 * @throws AuthenticationRequiredException
	 *             when this card is pin protected but <code>null</code> was
	 *             passed as the <code>credential</code> parameter.
	 * @throws CardException
	 */
	protected abstract void retrieveClaims(ICredential credential) throws AuthenticationRequiredException, AuthenticationException, CardException;
}
