/*******************************************************************************
 * Copyright (c) 2007-2008 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Valery Kokhan - Initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.icard.provider.cardspace.common.utils;

import java.net.URI;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.common.ClaimType;
import org.eclipse.higgins.icard.common.ClaimValue;
import org.eclipse.higgins.icard.provider.cardspace.common.CredentialDescriptor;
import org.eclipse.higgins.icard.provider.cardspace.common.EndpointReference;
import org.eclipse.higgins.icard.provider.cardspace.common.ManagedCard;
import org.eclipse.higgins.icard.provider.cardspace.common.TokenService;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class CardUtils {
	private static Log log = LogFactory.getLog(ManagedCard.class);
	
	public static URI parseCardId(Element card) throws CardException {
		String id = null;
		try {
			Element informationCardReference = XMLUtils.getChildElement(card, CardContext.IC_NS, CardContext.IC_INFORMATION_CARD_REFERENCE);
			Element cardId = XMLUtils.getChildElement(informationCardReference, CardContext.IC_NS, CardContext.IC_CARD_ID);
			id = XMLUtils.getTextContent(cardId);
		} catch (Exception e) {
			log.error(e);
		}
		if (isStringEmpty(id))
			throw new CardException("Couldn't get CardId element from InfoCard.");
		return URI.create(id);
	}

	public static String parseCardVersion(Element card) throws CardException {
		String version = null;
		try {
			Element informationCardReference = XMLUtils.getChildElement(card, CardContext.IC_NS, CardContext.IC_INFORMATION_CARD_REFERENCE);
			Element cardVersion = XMLUtils.getChildElement(informationCardReference, CardContext.IC_NS, CardContext.IC_CARD_VERSION);
			version = XMLUtils.getTextContent(cardVersion);
		} catch (Exception e) {
			log.error(e);
		}
		if (isStringEmpty(version))
			throw new CardException("Couldn't get CardVersion element");
		return version;
	}

	public static String parseCardName(Element card) throws CardException {
		Element cardNameElm = XMLUtils.getChildElement(card, CardContext.IC_NS, CardContext.IC_CARD_NAME);
		return (cardNameElm != null) ? XMLUtils.getTextContent(cardNameElm) : "";
	}

	public static byte[] parseCardImage(Element card) throws CardException {
		byte[] image = null;
		Element cardImageElm = XMLUtils.getChildElement(card, CardContext.IC_NS, CardContext.IC_CARD_IMAGE);
		if (cardImageElm != null) {
			image = CardCryptography.decodeBase64(XMLUtils.getTextContent(cardImageElm));
		}
		return image;
	}

	public static String parseCardImageType(Element card) throws CardException {
		String imageMimeType = "";
		Element cardImageElm = XMLUtils.getChildElement(card, CardContext.IC_NS, CardContext.IC_CARD_IMAGE);
		if (cardImageElm != null) {
			imageMimeType = cardImageElm.getAttribute("MimeType");
		}
		return imageMimeType;
	}

	public static String parseIssuer(Element card) throws CardException {
		String issuer = null;
		Element issuerElm = XMLUtils.getChildElement(card, CardContext.IC_NS, CardContext.IC_ISSUER);
		issuer = (issuerElm != null) ? XMLUtils.getTextContent(issuerElm) : "";
		if (isStringEmpty(issuer))
			throw new CardException("Couldn't get issuer information from card element");
		return issuer;
	}

	public static String parseIssuerName(Element card) throws CardException {
		String issuerName = null;
		Element issuerNameElm = XMLUtils.getChildElement(card, CardContext.IC_NS, CardContext.IC_ISSUER_NAME);
		issuerName = (issuerNameElm != null) ? XMLUtils.getTextContent(issuerNameElm) : "";
		if (isStringEmpty(issuerName)) {
			String sIssuer = parseIssuer(card);
			try {
				URI issuer = URI.create(sIssuer);
				if (issuer != null) {
					issuerName = issuer.getHost().toString();
				} else
					issuerName = sIssuer;
			} catch (Exception e) {
				issuerName = sIssuer;
				log.error(e);
			}
		}
		return issuerName;
	}

	public static byte[] parseIssuerID(Element card) throws CardException {
		Element issuerNameElm = XMLUtils.getChildElement(card, CardContext.IC_NS, CardContext.IC_ISSUER_ID);
		if (issuerNameElm != null) {
			String issuerID =  XMLUtils.getTextContent(issuerNameElm);
			if (issuerID != null)
				return CardCryptography.decodeBase64(issuerID);
		}
		return null;
	}

	public static Date parseTimeIssued(Element card) throws CardException {
		Element timeIssuedElm = XMLUtils.getChildElement(card, CardContext.IC_NS, CardContext.IC_TIME_ISSUED);
		String timeIssued = (timeIssuedElm != null) ? XMLUtils.getTextContent(timeIssuedElm) : "";
		if (isStringEmpty(timeIssued))
			throw new CardException("Couldn't get TimeIssued information from card element");
		return DateConvertor.parse(timeIssued);
	}

	public static Date parseTimeExpires(Element card) throws CardException {
		Date res = null;
		Element timeExpiresElm = XMLUtils.getChildElement(card, CardContext.IC_NS, CardContext.IC_TIME_EXPIRES);
		String timeExpires = (timeExpiresElm != null) ? XMLUtils.getTextContent(timeExpiresElm) : "";
		if (!isStringEmpty(timeExpires))
			res = DateConvertor.parse(timeExpires);
		return res;
	}

	public static List parseSupportedTokenTypes(Element card) throws CardException {
		List supportedTokenTypes = new ArrayList();
		Element supportedTokenTypeList = XMLUtils.getChildElement(card, CardContext.IC_NS, CardContext.IC_SUPPORTED_TOKEN_TYPE_LIST);
		if (supportedTokenTypeList == null)
			throw new CardException("Couldn't get \"SupportedTokenTypeList\" element");
		NodeList tokenTypes = supportedTokenTypeList.getChildNodes();
		int len = tokenTypes.getLength();
		for (int i = 0; i < len; i++) {
			Node nd = tokenTypes.item(i);
			if (nd.getNodeType() == Node.ELEMENT_NODE && CardContext.TRUST_TOKEN_TYPE.equals(nd.getLocalName()) && CardContext.TRUST_NS.equals(nd.getNamespaceURI())) {
				Element elm = (Element)nd;
				String uri = XMLUtils.getTextContent(elm);
				if (!isStringEmpty(uri)) {
					supportedTokenTypes.add(URI.create(uri));
				}
			}
		}
		if (supportedTokenTypes.size() == 0)
			throw new CardException("Couldn't get information about supported token types");
		return supportedTokenTypes;
	}

	public static List parseClaimTypes(Element card) throws CardException {
		List claimTypes_ = new ArrayList();
		Element supportedClaimTypeList = XMLUtils.getChildElement(card, CardContext.IC_NS, CardContext.IC_SUPPORTED_CLAIM_TYPE_LIST);
		if (supportedClaimTypeList == null)
			throw new CardException("Couldn't get \"SupportedClaimTypeList\" element");
		NodeList claimTypes = supportedClaimTypeList.getChildNodes();
		int len = claimTypes.getLength();
		for (int i = 0; i < len; i++) {
			Node node = claimTypes.item(i);
			if (node.getNodeType() == Node.ELEMENT_NODE && CardContext.IC_SUPPORTED_CLAIM_TYPE.equals(node.getLocalName()) && CardContext.IC_NS.equals(node.getNamespaceURI())) {
				Element elm = (Element) node;
				String uri = elm.getAttribute("Uri");
				if (isStringEmpty(uri))
					throw new CardException("Couldn't get \"uri\" attribute of SupportedClaimType element");
				Element displayTagElm = XMLUtils.getChildElement(elm, CardContext.IC_NS, CardContext.IC_DISPLAY_TAG);
				String displayTag = (displayTagElm != null) ? XMLUtils.getTextContent(displayTagElm) : "";
				Element descriptionElm = XMLUtils.getChildElement(elm, CardContext.IC_NS, CardContext.IC_DESCRIPTION);
				String description = (descriptionElm != null) ? XMLUtils.getTextContent(descriptionElm) : "";
				claimTypes_.add(new ClaimType(uri, displayTag, description));
			}
		}
		return claimTypes_;
	}

	private static EndpointReference parseEndpointReference(Element endpointRef) throws CardException {
		Element addressElm = XMLUtils.getChildElement(endpointRef, CardContext.WSA_NS, CardContext.WSA_ADDRESS);
		String addressStr = (addressElm != null) ? XMLUtils.getTextContent(addressElm) : "";
		URI address = (addressStr != null) ? URI.create(addressStr) : null;
		Element metadata = XMLUtils.getChildElement(endpointRef, CardContext.WSA_NS, CardContext.WSA_METADATA);
		Element identity = XMLUtils.getChildElement(endpointRef, CardContext.WSAI_NS, CardContext.WSAI_IDENTITY);
		return new EndpointReference(address, metadata, identity);
	}

	private static CredentialDescriptor parseUserCredential(Element credential) throws CardException {
		Element hintElm = XMLUtils.getChildElement(credential, CardContext.IC_NS, CardContext.IC_DISPLAY_CREDENTIAL_HINT);
		String hint = hintElm != null ? XMLUtils.getTextContent(hintElm) : "";
		return new CredentialDescriptor(hint, credential);
	}

	public static List parseTokenServiceList(Element card) throws CardException {
		List tokenServices_ = new ArrayList();
		Element tokenServiceList = XMLUtils.getChildElement(card, CardContext.IC_NS, CardContext.IC_TOKEN_SERVICE_LIST);
		if (tokenServiceList == null)
			throw new CardException("Couldn't get \"TokenServiceList\" element");
		NodeList tokenServices = tokenServiceList.getChildNodes();
		int len = tokenServices.getLength();
		for (int i = 0; i < len; i++) {
			Node node = tokenServices.item(i);
			if (node.getNodeType() == Node.ELEMENT_NODE && CardContext.IC_TOKEN_SERVICE.equals(node.getLocalName()) && CardContext.IC_NS.equals(node.getNamespaceURI())) {
				Element elm = (Element) node;
				Element endpointRef = XMLUtils.getChildElement(elm, CardContext.WSA_NS, CardContext.WSA_ENDPOINT_REFERENCE);
				if (endpointRef == null)
					throw new CardException("Couldn't get wsa:EndpointReference element");
				EndpointReference epRef = parseEndpointReference(endpointRef);
				Element credential = XMLUtils.getChildElement(elm, CardContext.IC_NS, CardContext.IC_USER_CREDENTIAL);
				if (credential == null)
					throw new CardException("Couldn't get ic:UserCredential element");
				CredentialDescriptor userCredential = parseUserCredential(credential);
				tokenServices_.add(new TokenService(epRef, userCredential));
			}
		}
		if (tokenServices_.size() == 0)
			throw new CardException("Couldn't get list of TokenService elements");
		return tokenServices_; 
	}

	public static Boolean parseRequireAppliesTo(Element card) throws CardException {
		Element requireAppliesTo = XMLUtils.getChildElement(card, CardContext.IC_NS, CardContext.IC_REQUIRE_APPLIES_TO);
//		String str = (requireAppliesTo != null) ? XMLUtils.getTextContent(requireAppliesTo) : "";
//		Boolean requireAppliesTo_ = null;
//		if (isStringEmpty(str) == false) {
//			if (str.equalsIgnoreCase("false"))
//				requireAppliesTo_ = new Boolean(false);
//			else if (str.equalsIgnoreCase("true"))
//				requireAppliesTo_ = new Boolean(true);
//		}
			
		Boolean requireAppliesTo_ = null;
		if(requireAppliesTo!=null){
			String optional = requireAppliesTo.getAttribute("Optional");
			if(optional==null || "".equals(optional) || "false".equalsIgnoreCase(optional)){
				requireAppliesTo_ = Boolean.FALSE;
			}
			if("true".equalsIgnoreCase(optional)){
				requireAppliesTo_ = Boolean.TRUE;
			}
		}
		
		System.out.println("---> RequiresAppliesTo field: " + requireAppliesTo_);
		
		return requireAppliesTo_;
	}

	public static Element parsePrivacyNotice(Element card) throws CardException {
		return XMLUtils.getChildElement(card, CardContext.IC_NS, CardContext.IC_PRIVACY_NOTICE);
	}

	public static byte[] parseHashSalt(Element card) throws CardException {
		Element hashSalt = XMLUtils.getChildElement(card, CardContext.IC_NS, CardContext.IC_HASH_SALT);
		String str = (hashSalt != null) ? XMLUtils.getTextContent(hashSalt) : "";
		if (isStringEmpty(str))
			throw new CardException("Couldn't retrieve HashSalt");
		return CardCryptography.decodeBase64(str);
	}

	public static Date parseTimeLastUpdated(Element card) throws CardException {
		Element timeLastUpdated = XMLUtils.getChildElement(card, CardContext.IC_NS, CardContext.IC_TIME_LAST_UPDATED);
		String str = (timeLastUpdated != null) ? XMLUtils.getTextContent(timeLastUpdated) : "";
		if (isStringEmpty(str))
			throw new CardException("Couldn't retrieve TimeLastUpdated");
		return DateConvertor.parse(str);
	}

	public static byte[] parseMasterKey(Element roamingCard) throws CardException {
		Element informationCardPrivateData = XMLUtils.getChildElement(roamingCard, CardContext.IC_NS, CardContext.IC_INFORMATION_CARD_PRIVATE_DATA);
		if (informationCardPrivateData != null) {
			Element masterKey = XMLUtils.getChildElement(informationCardPrivateData, CardContext.IC_NS, CardContext.IC_MASTER_KEY);
			String str = (masterKey != null) ? XMLUtils.getTextContent(masterKey) : "";
			if (isStringEmpty(str))
				throw new CardException("Couldn't get MasterKey element from RoamingInformationCard");
			return CardCryptography.decodeBase64(str);
		} else
			throw new CardException("Couldn't get \"InformationCardPrivateData\" element from RoamingInformationCard");
	}

	public static byte[] parsePinDigest(Element metaData) throws CardException {
		byte[] pinDigest_ = null;
		Element pinDigest = XMLUtils.getChildElement(metaData, CardContext.IC_NS, CardContext.IC_PIN_DIGEST);
		String str = (pinDigest != null) ? XMLUtils.getTextContent(pinDigest) : null;
		if (str != null)
			pinDigest_ = CardCryptography.decodeBase64(str);
		return pinDigest_;
	}

	public static HashMap parseClaimValues(Element privateData) throws CardException {
		HashMap tmpClaimValueMap_ = new HashMap();
		Element claimValueList = XMLUtils.getChildElement(privateData, CardContext.IC_NS, CardContext.IC_CLAIM_VALUE_LIST);
		if (claimValueList == null) 
			return tmpClaimValueMap_;
		// From the doc: optional element
		// throw new CardException("Couldn't get the list of \"ClaimValueList\" elements from InfoCard with id = " + id_);
		NodeList claimValues = claimValueList.getChildNodes();
		int len = claimValues.getLength();
		
		for (int i = 0; i < len; i++)
		{
			Node node = claimValues.item(i);
			if (node.getNodeType() == Node.ELEMENT_NODE && CardContext.IC_CLAIM_VALUE.equals(node.getLocalName()) && CardContext.IC_NS.equals(node.getNamespaceURI())) {
				Element elm = (Element) node;
				String uri = elm.getAttribute("Uri");
				if (uri == null || uri.trim().length() == 0)
					throw new CardException("Couldn't get \"uri\" attribute of ClaimValue element");
				Element valueElm = XMLUtils.getChildElement(elm, CardContext.IC_NS, CardContext.IC_VALUE);
				String value = (valueElm != null) ? XMLUtils.getTextContent(valueElm) : "";
				if (value != null) {
					ClaimValue claimValue = new ClaimValue(SelfIssuedCardClaims.getClaimType(uri), value);
					tmpClaimValueMap_.put(uri, claimValue);
				}
			}
		}
		
//		if (tmpClaimValueMap_.size() == 0)
//			throw new CardException("Couldn't get list of ClaimValue elements from private data section");
		
		/*
		ClaimValue claimValuePPID = new ClaimValue
			(SelfIssuedCardClaims.getClaimType
				("http://schemas.xmlsoap.org/ws/2005/05/identity/claims/privatepersonalidentifier"),
			"[Private]");
		tmpClaimValueList_.add(claimValuePPID);
		 */
		
		return tmpClaimValueMap_;
	}
	
	public static boolean isSelfIssued(Element card) throws CardException {
		boolean res = false;
		Element isSelfIssuedElm = XMLUtils.getChildElement(card, CardContext.IC_NS, CardContext.IC_IS_SELF_ISSUED);
		String boolValue = (isSelfIssuedElm != null) ? XMLUtils.getTextContent(isSelfIssuedElm) : "";
		if (isStringEmpty(boolValue) == false) {
			if ("true".equals(boolValue.trim().toLowerCase()))
				res = true;
		}
		return res;
	}
	
	public static boolean isStringEmpty(String str) {
		return str == null || str.trim().length() == 0;
	}
}
