/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.icard.provider.cardspace.common;

import java.io.IOException;
import java.io.StringReader;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.auth.ICredential;
import org.eclipse.higgins.icard.auth.ICredentialDescriptor;
import org.eclipse.higgins.icard.common.auth.SelfIssuedCredential;
import org.eclipse.higgins.icard.common.auth.UsernamePasswordCredential;
import org.eclipse.higgins.icard.common.auth.X509V3Credential;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.XMLUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

public class CredentialDescriptor implements ICredentialDescriptor {
	/*
	public static final String SELF_ISSUED_CREDENTIAL = "SelfIssuedCredential";

	public static final String X509V3_CREDENTIAL = "X509V3Credential";

	public static final String KERBEROS_V5_CREDENTIAL = "KerberosV5Credential";

	public static final String USERNAME_PASSWORD_CREDENTIAL = "UsernamePasswordCredential";

	public static final String OTHER_CREDENTIAL = "";
	*/
	
	private String type_;
	private String displayCredentialHint_;
	private Element credentialElm_;
	
	protected ICredential credential_;

	/**
	 * @param hint
	 * @param credential
	 * @throws ParserConfigurationException
	 * @throws SAXException
	 * @throws IOException
	 * @throws IllegalArgumentException
	 * @throws CardException 
	 */
	public CredentialDescriptor(String hint, String credential) throws ParserConfigurationException, SAXException, IOException, IllegalArgumentException, CardException {
		displayCredentialHint_ = hint;
		if (credential == null)
			throw new IllegalArgumentException("Parameter \"credential\" is null.");
		DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
		dbf.setIgnoringComments(true);
		dbf.setIgnoringElementContentWhitespace(true);
		dbf.setNamespaceAware(true);
		dbf.setValidating(false);
		DocumentBuilder db = dbf.newDocumentBuilder();
		StringReader sisCredential = new StringReader(credential);
		InputSource isCredential = new InputSource(sisCredential);
		Document mdDoc = db.parse(isCredential);
		credentialElm_ = mdDoc.getDocumentElement();
		credentialElm_.removeAttribute("xmlns");
		parseType();
	}

	/**
	 * @param hint
	 * @param credential
	 * @throws IllegalArgumentException
	 * @throws CardException 
	 */
	public CredentialDescriptor(String hint, Element credential) throws IllegalArgumentException, CardException {
		displayCredentialHint_ = hint;
		if (credential == null)
			throw new IllegalArgumentException("Parameter \"credential\" is null.");
		credentialElm_ = credential;
		parseType();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.icard.IUserCredential#getType()
	 */
	public String getType() {
		return type_;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.icard.IUserCredential#getDisplayCredentialHint()
	 */
	public String getDisplayCredentialHint() {
		return displayCredentialHint_;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.icard.IUserCredential#getCredential()
	 */
	public Element asXML() {
		return credentialElm_;
	}
	
	
	/**
	 * @throws CardException 
	 * 
	 */
	private void parseType() throws CardException {
		NodeList nl = credentialElm_.getChildNodes();
		int len = nl.getLength();
		for (int i = 0; i < len; i++) {
			Node node = nl.item(i);
			if (node.getNodeType() == Node.ELEMENT_NODE) {
				Element element = (Element) node;
				if ("DisplayCredentialHint".equals(node.getLocalName()) == false)
				{
					type_ = node.getLocalName();
					if (USERNAME_PASSWORD_CREDENTIAL.equals(type_)) {
						String name = null;
						Element e = XMLUtils.getChildElement(element, null, "Username");
						if (e != null) {
							name = e.getFirstChild().getNodeValue();
						}
//						credential = new TSUsernamePasswordCredential(
//								new TokenServiceCredential(epRef),
//								new UsernamePasswordCredential(name));
						credential_ = new UsernamePasswordCredential(name);
					} else if (SELF_ISSUED_CREDENTIAL.equals(type_)) {
						String ppid = null;
						Element e = XMLUtils.getChildElement(element, null, "PrivatePersonalIdentifier");
						if (e != null) {
							ppid = e.getFirstChild().getNodeValue();
						}
//						credential = new TSSelfIssuedCredential(
//								new TokenServiceCredential(epRef),
//								new SelfIssuedCredential(ppid));
						credential_ = new SelfIssuedCredential(ppid);
					} else if (KERBEROS_V5_CREDENTIAL.equals(type_)) {
						// TODO handle kerberos credential type
						throw new UnsupportedOperationException("The type \"" + type_ + "\" currently is not supported.");
					} else if (X509V3_CREDENTIAL.equals(type_)) {
						Element e = XMLUtils.getChildElement(element, null, "X509Data");
						if (e != null){
							Element keyIdentifier = XMLUtils.getChildElement(e, null, "KeyIdentifier");
							String value_type = keyIdentifier.getAttribute("ValueType");
							String encoding_type = keyIdentifier.getAttribute("EncodingType");
							String encoded_key = keyIdentifier.getFirstChild().getNodeValue();
							credential_ = new X509V3Credential(value_type, 
									encoding_type, encoded_key);
						}			
					} else {
						throw new IllegalArgumentException("Parameter \"credential\" is invalid.");
					}

				}
			}
		}
		
	}

	public ICredential getCredential() {
		return credential_;
	}
	

}
