/*******************************************************************************
 * Copyright (c) 2007 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Valery Kokhan - initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.icard.provider.cardspace.common;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import javax.security.auth.callback.CallbackHandler;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.AuthenticationException;
import org.eclipse.higgins.icard.CUID;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.ICard;
import org.eclipse.higgins.icard.ICardTemplate;
import org.eclipse.higgins.icard.IManagedInformationCard;
import org.eclipse.higgins.icard.ISimpleClaim;
import org.eclipse.higgins.icard.ISimpleClaimType;
import org.eclipse.higgins.icard.auth.ICredential;
import org.eclipse.higgins.icard.common.BasicCardProvider;
import org.eclipse.higgins.icard.common.ClaimValue;
import org.eclipse.higgins.icard.io.IElement;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.CardCryptography;
import org.eclipse.higgins.iss.ICardSelectorService;
import org.eclipse.higgins.iss.IDisplayToken;
import org.eclipse.higgins.iss.IICardSelector;
import org.eclipse.higgins.iss.IIdentityToken;
import org.eclipse.higgins.iss.SelectionANDofORs;
import org.eclipse.higgins.iss.SelectionANDofORsElm;
import org.w3c.dom.Element;

public abstract class ManagedCardProvider extends BasicCardProvider {
	protected Log log = LogFactory.getLog(ManagedCardProvider.class);

	/**
	 * To retrieve claim values for a card, card provider should send RST to
	 * appropriate STS. If STS requires "appliesTo" information about relying
	 * party, povider uses fake certificate and relying party defined below.
	 */
	protected String rpKeyStore = null;
	protected String rpKeyStorePassword = null;
	protected String rpCertificateAlias = null;
	protected String relyingPartyURI = "urn://higgins-selector";
	protected X509Certificate rpCertificate = null;

	protected static final String APPLIES_TO_SELECTOR_ID = "appliesTo.selectorIdentifier";
	protected static final String APPLIES_TO_KEYSTORE = "appliesTo.keystoreFile";
	protected static final String APPLIES_TO_KEYSTORE_PASS = "appliesTo.keystorePassword";
	protected static final String APPLIES_TO_CERTIFICATE_ALIAS = "appliesTo.certificateAlias";

	//	protected static String strCertificate = "MIIDHjCCAtwCBEcw9YAwCwYHKoZIzjgEAwUAMHUxCzAJBgNVBAYTAlVBMQswCQYDVQQIEwJETjEXMBUGA1UEBxMORG5lcHJvcGV0cm92c2sxDzANBgNVBAoTBlBhcml0eTEXMBUGA1UECxMOUGFyaXR5IFVrcmFpbmUxFjAUBgNVBAMTDVZhbGVyeSBLb2toYW4wHhcNMDcxMTA2MjMxNTEyWhcNMDgwMjA0MjMxNTEyWjB1MQswCQYDVQQGEwJVQTELMAkGA1UECBMCRE4xFzAVBgNVBAcTDkRuZXByb3BldHJvdnNrMQ8wDQYDVQQKEwZQYXJpdHkxFzAVBgNVBAsTDlBhcml0eSBVa3JhaW5lMRYwFAYDVQQDEw1WYWxlcnkgS29raGFuMIIBtzCCASwGByqGSM44BAEwggEfAoGBAP1/U4EddRIpUt9KnC7s5Of2EbdSPO9EAMMeP4C2USZpRV1AIlH7WT2NWPq/xfW6MPbLm1Vs14E7gB00b/JmYLdrmVClpJ+f6AR7ECLCT7up1/63xhv4O1fnxqimFQ8E+4P208UewwI1VBNaFpEy9nXzrith1yrv8iIDGZ3RSAHHAhUAl2BQjxUjC8yykrmCouuEC/BYHPUCgYEA9+GghdabPd7LvKtcNrhXuXmUr7v6OuqC+VdMCz0HgmdRWVeOutRZT+ZxBxCBgLRJFnEj6EwoFhO3zwkyjMim4TwWeotUfI0o4KOuHiuzpnWRbqN/C/ohNWLx+2J6ASQ7zKTxvqhRkImog9/hWuWfBpKLZl6Ae1UlZAFMO/7PSSoDgYQAAoGALQOE9rSSxFhktVM7whlol8WhgNpU1ZAKuJF0PYBK8fBHMH/+6RFpqrd44DyiPQ1vwtFzHaOLCNh0Q1u8aRJc3DwgX62v/4VU6U+1HHmU9JsLWxFGZgk2xCOcU3UJzE0HaQji3TvR3Punl2vdRbLjy/XWnxpcIF4yGb+wbeS2hoowCwYHKoZIzjgEAwUAAy8AMCwCFB9FoKbmWzoxjD4C0QmxbBAVY0FNAhQA3LbUiH6NU7FNuv1vhRI7OxQdkQ==";
	protected static String strCertificate = "MIICVDCCAb2gAwIBAgIERWUHrzANBgkqhkiG9w0BAQUFADBQMQswCQYDVQQGEwJVUzEYMBYGA1UEChMPSUJNIENvcnBvcmF0aW9uMRUwEwYDVQQLEwxJQk0gUmVzZWFyY2gxEDAOBgNVBAMTB1Jvb3QgQ0EwHhcNMDYxMTIzMDIyOTU3WhcNMjYxMTE4MDIyOTU3WjBFMQswCQYDVQQGEwJVUzEQMA4GA1UEChMHRWNsaXBzZTEQMA4GA1UECxMHSGlnZ2luczESMBAGA1UEAxMJbG9jYWxob3N0MIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQCUFDtM2/nLEhSJqF7zqayW8gQ183KCiKME53VkSjl6C7O9J2hEiBMWsI9a+qBm13VWKvmTc3h3EsLaR/dOreXyCLKEwDIpwHojW3+i037OOxL6U3Ij094rqfHBCJP2xEyLNpPXdIioEgP5H6MTRPXShsBsTQ7bEhZs0QKyXwTNHwIDAQABo0YwRDAdBgNVHQ4EFgQUTHM9O3KViNYsvuGeg4eANMCFHuMwEwYDVR0lBAwwCgYIKwYBBQUHAwEwDgYDVR0PAQH/BAQDAgSwMA0GCSqGSIb3DQEBBQUAA4GBADkZZ34T637gwxp/EmRlRd4ru1Y2ribGoLRpY1icJYlyrUB3552f1K7r69EQxuwnyUnFYVhB+4ZnzhralkKtNs8UVo7RZHSE8jKWOi4YU3RXSqEHKs10ttQLLcHPuH4o9t2OJLeaVtaijbdicbxUEwqhgcNnNrQHe6+f5fgwzSYM";
	protected X509Certificate certificate = null;

	protected void init(Hashtable config) {
		try {
			if (config.containsKey(APPLIES_TO_KEYSTORE))
				rpKeyStore = (String)config.get(APPLIES_TO_KEYSTORE);
			if (config.containsKey(APPLIES_TO_KEYSTORE_PASS))
				rpKeyStorePassword = (String)config.get(APPLIES_TO_KEYSTORE_PASS);
			if (config.containsKey(APPLIES_TO_CERTIFICATE_ALIAS))
				rpCertificateAlias = (String)config.get(APPLIES_TO_CERTIFICATE_ALIAS);
			if (config.containsKey(APPLIES_TO_SELECTOR_ID)) {
				String selectorID = (String)config.get(APPLIES_TO_SELECTOR_ID);
				if (selectorID != null && selectorID.trim().length() > 0)
					relyingPartyURI = selectorID;
			}
		}
		catch(Exception e) {
			log.error(e, e);
		}
	}

	public ICard getICardByCUID(CallbackHandler authHandler, CUID cuid, ICredential userCredential) throws AuthenticationException, CardException {
		ICard card = getICardByCUID(authHandler, cuid);
		if (card instanceof ManagedCard) {
			if (rpCertificate == null)
				initFakeRelyingPartyCertificate();
			ManagedCard mcard = (ManagedCard) card;
			CardSpacePolicy p = new CardSpacePolicy();
			p.setRequiredClaims(mcard.getSupportedClaimTypesUris());
			try {
				ICardSelectorService iss = ICardSelectorService.getInstance();
				IICardSelector s = iss.getICardSelector(authHandler, p);
				SelectionANDofORs selection = new SelectionANDofORs();
				selection.action = relyingPartyURI;
				selection.add(new SelectionANDofORsElm(mcard.getCUID().toString(), 0, 0));
				selection.setCredential(userCredential);
				selection.sslCertChain = new X509Certificate[]{rpCertificate};
				List claims = new LinkedList();
				IIdentityToken t = s.getIdentityToken(selection);
//				System.err.println(t.getAs(String.class));
				t.getAs(String.class);
				IDisplayToken dt = (IDisplayToken) t.getAs(IDisplayToken.class);
				System.err.println(dt.getAs(String.class));
				for (Iterator itr = mcard.getSupportedClaimTypes(); itr.hasNext(); ) {
					ISimpleClaimType ct = (ISimpleClaimType) itr.next();
					ISimpleClaim c = (ISimpleClaim) dt.getClaim(ct.getType());
					if (c == null) {
						c = new ClaimValue(ct, "");
					}
					claims.add(c);
				}
				mcard.setClaims(claims);
				return mcard;
			} catch (STSFaultException e) {
				throw e;
			} catch (Exception e) {
				throw new CardException(e);
			}
		} else {
			throw new CardException("Unsupported card type: " + card);
		}
	}

	/**
	 * Managed card provider doesn't support card creation. So, just return <code>null</code>
	 */
	public ICardTemplate[] getCardCreationTemplates(CallbackHandler authHandler) {
		return null;
	}
	
	/**
	 * Managed card provider doesn't support card creation. So, just return <code>null</code>
	 */
	public ICardTemplate getCardCreationTemplatesByID(CallbackHandler authHandler, String ID){
		return null;
	}

	/**
	 * The only supported type is {@link IManagedInformationCard}
	 */
	public Class[] getSupportedTypes() {
		return new Class[] {IManagedInformationCard.class};
	}

	protected X509Certificate getDefaultCertificate() {
		if (certificate == null) {
			try {
				CertificateFactory certificateFactory = CertificateFactory.getInstance("X.509");
				String strEncryptionCertificate = "-----BEGIN CERTIFICATE-----\n"
						+ strCertificate + "\n-----END CERTIFICATE-----\n";
				ByteArrayInputStream streamCertificate = new ByteArrayInputStream(strEncryptionCertificate.getBytes());
				certificate = (java.security.cert.X509Certificate) certificateFactory.generateCertificate(streamCertificate);
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		return certificate;
	}

	public boolean canImportICard(CallbackHandler authHandler, IElement element) {
		if (element == null)
			return false;
		try {
			Element card = (Element) element.getAs(Element.class);
			if (card == null) {
				return false;
			}
			Element crd = null;
			if ("Signature".equals(card.getLocalName()))
				crd = CardCryptography.getCardFromSignedEnvelop(card);
			else
				crd = card;
			DummyManagedCard c = new DummyManagedCard();
			c.initFromXML(crd);
			return true;
		} catch (Exception e) {
			e.printStackTrace();
			log.warn(e);
		}
		return false;
	}

	/**
	 * @return 	fake relying party certificate from provider configuration
	 */
	private X509Certificate getCertificateFromKeystore() {
		X509Certificate res = null;
		FileInputStream fis = null;
		try {
			char[] pass = null;
			if (rpKeyStorePassword != null) {
				pass = rpKeyStorePassword.toCharArray();
			}
			File rpksFile = new File(rpKeyStore);
			fis = new FileInputStream(rpksFile);
			java.security.KeyStore keyStore = java.security.KeyStore.getInstance("JKS");
			keyStore.load(fis, pass);
			res = (X509Certificate)keyStore.getCertificate(rpCertificateAlias);
		}
		catch(Exception e) {
			log.error("Can not init the fake RP certificate from keystore", e);
		}
		finally {
			if (fis != null) {
				try {
					fis.close();
				}
				catch(IOException e) {
					log.warn(e, e);
				}
			}
		}
		return res;
	}

	/**
	 * Initialize fake relying party certificate which could be used to retrieve claims
	 */
	private void initFakeRelyingPartyCertificate() {
		if (rpKeyStore == null) {
			log.warn("The file name of key store of fake RP certificate is not defined. The following configuration parameter should be defined: " + APPLIES_TO_KEYSTORE);
		}
		else if (rpCertificateAlias == null) {
			log.warn("The file name of key store of fake RP certificate is not defined. The following configuration parameter should be defined: " + APPLIES_TO_KEYSTORE);
		}
		else {
			rpCertificate = getCertificateFromKeystore();
		}
		if (rpCertificate == null)
			rpCertificate = getCertificateFromKeystore();
	}
	
}
