/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.icard.provider.cardspace.common.utils;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.SecureRandom;
import java.security.cert.X509Certificate;
import java.util.Arrays;

import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.xml.security.exceptions.Base64DecodingException;
import org.apache.xml.security.keys.KeyInfo;
import org.apache.xml.security.signature.XMLSignature;
import org.apache.xml.security.utils.Base64;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.common.utils.CardContext;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

public class CardCryptography {
	private static Log log = LogFactory.getLog(CardCryptography.class);

	private static final byte[] BOM = { (byte) 0xEF, (byte) 0xBB, (byte) 0xBF };

	private static final byte[] ENCRYPTION_KEY_ENTROPY = { (byte) 0xd9, (byte) 0x59, (byte) 0x7b, (byte) 0x26, (byte) 0x1e, (byte) 0xd8, (byte) 0xb3,
			(byte) 0x44, (byte) 0x93, (byte) 0x23, (byte) 0xb3, (byte) 0x96, (byte) 0x85, (byte) 0xde, (byte) 0x95, (byte) 0xfc };

	private static final byte[] INTEGRITY_KEY_ENTROPY = { (byte) 0xc4, (byte) 0x01, (byte) 0x7b, (byte) 0xf1, (byte) 0x6b, (byte) 0xad, (byte) 0x2f,
			(byte) 0x42, (byte) 0xaf, (byte) 0xf4, (byte) 0x97, (byte) 0x7d, (byte) 0x4, (byte) 0x68, (byte) 0x3, (byte) 0xdb };

	private static Document parseEncryptedStore(InputStream is) throws ParserConfigurationException, SAXException, IOException {
		is.skip(BOM.length);
		DocumentBuilder db = createDocumentBuilder();
		return db.parse(is);
	}

	/**
	 * @param data
	 * @return
	 * @throws ParserConfigurationException
	 * @throws SAXException
	 * @throws IOException
	 */
	private static Document parseDecryptedData(byte[] data) throws ParserConfigurationException, SAXException, IOException {
		DocumentBuilder db = createDocumentBuilder();
		ByteArrayInputStream bis = new ByteArrayInputStream(data);
		return db.parse(bis);
	}

	/**
	 * @return
	 * @throws ParserConfigurationException
	 */
	private static DocumentBuilder createDocumentBuilder() throws ParserConfigurationException {
		DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
		dbf.setIgnoringComments(true);
		dbf.setIgnoringElementContentWhitespace(true);
		dbf.setNamespaceAware(true);
		dbf.setValidating(false);
		return dbf.newDocumentBuilder();
	}

	/**
	 * @param doc
	 * @param xQuery
	 * @return
	 * @throws XPathExpressionException
	 * @throws CardException
	 * @throws Base64DecodingException
	 * @throws DOMException
/*
	private static byte[] extractEncryptedData(Document doc, String xQuery) throws XPathExpressionException, CardException, Base64DecodingException,
			DOMException {
		XPath xPath = org.eclipse.higgins.icard.provider.cardspace.common.utils.XMLUtils.createXPath();
		Element ss = (Element) xPath.evaluate(xQuery, doc, XPathConstants.NODE);
		if (ss == null)
			throw new CardException("Couldn't get node from encrypted store : " + xQuery);
		return Base64.decode(ss.getTextContent().getBytes());
	}
*/
	/**
	 * @param array
	 * @param offset
	 * @param len
	 * @return
	 */
	public static byte[] getSubArray(byte[] array, int offset, int len) {
		byte[] newArr = new byte[len];
		System.arraycopy(array, offset, newArr, 0, len);
		return newArr;
	}

	/**
	 * @param encryptedData
	 * @param initVector
	 * @param encryptionKey
	 * @return
	 * @throws NoSuchAlgorithmException
	 * @throws NoSuchProviderException
	 * @throws NoSuchPaddingException
	 * @throws InvalidKeyException
	 * @throws InvalidAlgorithmParameterException
	 * @throws IllegalBlockSizeException
	 * @throws CardException
	 * @throws BadPaddingException
	 */
	private static byte[] decryptData(byte[] encryptedData, byte[] initVector, Key encryptionKey) throws NoSuchAlgorithmException, NoSuchProviderException,
			NoSuchPaddingException, InvalidKeyException, InvalidAlgorithmParameterException, IllegalBlockSizeException, CardException {
		// Cipher cipher = Cipher.getInstance("AES/CBC/PKCS7Padding", "BC");
		Cipher cipher = Cipher.getInstance("AES/CBC/PKCS5Padding");
		IvParameterSpec ips = new IvParameterSpec(initVector);
		cipher.init(Cipher.DECRYPT_MODE, encryptionKey, ips);
		try {
			return cipher.doFinal(encryptedData);
		} catch (BadPaddingException e) {
			log.error(e);
			throw new CardException(e.getMessage() + ". Wrong password or corrupted data.");
		}
	}

	/**
	 * @param iv
	 * @param integrityKey
	 * @param decryptedData
	 * @return
	 * @throws NoSuchAlgorithmException
	 * @throws NoSuchProviderException
	 * @throws NoSuchPaddingException
	 * @throws InvalidKeyException
	 * @throws InvalidAlgorithmParameterException
	 * @throws IllegalBlockSizeException
	 * @throws BadPaddingException
	 */
	private static byte[] calculateIntegityHash(byte[] iv, byte[] integrityKey, byte[] decryptedData) throws NoSuchAlgorithmException, NoSuchProviderException,
			NoSuchPaddingException, InvalidKeyException, InvalidAlgorithmParameterException, IllegalBlockSizeException, BadPaddingException {
		byte[] lastDataBlock = getSubArray(decryptedData, decryptedData.length - 16, 16);
		byte[] key = concat(iv, integrityKey);
		key = concat(key, lastDataBlock);
		MessageDigest md = MessageDigest.getInstance("SHA-256");
		return md.digest(key);
	}

	/**
	 * Performs decryption of CardSpace backup file (*.crds file format) and
	 * returns <code>Document</code> with RoamingStore root element
	 * 
	 * @param is
	 *            <code>InputStream</code> of backup data
	 * @param password
	 *            Password used to encrypt backup data
	 * @return
	 * @throws Exception
	 */
	public static Document decrypt(InputStream is, String password) throws Exception {
		if (is == null)
			throw new IllegalArgumentException("Parameter \"is\" is null");
		if (password == null)
			throw new IllegalArgumentException("Parameter \"password\" is null");
		Document encriptedData = parseEncryptedStore(is);
		Element encryptedStoreElm = encriptedData.getDocumentElement();
		Element storeSaltElm = XMLUtils.getChildElement(encryptedStoreElm, CardContext.IC_NS, CardContext.IC_STORE_SALT);
		byte[] storeSalt = Base64.decode(XMLUtils.getTextContent(storeSaltElm)); 
		Element encryptedDataElm = XMLUtils.getChildElement(encryptedStoreElm, CardContext.XE_NS, CardContext.XE_ENCRYPTED_DATA);
		Element cipherDataElm = XMLUtils.getChildElement(encryptedDataElm, CardContext.XE_NS, CardContext.XE_CIPHER_DATA);
		Element cipherValueElm = XMLUtils.getChildElement(cipherDataElm, CardContext.XE_NS, CardContext.XE_CIPHER_VALUE);
		byte[] cipherValue = Base64.decode(XMLUtils.getTextContent(cipherValueElm));
		if (cipherValue.length < 48)
			throw new CardException("Cardspace backup file is garbled. Decoded CipherValue byte array is too short.");
		byte[] initVector = getSubArray(cipherValue, 0, 16);
		byte[] integrityHash = getSubArray(cipherValue, 16, 32);
		byte[] encryptedData = getSubArray(cipherValue, 48, cipherValue.length - 48);
		byte[] derivedKey = getDerivedKey(password.getBytes("UTF-16LE"), storeSalt);
		Key encryptionKey = getEncryptionKey(derivedKey);
		byte[] integrityKey = getIntegrityKey(derivedKey);
		byte[] decryptedData = decryptData(encryptedData, initVector, encryptionKey);
		decryptedData = getSubArray(decryptedData, BOM.length, decryptedData.length - BOM.length);
		byte[] calculatedIntegityHash = calculateIntegityHash(initVector, integrityKey, decryptedData);
		if (Arrays.equals(calculatedIntegityHash, integrityHash) == false)
			throw new CardException("Cardspace backup file is garbled. Integrity checking failed.");
		return parseDecryptedData(decryptedData);
	}

	/**
	 * Checks whether data in the provided <code>InputStream</code> represents
	 * CardSpace backup format (*.crds file format)
	 * 
	 * @param is
	 *            <code>InputStream</code> to check
	 */
	public static boolean isEncriptedStore(InputStream is) {
		boolean res = false;
		try {
			if (is == null)
				return res;
			Document encriptedData = parseEncryptedStore(is);
			Element encryptedStoreElm = encriptedData.getDocumentElement();
			Element storeSaltElm = XMLUtils.getChildElement(encryptedStoreElm, CardContext.IC_NS, "StoreSalt");
			byte[] storeSalt = Base64.decode(XMLUtils.getTextContent(storeSaltElm));
			if (storeSalt != null) {
				Element encryptedDataElm = XMLUtils.getChildElement(encryptedStoreElm, CardContext.XE_NS, CardContext.XE_ENCRYPTED_DATA);
				Element cipherDataElm = XMLUtils.getChildElement(encryptedDataElm, CardContext.XE_NS, CardContext.XE_CIPHER_DATA);
				Element cipherValueElm = XMLUtils.getChildElement(cipherDataElm, CardContext.XE_NS, CardContext.XE_CIPHER_VALUE);
				byte[] cipherValue = Base64.decode(XMLUtils.getTextContent(cipherValueElm));
				if (cipherValue.length < 48)
					res = false;
				else
					res = true;
			}
		} catch (Exception e) {
			log.trace(e);
			res = false;
		}
		return res;
	}
	
	/**
	 * @param data
	 * @param initVector
	 * @param encryptionKey
	 * @return
	 * @throws Exception
	 */
	public static byte[] encryptData(byte[] data, byte[] initVector, Key encryptionKey) throws Exception {
		// Cipher cipher = Cipher.getInstance("AES/CBC/PKCS7Padding", "BC");
		Cipher cipher = Cipher.getInstance("AES/CBC/PKCS5Padding");
		IvParameterSpec ips = new IvParameterSpec(initVector);
		cipher.init(Cipher.ENCRYPT_MODE, encryptionKey, ips);
		return cipher.doFinal(data);
	}

	/**
	 * @param doc
	 * @return
	 * @throws TransformerException
	 * @throws IOException
	 */
	private static byte[] convert(Document doc) throws TransformerException, IOException {
		ByteArrayOutputStream bos = new ByteArrayOutputStream(100000);
		bos.write(BOM);
		TransformerFactory transformerFactory = TransformerFactory.newInstance();
		Transformer transformer = transformerFactory.newTransformer();
		DOMSource source = new DOMSource(doc);
		StreamResult result = new StreamResult(bos);
		transformer.transform(source, result);
		return bos.toByteArray();
	}

	/**
	 * @param data
	 * @param salt
	 * @param password
	 * @return
	 * @throws Exception
	 */
	private static byte[] getCipherValue(byte[] data, byte[] salt, String password) throws Exception {
		byte[] initVector = SecureRandom.getSeed(16);
		byte[] derivedKey = getDerivedKey(password.getBytes("UTF-16LE"), salt);
		Key encryptionKey = getEncryptionKey(derivedKey);
		byte[] integrityKey = getIntegrityKey(derivedKey);
		byte[] encryptedData = encryptData(data, initVector, encryptionKey);
		byte[] integrityHash = calculateIntegityHash(initVector, integrityKey, data);
		byte[] cipherValue = concat(initVector, integrityHash);
		cipherValue = concat(cipherValue, encryptedData);
		return cipherValue;
	}

	/**
	 * @param salt
	 * @param cipherVal
	 * @return
	 * @throws ParserConfigurationException
	 */
	private static Document createEncryptedStore(byte[] salt, byte[] cipherVal) throws ParserConfigurationException {
		DocumentBuilder db = createDocumentBuilder();
		Document doc = db.newDocument();
		Element root = doc.createElementNS(CardContext.IC_NS, CardContext.IC_ENCRYPTED_STORE);
		doc.appendChild(root);
		Element storeSalt = doc.createElement(CardContext.IC_STORE_SALT);
		XMLUtils.setTextContent(storeSalt, new String(Base64.encode(salt)));
		root.appendChild(storeSalt);
		Element encryptedData = doc.createElementNS(CardContext.XE_NS, CardContext.XE_ENCRYPTED_DATA);
		root.appendChild(encryptedData);
		Element cipherData = doc.createElement(CardContext.XE_CIPHER_DATA);
		encryptedData.appendChild(cipherData);
		Element cipherValue = doc.createElement(CardContext.XE_CIPHER_VALUE);
		XMLUtils.setTextContent(cipherValue, new String(Base64.encode(cipherVal)));
		cipherData.appendChild(cipherValue);
		return doc;
	}

	/**
	 * @param cardsStore
	 * @param password
	 * @return
	 * @throws Exception
	 */
	private static Document encrypt(Document cardsStore, String password) throws Exception {
		byte[] data = convert(cardsStore);
		byte[] salt = SecureRandom.getSeed(16);
		byte[] cipherValue = getCipherValue(data, salt, password);
		return createEncryptedStore(salt, cipherValue);
	}

	/**
	 * Performs encryption of CardSpace-interoperable ICards
	 * 
	 * @param cardsStore
	 *            <code>Document</code> with list of cards in RoamingStore
	 *            format
	 * @param os
	 *            <code>OutputStream</code> for data output
	 * @param password
	 *            Password used to encrypt backup data
	 * @throws Exception
	 */
	public static void encrypt(Document cardsStore, OutputStream os, String password) throws Exception {
		Document encryptedStore = encrypt(cardsStore, password);
		os.write(BOM);
		TransformerFactory transformerFactory = TransformerFactory.newInstance();
		Transformer transformer = transformerFactory.newTransformer();
		transformer.setOutputProperty(OutputKeys.METHOD, "xml"); //$NON-NLS-1$
		DOMSource source = new DOMSource(encryptedStore);
		StreamResult result = new StreamResult(os);
		transformer.transform(source, result);
		os.flush();
	}

	/**
	 * @param arr1
	 * @param arr2
	 * @return
	 */
	private static byte[] concat(byte[] arr1, byte[] arr2) {
		byte[] newArr = new byte[arr1.length + arr2.length];
		System.arraycopy(arr1, 0, newArr, 0, arr1.length);
		System.arraycopy(arr2, 0, newArr, arr1.length, arr2.length);
		return newArr;
	}

	/**
	 * @param derivedKey
	 * @return
	 * @throws NoSuchAlgorithmException
	 */
	private static byte[] getIntegrityKey(byte[] derivedKey) throws NoSuchAlgorithmException {
		MessageDigest hash = MessageDigest.getInstance("SHA-256");
		byte[] key = concat(INTEGRITY_KEY_ENTROPY, derivedKey);
		return hash.digest(key);
	}

	/**
	 * @param password
	 * @param salt
	 * @return
	 * @throws NoSuchAlgorithmException
	 */
	public static byte[] getDerivedKey(byte[] password, byte[] salt) throws NoSuchAlgorithmException {
		MessageDigest hash = MessageDigest.getInstance("SHA-256");
		byte[] key = concat(password, salt);
		for (int i = 0; i < 1000; i++)
			key = hash.digest(key);
		return key;
	}

	/**
	 * @param derivedKey
	 * @return
	 * @throws NoSuchAlgorithmException
	 */
	private static Key getEncryptionKey(byte[] derivedKey) throws NoSuchAlgorithmException {
		MessageDigest hash = MessageDigest.getInstance("SHA-256");
		byte[] key = concat(ENCRYPTION_KEY_ENTROPY, derivedKey);
		key = hash.digest(key);
		return new SecretKeySpec(key, "AES");
	}

	/**
	 * @param signedCard
	 * @return
	 * @throws Exception
	 */
	public static Element getCardFromSignedEnvelop(InputStream signedCard) throws Exception {
		DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
		dbf.setIgnoringComments(true);
		dbf.setIgnoringElementContentWhitespace(true);
		dbf.setNamespaceAware(true);
		dbf.setValidating(false);
		DocumentBuilder db = dbf.newDocumentBuilder();
		Document doc = db.parse(signedCard);
		Element sigElement = doc.getDocumentElement();
		return getCardFromSignedEnvelop(sigElement);
	}

	/**
	 * @param signedCard
	 * @return
	 * @throws Exception
	 */
	public static Element getCardFromSignedEnvelop(Element sigElement) throws Exception {
		if (sigElement == null)
			throw new CardException("Parameter \"sigElement\" is null");
		XMLSignature signature = new XMLSignature(sigElement, "");
		KeyInfo ki = signature.getKeyInfo();
		if (ki == null)
			throw new CardException("Couldn't find a KeyInfo element in signed envelop.");
		if (ki.containsX509Data() == false)
			throw new CardException("Couldn't find a X509Data element in the KeyInfo.");
		X509Certificate cert = signature.getKeyInfo().getX509Certificate();
		if (cert == null)
			throw new CardException("Couldn't find a certificate.");
		if (signature.checkSignatureValue(cert) == false)
			throw new CardException("Signature is invalid.");
		if (signature.getObjectLength() != 1)
			throw new CardException("Signature contains wrong count of objects (only one object container expected).");
		else {
			Element obj = signature.getObjectItem(0).getElement();
			NodeList nl = obj.getChildNodes();
			Element card = null;
			int len = nl.getLength();
			for (int i = 0; i < len; i++) {
				Node node = nl.item(i);
				if (node.getNodeType() == Node.ELEMENT_NODE) {
					Element elm = (Element)node;
					if (CardContext.IC_INFORMATION_CARD.equals(elm.getLocalName())) {
						card = elm;
						break;
					}
				}
			}
			
			if (card == null)
				throw new CardException("Object container doesn't contain InformationCard element.");
			else
				return card;
		}
	}

	/**
	 * Encodes byte array to <code>String</code> using Base64
	 * 
	 * @param data
	 * @param wrappedStringlen
	 *            length of
	 * @return
	 */
	public static String encodeBase64(byte[] data, int wrappedStringlen) {
		return Base64.encode(data, wrappedStringlen);
	}

	/**
	 * Decodes <code>String</code> to byte array using Base64
	 * 
	 * @param data
	 * @return
	 * @throws CardException
	 */
	public static byte[] decodeBase64(String data) throws CardException {
		try {
			return Base64.decode(data);
		} catch (Base64DecodingException e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
	}

	/**
	 * @param bytesCount
	 * @return
	 */
	public static String getBase64Hash(int bytesCount) {
		return CardCryptography.encodeBase64(SecureRandom.getSeed(bytesCount), 0);
	}

	/**
	 * @param bytesCount
	 * @return
	 */
	public static byte[] getRandomBytes(int bytesCount) {
		return SecureRandom.getSeed(bytesCount);
	}

	/**
	 * @param encryptedData
	 * @param key
	 * @return
	 * @throws CardException
	 */
	public static byte[] decryptPersonalCardField(byte[] encryptedData, EncryptedMasterKey key) throws CardException {
		byte[] decryptedData = null;
		try {
			decryptedData = decryptData(encryptedData, key.getInitVector(), key.getKey());
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
		return decryptedData;
	}

	
/*	
	public static byte[] decryptPersonalCardField(byte[] data, String pin) throws CardException {
		if (data.length < 38)
			throw new CardException("Too short encrypted data array.");
		byte[] version = getSubArray(data, 0, 1);
		log.debug("decryptPersonalCardField() > Version " + String.valueOf(version));
		byte[] salt = getSubArray(data, 1, 16);
		byte[] iterationCount = getSubArray(data, 17, 4);
		log.debug("decryptPersonalCardField() > Version " + String.valueOf(iterationCount));
		byte[] initVector = getSubArray(data, 21, 16);
		byte[] encryptedData = getSubArray(data, 37, data.length - 37);
		byte[] derivedKey;
		try {
			derivedKey = getDerivedKey(password, salt);
		} catch (NoSuchAlgorithmException e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
		Key key = new SecretKeySpec(derivedKey, "AES");
		byte[] decryptedData = null;
		try {
			decryptedData = decryptData(encryptedData, initVector, key);
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
		return decryptedData;
	}
*/

	/**
	 * @param data
	 * @param pinCode
	 * @return
	 * @throws CardException
	 */
	public static byte[] encryptPersonalCardField(byte[] data, byte[] salt, byte[] initVector, byte[] pinCode) throws CardException {
		if (pinCode == null)
			throw new CardException("Parameter \"pinCode\" is null");
		byte[] derivedKey;
		try {
			derivedKey = getDerivedKey(pinCode, salt);
		} catch (NoSuchAlgorithmException e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
		Key key = new SecretKeySpec(derivedKey, "AES");
		byte[] encryptedData;
		try {
			encryptedData = encryptData(data, initVector, key);
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
		return encryptedData;
	}

	/**
	 * @param pinCode
	 * @return
	 * @throws CardException
	 */
	public static byte[] getPinDigest(byte[] pinCode) throws CardException {
		if (pinCode == null)
			throw new CardException("Parameter \"pinCode\" is null");
		try {
			MessageDigest hash = MessageDigest.getInstance("SHA-1");
			return hash.digest(pinCode);
		} catch (NoSuchAlgorithmException e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
	}

	static {
		org.apache.xml.security.Init.init();
		// Security.addProvider(new
		// org.bouncycastle.jce.provider.BouncyCastleProvider());
	}

}
