/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.icard.provider.cardspace.common.utils;

import java.net.URI;
import java.net.URISyntaxException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.IClaim;
import org.eclipse.higgins.icard.IClaimType;
import org.eclipse.higgins.icard.ISimpleClaim;
import org.eclipse.higgins.icard.ISimpleClaimType;
import org.eclipse.higgins.icard.InvalidClaimException;
import org.eclipse.higgins.icard.InvalidTypeException;
import org.eclipse.higgins.icard.common.ClaimType;
import org.eclipse.higgins.icard.common.ClaimValue;
import org.eclipse.higgins.icard.common.UIDescriptor;

import com.ibm.icu.text.SimpleDateFormat;

public class SelfIssuedCardClaims {

	public static final String GIVEN_NAME_TYPE = "http://schemas.xmlsoap.org/ws/2005/05/identity/claims/givenname";

	public static final String GIVEN_NAME_NAME = "First Name";

	public static final String GIVEN_NAME_DESC = "First Name";

	public static final String SURNAME_TYPE = "http://schemas.xmlsoap.org/ws/2005/05/identity/claims/surname";

	public static final String SURNAME_NAME = "Last Name";

	public static final String SURNAME_DESC = "Last Name";

	public static final String EMAIL_ADDRESS_TYPE = "http://schemas.xmlsoap.org/ws/2005/05/identity/claims/emailaddress";

	public static final String EMAIL_ADDRESS_NAME = "Email Address";

	public static final String EMAIL_ADDRESS_DESC = "Email Address";

	public static final String STREET_ADDRESS_TYPE = "http://schemas.xmlsoap.org/ws/2005/05/identity/claims/streetaddress";

	public static final String STREET_ADDRESS_NAME = "Street";

	public static final String STREET_ADDRESS_DESC = "Street";

	public static final String LOCALITY_TYPE = "http://schemas.xmlsoap.org/ws/2005/05/identity/claims/locality";

	public static final String LOCALITY_NAME = "City";

	public static final String LOCALITY_DESC = "City";

	public static final String STATE_OR_PROVINCE_TYPE = "http://schemas.xmlsoap.org/ws/2005/05/identity/claims/stateorprovince";

	public static final String STATE_OR_PROVINCE_NAME = "State";

	public static final String STATE_OR_PROVINCE_DESC = "State";

	public static final String POSTAL_CODE_TYPE = "http://schemas.xmlsoap.org/ws/2005/05/identity/claims/postalcode";

	public static final String POSTAL_CODE_NAME = "Postal Code";

	public static final String POSTAL_CODE_DESC = "Postal Code";

	public static final String COUNTRY_TYPE = "http://schemas.xmlsoap.org/ws/2005/05/identity/claims/country";

	public static final String COUNTRY_NAME = "Country/Region";

	public static final String COUNTRY_DESC = "Country/Region";

	public static final String HOME_PHONE_TYPE = "http://schemas.xmlsoap.org/ws/2005/05/identity/claims/homephone";

	public static final String HOME_PHONE_NAME = "Home Phone";

	public static final String HOME_PHONE_DESC = "Home Phone Number";

	public static final String OTHER_PHONE_TYPE = "http://schemas.xmlsoap.org/ws/2005/05/identity/claims/otherphone";

	public static final String OTHER_PHONE_NAME = "Other Phone";

	public static final String OTHER_PHONE_DESC = "Other Phone Number";

	public static final String MOBILE_PHONE_TYPE = "http://schemas.xmlsoap.org/ws/2005/05/identity/claims/mobilephone";

	public static final String MOBILE_PHONE_NAME = "Mobile Phone";

	public static final String MOBILE_PHONE_DESC = "Mobile Phone Number";

	public static final String DATE_OF_BIRTH_TYPE = "http://schemas.xmlsoap.org/ws/2005/05/identity/claims/dateofbirth";

	public static final String DATE_OF_BIRTH_NAME = "Date of Birth";

	public static final String DATE_OF_BIRTH_DESC = "Date of Birth";

	public static final String GENDER_TYPE = "http://schemas.xmlsoap.org/ws/2005/05/identity/claims/gender";

	public static final String GENDER_NAME = "Gender";

	public static final String GENDER_DESC = "Gender";

	public static final String WEB_PAGE_TYPE = "http://schemas.xmlsoap.org/ws/2005/05/identity/claims/webpage";

	public static final String WEB_PAGE_NAME = "Web Page";

	public static final String WEB_PAGE_DESC = "Web Page";

	public static final String PPID_TYPE = "http://schemas.xmlsoap.org/ws/2005/05/identity/claims/privatepersonalidentifier";

	public static final String PPID_NAME = "Site Specific Card ID";

	public static final String PPID_DESC = "Site Specific Card ID";

	private static Hashtable claimTypes_;

	private static ArrayList claimTypesList_;

	private Hashtable claimValues_;

	public static final String GENDER_MALE = "Male";

	public static final String GENDER_FEMALE = "Female";

	private static final SimpleDateFormat BIRTH_SDF = new SimpleDateFormat("yyyy-MM-dd"); 

	private static final ArrayList GENDER_LIST = new ArrayList();

	/**
	 * @param claimValues list of <code>IClaim</code>
	 * @throws CardException
	 */
	public SelfIssuedCardClaims(List claimValues) throws CardException {
		claimValues_ = getSupportedTypesHash();
		if (claimValues != null) {
			int size = claimValues.size();
			for (int i = 0; i < size; i++) {
				IClaim value = (IClaim)claimValues.get(i);
				IClaimType claimType = value.getType();
				if (claimType == null)
					throw new InvalidClaimException("Claim doestn't contain claim type.");
				if (!claimType.isSimple()) {
					throw new InvalidTypeException("Complex clime type: " + claimType.getType() + " is not supported.");
				}
				ISimpleClaimType simpleType = null;
				try {
					simpleType = (ISimpleClaimType) claimType;
				} catch (ClassCastException e) {
					throw new InvalidTypeException(e);
				}
				if (simpleType.getMaxCardinality() != 1) {
					throw new InvalidTypeException("Unsupported multivalued clime type: " + claimType.getType());
				}
				URI type = null;
				try {
					type = new URI(claimType.getType());
				} catch (URISyntaxException e) {
					throw new CardException(e);
				}
				if (type == null)
					throw new InvalidTypeException("Claim type contains null type URI.");
				String typeStr = type.toString();
				if (claimTypes_.containsKey(typeStr) == false)
					throw new InvalidTypeException("Unsupported claim type : " + typeStr);
				ISimpleClaim simple = null;
				try {
					simple = (ISimpleClaim) value;
				} catch (ClassCastException e) {
					throw new InvalidClaimException(e);
				}
				List values = simple.getValues();
				if (values.size() > 0) {
					claimValues_.put(typeStr, values.get(0));
				}
			}
		}
	}

	public void setGivenName(String value) {
		if ((value == null || value.trim().length() == 0))
			claimValues_.put(GIVEN_NAME_TYPE, "");
		else
			claimValues_.put(GIVEN_NAME_TYPE, value.trim());
	}

	public void setSurName(String value) {
		if ((value == null || value.trim().length() == 0))
			claimValues_.put(SURNAME_TYPE, "");
		else
			claimValues_.put(SURNAME_TYPE, value.trim());
	}

	public void setEmailAddress(String value) {
		if ((value == null || value.trim().length() == 0))
			claimValues_.put(EMAIL_ADDRESS_TYPE, "");
		else
			claimValues_.put(EMAIL_ADDRESS_TYPE, value.trim());
	}

	public void setStreetAddress(String value) {
		if ((value == null || value.trim().length() == 0))
			claimValues_.put(STREET_ADDRESS_TYPE, "");
		else
			claimValues_.put(STREET_ADDRESS_TYPE, value.trim());
	}

	public void setLocality(String value) {
		if ((value == null || value.trim().length() == 0))
			claimValues_.put(LOCALITY_TYPE, "");
		else
			claimValues_.put(LOCALITY_TYPE, value.trim());
	}

	public void setStateOrProvince(String value) {
		if ((value == null || value.trim().length() == 0))
			claimValues_.put(STATE_OR_PROVINCE_TYPE, "");
		else
			claimValues_.put(STATE_OR_PROVINCE_TYPE, value.trim());
	}

	public void setPostalCode(String value) {
		if ((value == null || value.trim().length() == 0))
			claimValues_.put(POSTAL_CODE_TYPE, "");
		else
			claimValues_.put(POSTAL_CODE_TYPE, value.trim());
	}

	public void setCountry(String value) {
		if ((value == null || value.trim().length() == 0))
			claimValues_.put(COUNTRY_TYPE, "");
		else
			claimValues_.put(COUNTRY_TYPE, value.trim());
	}

	public void setHomePhone(String value) {
		if ((value == null || value.trim().length() == 0))
			claimValues_.put(HOME_PHONE_TYPE, "");
		else
			claimValues_.put(HOME_PHONE_TYPE, value.trim());
	}

	public void setOtherPhone(String value) {
		if ((value == null || value.trim().length() == 0))
			claimValues_.put(OTHER_PHONE_TYPE, "");
		else
			claimValues_.put(OTHER_PHONE_TYPE, value.trim());
	}

	public void setMobilePhone(String value) {
		if ((value == null || value.trim().length() == 0))
			claimValues_.put(MOBILE_PHONE_TYPE, "");
		else
			claimValues_.put(MOBILE_PHONE_TYPE, value.trim());
	}

	public void setDateOfBirth(String value) {
		if ((value == null || value.trim().length() == 0))
			claimValues_.put(DATE_OF_BIRTH_TYPE, "");
		else
			claimValues_.put(DATE_OF_BIRTH_TYPE, value.trim());
	}

	public void setGender(String value) {
		if ((value == null || value.trim().length() == 0))
			claimValues_.put(GENDER_TYPE, "");
		else
			claimValues_.put(GENDER_TYPE, value.trim());
	}

	public void setWebPage(String value) {
		if ((value == null || value.trim().length() == 0))
			claimValues_.put(WEB_PAGE_TYPE, "");
		else
			claimValues_.put(WEB_PAGE_TYPE, value.trim());
	}

	public void setPrivatePersonalIdentifier(String value) {
		if ((value == null || value.trim().length() == 0))
			claimValues_.put(PPID_TYPE, "");
		else
			claimValues_.put(PPID_TYPE, value.trim());
	}

	public String getGivenName() {
		if (claimValues_.containsKey(GIVEN_NAME_TYPE))
			return claimValues_.get(GIVEN_NAME_TYPE).toString();
		else
			return "";
	}

	public String getSurName() {
		if (claimValues_.containsKey(SURNAME_TYPE))
			return claimValues_.get(SURNAME_TYPE).toString();
		else
			return "";
	}

	public String getEmailAddress() {
		if (claimValues_.containsKey(EMAIL_ADDRESS_TYPE))
			return claimValues_.get(EMAIL_ADDRESS_TYPE).toString();
		else
			return "";
	}

	public String getStreetAddress() {
		if (claimValues_.containsKey(STREET_ADDRESS_TYPE))
			return claimValues_.get(STREET_ADDRESS_TYPE).toString();
		else
			return "";
	}

	public String getLocality() {
		if (claimValues_.containsKey(LOCALITY_TYPE))
			return claimValues_.get(LOCALITY_TYPE).toString();
		else
			return "";
	}

	public String getStateOrProvince() {
		if (claimValues_.containsKey(STATE_OR_PROVINCE_TYPE))
			return claimValues_.get(STATE_OR_PROVINCE_TYPE).toString();
		else
			return "";
	}

	public String getPostalCode() {
		if (claimValues_.containsKey(POSTAL_CODE_TYPE))
			return claimValues_.get(POSTAL_CODE_TYPE).toString();
		else
			return "";
	}

	public String getCountry() {
		if (claimValues_.containsKey(COUNTRY_TYPE))
			return claimValues_.get(COUNTRY_TYPE).toString();
		else
			return "";
	}

	public String getHomePhone() {
		if (claimValues_.containsKey(HOME_PHONE_TYPE))
			return claimValues_.get(HOME_PHONE_TYPE).toString();
		else
			return "";
	}

	public String getOtherPhone() {
		if (claimValues_.containsKey(OTHER_PHONE_TYPE))
			return claimValues_.get(OTHER_PHONE_TYPE).toString();
		else
			return "";
	}

	public String getMobilePhone() {
		if (claimValues_.containsKey(MOBILE_PHONE_TYPE))
			return claimValues_.get(MOBILE_PHONE_TYPE).toString();
		else
			return "";
	}

	public String getDateOfBirth() {
		if (claimValues_.containsKey(DATE_OF_BIRTH_TYPE))
			return claimValues_.get(DATE_OF_BIRTH_TYPE).toString();
		else
			return "";
	}

	public String getGender() {
		if (claimValues_.containsKey(GENDER_TYPE))
			return claimValues_.get(GENDER_TYPE).toString();
		else
			return "";
	}

	public String getWebPage() {
		if (claimValues_.containsKey(WEB_PAGE_TYPE))
			return claimValues_.get(WEB_PAGE_TYPE).toString();
		else
			return "";
	}

	public String getPrivatePersonalIdentifier() {
		if (claimValues_.containsKey(PPID_TYPE))
			return claimValues_.get(PPID_TYPE).toString();
		else
			return "";
	}

	/**
	 * Returns list of <code>IClaim</code> instances
	 * 
	 * @param skipEmptyValues
	 *            If <code>true</code> only populated values will be returned,
	 *            otherwise all predefined self-issued values will be returned
	 * @return
	 * @throws CardException
	 */
	public ArrayList getClaimValueList(boolean skipEmptyValues) throws CardException {
		ArrayList list = new ArrayList();
		Enumeration types = claimValues_.keys();
		while (types.hasMoreElements()) {
			String type = (String)types.nextElement();
			String value = (String)claimValues_.get(type);
			if (value != null && value.trim().length() > 0) {
				ClaimType cType = getClaimType(type);
				list.add(new ClaimValue(cType, value));
			}
		}
		return list;
	}

	public static ArrayList getSupportedClaimTypeList() {
		ArrayList al = new ArrayList();
		al.addAll(claimTypesList_);
		return al;
	}

	public static ClaimType getClaimType(String type) throws CardException {
		if (type == null)
			throw new CardException("Parameter \"type\" is null.");
		if (claimTypes_.containsKey(type))
			return (ClaimType)((ClaimType)claimTypes_.get(type)).clone();
		else
			throw new CardException("Unsupported self-issued claim type : " + type);
	}

	public static ClaimType getClaimType(URI type) throws CardException {
		if (type == null)
			throw new CardException("Parameter \"type\" is null.");
		return getClaimType(type.toString());
	}

	public static Hashtable getSupportedTypesHash() {
		Hashtable claimValues = new Hashtable();
		claimValues.put(GIVEN_NAME_TYPE, "1");
		claimValues.put(SURNAME_TYPE, "2");
		claimValues.put(EMAIL_ADDRESS_TYPE, "3");
		claimValues.put(STREET_ADDRESS_TYPE, "4");
		claimValues.put(LOCALITY_TYPE, "5");
		claimValues.put(STATE_OR_PROVINCE_TYPE, "6");
		claimValues.put(POSTAL_CODE_TYPE, "7");
		claimValues.put(COUNTRY_TYPE, "8");
		claimValues.put(HOME_PHONE_TYPE, "9");
		claimValues.put(OTHER_PHONE_TYPE, "10");
		claimValues.put(MOBILE_PHONE_TYPE, "11");
		claimValues.put(DATE_OF_BIRTH_TYPE, "12");
		claimValues.put(GENDER_TYPE, "13");
		claimValues.put(WEB_PAGE_TYPE, "14");
		claimValues.put(PPID_TYPE, "15");
		return claimValues;
	}

	static {
		try {
			BIRTH_SDF.setLenient(false);
			GENDER_LIST.add("");
			GENDER_LIST.add(GENDER_MALE);
			GENDER_LIST.add(GENDER_FEMALE);
			UIDescriptor dateOfBirthDescriptor = UIDescriptor.createDateTimeDescriptor("YYYY-MM-DD", null);
			UIDescriptor genderDescriptor = UIDescriptor.createComboBoxDescriptor(GENDER_LIST);
			ClaimType givenName = new ClaimType(GIVEN_NAME_TYPE, GIVEN_NAME_NAME, GIVEN_NAME_DESC);
			ClaimType surName = new ClaimType(SURNAME_TYPE, SURNAME_NAME, SURNAME_DESC);
			ClaimType email = new ClaimType(EMAIL_ADDRESS_TYPE, EMAIL_ADDRESS_NAME, EMAIL_ADDRESS_DESC);
			ClaimType street = new ClaimType(STREET_ADDRESS_TYPE, STREET_ADDRESS_NAME, STREET_ADDRESS_DESC);
			ClaimType locality = new ClaimType(LOCALITY_TYPE, LOCALITY_NAME, LOCALITY_DESC);
			ClaimType state = new ClaimType(STATE_OR_PROVINCE_TYPE, STATE_OR_PROVINCE_NAME, STATE_OR_PROVINCE_DESC);
			ClaimType postalCode = new ClaimType(POSTAL_CODE_TYPE, POSTAL_CODE_NAME, POSTAL_CODE_DESC);
			ClaimType country = new ClaimType(COUNTRY_TYPE, COUNTRY_NAME, COUNTRY_DESC);
			ClaimType homePhone = new ClaimType(HOME_PHONE_TYPE, HOME_PHONE_NAME, HOME_PHONE_DESC);
			ClaimType otherPhone = new ClaimType(OTHER_PHONE_TYPE, OTHER_PHONE_NAME, OTHER_PHONE_DESC);
			ClaimType mobilePhone = new ClaimType(MOBILE_PHONE_TYPE, MOBILE_PHONE_NAME, MOBILE_PHONE_DESC);
			ClaimType dateOfBirth = new ClaimType(DATE_OF_BIRTH_TYPE, DATE_OF_BIRTH_NAME, DATE_OF_BIRTH_DESC);
			dateOfBirth.setUIDescriptor(dateOfBirthDescriptor);
			ClaimType gender = new ClaimType(GENDER_TYPE, GENDER_NAME, GENDER_DESC);
			gender.setUIDescriptor(genderDescriptor);
			ClaimType webPage = new ClaimType(WEB_PAGE_TYPE, WEB_PAGE_NAME, WEB_PAGE_DESC);
			ClaimType ppid = new ClaimType(PPID_TYPE, PPID_NAME, PPID_DESC, false, false);
			claimTypesList_ = new ArrayList();
			claimTypesList_.add(givenName);
			claimTypesList_.add(surName);
			claimTypesList_.add(email);
			claimTypesList_.add(street);
			claimTypesList_.add(locality);
			claimTypesList_.add(state);
			claimTypesList_.add(postalCode);
			claimTypesList_.add(country);
			claimTypesList_.add(homePhone);
			claimTypesList_.add(otherPhone);
			claimTypesList_.add(mobilePhone);
			claimTypesList_.add(dateOfBirth);
			claimTypesList_.add(gender);
			claimTypesList_.add(webPage);
			claimTypesList_.add(ppid);
			claimTypes_ = new Hashtable();
			for(int i = 0, j = claimTypesList_.size(); i < j; i++) {
				ClaimType ct = (ClaimType)claimTypesList_.get(i);
				claimTypes_.put(ct.getType(), ct);
			}
		} catch (CardException e) {
			Log log = LogFactory.getLog(SelfIssuedCardClaims.class);
			if (log != null)
				log.error(e);
		}
	}

	/**
	 * @param genderCode 0 - undefined gender, 1 - male, 2 - female
	 * @return the value of claim (male, female or undefined)
	 * @throws CardException
	 */
	public static String getGenderClaimValue(String genderCode) throws CardException {
		if (genderCode == null || "".equals(genderCode) || "0".equals(genderCode))
			return "";
		if ("1".equals(genderCode))
			return GENDER_MALE;
		if ("2".equals(genderCode))
			return GENDER_FEMALE;
		throw new CardException("Unsupported claim value :" + genderCode);
	}

	/**
	 * @param genderClaimValue the value of gender claim (male, female or undefined)
	 * @return the code of gender (1 for male, 2 for female or 0 for undefined)
	 * @throws CardException
	 */
	public static String getGenderCode(String genderClaimValue) throws CardException {
		if (GENDER_MALE.equalsIgnoreCase(genderClaimValue))
			return "1";
		if (GENDER_FEMALE.equalsIgnoreCase(genderClaimValue))
			return "2";
		if (genderClaimValue == null || genderClaimValue.trim().length() == 0)
			return "";
		throw new CardException("Unsupported claim value :" + genderClaimValue);
	}

	/**
	 * Validates the value of claim
	 * @param claimValue
	 * @param claimType
	 * @throws CardException
	 */
	public static void validateClaimValue(String claimValue, String claimType) throws CardException {
		if (GENDER_TYPE.equals(claimType)) {
			if (claimType != null) {
				if (!GENDER_LIST.contains(claimValue))
					throw new CardException("Unsupported claim value for claim type " + GENDER_TYPE);
			}
		}
		else if (DATE_OF_BIRTH_TYPE.equals(claimType)) {
			if (claimValue != null && claimValue.length() > 0) {
				try {
					BIRTH_SDF.parse(claimValue);
				} catch (ParseException e) {
					throw new CardException("Incorrect value of claim " + DATE_OF_BIRTH_TYPE);
				}
			}
		}
	}

	/**
	 * @return
	 */
	public static ArrayList getGenderValuesList() {
		ArrayList al = new ArrayList();
		al.addAll(GENDER_LIST);
		return al;
	}

	public static String convertICMDate(String dateICM) throws CardException {
		if (dateICM != null && dateICM.trim().length() > 0) {
			try {
				BIRTH_SDF.parse(dateICM); // if date setted during pin changing
				return dateICM;
			} catch (ParseException e) {
			}
			SimpleDateFormat sdf = new SimpleDateFormat ("MMMM dd, yyyy");
			sdf.setLenient(false);
			try {
				Date dt = sdf.parse(dateICM);
				return BIRTH_SDF.format(dt);
			} catch (ParseException e) {
				e.printStackTrace();
				return dateICM;
			}
		}
		else
			return dateICM;
	}

	public static void main(String[] args) throws CardException {
		System.out.println(convertICMDate("August 13, 2008"));
	}

		
}
