/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.icard.provider.cardspace.common.utils;

import java.io.ByteArrayOutputStream;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;

import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.apache.axiom.om.OMElement;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.CardException;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class XMLUtils {
	protected static final Log log = LogFactory.getLog(XMLUtils.class);

	/**
	 * @param elm
	 * @return
	 * @throws TransformerException
	 */
	public static String elementToString(Element elm) throws TransformerException {
		if (elm == null)
			return null;
		ByteArrayOutputStream bos = new ByteArrayOutputStream();
		TransformerFactory transformerFactory = TransformerFactory.newInstance();
		Transformer transformer = transformerFactory.newTransformer();
		DOMSource source = new DOMSource(elm);
		StreamResult result = new StreamResult(bos);
		transformer.transform(source, result);
		byte[] buf = bos.toByteArray();
		if (buf == null)
			return null;
		else
			return new String(buf);
	}

	public static String toString(OMElement omElement) throws Exception {
		if (null == omElement) {
			return null;
		}
		ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
		omElement.serialize(outputStream);
		String strElement = null;
		try {
			strElement = outputStream.toString("UTF-8");
			strElement = StripNamespaceDeclaration(strElement);
		} catch (UnsupportedEncodingException uee) {
			log.error(uee);
			throw uee;
		}
		return strElement;
	}

	public static String StripNamespaceDeclaration(String strElement) {
		boolean bNamespaceDeclaration = strElement.startsWith("<?xml");
		if (!bNamespaceDeclaration) {
			return strElement;
		}
		String strResult = strElement.substring(1);
		int nBegin = strResult.indexOf("<");
		return strResult.substring(nBegin);
	}

	public static ArrayList getChildElements(Element owner, String nameSpace, String localName) throws CardException {
		ArrayList children = new ArrayList();
		if (owner == null)
			throw new CardException("Parameter \"owner\" is null.");
		NodeList nl = owner.getChildNodes();
		int len = nl.getLength();
		for (int i = 0; i < len; i++) {
			Node nd = nl.item(i);
			if (nd.getNodeType() == Node.ELEMENT_NODE) {
				if (nameSpace != null && nameSpace.equals(nd.getNamespaceURI()) == false)
					continue;
				if (localName != null && localName.equals(nd.getLocalName()) == false)
					continue;
				children.add(nd);
			}
		}
		return children;
	}

	public static Element getChildElement(Element owner, String nameSpace, String localName) throws CardException {
		ArrayList children = getChildElements(owner, nameSpace, localName);
		switch (children.size()) {
		case 0:
			return null;
		case 1:
			return (Element) children.get(0);
		default:
			throw new CardException("There are more then one child element with local name = " + localName + " and nameSpace = " + nameSpace);
		}
	}

	public static String getTextContent(final org.w3c.dom.Element element)
	{
		org.w3c.dom.NodeList nlChildren = element.getChildNodes();
		for (int i = 0; i < nlChildren.getLength(); ++i)
		{
			org.w3c.dom.Node nodeChild = nlChildren.item(i);
			if (org.w3c.dom.Node.TEXT_NODE == nodeChild.getNodeType())
			{
				org.w3c.dom.Text textChild = (org.w3c.dom.Text)nodeChild;
				return textChild.getData();
			}
		}
		return null;
	}

	public static void setTextContent(final org.w3c.dom.Element element, final String value)
	{
		org.w3c.dom.NodeList nlChildren = element.getChildNodes();
		for (int i = 0; i < nlChildren.getLength(); ++i)
		{
			org.w3c.dom.Node nodeChild = nlChildren.item(i);
			if (org.w3c.dom.Node.TEXT_NODE == nodeChild.getNodeType())
			{
				org.w3c.dom.Text textChild = (org.w3c.dom.Text)nodeChild;
				if (value == null)
					element.removeChild(textChild);
				else
					textChild.setData(value);
				return;
			}
		}
		if (value != null) {
			org.w3c.dom.Text textNode = element.getOwnerDocument().createTextNode(value);
			element.appendChild(textNode);
		}
	}

	/*
	 * public static XPath createXPath() { NamespaceContext ctx = new
	 * NamespaceContext(); ctx.addNamespace("ic", CardContext.IC_NS);
	 * ctx.addNamespace("trust", CardContext.TRUST_NS); ctx.addNamespace("wsa",
	 * CardContext.WSA_NS); ctx.addNamespace("mex", CardContext.MEX_NS);
	 * ctx.addNamespace("wsai", CardContext.WSAI_NS); ctx.addNamespace("dsig",
	 * CardContext.DSIG_NS); ctx.addNamespace("xe",
	 * "http://www.w3.org/2001/04/xmlenc#"); XPathFactory factory =
	 * XPathFactory.newInstance(); XPath xpath = factory.newXPath();
	 * xpath.setNamespaceContext(ctx); return xpath; }
	 */

}
