package org.eclipse.higgins.icard.provider.cardspace.common.utils;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.security.Key;
import java.security.NoSuchAlgorithmException;

import javax.crypto.spec.SecretKeySpec;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.CardException;

/**
 * This class helps to generate symmethric key used to encrypt/decrypt masterkey and claim values
 * (the salt and initVector bytes of key are stored within encrypted masterkey bytes)
 *
 */
public class EncryptedMasterKey {

	private static Log log = LogFactory.getLog(EncryptedMasterKey.class);

	private byte[] encryptedKey_ = null;
	private byte[] version_ = null;
	private byte[] salt_ = null;
	private byte[] iterationCount_ = null;
	private byte[] initVector_ = null;
	private byte[] encryptedData_ = null;
	private byte[] derivedKey_ = null;
	private byte[] pinCode_ = null;
	private Key key_ = null;
	private byte[] decryptedMasterKey_ = null;

//	private static final byte[] PIN_ENCRYPTION_VERSION = { (byte) 0x01 };

//	private static final byte[] KEY_DERIVATION_ITERATION_COUNT = { (byte) 0x01, (byte) 0x00, (byte) 0x00, (byte) 0x00 };

	private static final byte[] PIN_ENCRYPTION_VERSION = { (byte) 0x02 };

	private static final byte[] KEY_DERIVATION_ITERATION_COUNT = { (byte) 0xE8, (byte) 0x03, (byte) 0x00, (byte) 0x00 };

	public EncryptedMasterKey(byte[] encryptedMasterKey, byte[] pinCode) throws CardException {
		if (pinCode == null)
			throw new CardException("Parameter \"pinCode\" is null.");
		if (encryptedMasterKey == null)
			throw new CardException("Parameter \"encryptedMasterKey\" is null.");
		if (encryptedMasterKey.length < 38)
			throw new CardException("EncryptedMasterKey array is too short. Most likely the master key is not encrypted.");
		pinCode_ = pinCode;
		encryptedKey_ = encryptedMasterKey;
		version_ = CardCryptography.getSubArray(encryptedKey_, 0, 1);
		log.debug("decryptPersonalCardField() > Version " + String.valueOf(version_));
		salt_ = CardCryptography.getSubArray(encryptedKey_, 1, 16);
		iterationCount_ = CardCryptography.getSubArray(encryptedKey_, 17, 4);
		log.debug("decryptPersonalCardField() > Version " + String.valueOf(iterationCount_));
		initVector_ = CardCryptography.getSubArray(encryptedKey_, 21, 16);
		encryptedData_ = CardCryptography.getSubArray(encryptedKey_, 37, encryptedKey_.length - 37);
		try {
			derivedKey_ = CardCryptography.getDerivedKey(pinCode, salt_);
		} catch (NoSuchAlgorithmException e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
		key_ = new SecretKeySpec(derivedKey_, "AES");
	}

	
	public EncryptedMasterKey(byte[] decryptedMasterKey, byte[] salt, byte[] initVector, byte[] pinCode) throws CardException {
		if (decryptedMasterKey == null)
			throw new CardException("Parameter \"decryptedMasterKey\" is null");
		decryptedMasterKey_ = decryptedMasterKey;
		if (salt == null)
			throw new CardException("Parameter \"salt\" is null");
		salt_ = salt;
		if (initVector == null)
			throw new CardException("Parameter \"initVector\" is null");
		initVector_ = initVector;
		if (pinCode == null)
			throw new CardException("Parameter \"pinCode\" is null");
		pinCode_ = pinCode;
		byte[] derivedKey;
		try {
			derivedKey = CardCryptography.getDerivedKey(pinCode_, salt_);
		} catch (NoSuchAlgorithmException e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
		key_ = new SecretKeySpec(derivedKey, "AES");
		try {
			encryptedData_ = CardCryptography.encryptData(decryptedMasterKey_, initVector_, key_);
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
		ByteArrayOutputStream os = new ByteArrayOutputStream();
		try {
			os.write(PIN_ENCRYPTION_VERSION);
			os.write(salt_);
			os.write(KEY_DERIVATION_ITERATION_COUNT);
			os.write(initVector_);
			os.write(encryptedData_);
		} catch (IOException e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
		encryptedKey_ = os.toByteArray();
	}

	public Key getKey() {
		return key_;
	}

	public byte[] getInitVector() {
		return initVector_;
	}

	public byte[] getSalt() {
		return salt_;
	}

	public byte[] getPinCode() {
		return pinCode_;
	}

	public byte[] getDecryptedMasterKey() throws CardException {
		if (decryptedMasterKey_ == null) {
			try {
				decryptedMasterKey_ = CardCryptography.decryptPersonalCardField(encryptedData_, this);
			} catch (Exception e) {
				log.error(e);
				throw new CardException(e.getMessage());
			}
		}
		return decryptedMasterKey_;
	}

	public byte[] getEncryptedMasterKey() throws CardException {
		return encryptedKey_;
	}

}