/*******************************************************************************
 * Copyright (c) 20062007 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Bjoern Assmann (IBM Research) - initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.icard.provider.cardspace.common.utils;

import java.io.File;
import java.io.FileInputStream;
import java.security.KeyStore;
import java.security.cert.CertPath;
import java.security.cert.CertPathValidator;
import java.security.cert.CertPathValidatorResult;
import java.security.cert.CertificateFactory;
import java.security.cert.PKIXCertPathValidatorResult;
import java.security.cert.PKIXParameters;
import java.security.cert.TrustAnchor;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLSocket;
import javax.net.ssl.SSLSocketFactory;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class RelyingPartyIdentifierHelper {
	protected static final Log log = LogFactory.getLog(RelyingPartyIdentifierHelper.class);

	public static String ev_policies_oids[] = { "1.3.6.1.4.1.6449.1.2.1.5.1",
			"1.3.6.1.4.1.6334.1.100.1", "2.16.840.1.114412.2.1", // DigiCert
			// EV CPS v.
			// 1.0.1, p.
			// 47
			"2.16.840.1.114028.10.1.2", // Entrust EV CPS, p. 37
			"1.3.6.1.4.1.14370.1.6", // GeoTrust EV CPS v. 2.6, p. 28
			"2.16.840.1.114413.1.7.23.3", // Go Daddy EV CPS v. 2.0, p.42
			"1.3.6.1.4.1.8024.0.2.100.1.2", // QuoVadis Repository , p.33
			"2.16.840.1.114414.1.7.23.3", // Starfield EV CPS v. 2.0, p.42
			"2.16.840.1.113733.1.7.48.1", // Thawte EV CPS v. 3.3, p.95
			"2.16.840.1.113733.1.7.23.6" };

	public static class X500_Oid_HashTable {
		static Map map = new HashMap();

		public X500_Oid_HashTable() {
			// taken from
			// http://javadoc.iaik.tugraz.at/iaik_jce/3.13/iaik/utils/RFC2253NameParser.html
			map.put("2.5.4.3", "CN");
			map.put("2.5.4.4", "SN");
			map.put("2.5.4.6", "C");
			map.put("2.5.4.7", "L");
			map.put("2.5.4.8", "S");
			map.put("2.5.4.9", "street");
			map.put("2.5.4.10", "O");
			map.put("2.5.4.11", "OU");
			map.put("2.5.4.12", "T");
			map.put("0.9.2342.19200300.100.1.25", "DC");
			map.put("0.9.2342.19200300.100.1.1", "UID");
			map.put("1.2.840.113549.1.9.1", "E");
			// removed following entry after experience with xmldap.org
			// map.put("2.5.4.5", "serialNumber");

			// map.put("2.5.4.16", "postalAddress");
			// map.put("2.5.4.17", "postalCode");
			// map.put("2.5.4.20", "telephoneNumber");
			// map.put("2.5.4.21", "telexNumber");
			// map.put("2.5.4.13", "description");
			// map.put("2.5.4.42", "givenName");
			// map.put("2.5.4.43", "initials");
			// map.put("2.5.4.44", "generationQualifier");
			// map.put("2.5.4.45", "uniqueIdentifier");
			// map.put("2.5.4.46", "dnQualifier");
			// map.put("2.5.4.65", "pseudonym");

		}

		public static String transform(String attrID) {
			if (attrID.startsWith("OID.")) {
				String oid = attrID.substring(4);
				String keyName = (String) map.get(oid);
				if (keyName == null) {
					// System.err.println("unknown oid: " + oid);
					return attrID;
				} else {
					return keyName;
				}
			} else if (attrID.equals("ST")) {
				return "S";
			} else {
				return attrID;
			}
		}

	}

	public static String getOrgIdString(final X509Certificate rp) {

		String dnName = rp.getSubjectDN().getName();
		log.trace("Subject DN of certificate: " + dnName);

		String O_string = "O=\"\"";
		String L_string = "L=\"\"";
		String S_string = "S=\"\"";
		String C_string = "C=\"\"";

		List strArr = splitAttributesOfSubjectName(dnName);
		for (int i = 0; i < strArr.size(); i++) {
			String temp = (String) strArr.get(i);
			// System.err.println("temp: " + temp);

			if (temp.startsWith("O=")) {
				if (temp.charAt(2) == '"') {
					O_string = temp;
				} else {
					O_string = "O=\"" + temp.substring(2) + "\"";
				}
			} else if (temp.startsWith("L=")) {
				if (temp.charAt(2) == '"') {
					L_string = temp;
				} else {
					L_string = "L=\"" + temp.substring(2) + "\"";
				}
			} else if (temp.startsWith("S=")) {
				if (temp.charAt(2) == '"') {
					S_string = temp;
				} else {
					S_string = "S=\"" + temp.substring(2) + "\"";
				}
			} else if (temp.startsWith("ST=")) {
				if (temp.charAt(3) == '"') {
					S_string = temp;
				} else {
					S_string = "S=\"" + temp.substring(3) + "\"";
				}
			} else if (temp.startsWith("C=")) {
				if (temp.charAt(2) == '"') {
					C_string = temp;
				} else {
					C_string = "C=\"" + temp.substring(2) + "\"";
				}
			}
		}
		String res = "|" + O_string + "|" + L_string + "|" + S_string + "|"
				+ C_string + "|";
		log.trace("org id string is: " + res);
		return res;
	}

	public static List findSplitCommaPositions(String n) {

		char[] chars = n.toCharArray();
		List res = new ArrayList();
		boolean withinQuotes = false;
		for (int i = 0; i < chars.length; i++) {
			char c = chars[i];
			// check whether we enter or leave quotes
			if (c == '"') {
				withinQuotes = !withinQuotes;
			} else {
				// add comma pos if comma found outside of quotes
				if (c == ',') {
					if (!withinQuotes) {
						res.add(new Integer(i));
					}
				}
			}
		}
		return res;
	}

	public static List splitAttributesOfSubjectName(String name) {

		List res = new ArrayList();
		List commaPos = findSplitCommaPositions(name);
		log.trace("Comma pos: " + commaPos);

		// catch special case
		if (commaPos.size() == 0) {
			res.add(name);
			return res;
		}

		int beginIndex = 0;
		int endIndex = -1;
		for (Iterator iter = commaPos.iterator(); iter.hasNext();) {
			Integer element = (Integer) iter.next();

			// get substring
			endIndex = element.intValue();
			String attr = name.substring(beginIndex, endIndex);
			res.add(attr);
			log.trace("Found attribute: " + attr);

			// update beginIndex
			beginIndex = endIndex + 2;
		}
		String attr = name.substring(beginIndex);
		res.add(attr);
		log.trace("Found attribute: " + attr);

		return res;
	}

	// The AttributeType OID is converted to its X.500 key name. If an OID does
	// not
	// have a corresponding X.500 name, it is encoded as the prefix string
	// "OID."
	// followed by the dotted-decimal encoding of the OID.
	public static String getCertPathString(X509Certificate cert) {
		String name_RFC1779 = cert.getSubjectX500Principal().getName(
				javax.security.auth.x500.X500Principal.RFC1779);
		String name_toString = cert.getSubjectX500Principal().toString();
		log.trace("name_RFC1779:  " + name_RFC1779);
		log.trace("name_toString: " + name_toString);
		log.trace("subject dn:    " + cert.getSubjectDN());
		int pathLength = cert.getBasicConstraints();
		log.trace("Path length: " + pathLength);

		// get attributes
		List attributes = splitAttributesOfSubjectName(name_RFC1779);

		// transform attribute types and attribute values
		List attributesTransformed = new ArrayList();
		for (Iterator iter = attributes.iterator(); iter.hasNext();) {
			String element = (String) iter.next();
			int pos_equalSign = element.indexOf("=");
			String attrID = element.substring(0, pos_equalSign);
			String attrValue = element.substring(pos_equalSign + 1);

			// transform attribute id
			X500_Oid_HashTable x500Table = new X500_Oid_HashTable();
			String attrIDTrans = x500Table.transform(attrID);

			// transform attribute value, can probably be omitted
			String attrValueTrans = attrValue;

			// put back together
			String transformed = attrIDTrans + "=" + attrValueTrans;

			log.trace("transformed is: " + transformed);

			attributesTransformed.add(transformed);
		}

		String res = (String) attributesTransformed.get(0);
		for (int i = 1; i < attributesTransformed.size(); i++) {
			res = res + ", " + (String) attributesTransformed.get(i);
		}
		return "|ChainElement=\"" + res + "\"";
	}

	// TODO: this is not very elegant
	public static boolean isExtendendValidationCertificate(
			final X509Certificate rp) {

		String s_rp = rp.toString();
		for (int i = 0; i < ev_policies_oids.length; i++) {
			if (s_rp.indexOf(ev_policies_oids[i]) != -1) {
				log.trace("certificate is ev and contains oid: "
						+ ev_policies_oids[i]);
				return true;
			}
		}
		log.trace("Certificate is not ev");
		return false;
	}

	public static boolean hasNoOrganizationalIdentifierAttributes(
			final X509Certificate rp) {
		String orgIdString = getOrgIdString(rp);
		if (orgIdString.equals("|O=\"\"|L=\"\"|S=\"\"|C=\"\"|")) {
			return true;
		} else {
			return false;
		}

	}

	public static boolean isRootCertificate(X509Certificate cert) {

		// check whether certificate authority
		// int pathLength = cert.getBasicConstraints();
		// if (pathLength == -1){
		// return false;
		// }
		String issuer_dn = cert.getIssuerDN().toString();
		String subject_dn = cert.getSubjectDN().toString();
		return issuer_dn.equals(subject_dn);
	}

	public static String getQualifiedOrgIdString(List certPath)
			throws Exception {

		// get OrgIdString
		if (certPath.size() == 0) {
			throw new Exception("certificate path is empty");
		}
		X509Certificate rp = (X509Certificate) certPath.get(0);
		String orgIdString = getOrgIdString(rp);

		// get CertPathString of each element of the certificate path and
		// concatenate it to the result
		String res = "";
		for (int i = 1; i < certPath.size(); i++) {
			X509Certificate cert = (X509Certificate) certPath.get(i);
			String certPathString = getCertPathString(cert);
			// log.trace("processing cert: \n" +cert );
			log.trace("next chain element for qualified org id string: "
							+ certPathString);
			res = certPathString + res;
		}
		res = res + orgIdString;
		log.trace("Qualified org id string is: " + res);
		return res;
	}

	// extends list of elms of type java.security.cert.X509Certificate using the
	// keystore
	// so that the last elm is a root cert.
	public static List extendPath(List path) {
		try {
			log.trace("Extending path ...");
			// Load the JDK's cacerts keystore file
			String filename = System.getProperty("java.home")
					+ "/lib/security/cacerts".replace('/', File.separatorChar);
			FileInputStream is = new FileInputStream(filename);
			KeyStore keystore = KeyStore.getInstance(KeyStore.getDefaultType());
			String password = "changeit";
			keystore.load(is, password.toCharArray());

			// Create the parameters for the validator
			PKIXParameters params = new PKIXParameters(keystore);

			// Disable CRL checking since we are not supplying any CRLs
			params.setRevocationEnabled(false);

			// create list which contains only last element
			List ll = new ArrayList();
			ll.add(path.get(path.size() - 1));
			CertificateFactory certFact = CertificateFactory
					.getInstance("X.509");
			CertPath certPath = certFact.generateCertPath(ll);

			// Create the validator and validate the path
			CertPathValidator certPathValidator = CertPathValidator
					.getInstance(CertPathValidator.getDefaultType());
			CertPathValidatorResult result = certPathValidator.validate(
					certPath, params);

			// Get the trust anchor used to validate this path
			PKIXCertPathValidatorResult pkixResult = (PKIXCertPathValidatorResult) result;
			TrustAnchor ta = pkixResult.getTrustAnchor();
			X509Certificate cert = ta.getTrustedCert();

			// extend path
			log.trace("Extending path by: " + cert + "\n");
			path.add(cert);

			// extend further if necessary
			if (!isRootCertificate(cert)) {
				return extendPath(path);
			} else {
				return path;
			}

		} catch (Exception e) {
			log.error(e,e);
			return path;
		}

	}

	public static String hostnameByCertficate(X509Certificate cert)
			throws Exception {
		String dnName = cert.getSubjectDN().getName();
		log.trace("Subject DN of certificate: " + dnName);

		String hostname = null;
		List strArr = splitAttributesOfSubjectName(dnName);
		for (int i = 0; i < strArr.size(); i++) {
			String temp = (String) strArr.get(i);
			// System.err.println("temp: " + temp);

			if (temp.startsWith("CN=")) {
				hostname = temp.substring(3);
				break;
			}
		}
		if (hostname == null) {
			throw new Exception("host not found in x509 certificate");
		}
		return hostname;

	}

	public static byte[] getRelyingPartyIdentifier(String hostname)
			throws Exception {

		// get certficate path
		int port = 443;
		SSLSocketFactory factory = HttpsURLConnection
				.getDefaultSSLSocketFactory();
		SSLSocket socket = (SSLSocket) factory.createSocket(hostname, port);
		socket.startHandshake();
		java.security.cert.Certificate[] certPath = socket.getSession()
				.getPeerCertificates();
		if (certPath.length == 0) {
			throw new Exception("no certificates in path of host: " + hostname);
		}

		// get x509 certificate of host
		X509Certificate rp = (X509Certificate) certPath[0];

		byte[] idBytes;
		if (isExtendendValidationCertificate(rp)) {
			String orgIdString = getOrgIdString(rp);
			log.trace("orgIdString: " + orgIdString);
			idBytes = orgIdString.getBytes("UTF-16LE");
		} else if (hasNoOrganizationalIdentifierAttributes(rp)) {
			idBytes = rp.getPublicKey().getEncoded();
		} else {
			// convert path to list
			List path_asList = new ArrayList();
			log.trace("Used certificate path: \n");
			for (int i = 0; i < certPath.length; i++) {
				path_asList.add(certPath[i]);
				log.trace("Certificate " + i + " :" + certPath[i]);
			}
			log.trace("end certificate path");

			// extend path if necessary
			X509Certificate last = (X509Certificate) certPath[certPath.length - 1];
			if (!isRootCertificate(last)) {
				path_asList = extendPath(path_asList);
			}

			String qualifiedOrgIdString = getQualifiedOrgIdString(path_asList);
			log.trace("qualifiedOrgIdString: " + qualifiedOrgIdString);
			idBytes = qualifiedOrgIdString.getBytes("UTF-16LE");
		}

		socket.close();

		java.security.MessageDigest md = java.security.MessageDigest
				.getInstance("SHA-256");
		md.update(idBytes);
		byte[] res = md.digest();
		// for (int i = 0; i < res.length; i++) {
		// log.trace("byte " + i + " : " + res[i] );
		//			
		// }
		// log.trace("Relying party identifier: " + res);
		// log.trace("Relying party identifier: " + new String( res ));

		return res;
	}

	public static byte[] getRelyingPartyIdentifier(X509Certificate rp)
			throws Exception {
		String hostname = hostnameByCertficate(rp);
		byte[] rpid = getRelyingPartyIdentifier(hostname);
		return rpid;
	}

}
