/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.icard.provider.cardspace.common;

import java.util.Date;
import java.util.Iterator;
import java.util.List;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.AuthenticationRequiredException;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.ICardConstants;
import org.eclipse.higgins.icard.IClaim;
import org.eclipse.higgins.icard.IClaimType;
import org.eclipse.higgins.icard.IEndpointReference;
import org.eclipse.higgins.icard.IInformationCardExtension;
import org.eclipse.higgins.icard.IManagedInformationCard;
import org.eclipse.higgins.icard.ITokenService;
import org.eclipse.higgins.icard.InvalidTypeException;
import org.eclipse.higgins.icard.auth.ICredentialDescriptor;
import org.eclipse.higgins.icard.common.io.IOElement;
import org.eclipse.higgins.icard.io.CardIOException;
import org.eclipse.higgins.icard.io.IElement;
import org.eclipse.higgins.icard.io.IElementFormat;
import org.eclipse.higgins.icard.io.UnsupportedElementFormatException;
import org.eclipse.higgins.icard.common.utils.CardContext;
import org.eclipse.higgins.icard.policy.ISTSPrivacyPolicy;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.CardCryptography;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.CardUtils;
import org.eclipse.higgins.icard.common.utils.DateConvertor;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.XMLUtils;
import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

public abstract class ManagedCard extends InformationCard implements IManagedInformationCard {
	protected static Log log = LogFactory.getLog(ManagedCard.class);

	protected List tokenServices_ = null;

	protected Boolean requireAppliesTo_ = null;

	protected ISTSPrivacyPolicy privacyNotice_ = null;

	protected ICredentialDescriptor[] requiredCredentials_ = null;

	protected String ic07IssuerInformation = null;

	protected boolean requireStrongRecipientIdentity = false;

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.IManagedInformationCard#getTokenServices()
	 */
	public List getTokenServices() {
		return tokenServices_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.IManagedInformationCard#getRequireAppliesTo()
	 */
	public Boolean getRequireAppliesTo() {
		return requireAppliesTo_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.IManagedInformationCard#getPrivacyNotice()
	 */
	public ISTSPrivacyPolicy getPrivacyNotice() {
		return privacyNotice_;
	}

	/**
	 * @param crd
	 * @throws CardException
	 * @throws DOMException
	 */
	public void initFromXML(Element crd) throws CardException, DOMException {
		if (crd == null)
			throw new CardException("Parameter card is null.");
		isSelfIssued_ = false;
		Element cardMetaData = null;
		boolean isRoamingCard = false;
		if (CardContext.IC_INFORMATION_CARD.equals(crd.getLocalName()))
			cardMetaData = crd;
		else if (CardContext.IC_ROAMING_INFORMATION_CARD.equals(crd.getLocalName())) {
			cardMetaData = XMLUtils.getChildElement(crd, CardContext.IC_NS, CardContext.IC_INFORMATION_CARD_META_DATA);
			isRoamingCard = true;
		} else
			throw new CardException("Couldn't import the card. Unexpected tag name: " + crd.getTagName());
		if (cardMetaData == null)
			throw new CardException("Couldn't import the card. Can not find ic:InformationCardMetaData element");
		if (cardMetaData.hasAttribute("xml:lang"))
			language_ = cardMetaData.getAttribute("xml:lang");
		else
			language_ = null;
		checkIsSelfIssued(cardMetaData);
		/*
		 * parseCardId(cardMetaData); parseCardVersion(cardMetaData);
		 * parseCardName(cardMetaData); parseCardImage(cardMetaData);
		 * parseIssuer(cardMetaData); parseIssuerName(cardMetaData);
		 * parseTimeIssued(cardMetaData); parseTimeExpires(cardMetaData);
		 * parseSupportedTokenTypes(cardMetaData);
		 * parseClaimTypes(cardMetaData); parseTokenServiceList(cardMetaData);
		 * parseRequireAppliesTo(cardMetaData);
		 * parsePrivacyNotice(cardMetaData); if (isRoamingCard == false) {
		 * hashSalt_ = SecureRandom.getSeed(32); masterKey_ =
		 * SecureRandom.getSeed(32); timeLastUpdated_ = new Date(); } else {
		 * parseHashSalt(cardMetaData); parseTimeLastUpdated(cardMetaData);
		 * parseMasterKey(crd); }
		 */
		id_ = CardUtils.parseCardId(cardMetaData);
		version_ = CardUtils.parseCardVersion(cardMetaData);
		name_ = CardUtils.parseCardName(cardMetaData);
		image_ = CardUtils.parseCardImage(cardMetaData);
		imageMimeType_ = CardUtils.parseCardImageType(cardMetaData);
		issuer_ = CardUtils.parseIssuer(cardMetaData);
		issuerName_ = CardUtils.parseIssuerName(cardMetaData);
		timeIssued_ = CardUtils.parseTimeIssued(cardMetaData);
		timeExpires_ = CardUtils.parseTimeExpires(cardMetaData);
		supportedTokenTypes_ = CardUtils.parseSupportedTokenTypes(cardMetaData);
		claimTypes_ = CardUtils.parseClaimTypes(cardMetaData);
		tokenServices_ = CardUtils.parseTokenServiceList(cardMetaData);
		requireAppliesTo_ = CardUtils.parseRequireAppliesTo(cardMetaData);
		privacyNotice_ = CardUtils.parsePrivacyNotice(cardMetaData);
		extensions_ = CardUtils.parseExtensions(cardMetaData);
		initStandardExtensionFields();
		if (isRoamingCard == false) {
			hashSalt_ = CardCryptography.getRandomBytes(32);
			masterKey_ = CardCryptography.getRandomBytes(32);
			timeLastUpdated_ = new Date();
		} else {
			hashSalt_ = CardUtils.parseHashSalt(cardMetaData);
			timeLastUpdated_ = CardUtils.parseTimeLastUpdated(cardMetaData);
			masterKey_ = CardUtils.parseMasterKey(crd);
		}
		issuerID_ = CardUtils.parseIssuerID(cardMetaData);
		if (issuerID_ == null) {
			//TODO issuerID should be calculated using the same algorithm as for PPID 
			issuerID_ = CardCryptography.getRandomBytes(32);
		}
	}

	/**
	 * Initialize card fields from extension elements (like
	 * ic07IssuerInformation and hasStrongRecipientIdentity). Extensions should
	 * be initialized before this method invocation.
	 * 
	 * @throws CardException
	 */
	protected void initStandardExtensionFields() throws CardException {
		requireStrongRecipientIdentity = false;
		ic07IssuerInformation = null;
		IInformationCardExtension[] exts = getExtensions();
		if (hasExtensions()) {
			for (int i = 0, s = exts.length; i < s; i++) {
				IInformationCardExtension ext = exts[i];
				IElement iElm = ext.getElement();
				try {
					Element elm = (Element)iElm.getAs(Element.class);
					if (elm != null) {
						if (CardContext.IC07_NS.equals(elm.getNamespaceURI())) {
							if (CardContext.IC07_REQUIRE_STRONG_RECIPIENT_IDENTITY.equals(elm.getLocalName())) {
								requireStrongRecipientIdentity = true;
							}
							else if (CardContext.IC07_ISSUER_INFORMATION.equals(elm.getLocalName())) {
								ic07IssuerInformation = XMLUtils.getTextContent(elm);
							}
						}
					}
				}
				catch(Exception e) {
					throw new CardException(e);
				}
			}
		}
	}

	/**
	 * @param card
	 * @throws CardException
	 */
	private void checkIsSelfIssued(Element card) throws CardException {
		// Element isSelfIssuedElm = XMLUtils.getChildElement(card,
		// CardContext.IC_NS, "IsSelfIssued");
		// String boolValue = (isSelfIssuedElm != null) ?
		// XMLUtils.getTextContent(isSelfIssuedElm) : "";
		// if (isStringEmpty(boolValue) == false) {
		// if ("true".equals(boolValue.trim().toLowerCase()))
		// throw new CardException("Can not import self issued card.");
		// }
		if (CardUtils.isSelfIssued(card)) {
			throw new CardException("Can not import self issued card.");
		}
	}

	/**
	 * @param card
	 * @throws CardException
	 */
	// private void parseCardId(Element card) throws CardException {
	// String id = null;
	// try {
	// Element informationCardReference = XMLUtils.getChildElement(card,
	// CardContext.IC_NS, "InformationCardReference");
	// Element cardId = XMLUtils.getChildElement(informationCardReference,
	// CardContext.IC_NS, "CardId");
	// id = XMLUtils.getTextContent(cardId);
	// } catch (Exception e) {
	// log.error(e);
	// }
	// if (isStringEmpty(id))
	// throw new CardException("Couldn't get CardId element from InfoCard.");
	// id_ = URI.create(id);
	// }
	/**
	 * @param card
	 * @throws CardException
	 */
	// private void parseCardVersion(Element card) throws CardException {
	// try {
	// Element informationCardReference = XMLUtils.getChildElement(card,
	// CardContext.IC_NS, "InformationCardReference");
	// Element cardVersion = XMLUtils.getChildElement(informationCardReference,
	// CardContext.IC_NS, "CardVersion");
	// version_ = XMLUtils.getTextContent(cardVersion);
	// } catch (Exception e) {
	// log.error(e);
	// }
	// if (isStringEmpty(version_))
	// throw new CardException("Couldn't get CardVersion element from InfoCard
	// with id = " + id_);
	// }
	/**
	 * @param card
	 * @throws CardException
	 */
	// private void parseCardName(Element card) throws CardException {
	// Element cardName = XMLUtils.getChildElement(card, CardContext.IC_NS,
	// "CardName");
	// name_ = (cardName != null) ? XMLUtils.getTextContent(cardName) : "";
	// }
	/**
	 * @param card
	 * @throws CardException
	 * @throws DOMException
	 */
	// private void parseCardImage(Element card) throws CardException,
	// DOMException {
	// Element cardImg = XMLUtils.getChildElement(card, CardContext.IC_NS,
	// "CardImage");
	// if (cardImg != null) {
	// image_ = CardCryptography.decodeBase64(XMLUtils.getTextContent(cardImg));
	// imageMimeType_ = cardImg.getAttribute("MimeType");
	// }
	// }
	/**
	 * @param card
	 * @throws CardException
	 */
	// private void parseIssuer(Element card) throws CardException {
	// Element issuer = XMLUtils.getChildElement(card, CardContext.IC_NS,
	// "Issuer");
	// issuer_ = (issuer != null) ? XMLUtils.getTextContent(issuer) : "";
	// if (isStringEmpty(issuer_))
	// throw new CardException("Couldn't get Issuer element from InfoCard with
	// id = " + id_);
	// }
	/**
	 * @param card
	 * @throws CardException
	 */
	// private void parseIssuerName(Element card) throws CardException {
	// Element issuerName = XMLUtils.getChildElement(card, CardContext.IC_NS,
	// "IssuerName");
	// issuerName_ = (issuerName != null) ? XMLUtils.getTextContent(issuerName)
	// : "";
	// if (isStringEmpty(issuerName_)) {
	// try {
	// URI issuer = URI.create(issuer_);
	// if (issuer != null) {
	// issuerName_ = issuer.getHost().toString();
	// } else
	// issuerName_ = issuer_;
	// } catch (Exception e) {
	// issuerName_ = issuer_;
	// log.error(e);
	// }
	// }
	//
	// }
	/**
	 * @param card
	 * @throws CardException
	 */
	// private void parseTimeIssued(Element card) throws CardException {
	// Element timeIssuedElm = XMLUtils.getChildElement(card, CardContext.IC_NS,
	// "TimeIssued");
	// String timeIssued = (timeIssuedElm != null) ?
	// XMLUtils.getTextContent(timeIssuedElm) : "";
	// if (isStringEmpty(timeIssued))
	// throw new CardException("Couldn't get TimeIssued element from InfoCard
	// with id = " + id_);
	// timeIssued_ = DateConvertor.parse(timeIssued);
	// }
	/**
	 * @param card
	 * @throws CardException
	 */
	// private void parseTimeExpires(Element card) throws CardException {
	// Element timeExpiresElm = XMLUtils.getChildElement(card,
	// CardContext.IC_NS, "TimeExpires");
	// String timeExpires = (timeExpiresElm != null) ?
	// XMLUtils.getTextContent(timeExpiresElm) : "";
	// if (isStringEmpty(timeExpires) == false)
	// timeExpires_ = DateConvertor.parse(timeExpires);
	// }
	/**
	 * @param card
	 * @throws CardException
	 */
	// private void parseSupportedTokenTypes(Element card) throws CardException
	// {
	// supportedTokenTypes_ = new ArrayList();
	// Element supportedTokenTypeList = XMLUtils.getChildElement(card,
	// CardContext.IC_NS, "SupportedTokenTypeList");
	// if (supportedTokenTypeList == null)
	// throw new CardException("Couldn't get \"SupportedTokenTypeList\" element
	// from InfoCard with id = " + id_);
	// NodeList tokenTypes = supportedTokenTypeList.getChildNodes();
	// int len = tokenTypes.getLength();
	// for (int i = 0; i < len; i++) {
	// Node nd = tokenTypes.item(i);
	// if (nd.getNodeType() == Node.ELEMENT_NODE &&
	// "TokenType".equals(nd.getLocalName()) &&
	// CardContext.TRUST_NS.equals(nd.getNamespaceURI())) {
	// Element elm = (Element)nd;
	// supportedTokenTypes_.add(URI.create(XMLUtils.getTextContent(elm)));
	// }
	// }
	// if (supportedTokenTypes_.size() == 0)
	// throw new CardException("Couldn't get the list of TokenType elements from
	// InfoCard with id = " + id_);
	// }
	/**
	 * @param card
	 * @throws CardException
	 */
	// private void parseClaimTypes(Element card) throws CardException {
	// claimTypes_ = new ArrayList();
	// Element supportedClaimTypeList = XMLUtils.getChildElement(card,
	// CardContext.IC_NS, "SupportedClaimTypeList");
	// if (supportedClaimTypeList == null)
	// throw new CardException("Couldn't get \"SupportedClaimTypeList\" element
	// from InfoCard with id = " + id_);
	// NodeList claimTypes = supportedClaimTypeList.getChildNodes();
	// int len = claimTypes.getLength();
	// for (int i = 0; i < len; i++) {
	// Node node = claimTypes.item(i);
	// if (node.getNodeType() == Node.ELEMENT_NODE &&
	// "SupportedClaimType".equals(node.getLocalName()) &&
	// CardContext.IC_NS.equals(node.getNamespaceURI())) {
	// Element elm = (Element) node;
	// String uri = elm.getAttribute("Uri");
	// if (isStringEmpty(uri))
	// throw new CardException("Couldn't get \"uri\" attribute of
	// SupportedClaimType element for InfoCard with id = " + id_);
	// Element displayTagElm = XMLUtils.getChildElement(elm, CardContext.IC_NS,
	// "DisplayTag");
	// String displayTag = (displayTagElm != null) ?
	// XMLUtils.getTextContent(displayTagElm) : "";
	// Element descriptionElm = XMLUtils.getChildElement(elm, CardContext.IC_NS,
	// "Description");
	// String description = (descriptionElm != null) ?
	// XMLUtils.getTextContent(descriptionElm) : "";
	// claimTypes_.add(new ClaimType(uri, displayTag, description));
	// }
	// }
	// }
	/**
	 * @param endpointRef
	 * @return
	 * @throws CardException
	 */
	// private EndpointReference parseEndpointReference(Element endpointRef)
	// throws CardException {
	// Element addressElm = XMLUtils.getChildElement(endpointRef,
	// CardContext.WSA_NS, "Address");
	// String addressStr = (addressElm != null) ?
	// XMLUtils.getTextContent(addressElm) : "";
	// URI address = (addressStr != null) ? URI.create(addressStr) : null;
	// Element metadata = XMLUtils.getChildElement(endpointRef,
	// CardContext.WSA_NS, "Metadata");
	// Element identity = XMLUtils.getChildElement(endpointRef,
	// CardContext.WSAI_NS, "Identity");
	// return new EndpointReference(address, metadata, identity);
	// }
	/**
	 * @param credential
	 * @return
	 * @throws CardException
	 */
	// private CredentialDescriptor parseUserCredential(Element credential)
	// throws CardException {
	// Element hintElm = XMLUtils.getChildElement(credential, CardContext.IC_NS,
	// "DisplayCredentialHint");
	// String hint = hintElm != null ? XMLUtils.getTextContent(hintElm) : "";
	// return new CredentialDescriptor(hint, credential);
	// }
	/**
	 * @param card
	 * @throws CardException
	 */
	// private void parseTokenServiceList(Element card) throws CardException {
	// tokenServices_ = new ArrayList();
	// Element tokenServiceList = XMLUtils.getChildElement(card,
	// CardContext.IC_NS, "TokenServiceList");
	// if (tokenServiceList == null)
	// throw new CardException("Couldn't get \"TokenServiceList\" element from
	// InfoCard with id = " + id_);
	// NodeList tokenServices = tokenServiceList.getChildNodes();
	// int len = tokenServices.getLength();
	// for (int i = 0; i < len; i++) {
	// Node node = tokenServices.item(i);
	// if (node.getNodeType() == Node.ELEMENT_NODE &&
	// "TokenService".equals(node.getLocalName()) &&
	// CardContext.IC_NS.equals(node.getNamespaceURI())) {
	// Element elm = (Element) node;
	// Element endpointRef = XMLUtils.getChildElement(elm, CardContext.WSA_NS,
	// "EndpointReference");
	// if (endpointRef == null)
	// throw new CardException("Couldn't get wsa:EndpointReference element");
	// EndpointReference epRef = parseEndpointReference(endpointRef);
	// Element credential = XMLUtils.getChildElement(elm, CardContext.IC_NS,
	// "UserCredential");
	// if (credential == null)
	// throw new CardException("Couldn't get ic:UserCredential element");
	// CredentialDescriptor userCredential = parseUserCredential(credential);
	// tokenServices_.add(new TokenService(epRef, userCredential));
	// }
	// }
	// if (tokenServices_.size() == 0)
	// throw new CardException("Couldn't get list of TokenService elements from
	// InfoCard with id = " + id_);
	// }
	/**
	 * @param card
	 * @throws CardException
	 */
	// private void parseRequireAppliesTo(Element card) throws CardException {
	// Element requireAppliesTo = XMLUtils.getChildElement(card,
	// CardContext.IC_NS, "RequireAppliesTo");
	// String str = (requireAppliesTo != null) ?
	// XMLUtils.getTextContent(requireAppliesTo) : "";
	// requireAppliesTo_ = null;
	// if (isStringEmpty(str) == false) {
	// if (str.equalsIgnoreCase("false"))
	// requireAppliesTo_ = new Boolean(false);
	// else if (str.equalsIgnoreCase("true"))
	// requireAppliesTo_ = new Boolean(true);
	// }
	// }
	/**
	 * @param card
	 * @throws CardException
	 */
	// private void parsePrivacyNotice(Element card) throws CardException {
	// privacyNotice_ = XMLUtils.getChildElement(card, CardContext.IC_NS,
	// "PrivacyNotice");
	// }
	/**
	 * @param card
	 * @throws CardException
	 */
	// private void parseHashSalt(Element card) throws CardException {
	// Element hashSalt = XMLUtils.getChildElement(card, CardContext.IC_NS,
	// "HashSalt");
	// String str = (hashSalt != null) ? XMLUtils.getTextContent(hashSalt) : "";
	// if (isStringEmpty(str))
	// throw new CardException("Couldn't get HashSalt element from InfoCard with
	// id = " + id_);
	// hashSalt_ = CardCryptography.decodeBase64(str);
	// }
	/**
	 * @param card
	 * @throws CardException
	 */
	// private void parseTimeLastUpdated(Element card) throws CardException {
	// Element timeLastUpdated = XMLUtils.getChildElement(card,
	// CardContext.IC_NS, "TimeLastUpdated");
	// String str = (timeLastUpdated != null) ?
	// XMLUtils.getTextContent(timeLastUpdated) : "";
	// if (isStringEmpty(str))
	// throw new CardException("Couldn't get TimeLastUpdated element from
	// InfoCard with id = " + id_);
	// timeLastUpdated_ = DateConvertor.parse(str);
	// }
	/**
	 * @param roamingCard
	 * @throws CardException
	 */
	// private void parseMasterKey(Element roamingCard) throws CardException {
	// Element informationCardPrivateData =
	// XMLUtils.getChildElement(roamingCard, CardContext.IC_NS,
	// "InformationCardPrivateData");
	// if (informationCardPrivateData != null) {
	// Element masterKey = XMLUtils.getChildElement(informationCardPrivateData,
	// CardContext.IC_NS, "MasterKey");
	// String str = (masterKey != null) ? XMLUtils.getTextContent(masterKey) :
	// "";
	// if (isStringEmpty(str))
	// throw new CardException("Couldn't get MasterKey element from
	// RoamingInformationCard with id = " + id_);
	// masterKey_ = CardCryptography.decodeBase64(str);
	// } else
	// throw new CardException("Couldn't get \"InformationCardPrivateData\"
	// element from RoamingInformationCard with id = " + id_);
	// }
	/**
	 * @param str
	 * @return
	 */
	// private static boolean isStringEmpty(String str) {
	// return str == null || str.trim().length() == 0;
	// }
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.IInformationCard#toXML(org.w3c.dom.Document)
	 */
	public IElement toElement(IElementFormat format) throws CardException, CardIOException, UnsupportedElementFormatException {
		if (!isFormatSupported(format)) {
			throw new CardException("Unsupported element format: " + format);
		}

		try {
			DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
			dbf.setIgnoringComments(true);
			dbf.setIgnoringElementContentWhitespace(true);
			dbf.setNamespaceAware(true);
			dbf.setValidating(false);
			DocumentBuilder db = dbf.newDocumentBuilder();

			Document doc = db.newDocument();
			Element card = doc.createElementNS(CardContext.IC_NS, CardContext.IC_ROAMING_INFORMATION_CARD);

			Element metaData = doc.createElementNS(CardContext.IC_NS, CardContext.IC_INFORMATION_CARD_META_DATA);
			if (language_ != null)
				metaData.setAttribute("xml:lang", language_);
			card.appendChild(metaData);

			addCardReference(doc, metaData);
			addCardName(doc, metaData);
			addCardImage(doc, metaData);
			addIssuer(doc, metaData);
			// addIssuerName(doc, metaData);
			addTimeIssued(doc, metaData);
			addTimeExpires(doc, metaData);
			addTokenServiceList(doc, metaData);
			addSupportedTokenTypes(doc, metaData);
			addClaimTypes(doc, metaData);
			addRequireAppliesTo(doc, metaData);
			addPrivacyNotice(doc, metaData);
			addExtensions(doc, metaData);
			addIsSelfIssued(doc, metaData);
			addHashSalt(doc, metaData);
			addTimeLastUpdated(doc, metaData);
			addIssuerId(doc, metaData);
			addIssuerName(doc, metaData);
			addBackgroundColor(doc, metaData);

			Element privateData = doc.createElementNS(CardContext.IC_NS, CardContext.IC_INFORMATION_CARD_PRIVATE_DATA);
			card.appendChild(privateData);

			addMasterKey(doc, privateData);

			IElement element = new IOElement();
			element.set(card);
			return element;
		} catch (Exception e) {
			log.error("Can't export card", e);
			throw new CardException("Can't export card", e);
		}
	}

	/**
	 * @param doc
	 * @param metaData
	 */
	private void addCardReference(Document doc, Element metaData) {
		Element ref = doc.createElementNS(CardContext.IC_NS, CardContext.IC_INFORMATION_CARD_REFERENCE);
		metaData.appendChild(ref);
		Element id = doc.createElementNS(CardContext.IC_NS, CardContext.IC_CARD_ID);
		XMLUtils.setTextContent(id, id_.toString());
		ref.appendChild(id);
		Element version = doc.createElementNS(CardContext.IC_NS, CardContext.IC_CARD_VERSION);
		XMLUtils.setTextContent(version, version_);
		ref.appendChild(version);
	}

	/**
	 * @param doc
	 * @param metaData
	 */
	private void addCardName(Document doc, Element metaData) {
		if (name_ != null) {
			Element name = doc.createElementNS(CardContext.IC_NS, CardContext.IC_CARD_NAME);
			XMLUtils.setTextContent(name, name_);
			metaData.appendChild(name);
		}
	}

	/**
	 * @param doc
	 * @param metaData
	 */
	private void addCardImage(Document doc, Element metaData) {
		if (image_ != null) {
			Element image = doc.createElementNS(CardContext.IC_NS, CardContext.IC_CARD_IMAGE);
			if (imageMimeType_ != null)
				image.setAttribute("MimeType", imageMimeType_);
			XMLUtils.setTextContent(image, new String(CardCryptography.encodeBase64(image_, 0)));
			metaData.appendChild(image);
		}
	}

	/**
	 * @param doc
	 * @param metaData
	 */
	private void addIssuer(Document doc, Element metaData) {
		Element issuer = doc.createElementNS(CardContext.IC_NS, CardContext.IC_ISSUER);
		XMLUtils.setTextContent(issuer, issuer_);
		metaData.appendChild(issuer);
	}

	/**
	 * @param doc
	 * @param metaData
	 */
	private void addTimeIssued(Document doc, Element metaData) {
		String time = DateConvertor.format(timeIssued_);
		Element timeIssued = doc.createElementNS(CardContext.IC_NS, CardContext.IC_TIME_ISSUED);
		XMLUtils.setTextContent(timeIssued, time);
		metaData.appendChild(timeIssued);
	}

	/**
	 * @param doc
	 * @param metaData
	 */
	private void addTimeExpires(Document doc, Element metaData) {
		if (timeExpires_ != null) {
			String time = DateConvertor.format(timeExpires_);
			Element timeExpires = doc.createElementNS(CardContext.IC_NS, CardContext.IC_TIME_EXPIRES);
			XMLUtils.setTextContent(timeExpires, time);
			metaData.appendChild(timeExpires);
		}
	}

	/**
	 * @param doc
	 * @param metaData
	 */
	private void addTokenServiceList(Document doc, Element metaData) {
		Element tokenServiceList = doc.createElementNS(CardContext.IC_NS, CardContext.IC_TOKEN_SERVICE_LIST);
		metaData.appendChild(tokenServiceList);
		Iterator it = tokenServices_.iterator();
		while (it.hasNext()) {
			ITokenService tokenSrv = (ITokenService) it.next();
			Element tokenService = doc.createElementNS(CardContext.IC_NS, CardContext.IC_TOKEN_SERVICE);
			tokenServiceList.appendChild(tokenService);
			IEndpointReference endpoint = tokenSrv.getEndpointReference();
			Element endpointReference = doc.createElementNS(CardContext.WSA_NS, CardContext.WSA_ENDPOINT_REFERENCE);
			tokenService.appendChild(endpointReference);
			Element address = doc.createElementNS(CardContext.WSA_NS, CardContext.WSA_ADDRESS);
			XMLUtils.setTextContent(address, endpoint.getAddress().toString());
			endpointReference.appendChild(address);
			// /wsa:EndpointReference/wsa:Metadata is optional, ie:
			// http://www.w3.org/TR/2005/CR-ws-addr-core-20050817/
			if (endpoint.getMetadata() != null) {
				Node metadata = doc.importNode(endpoint.getMetadata(), true);
				endpointReference.appendChild(metadata);
			}
			// from the old InformationCard.java branch Zurich-Catalyst
			if (endpoint.getIdentity() != null) {
				Node identity = doc.importNode(endpoint.getIdentity(), true);
				endpointReference.appendChild(identity);
			}
			ICredentialDescriptor uCredential = tokenSrv.getUserCredential();
			Node credential = doc.importNode(uCredential.asXML(), true);
			tokenService.appendChild(credential);
		}
	}

	/**
	 * @param doc
	 * @param metaData
	 */
	private void addSupportedTokenTypes(Document doc, Element metaData) {
		Element tokenList = doc.createElementNS(CardContext.IC_NS, CardContext.IC_SUPPORTED_TOKEN_TYPE_LIST);
		metaData.appendChild(tokenList);
		Iterator it = supportedTokenTypes_.iterator();
		while (it.hasNext()) {
			String uri = it.next().toString();
			Element tokenType = doc.createElementNS(CardContext.TRUST_NS, CardContext.TRUST_TOKEN_TYPE);
			XMLUtils.setTextContent(tokenType, uri);
			tokenList.appendChild(tokenType);
		}
	}

	/**
	 * @param doc
	 * @param metaData
	 */
	private void addClaimTypes(Document doc, Element metaData) {
		Element claimTypeList = doc.createElementNS(CardContext.IC_NS, CardContext.IC_SUPPORTED_CLAIM_TYPE_LIST);
		metaData.appendChild(claimTypeList);
		Iterator it = claimTypes_.iterator();
		while (it.hasNext()) {
			IClaimType type = (IClaimType) it.next();
			Element claimType = doc.createElementNS(CardContext.IC_NS, CardContext.IC_SUPPORTED_CLAIM_TYPE);
			claimType.setAttribute("Uri", type.getType().toString());
			claimTypeList.appendChild(claimType);
			String display = type.getDisplayName();
			if (display != null && display.trim().length() > 0) {
				Element displayTag = doc.createElementNS(CardContext.IC_NS, CardContext.IC_DISPLAY_TAG);
				XMLUtils.setTextContent(displayTag, display);
				claimType.appendChild(displayTag);
			}
			String descript = type.getDescription();
			if (descript != null && descript.trim().length() > 0) {
				Element description = doc.createElementNS(CardContext.IC_NS, CardContext.IC_DESCRIPTION);
				XMLUtils.setTextContent(description, descript);
				claimType.appendChild(description);
			}
		}
	}

	/**
	 * @param doc
	 * @param metaData
	 */
	private void addRequireAppliesTo(Document doc, Element metaData) {
		if (requireAppliesTo_ != null) {
			Element requireAppliesTo = doc.createElementNS(CardContext.IC_NS, CardContext.IC_REQUIRE_APPLIES_TO);
			if (requireAppliesTo_.booleanValue()) {
				requireAppliesTo.setAttribute(CardContext.IC_OPTIONAL_ATTR, "true");
			}
			metaData.appendChild(requireAppliesTo);
		}
	}

	/**
	 * @param doc
	 * @param metaData
	 */
	private void addPrivacyNotice(Document doc, Element metaData) {
		if (privacyNotice_ != null) {
			IElement elm = privacyNotice_.getPrivacyElement();
			if (elm != null) {
				Element element = null;
				try {
					element = (Element) elm.getAs(Element.class);
				} catch (Exception e) {
					log.error(e, e);
				}
				if (element != null) {
					Element privacyNotice = (Element) doc.importNode(element, true);
					metaData.appendChild(privacyNotice);
				}
				else
					log.error("Privacy notice was not added to XML card.");
			}
		}
	}

	private void addExtensions(Document doc, Element metaData) {
		if (extensions_ != null && extensions_.size() > 0) {
			for(int i = 0, j = extensions_.size(); i < j; i++) {
				IInformationCardExtension ext = (IInformationCardExtension)extensions_.get(i);
				IElement elm = ext.getElement();
				if (elm != null) {
					Element element = null;
					try {
						element = (Element) elm.getAs(Element.class);
					} catch (Exception e) {
						log.error(e, e);
					}
					if (element != null) {
						Element extension = (Element) doc.importNode(element, true);
						metaData.appendChild(extension);
					}
					else
						log.error("Extension element was not added to XML card.");
				}
			}
		}
	}
	
	
	/**
	 * @param doc
	 * @param metaData
	 */
	private void addIsSelfIssued(Document doc, Element metaData) {
		Element isSelfIssued = doc.createElementNS(CardContext.IC_NS, CardContext.IC_IS_SELF_ISSUED);
		XMLUtils.setTextContent(isSelfIssued, String.valueOf(isSelfIssued()));
		metaData.appendChild(isSelfIssued);
	}

	/**
	 * @param doc
	 * @param metaData
	 */
	private void addHashSalt(Document doc, Element metaData) throws CardException {
		if (hashSalt_ != null) {
			Element hashSalt = doc.createElementNS(CardContext.IC_NS, CardContext.IC_HASH_SALT);
			XMLUtils.setTextContent(hashSalt, CardCryptography.encodeBase64(hashSalt_, 1));
			metaData.appendChild(hashSalt);
		} else
			throw new CardException("Card does not have HashSalt value");
	}

	/**
	 * @param doc
	 * @param metaData
	 */
	private void addTimeLastUpdated(Document doc, Element metaData) throws CardException {
		if (timeLastUpdated_ != null) {
			Element timeLastUpdated = doc.createElementNS(CardContext.IC_NS, CardContext.IC_TIME_LAST_UPDATED);
			XMLUtils.setTextContent(timeLastUpdated, DateConvertor.formatMS(timeLastUpdated_));
			metaData.appendChild(timeLastUpdated);
		} else
			throw new CardException("Card does not have TimeLastUpdated value");
	}

	/**
	 * @param doc
	 * @param metaData
	 */
	private void addIssuerId(Document doc, Element metaData) {
		Element issuerId = doc.createElementNS(CardContext.IC_NS, CardContext.IC_ISSUER_ID);
		metaData.appendChild(issuerId);
		if (issuerID_ != null)
			XMLUtils.setTextContent(issuerId, CardCryptography.encodeBase64(issuerID_, 0));
	}

	/**
	 * @param doc
	 * @param metaData
	 */
	private void addIssuerName(Document doc, Element metaData) {
		Element issuerName = doc.createElementNS(CardContext.IC_NS, CardContext.IC_ISSUER_NAME);
		XMLUtils.setTextContent(issuerName, getIssuerName());
		metaData.appendChild(issuerName);
	}

	/**
	 * @param doc
	 * @param metaData
	 */
	private void addBackgroundColor(Document doc, Element metaData) {
		Element color = doc.createElementNS(CardContext.IC_NS, CardContext.IC_BACK_GROUND_COLOR);
		XMLUtils.setTextContent(color, "0");
		metaData.appendChild(color);
	}

	/**
	 * @param doc
	 * @param privateData
	 */
	private void addMasterKey(Document doc, Element privateData) throws CardException {
		if (masterKey_ != null) {
			Element masterKey = doc.createElementNS(CardContext.IC_NS, CardContext.IC_MASTER_KEY);
			XMLUtils.setTextContent(masterKey, CardCryptography.encodeBase64(masterKey_, 0));
			privateData.appendChild(masterKey);
		} else
			throw new CardException("Card does not have MasterKey value");
	}

	public ICredentialDescriptor[] getRequiredCredentials() {
		if (requiredCredentials_ == null) {
			List tsl = getTokenServices();
			requiredCredentials_ = new ICredentialDescriptor[tsl.size()];
			for (int i = 0; i < tsl.size(); i++) {
				ITokenService ts = (ITokenService) tsl.get(i);
				requiredCredentials_[i] = new TSCredentialDescriptor(ts);
			}
		}
		return requiredCredentials_;
	}

	public IClaim getClaim(String type) throws AuthenticationRequiredException, InvalidTypeException, CardException {
		List supportedTypes = getSupportedClaimTypesUris();
		if (type == null || supportedTypes.contains(type) == false)
			throw new InvalidTypeException("Claim type: " + type + " is not supported.");
		if (claimValues_ == null) {
			throw new AuthenticationRequiredException();
		}
		if (claimValues_.containsKey(type)) {
			return (IClaim) claimValues_.get(type);
		} else
			return null;
	}

	public IClaim getClaimByLocalName(String shortTypeName) throws AuthenticationRequiredException, InvalidTypeException, CardException {
		if (claimValuesByLocalName_ == null) {
			throw new AuthenticationRequiredException();
		}
		if (claimValuesByLocalName_.containsKey(shortTypeName)) {
			return (IClaim) claimValuesByLocalName_.get(shortTypeName);
		}
		throw new InvalidTypeException("Claim with local type name: " + shortTypeName + " is not supported.");
	}

	public Iterator getClaims() throws CardException {
		if (claimValues_ == null) {
			throw new AuthenticationRequiredException();
		}
		return claimValues_.values().iterator();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#getType()
	 */
	public String getType() {
		return ICardConstants.I_CARD_TYPE_MANAGED_CARDSPACE;
	}

	public String getIc07IssuerInformation() {
		return ic07IssuerInformation;
	}

	public boolean isStrongRecipientIdentityRequired() {
		return requireStrongRecipientIdentity;
	}

}
