/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.icard.provider.cardspace.common;

import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.security.NoSuchAlgorithmException;
import java.util.Iterator;
import java.util.List;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.AuthenticationException;
import org.eclipse.higgins.icard.AuthenticationRequiredException;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.ICardConstants;
import org.eclipse.higgins.icard.ICardProvider;
import org.eclipse.higgins.icard.IClaim;
import org.eclipse.higgins.icard.IClaimType;
import org.eclipse.higgins.icard.IPersonalInformationCard;
import org.eclipse.higgins.icard.InvalidTypeException;
import org.eclipse.higgins.icard.auth.ICredential;
import org.eclipse.higgins.icard.auth.ICredentialDescriptor;
import org.eclipse.higgins.icard.auth.IPinCodeCredential;
import org.eclipse.higgins.icard.common.ClaimType;
import org.eclipse.higgins.icard.common.auth.BasicCredentialDescriptor;
import org.eclipse.higgins.icard.common.auth.PinCodeCredential;
import org.eclipse.higgins.icard.common.io.IOElement;
import org.eclipse.higgins.icard.io.CardIOException;
import org.eclipse.higgins.icard.io.IElement;
import org.eclipse.higgins.icard.io.IElementFormat;
import org.eclipse.higgins.icard.io.UnsupportedElementFormatException;
import org.eclipse.higgins.icard.common.utils.CardContext;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.CardCryptography;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.CardUtils;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.SelfIssuedCardClaims;
import org.eclipse.higgins.icard.common.utils.DateConvertor;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.XMLUtils;
import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

public abstract class PersonalCard extends InformationCard implements IPersonalInformationCard {
	protected static Log log = LogFactory.getLog(PersonalCard.class);

	// can contain pin-protected claim values as it stored within *.crds file.
	// Used ONLY for parsing/creation the card. Should not be used to return/set
	// claim value in methods getClaim(), setClaim() etc. because claims can be
	// encrypted

	public static String ISSUER_SELF = "http://schemas.xmlsoap.org/ws/2005/05/identity/issuer/self";

	/**
	 * Contains "dirty" (as it is in ic:InformationCardPrivateData of
	 * ic:RoamingInformationCard) claim values map (real value of claim for not
	 * pin-protected card or Base64-encoded encrypted value for pin-protected
	 * card)represented as <code>String</code> by their claim types
	 * represented as <code>String</code>. Contains only those of standard
	 * personal card claims which calue is not empty
	 */
	protected java.util.HashMap dirtyClaimValueMap_ = null;

	protected URI claimListContextID_ = null;

	protected String claimListEntityID_ = null;

	protected byte[] pinDigest_ = null;
	
	protected int pinStatus;
	
	public static final int NO_PIN = 0;
	public static final int LOCKED = 1;
	public static final int UNLOCKED = 2; 
	
	/**
	 * @param card
	 * @throws CardException
	 * @throws DOMException
	 */
	public void initFromXML(Element card) throws CardException, DOMException {
		if (CardContext.IC_ROAMING_INFORMATION_CARD.equals(card.getNodeName()) == false)
			throw new CardException("Can not init personal card from element " + card.getNodeName());
		isSelfIssued_ = true;
		issuer_ = ISSUER_SELF;
		issuerName_ = "Self";
		Element informationCardMetaData = XMLUtils.getChildElement(card, CardContext.IC_NS, CardContext.IC_INFORMATION_CARD_META_DATA);
		if (informationCardMetaData == null)
			throw new CardException("Can not find \"InformationCardMetaData\" element.");
		language_ = null;
		if (informationCardMetaData.hasAttribute("xml:lang"))
			language_ = informationCardMetaData.getAttribute("xml:lang");

		Element informationCardPrivateData = XMLUtils.getChildElement(card, CardContext.IC_NS, CardContext.IC_INFORMATION_CARD_PRIVATE_DATA);

//		checkIsSelfIssued(informationCardMetaData);
//		parseCardId(informationCardMetaData);
//		parseCardVersion(informationCardMetaData);
//		parseCardName(informationCardMetaData);
//		parseCardImage(informationCardMetaData);
//		parseTimeIssued(informationCardMetaData);
//		parseTimeExpires(informationCardMetaData);
//		parseSupportedTokenTypes(informationCardMetaData);
//		parseClaimTypes(informationCardMetaData);
//		parseHashSalt(informationCardMetaData);
//		parsePinDigest(informationCardMetaData);
//		parseTimeLastUpdated(informationCardMetaData);
//		parseClaimValues(informationCardPrivateData);
//		parseMasterKey(informationCardPrivateData);

		checkIsSelfIssued(informationCardMetaData);
		id_ = CardUtils.parseCardId(informationCardMetaData);
		version_ = CardUtils.parseCardVersion(informationCardMetaData);
		name_ = CardUtils.parseCardName(informationCardMetaData);
		image_ = CardUtils.parseCardImage(informationCardMetaData);
		imageMimeType_ = CardUtils.parseCardImageType(informationCardMetaData);
		timeIssued_ = CardUtils.parseTimeIssued(informationCardMetaData);
		timeExpires_ = CardUtils.parseTimeExpires(informationCardMetaData);
		supportedTokenTypes_ = CardUtils.parseSupportedTokenTypes(informationCardMetaData);
		claimTypes_ = parseClaimTypes(informationCardMetaData);
		hashSalt_ = CardUtils.parseHashSalt(informationCardMetaData);
		pinDigest_ = CardUtils.parsePinDigest(informationCardMetaData);
		issuerID_ = CardUtils.parseIssuerID(informationCardMetaData);
		timeLastUpdated_ = CardUtils.parseTimeLastUpdated(informationCardMetaData);
		extensions_ = CardUtils.parseExtensions(informationCardMetaData);
		dirtyClaimValueMap_ = CardUtils.parseClaimValues(informationCardPrivateData);
		masterKey_ = CardUtils.parseMasterKey(card);
		
		if(pinDigest_!=null && pinDigest_.length>0){
			pinStatus = LOCKED;
		} else {
			pinStatus = NO_PIN;
		}
	}

	private void checkIsSelfIssued(Element metaData) throws CardException {
		Element isSelfIssued = XMLUtils.getChildElement(metaData, CardContext.IC_NS, CardContext.IC_IS_SELF_ISSUED);
		String boolValue = (isSelfIssued != null) ? XMLUtils.getTextContent(isSelfIssued) : "";
		if (boolValue == null || "true".equals(boolValue.trim().toLowerCase()) == false)
			throw new CardException("Can not import non-self issued card.");
	}

	/**
	 * @param metaData
	 * @throws CardException
	 */
//	private void parseCardId(Element metaData) throws CardException {
//		Element informationCardReference = XMLUtils.getChildElement(metaData, CardContext.IC_NS, "InformationCardReference");
//		if (informationCardReference != null) {
//			Element cardId = XMLUtils.getChildElement(informationCardReference, CardContext.IC_NS, "CardId");
//			String id = (cardId != null) ? XMLUtils.getTextContent(cardId) : "";
//			if (id == null || id.trim().length() == 0)
//				throw new CardException("Couldn't get CardId from InfoCard.");
//			id_ = URI.create(id);
//		} else
//			throw new CardException("Couldn't get \"InformationCardReference\" from InfoCard.");
//	}

	/**
	 * @param metaData
	 * @throws CardException
	 */
//	private void parseCardVersion(Element metaData) throws CardException {
//		Element informationCardReference = XMLUtils.getChildElement(metaData, CardContext.IC_NS, "InformationCardReference");
//		if (informationCardReference != null) {
//			Element cardVersion = XMLUtils.getChildElement(informationCardReference, CardContext.IC_NS, "CardVersion");
//			version_ = (cardVersion != null) ? XMLUtils.getTextContent(cardVersion) : "";
//			if (version_ == null || version_.trim().length() == 0)
//				throw new CardException("Couldn't get CardVersion from InfoCard with id = " + id_);
//		} else
//			throw new CardException("Couldn't get \"InformationCardReference\" from InfoCard.");
//	}

	/**
	 * @param metaData
	 * @throws CardException
	 */
//	private void parseCardName(Element metaData) throws CardException {
//		Element cardName = XMLUtils.getChildElement(metaData, CardContext.IC_NS, "CardName");
//		name_ = (cardName != null) ? XMLUtils.getTextContent(cardName) : "";
//	}

	/**
	 * @param metaData
	 * @throws CardException
	 * @throws DOMException
	 */
//	private void parseCardImage(Element metaData) throws CardException, DOMException {
//		Element cardImg = XMLUtils.getChildElement(metaData, CardContext.IC_NS, "CardImage");
//		if (cardImg == null)
//			throw new CardException("Couldn't get CardImage from InfoCard with id = " + id_);
//		if (cardImg != null) {
//			image_ = CardCryptography.decodeBase64(XMLUtils.getTextContent(cardImg));
//			imageMimeType_ = cardImg.getAttribute("MimeType");
//		}
//	}

	/**
	 * @param metaData
	 * @throws CardException
	 */
//	private void parseTimeIssued(Element metaData) throws CardException {
//		Element timeIssuedElm = XMLUtils.getChildElement(metaData, CardContext.IC_NS, "TimeIssued");
//		String timeIssued = (timeIssuedElm != null) ? XMLUtils.getTextContent(timeIssuedElm) : "";
//		if (timeIssued == null || timeIssued.trim().length() == 0)
//			throw new CardException("Couldn't get TimeIssued from InfoCard with id = " + id_);
//		timeIssued_ = DateConvertor.parse(timeIssued);
//	}

	/**
	 * @param metaData
	 * @throws CardException
	 */
//	private void parseTimeExpires(Element metaData) throws CardException {
//		Element timeExpiresElm = XMLUtils.getChildElement(metaData, CardContext.IC_NS, "TimeExpires");
//		String timeExpires = (timeExpiresElm != null) ? XMLUtils.getTextContent(timeExpiresElm) : "";
//		if (timeExpires != null && timeExpires.trim().length() == 0)
//			timeExpires_ = DateConvertor.parse(timeExpires);
//	}

//	private void parseSupportedTokenTypes(Element metaData) throws CardException {
//		supportedTokenTypes_ = new ArrayList();
//		Element supportedTokenTypeList = XMLUtils.getChildElement(metaData, CardContext.IC_NS, "SupportedTokenTypeList");
//		if (supportedTokenTypeList == null)
//			throw new CardException("Couldn't get \"SupportedTokenTypeList\" element from InfoCard with id = " + id_);
//		NodeList tokenTypes = supportedTokenTypeList.getChildNodes();
//		int len = tokenTypes.getLength();
//		for (int i = 0; i < len; i++) {
//			Node nd = tokenTypes.item(i);
//			if (nd.getNodeType() == Node.ELEMENT_NODE && "TokenType".equals(nd.getLocalName()) && CardContext.TRUST_NS.equals(nd.getNamespaceURI())) {
//				Element elm = (Element)nd;
//				supportedTokenTypes_.add(URI.create(XMLUtils.getTextContent(elm)));
//			}
//		}
//		if (supportedTokenTypes_.size() == 0)
//			throw new CardException("Couldn't get the list of TokenType elements from InfoCard with id = " + id_);
//	}

	/**
	 * @param metaData
	 * @throws CardException
	 */
//	private void parseClaimTypes(Element metaData) throws CardException {
//		claimTypes_ = new ArrayList();
//		Element supportedClaimTypeList = XMLUtils.getChildElement(metaData, CardContext.IC_NS, "SupportedClaimTypeList");
//		if (supportedClaimTypeList == null)
//			throw new CardException("Couldn't get \"SupportedClaimTypeList\" element from InfoCard with id = " + id_);
//		NodeList claimTypes = supportedClaimTypeList.getChildNodes();
//		int len = claimTypes.getLength();
//		for (int i = 0; i < len; i++)
//		{
//			Node node = claimTypes.item(i);
//			if (node.getNodeType() == Node.ELEMENT_NODE && "SupportedClaimType".equals(node.getLocalName()) && CardContext.IC_NS.equals(node.getNamespaceURI())) {
//				Element elm = (Element) node;
//				String uri = elm.getAttribute("Uri");
//				if (uri == null || uri.trim().length() == 0)
//					throw new CardException("Couldn't get \"uri\" attribute of SupportedClaimType element for InfoCard with id = " + id_);
//				Element displayTagElm = XMLUtils.getChildElement(elm, CardContext.IC_NS, "DisplayTag");
//				String displayTag = (displayTagElm != null) ? XMLUtils.getTextContent(displayTagElm) : "";
//				Element descriptionElm = XMLUtils.getChildElement(elm, CardContext.IC_NS, "Description");
//				String description = (descriptionElm != null) ? XMLUtils.getTextContent(descriptionElm) : "";
//				claimTypes_.add(new ClaimType(uri, displayTag, description));
//			}
//		}
//		String displayTag = "Private Personal Identifier";
//		String description = "Private Personal Identifier";
//		claimTypes_.add(new ClaimType
//			("http://schemas.xmlsoap.org/ws/2005/05/identity/claims/privatepersonalidentifier",
//			displayTag,
//			description));
//	}
	
	protected List parseClaimTypes(Element metaData) throws CardException {
		List res = CardUtils.parseClaimTypes(metaData);
		res.add(new ClaimType
				("http://schemas.xmlsoap.org/ws/2005/05/identity/claims/privatepersonalidentifier",
						"Private Personal Identifier",
						"Private Personal Identifier", false, false));
		return res;
	}


	/**
	 * @param privateData
	 * @throws CardException
	 */
//	private void parseClaimValues(Element privateData) throws CardException {
//		java.util.HashMap tmpClaimValueMap_ = new java.util.HashMap();
//		Element claimValueList = XMLUtils.getChildElement(privateData, CardContext.IC_NS, "ClaimValueList");
//		if (claimValueList == null) return;
//		// From the doc: optional element
////			throw new CardException("Couldn't get the list of \"ClaimValueList\" elements from InfoCard with id = " + id_);
//		NodeList claimValues = claimValueList.getChildNodes();
//		int len = claimValues.getLength();
//		for (int i = 0; i < len; i++)
//		{
//			Node node = claimValues.item(i);
//			if (node.getNodeType() == Node.ELEMENT_NODE && "ClaimValue".equals(node.getLocalName()) && CardContext.IC_NS.equals(node.getNamespaceURI())) {
//				Element elm = (Element) node;
//				String uri = elm.getAttribute("Uri");
//				if (uri == null || uri.trim().length() == 0)
//					throw new CardException("Couldn't get \"uri\" attribute of ClaimValue element for InfoCard with id = " + id_);
//				Element valueElm = XMLUtils.getChildElement(elm, CardContext.IC_NS, "Value");
//				String value = (valueElm != null) ? XMLUtils.getTextContent(valueElm) : "";
//				if (value != null) {
//					ClaimValue claimValue = new ClaimValue(SelfIssuedCardClaims.getClaimType(uri), value);
//					tmpClaimValueMap_.put(uri, claimValue);
//				}
//			}
//		}
//		if (tmpClaimValueMap_.size() == 0)
//			throw new CardException("Couldn't get list of ClaimValue elements from InfoCard with id = " + id_);
///*
//	ClaimValue claimValuePPID = new ClaimValue
//			(SelfIssuedCardClaims.getClaimType
//				("http://schemas.xmlsoap.org/ws/2005/05/identity/claims/privatepersonalidentifier"),
//			"[Private]");
//		tmpClaimValueList_.add(claimValuePPID);
//*/	
//		claimValues_ = tmpClaimValueMap_;
//	}

	/**
	 * @param metaData
	 * @throws CardException
	 */
//	private void parseHashSalt(Element metaData) throws CardException {
//		Element hashSalt = XMLUtils.getChildElement(metaData, CardContext.IC_NS, "HashSalt");
//		String str = (hashSalt != null) ? XMLUtils.getTextContent(hashSalt) : "";
//		if (str == null || str.trim().length() == 0)
//			throw new CardException("Couldn't get HashSalt element from InfoCard with id = " + id_);
//		hashSalt_ = CardCryptography.decodeBase64(str);
//	}

	/**
	 * @param metaData
	 * @throws CardException
	 */
//	private void parsePinDigest(Element metaData) throws CardException {
//		pinDigest_ = null;
//		Element pinDigest = XMLUtils.getChildElement(metaData, CardContext.IC_NS, "PinDigest");
//		String str = (pinDigest != null) ? XMLUtils.getTextContent(pinDigest) : "";
//		if (str != null)
//			pinDigest_ = CardCryptography.decodeBase64(str);
//	}

	/**
	 * @param metaData
	 * @throws CardException
	 */
//	private void parseTimeLastUpdated(Element metaData) throws CardException {
//		Element timeLastUpdated = XMLUtils.getChildElement(metaData, CardContext.IC_NS, "TimeLastUpdated");
//		String str = (timeLastUpdated != null) ? XMLUtils.getTextContent(timeLastUpdated) : "";
//		if (str == null || str.trim().length() == 0)
//			throw new CardException("Couldn't get TimeLastUpdated element from InfoCard with id = " + id_);
//		timeLastUpdated_ = DateConvertor.parse(str);
//	}

	/**
	 * @param privateData
	 * @throws CardException
	 */
//	private void parseMasterKey(Element privateData) throws CardException {
//		Element masterKey = XMLUtils.getChildElement(privateData, CardContext.IC_NS, "MasterKey");
//		String str = (masterKey != null) ? XMLUtils.getTextContent(masterKey) : "";
//		if (str == null || str.trim().length() == 0)
//			throw new CardException("Couldn't get MasterKey element from RoamingInformationCard with id = " + id_);
//		masterKey_ = CardCryptography.decodeBase64(str);
//	}

	public URI getClaimListContextID() {
		return claimListContextID_;
	}

	public String getClaimListEntityID() {
		return claimListEntityID_;
	}

	public IElement toElement(IElementFormat format) throws CardException, CardIOException, UnsupportedElementFormatException {
		if (!isFormatSupported(format)) {
			throw new CardException("Unsupported element format: " + format);
		}
		
		try {
			DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
			dbf.setIgnoringComments(true);
			dbf.setIgnoringElementContentWhitespace(true);
			dbf.setNamespaceAware(true);
			dbf.setValidating(false);
			DocumentBuilder db = dbf.newDocumentBuilder();

			Document doc = db.newDocument();
			Element card = doc.createElementNS(CardContext.IC_NS, CardContext.IC_ROAMING_INFORMATION_CARD);

			Element metaData = doc.createElementNS(CardContext.IC_NS, CardContext.IC_INFORMATION_CARD_META_DATA);
			if (language_ != null)
				metaData.setAttribute("xml:lang", language_);
			else
				metaData.setAttribute("xml:lang", "en");
			card.appendChild(metaData);
			addCardReference(doc, metaData);
			addCardName(doc, metaData);
			addCardImage(doc, metaData);
			addIssuer(doc, metaData);
			addTimeIssued(doc, metaData);
			addTimeExpires(doc, metaData);
//			addTokenServiceList(doc, metaData);
			addSupportedTokenTypes(doc, metaData);
			addClaimTypes(doc, metaData);
			addIsSelfIssued(doc, metaData);
			addPinDigest(doc, metaData);
			addHashSalt(doc, metaData);
			addTimeLastUpdated(doc, metaData);
			addIssuerID(doc, metaData);
			addIssuerName(doc, metaData);
			addBackgroundColor(doc, metaData);
			Element privateData = doc.createElementNS(CardContext.IC_NS, CardContext.IC_INFORMATION_CARD_PRIVATE_DATA);
			card.appendChild(privateData);
			addMasterKey(doc, privateData);
			addClaims(doc, privateData);
			IElement element = new IOElement();
			element.set(card);
			return element;
		} catch (Exception e) {
			log.error("Can't export card", e);
			throw new CardException("Can't export card", e);
		}
	}

	/**
	 * @param doc
	 * @param metaData
	 */
	private void addIsSelfIssued(Document doc, Element informationCardMetaData) {
		Element IsSelfIssued = doc.createElementNS(CardContext.IC_NS, CardContext.IC_IS_SELF_ISSUED);
		informationCardMetaData.appendChild(IsSelfIssued);
		XMLUtils.setTextContent(IsSelfIssued, "true");
	}

	/**
	 * @param doc
	 * @param metaData
	 */
	private void addIssuerName(Document doc, Element informationCardMetaData) {
		Element IssuerName = doc.createElementNS(CardContext.IC_NS, CardContext.IC_ISSUER_NAME);
		informationCardMetaData.appendChild(IssuerName);
		XMLUtils.setTextContent(IssuerName, "Self");
	}

	/**
	 * @param doc
	 * @param metaData
	 */
	private void addIssuerID(Document doc, Element informationCardMetaData) {
		Element issuerID = doc.createElementNS(CardContext.IC_NS, CardContext.IC_ISSUER_ID);
		informationCardMetaData.appendChild(issuerID);
		if (issuerID_ != null)
			XMLUtils.setTextContent(issuerID, CardCryptography.encodeBase64(issuerID_, 0));
	}

	/**
	 * @param doc
	 * @param metaData
	 */
	private void addHashSalt(Document doc, Element informationCardMetaData) {
		Element hashSalt = doc.createElementNS(CardContext.IC_NS, CardContext.IC_HASH_SALT);
		informationCardMetaData.appendChild(hashSalt);
		XMLUtils.setTextContent(hashSalt, CardCryptography.encodeBase64(hashSalt_, 0));
	}

	/**
	 * @param doc
	 * @param informationCardMetaData
	 */
	private void addPinDigest(Document doc, Element informationCardMetaData) {
		if (pinDigest_ != null) {
			Element pinDigest = doc.createElementNS(CardContext.IC_NS, 
					CardContext.IC_PIN_DIGEST);
			informationCardMetaData.appendChild(pinDigest);
			XMLUtils.setTextContent(pinDigest, CardCryptography.encodeBase64(pinDigest_, 0));
		}
	}

	/**
	 * @param doc
	 * @param metaData
	 */
	private void addTimeLastUpdated(Document doc, Element informationCardMetaData) {
		Element timeLastUpdated = doc.createElementNS(CardContext.IC_NS, CardContext.IC_TIME_LAST_UPDATED);
		informationCardMetaData.appendChild(timeLastUpdated);
		String time = DateConvertor.format(timeLastUpdated_);
		XMLUtils.setTextContent(timeLastUpdated, time);
	}

	/**
	 * @param doc
	 * @param metaData
	 */
	private void addBackgroundColor(Document doc, Element informationCardMetaData) {
		Element bgcolor = doc.createElementNS(CardContext.IC_NS, CardContext.IC_BACK_GROUND_COLOR);
		informationCardMetaData.appendChild(bgcolor);
		// TODO a good value
		XMLUtils.setTextContent(bgcolor, "0");
	}

	
	/**
	 * @param doc
	 * @param privateData
	 */
	private void addMasterKey(Document doc, Element privateData) {
		Element masterKey = doc.createElementNS(CardContext.IC_NS, CardContext.IC_MASTER_KEY);
		String base64Key = CardCryptography.encodeBase64(masterKey_, 0);
		masterKey.appendChild(doc.createTextNode(base64Key));
		privateData.appendChild(masterKey);
	}
	
	/**
	 * @param doc
	 * @param privateData
	 */
	private void addClaims(Document doc, Element privateData) {
		Element claimList = doc.createElementNS(CardContext.IC_NS, CardContext.IC_CLAIM_VALUE_LIST);
		privateData.appendChild(claimList);
		Iterator types = SelfIssuedCardClaims.getSupportedClaimTypeList().iterator();
		while (types.hasNext()) {
			ClaimType cType = (ClaimType)types.next();
			String sType = cType.getType();
			if (SelfIssuedCardClaims.PPID_TYPE.equals(sType))
				continue;
			String sValue = (dirtyClaimValueMap_.containsKey(sType)) ? (String)dirtyClaimValueMap_.get(sType) : "";
			Element claimValue = doc.createElementNS(CardContext.IC_NS, CardContext.IC_CLAIM_VALUE);
			claimValue.setAttribute("Uri", sType);
			claimList.appendChild(claimValue);
			Element value = doc.createElementNS(CardContext.IC_NS, CardContext.IC_VALUE);
			XMLUtils.setTextContent(value, sValue);
			claimValue.appendChild(value);
		}
	}

	/**
	 * @param doc
	 * @param root
	 */
	private void addCardReference(Document doc, Element root) {
		Element ref = doc.createElementNS(CardContext.IC_NS, CardContext.IC_INFORMATION_CARD_REFERENCE);
		root.appendChild(ref);
		Element id = doc.createElementNS(CardContext.IC_NS, CardContext.IC_CARD_ID);
		XMLUtils.setTextContent(id, id_.toString());
		ref.appendChild(id);
		Element version = doc.createElementNS(CardContext.IC_NS, CardContext.IC_CARD_VERSION);
		XMLUtils.setTextContent(version, version_);
		ref.appendChild(version);
	}

	/**
	 * @param doc
	 * @param root
	 */
	private void addCardName(Document doc, Element root) {
		if (name_ != null) {
			Element name = doc.createElementNS(CardContext.IC_NS, CardContext.IC_CARD_NAME);
			XMLUtils.setTextContent(name, name_);
			root.appendChild(name);
		}
	}

	/**
	 * @param doc
	 * @param root
	 */
	private void addCardImage(Document doc, Element root) {
		if (image_ != null) {
			Element image = doc.createElementNS(CardContext.IC_NS, CardContext.IC_CARD_IMAGE);
			image.setAttribute("MimeType", imageMimeType_);
			XMLUtils.setTextContent(image, CardCryptography.encodeBase64(image_, 0));
			root.appendChild(image);
		}
	}

	/**
	 * @param doc
	 * @param root
	 */
	private void addIssuer(Document doc, Element root) {
		Element issuer = doc.createElementNS(CardContext.IC_NS, CardContext.IC_ISSUER);
		XMLUtils.setTextContent(issuer, issuer_);
		root.appendChild(issuer);
	}

	/**
	 * @param doc
	 * @param root
	 */
	private void addTimeIssued(Document doc, Element root) {
		String time = DateConvertor.format(timeIssued_);
		Element timeIssued = doc.createElementNS(CardContext.IC_NS, CardContext.IC_TIME_ISSUED);
		XMLUtils.setTextContent(timeIssued, time);
		root.appendChild(timeIssued);
	}

	/**
	 * @param doc
	 * @param root
	 */
	private void addTimeExpires(Document doc, Element root) {
		if (timeExpires_ != null) {
			String time = DateConvertor.format(timeExpires_);
			Element timeExpires = doc.createElementNS(CardContext.IC_NS, CardContext.IC_TIME_EXPIRES);
			XMLUtils.setTextContent(timeExpires, time);
			root.appendChild(timeExpires);
		}
	}
	

	/**
	 * @param doc
	 * @param root
	 */
	private void addSupportedTokenTypes(Document doc, Element root) {
		Element tokenList = doc.createElementNS(CardContext.IC_NS, CardContext.IC_SUPPORTED_TOKEN_TYPE_LIST);
		root.appendChild(tokenList);
		Iterator it = supportedTokenTypes_.iterator();
		while (it.hasNext()) {
			String uri = it.next().toString();
			Element tokenType = doc.createElementNS(CardContext.TRUST_NS, CardContext.TRUST_TOKEN_TYPE);
			XMLUtils.setTextContent(tokenType, uri);
			tokenList.appendChild(tokenType);
		}
	}

	/**
	 * @param doc
	 * @param root
	 */
	private void addClaimTypes(Document doc, Element root) {
		Element claimTypeList = doc.createElementNS(CardContext.IC_NS, CardContext.IC_SUPPORTED_CLAIM_TYPE_LIST);
		root.appendChild(claimTypeList);
		Iterator it = claimTypes_.iterator();
		while (it.hasNext()) {
			IClaimType type = (IClaimType) it.next();
			if ("http://schemas.xmlsoap.org/ws/2005/05/identity/claims/privatepersonalidentifier".equals(type.getType()))
				continue;
			Element claimType = doc.createElementNS(CardContext.IC_NS, CardContext.IC_SUPPORTED_CLAIM_TYPE);
			claimType.setAttribute("Uri", type.getType().toString());
			claimTypeList.appendChild(claimType);
			Element displayTag = doc.createElementNS(CardContext.IC_NS, CardContext.IC_DISPLAY_TAG);
			XMLUtils.setTextContent(displayTag, type.getDisplayName());
			claimType.appendChild(displayTag);
			Element description = doc.createElementNS(CardContext.IC_NS, CardContext.IC_DESCRIPTION);
			XMLUtils.setTextContent(description, type.getDescription());
			claimType.appendChild(description);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.IPersonalInformationCard#getPinDigest()
	 */
	public byte[] getPinDigest() throws CardException {
		return pinDigest_;
	}

	public IClaim getClaim(String type) throws AuthenticationRequiredException, InvalidTypeException, CardException {
		List supportedTypes = getSupportedClaimTypesUris();
		if (type == null || supportedTypes.contains(type) == false)
			throw new InvalidTypeException("Claim type: " + type + " is not supported.");
		if (claimValues_ == null) {
			retrieveClaims(null);
		}
		if (claimValues_ == null) {
			throw new AuthenticationRequiredException();
		}
		if (claimValues_.containsKey(type))
			return (IClaim) claimValues_.get(type);
		else
			return null;
	}

	public IClaim getClaimByLocalName(String shortTypeName) throws AuthenticationRequiredException, InvalidTypeException, CardException {
		if (claimValuesByLocalName_ == null) {
			retrieveClaims(null);
		}
		if (claimValuesByLocalName_ == null) {
			throw new AuthenticationRequiredException();
		}
		if (claimValuesByLocalName_.containsKey(shortTypeName)) {
			return (IClaim) claimValuesByLocalName_.get(shortTypeName);
		}
		throw new InvalidTypeException("Claim with local type name: " + shortTypeName + " is not supported.");
	}

	public Iterator getClaims() throws CardException {
		if (claimValues_ == null) {
			retrieveClaims(null);
		}
		if (claimValues_ == null) {
			throw new AuthenticationRequiredException();
		}
		return claimValues_.values().iterator();
	}
	
	public ICredentialDescriptor[] getRequiredCredentials() {
		ICredentialDescriptor[] requiredCredentials = null;
		if (pinDigest_ != null) {
			requiredCredentials = new ICredentialDescriptor[1];
			requiredCredentials[0] = new BasicCredentialDescriptor("PinCodeCredential", "Please provide PIN code", new PinCodeCredential());
		}
		return requiredCredentials;
	}

	public int getPinStatus(){
		return pinStatus;
	}
	
	/**
	 * Validate a user input pin code
	 * 
	 * @param pinCode
	 * @return
	 * @throws UnsupportedEncodingException
	 * @throws NoSuchAlgorithmException
	 */
	public abstract boolean validatePINCode(String pinCode) throws Exception;
	
	/**
	 * If the card is not locked, this method protect the card by PIN code.
	 * 
	 * @param credential
	 * @throws Exception
	 */
	public abstract void lock(IPinCodeCredential credential) throws Exception;
	
	/**
	 * If the card is locked by PIN code, this method unlock the claim values
	 * and the master key. 
	 * 
	 * @param credential
	 * @throws Exception
	 */
	public abstract void unlock(IPinCodeCredential credential) throws Exception;
	
	/**
	 * This method should be used by
	 * {@link ICardProvider#getICardByCUID(javax.security.auth.callback.CallbackHandler, org.eclipse.higgins.icard.CUID, ICredential)}
	 * to retrieve fully loaded card instance.
	 * 
	 * This methid should use {@link InformationCard#setClaims(List)} to set
	 * cached list of claim values.
	 * 
	 * @param credential
	 *            the <code>ICredential</code> object used as an
	 *            authentication materials. Can be <code>null</code> if this
	 *            card is not pin protected.
	 * @throws AuthenticationRequiredException
	 *             when this card is pin protected but <code>null</code> was
	 *             passed as the <code>credential</code> parameter.
	 * @throws CardException
	 */
	protected abstract void retrieveClaims(ICredential credential) 
	throws AuthenticationRequiredException, AuthenticationException, CardException;
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#getType()
	 */
	public String getType() {
		return ICardConstants.I_CARD_TYPE_PERSONAL_CARDSPACE;
	}
}
