/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.icard.provider.cardspace.db.mysql;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.provider.cardspace.db.DAOStoredEvent;
import org.eclipse.higgins.icard.provider.cardspace.db.ICardContext;
import org.eclipse.higgins.icard.provider.cardspace.db.IConnectionFactory;
import org.eclipse.higgins.icard.provider.cardspace.db.IDAO;
import org.eclipse.higgins.icard.provider.cardspace.db.IDaoMCard;
import org.eclipse.higgins.icard.provider.cardspace.db.IDaoPCard;

import org.eclipse.higgins.cache.CacheException;
import org.eclipse.higgins.cache.CacheProviderFactory;
import org.eclipse.higgins.cache.api.ICache;
import org.eclipse.higgins.cache.api.ICacheKey;
import org.eclipse.higgins.cache.api.key.UserCacheKey;
import org.eclipse.higgins.cache.nocache.NOCache;

public class CardContext implements ICardContext {
	private Log log_ = LogFactory.getLog(CardContext.class);

	public final static String JNDI_CONTEXT = "jndi.context";

	public final static String SOURCE_NAME = "source.name";

	public final static String DRIVER_CLASS_NAME = "driver.classname";

	public final static String DB_USER_NAME = "db.username";

	public final static String DB_PASSWORD = "db.password";

	public final static String DB_URL = "db.url";

	public final static String USE_POOL = "db.pool";

	protected boolean isPoolUsed_;

	protected String driverName_;

	protected String dbUserName_;

	protected String dbPassword_;

	protected String dbURL_;

	protected String jndiContext_;

	protected String sourceName_;

	protected ConnectionFactory connectionFactory_ = null;

	private static ICache cardCache = CacheProviderFactory.getCacheProvider().getCache("cardcontext.mysql");
	private static boolean isNoCache = cardCache instanceof NOCache;

	private class CachedCards {
		private Map mCards = new HashMap();
		private Map pCards = new HashMap();

		public CachedCards() {
		}

		public Map getMCards() {
			return mCards;
		}

		public Map getPCards() {
			return pCards;
		}
	}

	public CardContext() {
	}

	public void init(Map config) throws CardException {
		String useConnectionPool = config.containsKey(USE_POOL) ? (String) config.get(USE_POOL) : null;
		isPoolUsed_ = ("yes".equalsIgnoreCase(useConnectionPool)) ? true : false;
		if (isPoolUsed_) {
			jndiContext_ = config.containsKey(JNDI_CONTEXT) ? (String) config.get(JNDI_CONTEXT) : null;
			sourceName_ = config.containsKey(SOURCE_NAME) ? (String) config.get(SOURCE_NAME) : null;
			if (jndiContext_ == null)
				throw new CardException("Property \"" + JNDI_CONTEXT + "\" not found.");
			if (sourceName_ == null)
				throw new CardException("Property \"" + SOURCE_NAME + "\" not found.");
			connectionFactory_ = new ConnectionFactory(jndiContext_, sourceName_);
		} else {
			driverName_ = config.containsKey(DRIVER_CLASS_NAME) ? (String) config.get(DRIVER_CLASS_NAME) : "com.mysql.jdbc.Driver";
			dbUserName_ = config.containsKey(DB_USER_NAME) ? (String) config.get(DB_USER_NAME) : null;
			dbPassword_ = config.containsKey(DB_PASSWORD) ? (String) config.get(DB_PASSWORD) : "";
			dbURL_ = config.containsKey(DB_URL) ? (String) config.get(DB_URL) : null;
			if (dbUserName_ == null)
				throw new CardException("Property \"" + DB_USER_NAME + "\" not found.");
			if (dbURL_ == null)
				throw new CardException("Property \"" + DB_URL + "\" not found.");
			connectionFactory_ = new ConnectionFactory(driverName_, dbUserName_, dbPassword_, dbURL_);
		}
	}

	public IConnectionFactory getConnectionFactory() {
		return connectionFactory_;
	}

	public IDaoMCard createMCard(String userID) {
		DaoMCard card = new DaoMCard(this, userID);
		return card;
	}

	public IDaoPCard createPCard(String userID) {
		DaoPCard card = new DaoPCard(this, userID);
		return card;
	}

	public IDaoMCard getMCard(String cardID, String userID) throws CardException {
		IDaoMCard card = null;
		if (isNoCache) {
			card = getMCardFromDB(cardID, userID);
		} else {
			ensureInCache(userID);
			card = getMCardFromCache(userID, cardID);
		}
		return card;
	}

	private IDaoMCard getMCardFromDB(String cardID, String userID) throws CardException {
		if (cardID == null || cardID.trim().length() == 0)
			throw new CardException("Can not return a card for null or empty card ID.");
		if (userID == null || userID.trim().length() == 0)
			throw new CardException("Can not return a card for null or empty user ID.");
		String query = "SELECT * FROM ManagedCard WHERE cardID_ = ? AND userID_ = ?";
		Connection con = null;
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			con = connectionFactory_.getConnection();
			ps = con.prepareStatement(query);
			ps.setString(1, cardID);
			ps.setString(2, userID);
			rs = ps.executeQuery();
			if (rs.next())
				return new DaoMCard(this, userID, rs, con);
			else
				return null;
		} catch (Exception e) {
			log_.error(e);
			throw new CardException(e);
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
			try {
				if (con != null && !con.isClosed()) {
					con.close();
					con = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	public ArrayList getMCards(String userID) throws CardException {
		if (isNoCache) {
			return getMCardsFromDB(userID);
		} else {
			ensureInCache(userID);
			CachedCards cards = getCachedCards(userID);
			return new ArrayList(cards.getMCards().values());
		}
	}

	private ArrayList getMCardsFromDB(String userID) throws CardException {
		if (userID == null || userID.trim().length() == 0)
			throw new CardException("Can not return a card for null or empty user ID.");
		String query = "SELECT * FROM ManagedCard WHERE userID_ = ? ORDER BY id_";
		ArrayList list = new ArrayList();
		Connection con = null;
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			con = connectionFactory_.getConnection();
			ps = con.prepareStatement(query);
			ps.setString(1, userID);
			rs = ps.executeQuery();
			while (rs.next()) {
				DaoMCard card = new DaoMCard(this, userID, rs, con);
				list.add(card);
			}
			return list;
		} catch (Exception e) {
			log_.error(e);
			throw new CardException(e);
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
			try {
				if (con != null && !con.isClosed()) {
					con.close();
					con = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	public IDaoPCard getPCard(String cardID, String userID) throws CardException {
		IDaoPCard card = null;
		if (isNoCache) {
			card = getPCardFromDB(cardID, userID);
		} else {
			ensureInCache(userID);
			card = getPCardFromCache(userID, cardID);
		}
		return card;
	}

	private IDaoPCard getPCardFromDB(String cardID, String userID) throws CardException {
		if (cardID == null || cardID.trim().length() == 0)
			throw new CardException("Can not return a card for null or empty card ID.");
		if (userID == null || userID.trim().length() == 0)
			throw new CardException("Can not return a card for null or empty user ID.");
		String query = "SELECT * FROM PersonalCard WHERE cardID_ = ? AND userID_ = ?";
		Connection con = null;
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			con = connectionFactory_.getConnection();
			ps = con.prepareStatement(query);
			ps.setString(1, cardID);
			ps.setString(2, userID);
			rs = ps.executeQuery();
			if (rs.next())
				return new DaoPCard(this, userID, rs, con);
			else
				return null;
		} catch (Exception e) {
			log_.error(e);
			throw new CardException(e);
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
			try {
				if (con != null && !con.isClosed()) {
					con.close();
					con = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	public ArrayList getPCards(String userID) throws CardException {
		if (isNoCache) {
			return getPCardsFromDB(userID);
		} else {
			ensureInCache(userID);
			CachedCards cards = getCachedCards(userID);
			return new ArrayList(cards.getPCards().values());
		}
	}

	private ArrayList getPCardsFromDB(String userID) throws CardException {
		if (userID == null || userID.trim().length() == 0)
			throw new CardException("Can not return a card for null or empty user ID.");
		String query = "SELECT * FROM PersonalCard WHERE userID_ = ? ORDER BY id_";
		ArrayList list = new ArrayList();
		Connection con = null;
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			con = connectionFactory_.getConnection();
			ps = con.prepareStatement(query);
			ps.setString(1, userID);
			rs = ps.executeQuery();
			while (rs.next()) {
				DaoPCard card = new DaoPCard(this, userID, rs, con);
				list.add(card);
			}
			return list;
		} catch (Exception e) {
			log_.error(e);
			throw new CardException(e);
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
			try {
				if (con != null && !con.isClosed()) {
					con.close();
					con = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	public void objectStored(DAOStoredEvent event) {
		IDAO obj = event.getObject();
		if (obj.getState() == IDAO.DELETED_OBJ) {
			if (!isNoCache) {
				if (obj instanceof IDaoPCard)
					removeDeletedPCardsFromCache((IDaoPCard) obj);
				else if (obj instanceof IDaoMCard)
					removeDeletedMCardsFromCache((IDaoMCard) obj);	
			}
		}
		if (obj.getState() == IDAO.STORED_OBJ) {
			if (!isNoCache) {
				if (obj instanceof DaoPCard) {
					DaoPCard card = (DaoPCard) obj;
					String userID = card.getUserID();
					try {
						ensureInCache(userID);
						updateCache(card, userID);
					} catch (CardException e) {
						log_.error(e);
					}	
				} else if (obj instanceof DaoMCard) {
					try {
						DaoMCard card = (DaoMCard) obj;
						String userID = card.getUserID();
						ensureInCache(userID);
						updateCache(card, userID);
					} catch (CardException e) {
						log_.error(e);
					}
				}
			}
		}
	}

	private void removeDeletedMCardsFromCache(IDaoMCard card) {
		try {
			String userID = card.getUserID();
			ensureInCache(userID);
			CachedCards cards = getCachedCards(userID);
			Object removedCard = cards.getMCards().remove(card.getCardID());
			if (removedCard != null) {
				log_.debug("M-card has been removed from cache");
				updateCache(cards, userID);
			}
		} catch (CardException e) {
			log_.error(e);
		}
	}

	private void removeDeletedPCardsFromCache(IDaoPCard card) {
		try {
			String userID = card.getUserID();
			ensureInCache(userID);
			CachedCards cards = getCachedCards(userID);
			Object removedCard = cards.getPCards().remove(card.getCardID());
			if (removedCard != null) {
				log_.debug("M-card has been removed from cache");
				updateCache(cards, userID);
			}
		} catch (CardException e) {
			log_.error(e);
		}
	}

	/**
	 * Preloads all Cards of current user and put them in cache
	 * 
	 * @param dsUserProfile
	 * @param cardInformationBinding
	 * @throws CardException
	 */
	private void ensureInCache(String userID) throws CardException {
		ICacheKey key = new UserCacheKey(userID);
		if (!cardCache.isKeyInCache(key)) {
			log_.info("Load all managed ICards by userID:" + key.getKey().toString());
			List lstMCards = getMCardsFromDB(userID);
			log_.info("Load all personal ICards by userID:" + key.getKey().toString());
			List lstPCards = getPCardsFromDB(userID);
			CachedCards cachedCards = new CachedCards();
			Map m = cachedCards.getMCards();
			for (Iterator iterator = lstMCards.iterator(); iterator.hasNext();) {
				DaoMCard card = (DaoMCard) iterator.next();
				m.put(card.getCardID(), card);
			}
			m = cachedCards.getPCards();
			for (Iterator iterator = lstPCards.iterator(); iterator.hasNext();) {
				DaoPCard card = (DaoPCard) iterator.next();
				m.put(card.getCardID(), card);
			}
			cardCache.put(key, cachedCards); // store the hash map of cards
			log_.info(key.getKey().toString() + " cards cached.");
		}
	}

	private CachedCards getCachedCards(String userId) {
		ICacheKey key = new UserCacheKey(userId);
		if (cardCache.isKeyInCache(key)) {
			return (CachedCards) cardCache.get(key);
		} else
			throw new CacheException("Card Cache for user " + userId + " was not inited!");
	}

	private IDaoMCard getMCardFromCache(String userId, String cardID) {
		CachedCards cachedCards = getCachedCards(userId);
		return (DaoMCard) cachedCards.getMCards().get(cardID);
	}

	private IDaoPCard getPCardFromCache(String userId, String cardID) {
		CachedCards cachedCards = getCachedCards(userId);
		return (DaoPCard) cachedCards.getPCards().get(cardID);
	}

	private void updateCache(CachedCards cards, String userId) {
		ICacheKey key = new UserCacheKey(userId);
		cardCache.put(key, cards);
	}

	private void updateCache(DaoMCard crd, String userId) {
		CachedCards cards = getCachedCards(userId);
		cards.getMCards().put(crd.getCardID(), crd);
		updateCache(cards, userId);
	}

	private void updateCache(DaoPCard crd, String userId) {
		CachedCards cards = getCachedCards(userId);
		cards.getPCards().put(crd.getCardID(), crd);
		updateCache(cards, userId);
	}

}
