/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.icard.provider.cardspace.db.mysql;

import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Arrays;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.InvalidStateException;
import org.eclipse.higgins.icard.provider.cardspace.db.IDAO;
import org.eclipse.higgins.icard.provider.cardspace.db.IDaoInformationCardExtension;
import org.eclipse.higgins.icard.provider.cardspace.db.IDaoMCard;
import org.eclipse.higgins.icard.provider.cardspace.db.IDaoSupportedClaimType;
import org.eclipse.higgins.icard.provider.cardspace.db.IDaoSupportedTokenType;
import org.eclipse.higgins.icard.provider.cardspace.db.IDaoTokenService;

public class DaoMCard extends DAO implements IDaoMCard {

	private Log log_ = LogFactory.getLog(DaoMCard.class);

	protected CardContext context_;

	protected String userID_;

	protected String cardID_ = null;

	protected String language_ = null;

	protected String name_ = null;

	protected int version_ = 0;

	protected String issuer_ = null;

	protected byte[] issuerID_ = null;

	protected String issuerName_ = null;

	protected byte[] image_ = null;

	protected String imageType_ = null;

	protected Date timeIssued_ = null;

	protected Date timeExpires_ = null;

	protected Date timeLastUpdated_ = null;

	protected byte[] hashSalt_ = null;

	protected byte[] masterKey_ = null;

	protected byte[] initialMasterKey_ = null;

	protected Boolean requireAppliesTo_ = null;

	protected String privacyNotice_ = null;

	protected ArrayList tokenServiceList_ = new ArrayList();

	protected ArrayList supportedTokenTypeList_ = new ArrayList();

	protected ArrayList supportedClaimTypeList_ = new ArrayList();

	protected ArrayList extensionList_ = new ArrayList();

	protected byte[] initialImage_ = null;

	protected String initialName_ = null;

	protected String initialImageType_ = null;

	public DaoMCard(CardContext context, String userID) {
		state_ = NEW_OBJ;
		userID_ = userID;
		context_ = context;
		connectionFactory_ = context.getConnectionFactory();
		addListener(context);
	}

	public DaoMCard(CardContext context, String userID, ResultSet rs, Connection con) throws SQLException {
		state_ = STORED_OBJ;
		userID_ = userID;
		context_ = context;
		connectionFactory_ = context.getConnectionFactory();
		addListener(context);
		init(rs, con);
	}

	public void init(ResultSet rs, Connection con) throws SQLException {
		id_ = rs.getInt("id_");
		cardID_ = rs.getString("cardID_");
		language_ = rs.getString("language_");
		initialName_ = name_ = rs.getString("name_");
		version_ = rs.getInt("version_");
		issuer_ = rs.getString("issuer_");
		issuerName_ = rs.getString("issuerName_");
		initialImage_ = image_ = rs.getBytes("image_");
		initialImageType_ = imageType_ = rs.getString("imageType_");
		timeIssued_ = rs.getDate("timeIssued_");
		timeExpires_ = rs.getDate("timeExpires_");
		timeLastUpdated_ = rs.getDate("timeLastUpdated_");
		hashSalt_ = rs.getBytes("hashSalt_");
		initialMasterKey_ = masterKey_ = rs.getBytes("masterKey_");
		issuerID_ = rs.getBytes("issuerID_");
		BigDecimal rat = rs.getBigDecimal("requireAppliesTo_");
		if (rat != null)
			requireAppliesTo_ = (rat.intValue() == 0) ? Boolean.TRUE : Boolean.FALSE;
		else
			requireAppliesTo_ = null;
		privacyNotice_ = rs.getString("privacyNotice_");
		initTokenServiceList(con);
		initSupportedTokenTypeList(con);
		initSupportedClaimTypeList(con);
		initExtensionList(con);
	}

	public int getState() {
		return state_;
	}

	public String getCardID() {
		return cardID_;
	}

	public byte[] getHashSalt() {
		return hashSalt_;
	}

	public byte[] getImage() {
		return image_;
	}

	public String getImageType() {
		return imageType_;
	}

	public String getIssuer() {
		return issuer_;
	}

	public byte[] getIssuerID() {
		return issuerID_;
	}

	public String getIssuerName() {
		return issuerName_;
	}

	public String getLanguage() {
		return language_;
	}

	public byte[] getMasterKey() {
		return masterKey_;
	}

	public String getName() {
		return name_;
	}

	public String getPrivacyNotice() {
		return privacyNotice_;
	}

	public Boolean getRequireAppliesTo() {
		return requireAppliesTo_;
	}

	public ArrayList getSupportedClaimType() {
		return supportedClaimTypeList_;
	}

	public ArrayList getSupportedTokenType() {
		return supportedTokenTypeList_;
	}

	public Date getTimeExpires() {
		return timeExpires_;
	}

	public Date getTimeIssued() {
		return timeIssued_;
	}

	public Date getTimeLastUpdated() {
		return timeLastUpdated_;
	}

	public int getVersion() {
		return version_;
	}

	public void setCardID(String cardID) throws InvalidStateException {
		if (id_ != -1)
			throw new InvalidStateException("Unsupported operation for existent card");
		cardID_ = cardID;
	}

	public void setHashSalt(byte[] hashSalt) throws InvalidStateException {
		if (id_ != -1)
			throw new InvalidStateException("Unsupported operation for existent card");
		hashSalt_ = hashSalt;
	}

	public void setImage(byte[] image) throws CardException {
		image_ = image;
		setChanged();
	}

	public void setImageType(String imageType) throws CardException {
		imageType_ = imageType;
		setChanged();
	}

	public void setIssuer(String issuer) throws InvalidStateException {
		if (id_ != -1)
			throw new InvalidStateException("Unsupported operation for existent card");
		issuer_ = issuer;
	}

	public void setIssuerID(byte[] issuerID) throws CardException {
		if (id_ != -1)
			throw new InvalidStateException("Unsupported operation for existent card");
		issuerID_ = issuerID;
	}

	public void setIssuerName(String issuerName) throws InvalidStateException {
		if (id_ != -1)
			throw new InvalidStateException("Unsupported operation for existent card");
		issuerName_ = issuerName;
	}

	public void setLanguage(String language) throws InvalidStateException {
		if (id_ != -1)
			throw new InvalidStateException("Unsupported operation for existent card");
		language_ = language;
	}

	public void setMasterKey(byte[] masterKey) throws InvalidStateException {
		if (id_ != -1)
			throw new InvalidStateException("Unsupported operation for existent card");
		masterKey_ = masterKey;
	}

	public void setName(String name) throws CardException {
		name_ = name;
		setChanged();
	}

	public void setPrivacyNotice(String privacyNotice) throws InvalidStateException {
		if (id_ != -1)
			throw new InvalidStateException("Unsupported operation for existent card");
		privacyNotice_ = privacyNotice;
	}

	public void setRequireAppliesTo(Boolean requireAppliesTo) throws InvalidStateException {
		if (id_ != -1)
			throw new InvalidStateException("Unsupported operation for existent card");
		requireAppliesTo_ = requireAppliesTo;
	}

	public void setTimeExpires(Date timeExpires) throws InvalidStateException {
		if (id_ != -1)
			throw new InvalidStateException("Unsupported operation for existent card");
		timeExpires_ = checkMaxSQLDate(timeExpires);
	}

	public void setTimeIssued(Date timeIssued) throws InvalidStateException {
		if (id_ != -1)
			throw new InvalidStateException("Unsupported operation for existent card");
		timeIssued_ = timeIssued;
	}

	public void setTimeLastUpdated(Date timeLastUpdated) throws InvalidStateException {
		timeLastUpdated_ = timeLastUpdated;
	}

	public void setVersion(int version) throws InvalidStateException {
		if (id_ != -1)
			throw new InvalidStateException("Unsupported operation for existent card");
		version_ = version;
	}

	public void addSupportedClaimType(IDaoSupportedClaimType supportedClaimType) throws CardException {
		if (id_ != -1)
			throw new InvalidStateException("Unsupported operation for existent card");
		supportedClaimTypeList_.add(supportedClaimType);
	}

	public void addSupportedTokenType(IDaoSupportedTokenType supportedTokenType) throws CardException {
		if (id_ != -1)
			throw new InvalidStateException("Unsupported operation for existent card");
		supportedTokenTypeList_.add(supportedTokenType);
	}

	public void addTokenService(IDaoTokenService tokenService) throws CardException {
		if (id_ != -1)
			throw new InvalidStateException("Unsupported operation for existent card");
		tokenServiceList_.add(tokenService);
	}

	public IDaoSupportedClaimType createSupportedClaimType() {
		return new SupportedClaimType(this);
	}

	public IDaoSupportedTokenType createSupportedTokenType() {
		return new SupportedTokenType(this);
	}

	public IDaoTokenService createTokenService() {
		return new TokenService(this);
	}

	public ArrayList getTokenServiceList() {
		return tokenServiceList_;
	}

	public ArrayList getSupportedClaimTypeList() {
		return supportedClaimTypeList_;
	}

	public ArrayList getSupportedTokenTypeList() {
		return supportedTokenTypeList_;
	}

	private void initExtensionList(Connection con) throws SQLException {
		extensionList_.clear();
		String query = "SELECT * FROM MCardExtension WHERE cardID_ = ?";
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, id_);
			rs = ps.executeQuery();
			while (rs.next()) {
				DaoInformationCardExtension ts = new DaoInformationCardExtension(this, rs, con);
				extensionList_.add(ts);
			}
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	private void initTokenServiceList(Connection con) throws SQLException {
		tokenServiceList_.clear();
		String query = "SELECT * FROM TokenService WHERE cardID_ = ?";
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, id_);
			rs = ps.executeQuery();
			while (rs.next()) {
				TokenService ts = new TokenService(this, rs, con);
				tokenServiceList_.add(ts);
			}
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	private void initSupportedTokenTypeList(Connection con) throws SQLException {
		supportedTokenTypeList_.clear();
		String query = "SELECT stt.id_, stt.tokenTypeID_, tt.type_ FROM SupportedTokenType AS stt, TokenType AS tt WHERE stt.cardID_ = ? AND stt.tokenTypeID_ = tt.id_ ORDER BY stt.id_";
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, id_);
			rs = ps.executeQuery();
			while (rs.next()) {
				SupportedTokenType tt = new SupportedTokenType(this, rs, con);
				supportedTokenTypeList_.add(tt);
			}
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	private void initSupportedClaimTypeList(Connection con) throws SQLException {
		supportedClaimTypeList_.clear();
		String query = "SELECT sct.id_, sct.claimTypeID_, ct.type_ , ct.typeLocalName_ , ct.displayName_ , ct.description_ FROM SupportedClaimType AS sct, ClaimType AS ct WHERE sct.cardID_ = ? AND sct.claimTypeID_ = ct.id_ ORDER BY sct.id_";
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, id_);
			rs = ps.executeQuery();
			while (rs.next()) {
				SupportedClaimType ct = new SupportedClaimType(this, rs, con);
				supportedClaimTypeList_.add(ct);
			}
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	protected void insert(Connection con) throws Exception {
		String query = "INSERT INTO ManagedCard (cardID_, language_, name_, version_, issuer_, issuerName_, image_, imageType_, timeIssued_, timeExpires_, timeLastUpdated_, hashSalt_, masterKey_, requireAppliesTo_, privacyNotice_, userID_, issuerID_) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query, Statement.RETURN_GENERATED_KEYS);
			ps.setString(1, cardID_);
			ps.setString(2, language_);
			ps.setString(3, name_);
			ps.setInt(4, version_);
			ps.setString(5, issuer_);
			ps.setString(6, issuerName_);
			ps.setBytes(7, image_);
			ps.setString(8, imageType_);
			ps.setDate(9, timeIssued_);
			ps.setDate(10, timeExpires_);
			ps.setDate(11, timeLastUpdated_);
			ps.setBytes(12, hashSalt_);
			ps.setBytes(13, masterKey_);
			ps.setBigDecimal(14, boolToInt(requireAppliesTo_));
			ps.setString(15, privacyNotice_);
			ps.setString(16, userID_);
			ps.setBytes(17, issuerID_);
			ps.execute();
			rs = ps.getGeneratedKeys();
			if (rs.next())
				id_ = rs.getInt(1);
			else
				throw new CardException("Prepared statement did not return primary key for inserted row.");
		} catch (SQLException e) {
			log_.error(e);
			throw new CardException(e);
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	protected void update(Connection con) throws SQLException {
		updateName(con);
		updateImage(con);
		updateImageType(con);
		updateTimeLastUpdated(con);
	}

	protected void updateName(Connection con) throws SQLException {
		if (initialName_ == null && name_ == null)
			return;
		String n1 = (initialName_ == null) ? "" : initialName_;
		String n2 = (name_ == null) ? "" : name_;
		if (n1.equals(n2) == false) {
			String query = "UPDATE ManagedCard SET name_ = ? WHERE id_ = ?";
			PreparedStatement ps = null;
			try {
				ps = con.prepareStatement(query);
				ps.setString(1, name_);
				ps.setInt(2, id_);
				ps.execute();
			} finally {
				try {
					if (ps != null) {
						ps.close();
						ps = null;
					}
				} catch (SQLException e) {
					log_.error(e);
				}
			}
		}
	}

	protected void updateImage(Connection con) throws SQLException {
		if (Arrays.equals(initialImage_, image_))
			return;
		String query = "UPDATE ManagedCard SET image_ = ? WHERE id_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setBytes(1, image_);
			ps.setInt(2, id_);
			ps.execute();
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	private void removeDeletedExtensions() {
		ArrayList deleted = new ArrayList();
		for (int i = 0, j = extensionList_.size(); i < j; i++) {
			IDAO obj = (IDAO) extensionList_.get(i);
			if (obj.getState() == IDAO.DELETED_OBJ) {
				deleted.add(obj);
			}
		}
		extensionList_.removeAll(deleted);
	}

	private void updateInitialValues() {
		initialName_ = name_;
		initialImage_ = image_;
		initialImageType_ = imageType_;
		initialMasterKey_ = masterKey_;
	}

	public void commitState() {
		super.commitState();
		removeDeletedExtensions();
		updateInitialValues();
	}
	
	
	protected void updateImageType(Connection con) throws SQLException {
		if (initialImageType_ == null && imageType_ == null)
			return;
		String n1 = (initialImageType_ == null) ? "" : initialImageType_;
		String n2 = (imageType_ == null) ? "" : imageType_;
		if (n1.equals(n2) == false) {
			String query = "UPDATE ManagedCard SET imageType_ = ? WHERE id_ = ?";
			PreparedStatement ps = null;
			try {
				ps = con.prepareStatement(query);
				ps.setString(1, imageType_);
				ps.setInt(2, id_);
				ps.execute();
			} finally {
				try {
					if (ps != null) {
						ps.close();
						ps = null;
					}
				} catch (SQLException e) {
					log_.error(e);
				}
			}
		}
	}

	protected void updateTimeLastUpdated(Connection con) throws SQLException {
		String query = "UPDATE ManagedCard SET timeLastUpdated_ = ? WHERE id_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setDate(1, timeLastUpdated_);
			ps.setInt(2, id_);
			ps.execute();
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	protected void delete(Connection con) throws SQLException {
		String query = "DELETE FROM ManagedCard WHERE id_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, id_);
			ps.execute();
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	public ArrayList getChildren() {
		ArrayList children = new ArrayList();
		children.addAll(tokenServiceList_);
		children.addAll(supportedClaimTypeList_);
		children.addAll(supportedTokenTypeList_);
		children.addAll(extensionList_);
		return children;
	}

	private BigDecimal boolToInt(Boolean bool) {
		if (bool != null)
			return (bool.booleanValue()) ? new BigDecimal(0) : new BigDecimal(1);
		else
			return null;
	}

	public String getUserID() {
		return userID_;
	}

	public void addInformationCardExtension(IDaoInformationCardExtension extension) throws CardException {
		if (extension == null)
			throw new CardException("Parameter \"extension\" is null.");
		extensionList_.add(extension);
		setChanged();
	}

	public IDaoInformationCardExtension createInformationCardExtension() {
		return new DaoInformationCardExtension(this);
	}

	public ArrayList getExtensions() {
		return extensionList_;
	}

}