/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.icard.provider.cardspace.db.mysql;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Hashtable;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.InvalidStateException;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.SelfIssuedCardClaims;
import org.eclipse.higgins.icard.provider.cardspace.db.IDaoPCardClaim;

public class DaoPCardClaim extends DAO implements IDaoPCardClaim {

	private Log log_ = LogFactory.getLog(DaoPCard.class);

	private final static Hashtable typeToId_;

	private final static Hashtable idToType_;

	private int typeIndex_ = -1;

	private String type_ = null;

	private String value_ = null;

	private DaoPCard card_ = null;

	public DaoPCardClaim(DaoPCard card) {
		state_ = NEW_OBJ;
		connectionFactory_ = card.getConnectionFactory();
		card_ = card;
	}

	public DaoPCardClaim(DaoPCard card, ResultSet rs, Connection con)
			throws SQLException, CardException {
		state_ = STORED_OBJ;
		connectionFactory_ = card.getConnectionFactory();
		card_ = card;
		init(rs, con);
	}

	private void init(ResultSet rs, Connection con) throws SQLException,
			CardException {
		id_ = rs.getInt("id_");
		typeIndex_ = rs.getInt("claimTypeID_");
		value_ = rs.getString("claimValue_");
		String id = String.valueOf(typeIndex_);
		if (idToType_.containsKey(id))
			type_ = (String) idToType_.get(id);
		else
			throw new CardException(
					"Can not find claim type in dictionary by id = " + id);
	}

	protected ArrayList getChildren() {
		return null;
	}

	public int getCardID() {
		return card_.getID();
	}

	public String getType() {
		return type_;
	}

	public int getTypeID() {
		return typeIndex_;
	}

	public String getValue() {
		return value_;
	}

	public void setType(String type) throws CardException {
		if (id_ != -1)
			throw new InvalidStateException(
					"Unsupported operation for existent card");
		if (type == null)
			throw new CardException("Parameter \"type\" is null.");
		if (typeToId_.containsKey(type)) {
			String id = (String) typeToId_.get(type);
			try {
				typeIndex_ = Integer.parseInt(id);
			} catch (NumberFormatException e) {
				log_.error(e);
				throw new CardException(e);
			}
			type_ = type;
		} else
			throw new CardException(
					"SelfSigned card does not support claim value type = "
							+ type);
	}

	public void setValue(String value) throws CardException {
		value_ = value;
		setChanged();
	}

	protected void delete(Connection con) throws Exception {
		String query = "DELETE FROM PersonalCardClaim WHERE id_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, id_);
			ps.execute();
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	protected void insert(Connection con) throws Exception {
		if (typeIndex_ == -1)
			throw new CardException(
					"Can not store PersonalCardClaim because its type of claim undefined.");
		if (value_ == null || value_.trim().length() == 0) {
			log_.info("Claim value with type = " + type_
					+ " was skipped because its value is empty.");
			return;
		}
		int cardID = card_.getID();
		if (cardID == -1)
			throw new CardException(
					"Can not store PersonalCardClaim for non-stored ICard.");

		String query = "INSERT INTO PersonalCardClaim (cardID_, claimTypeID_, claimValue_) VALUES (?, ?, ?)";
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query, Statement.RETURN_GENERATED_KEYS);
			ps.setInt(1, cardID);
			ps.setInt(2, typeIndex_);
			ps.setString(3, value_);
			ps.execute();
			rs = ps.getGeneratedKeys();
			if (rs.next()) {
				id_ = rs.getInt(1);
			} else
				throw new CardException(
						"Prepared statement did not return primary key for inserted row.");
		} catch (SQLException e) {
			log_.error(e);
			throw new CardException(e);
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	protected void update(Connection con) throws Exception {
		String query = "UPDATE PersonalCardClaim SET claimValue_ = ? WHERE id_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setString(1, value_);
			ps.setInt(2, id_);
			ps.execute();
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	static {
		typeToId_ = SelfIssuedCardClaims.getSupportedTypesHash();
		idToType_ = new Hashtable();
		Enumeration keys = typeToId_.keys();
		while (keys.hasMoreElements()) {
			Object key = keys.nextElement();
			Object val = typeToId_.get(key);
			idToType_.put(val, key);
		}
	}

}
