/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.icard.provider.cardspace.db.mysql;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.InvalidStateException;
import org.eclipse.higgins.icard.provider.cardspace.db.IDaoSupportedClaimType;

public class SupportedClaimType extends DAO implements IDaoSupportedClaimType {
	private Log log_ = LogFactory.getLog(SupportedClaimType.class);

	protected DaoMCard card_ = null;

	protected int claimTypeID_ = -1;

	protected String description_ = null;

	protected String displayName_ = null;

	protected String type_ = null;

	protected String typeLocalName_ = null;

	public SupportedClaimType(DaoMCard card) {
		state_ = NEW_OBJ;
		connectionFactory_ = card.getConnectionFactory();
		card_ = card;
	}

	public SupportedClaimType(DaoMCard card, ResultSet rs, Connection con) throws SQLException {
		state_ = STORED_OBJ;
		connectionFactory_ = card.getConnectionFactory();
		card_ = card;
		init(rs, con);
	}

	private void init(ResultSet rs, Connection con) throws SQLException {
		id_ = rs.getInt("id_");
		claimTypeID_ = rs.getInt("claimTypeID_");
		type_ = rs.getString("type_");
		description_ = rs.getString("description_");
		displayName_ = rs.getString("displayName_");
		typeLocalName_ = rs.getString("typeLocalName_");
	}

	public int getCardID() {
		return card_.getID();
	}

	public int getTypeID() {
		return id_;
	}

	public String getDescription() {
		return description_;
	}

	public String getDisplayName() {
		return displayName_;
	}

	public String getType() {
		return type_;
	}

	public String getTypeLocalName() {
		return typeLocalName_;
	}

	public void setDescription(String description) throws CardException {
		if (id_ != -1)
			throw new InvalidStateException("Unsupported operation for existent card");
		description_ = description;
	}

	public void setDisplayName(String displayName) throws CardException {
		if (id_ != -1)
			throw new InvalidStateException("Unsupported operation for existent card");
		displayName_ = displayName;
	}

	public void setType(String type) throws CardException {
		if (id_ != -1)
			throw new InvalidStateException("Unsupported operation for existent card");
		type_ = type;
	}

	public void setTypeLocalName(String typeLocalName) throws CardException {
		if (id_ != -1)
			throw new InvalidStateException("Unsupported operation for existent card");
		typeLocalName_ = typeLocalName;
	}

	private int initClaimType(Connection con) throws CardException {
		int tokenTypeID = getClaimType(con);
		if (tokenTypeID == -1)
			tokenTypeID = insertClaimType(con);
		return tokenTypeID;
	}

	private int getClaimType(Connection con) throws CardException {
		String query = "SELECT id_ FROM ClaimType WHERE type_ = ? AND typeLocalName_ = ? AND displayName_ = ? AND description_ = ?";
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query);
			ps.setString(1, type_);
			ps.setString(2, typeLocalName_);
			ps.setString(3, displayName_);
			ps.setString(4, description_);
			rs = ps.executeQuery();
			if (rs.next()) {
				int id = rs.getInt("id_");
				return id;
			} else
				return -1;
		} catch (SQLException e) {
			log_.error(e);
			throw new CardException(e);
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	private int insertClaimType(Connection con) throws CardException {
		String query = "INSERT INTO ClaimType (type_, typeLocalName_, displayName_, description_) VALUES (?, ?, ?, ?)";
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query, Statement.RETURN_GENERATED_KEYS);
			ps.setString(1, type_);
			ps.setString(2, typeLocalName_);
			ps.setString(3, displayName_);
			ps.setString(4, description_);
			ps.execute();
			rs = ps.getGeneratedKeys();
			if (rs.next()) {
				int id = rs.getInt(1);
				return id;
			} else
				throw new CardException("Prepared statement did not return primary key for inserted row.");
		} catch (SQLException e) {
			log_.error(e);
			throw new CardException(e);
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	private void insertSupportedClaimType(int cardID, int claimTypeID, Connection con) throws Exception {
		String query = "INSERT INTO SupportedClaimType (cardID_, claimTypeID_) VALUES (?, ?)";
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query, Statement.RETURN_GENERATED_KEYS);
			ps.setInt(1, cardID);
			ps.setInt(2, claimTypeID);
			ps.execute();
			rs = ps.getGeneratedKeys();
			if (rs.next())
				id_ = rs.getInt(1);
			else
				throw new CardException("Prepared statement did not return primary key for inserted row.");
		} catch (SQLException e) {
			log_.error(e);
			throw new CardException(e);
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	protected void delete(Connection con) throws Exception {
		String query = "DELETE FROM SupportedClaimType WHERE id_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, id_);
			ps.execute();
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	protected ArrayList getChildren() {
		// TODO Auto-generated method stub
		return null;
	}

	protected void insert(Connection con) throws Exception {
		if (type_ == null || type_.trim().length() == 0)
			throw new CardException("Claim type is empty.");
		int cardID = card_.getID();
		if (cardID == -1)
			throw new CardException("Can not store SupportedClaimType for non-stored ICard.");
		claimTypeID_ = initClaimType(con);
		if (claimTypeID_ == -1)
			throw new CardException("Can not store SupportedClaimType for non-initialized claimTypeID.");
		insertSupportedClaimType(cardID, claimTypeID_, con);
	}

	protected void update(Connection con) throws SQLException {
	}

}
