/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.icard.provider.cardspace.db.mysql;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.InvalidStateException;
import org.eclipse.higgins.icard.provider.cardspace.db.IDaoSupportedTokenType;

public class SupportedTokenType extends DAO implements IDaoSupportedTokenType {
	private Log log_ = LogFactory.getLog(SupportedTokenType.class);

	protected DaoMCard card_ = null;

	protected String type_ = null;

	protected int tokenTypeID_ = -1;

	public SupportedTokenType(DaoMCard card) {
		state_ = NEW_OBJ;
		connectionFactory_ = card.getConnectionFactory();
		card_ = card;
	}

	public SupportedTokenType(DaoMCard card, ResultSet rs, Connection con)
	        throws SQLException {
		state_ = STORED_OBJ;
		connectionFactory_ = card.getConnectionFactory();
		card_ = card;
		init(rs, con);
	}

	private void init(ResultSet rs, Connection con) throws SQLException {
		id_ = rs.getInt("id_");
		tokenTypeID_ = rs.getInt("tokenTypeID_");
		type_ = rs.getString("type_");
	}

	public int getCardID() {
		return card_.getID();
	}

	public int getTypeID() {
		return tokenTypeID_;
	}

	public String getType() {
		return type_;
	}

	public void setType(String type) throws CardException {
		if (id_ != -1)
			throw new InvalidStateException("Unsupported operation for existent card");
		type_ = type;
	}

	private int initTokenType(Connection con) throws CardException {
		int tokenTypeID = getTokenType(con);
		if (tokenTypeID == -1)
			tokenTypeID = insertTokenType(con);
		return tokenTypeID;
	}

	private int getTokenType(Connection con) throws CardException {
		String query = "SELECT id_ FROM TokenType WHERE type_ = ?";
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query);
			ps.setString(1, type_);

			rs = ps.executeQuery();
			if (rs.next()) {
				int id = rs.getInt("id_");
				return id;
			} else
				return -1;
		} catch (SQLException e) {
			log_.error(e);
			throw new CardException(e);
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	private int insertTokenType(Connection con) throws CardException {
		String query = "INSERT INTO TokenType (type_) VALUES (?)";
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query, Statement.RETURN_GENERATED_KEYS);
			ps.setString(1, type_);
			ps.execute();
			rs = ps.getGeneratedKeys();
			if (rs.next()) {
				int id = rs.getInt(1);
				return id;
			} else
				throw new CardException("Prepared statement did not return primary key for inserted row.");
		} catch (SQLException e) {
			log_.error(e);
			throw new CardException(e);
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	private void insertSupportedTokenType(int cardID, int tokenTypeID, Connection con) throws Exception {
		String query = "INSERT INTO SupportedTokenType (cardID_, tokenTypeID_) VALUES (?, ?)";
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query, Statement.RETURN_GENERATED_KEYS);
			ps.setInt(1, cardID);
			ps.setInt(2, tokenTypeID);
			ps.execute();
			rs = ps.getGeneratedKeys();
			if (rs.next())
				id_ = rs.getInt(1);
			else
				throw new CardException("Prepared statement did not return primary key for inserted row.");
		} catch (SQLException e) {
			log_.error(e);
			throw new CardException(e);
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	protected void delete(Connection con) throws Exception {
		String query = "DELETE FROM SupportedTokenType WHERE id_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, id_);
			ps.execute();
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
    }

	protected ArrayList getChildren() {
	    return null;
    }

	protected void insert(Connection con) throws Exception {
		if (type_ == null || type_.trim().length() == 0)
			throw new CardException("Token type is empty.");
		int cardID = card_.getID();
		if (cardID == -1)
			throw new CardException("Can not store SupportedTokenType for non-stored ICard.");
		tokenTypeID_ = initTokenType(con);
		if (tokenTypeID_ == -1)
			throw new CardException("Can not store SupportedTokenType for non-initialized tokenTypeID.");
		insertSupportedTokenType(cardID, tokenTypeID_, con);
    }

	protected void update(Connection con) throws Exception {
	    // TODO Auto-generated method stub
	    
    }

}
