/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.icard.provider.cardspace.db.mysql;

import java.sql.Connection;
import java.sql.Date;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Vector;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.provider.cardspace.db.DAOStoredEvent;
import org.eclipse.higgins.icard.provider.cardspace.db.IConnectionFactory;
import org.eclipse.higgins.icard.provider.cardspace.db.IDAO;
import org.eclipse.higgins.icard.provider.cardspace.db.IDAOListener;

public abstract class DAO implements IDAO {
	private Log log_ = LogFactory.getLog(DAO.class);

	private static final long maxMySQLDate_ = 253402253999000l;

	protected int state_ = NEW_OBJ;

	protected int id_ = -1;

	protected IConnectionFactory connectionFactory_ = null;

	protected Vector listeners_ = new Vector();

	/**
	 * Change the state of children objects after persisting
	 */
	private void commitChildrenState() {
		ArrayList children = getChildren();
		if (children != null) {
			for (int i = 0, j = children.size(); i < j; i++) {
				DAO obj = (DAO) children.get(i);
				obj.commitState();
			}
		}
	}

	/**
	 * Change the state of the object after persisting
	 */
	protected void commitState() {
		switch (state_) {
		case NEW_OBJ:
			if (id_ != -1) // do not commit skipped object
				state_ = STORED_OBJ;
			break;
		case CHANGED_OBJ:
			state_ = STORED_OBJ;
			break;
		case PRE_DELETED_NEW_OBJ:
			state_ = DELETED_OBJ;
			break;
		case PRE_DELETED_STORED_OBJ:
			state_ = DELETED_OBJ;
			break;
		}
		commitChildrenState();
	}

	/**
	 * @return Connection factory associated with this DAO object
	 */
	public IConnectionFactory getConnectionFactory() {
		return connectionFactory_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.provider.cardspace.db.IDAO#getID()
	 */
	public int getID() {
		return id_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.provider.cardspace.db.IDAO#getState()
	 */
	public int getState() {
		return state_;
	}

	/**
	 * @return <code>true</code> if the object requires changes to database,
	 *         otherwise returns <code>false</code>
	 */
	public boolean isChanged() {
		return (state_ == NEW_OBJ || state_ == CHANGED_OBJ || state_ == PRE_DELETED_STORED_OBJ);
	}

	/**
	 * This method should be invoked in methods of descendants of this class
	 * which change the state of DAO object
	 * 
	 * @throws CardException
	 */
	protected void setChanged() throws CardException {
		if (state_ == PRE_DELETED_NEW_OBJ || state_ == PRE_DELETED_STORED_OBJ)
			throw new CardException("Can not change data for predeleted object.");
		if (state_ == DELETED_OBJ)
			throw new CardException("Can not change data for deleted object.");
		if (state_ == STORED_OBJ)
			state_ = CHANGED_OBJ;
	}

	/**
	 * This method need to be invoked to delete the object
	 * 
	 * @throws CardException
	 */
	public void setDeleteState() {
		if (state_ == NEW_OBJ)
			state_ = PRE_DELETED_NEW_OBJ;
		if (state_ == STORED_OBJ || state_ == CHANGED_OBJ)
			state_ = PRE_DELETED_STORED_OBJ;
		ArrayList children = getChildren();
		if (children != null) {
			for (int i = 0, j = children.size(); i < j; i++) {
				DAO obj = (DAO) children.get(i);
				obj.setDeleteState();
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.provider.cardspace.db.IDAO#store()
	 */
	public void store() throws CardException {
		Connection con = null;
		try {
			con = getConnectionFactory().getConnection();
			updateState(con);
			if (con.getAutoCommit() == false)
				con.commit();
			commitState();
			sentStoredEvent();

		} catch (Exception e) {
			log_.error(e);
			try {
				con.rollback();
			} catch (SQLException ex) {
				log_.error(ex);
				throw new CardException(ex);
			}
			throw new CardException(e);
		} finally {
			try {
				if (con != null) {
					con.close();
					con = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	private void sentStoredEvent() {
		int size = listeners_.size();
		if (size == 0)
			return;
		DAOStoredEvent event = new DAOStoredEvent(this);
		for (int i = 0; i < size; i++) {
			IDAOListener lst = (IDAOListener) listeners_.get(i);
			lst.objectStored(event);
		}
	}

	/**
	 * @param con
	 * @throws Exception
	 */
	private void updateChildrenState(Connection con) throws Exception {
		ArrayList children = getChildren();
		if (children != null) {
			for (int i = 0, j = children.size(); i < j; i++) {
				DAO obj = (DAO) children.get(i);
				obj.updateState(con);
			}
		}
	}

	/**
	 * Update
	 * 
	 * @param con
	 * @throws Exception
	 */
	protected void updateState(Connection con) throws Exception {
		if (state_ == NEW_OBJ)
			insert(con);
		else if (state_ == CHANGED_OBJ)
			update(con);
		else if (state_ == PRE_DELETED_STORED_OBJ)
			delete(con);
		updateChildrenState(con);
	}

	/**
	 * @param date
	 * @return
	 */
	protected Date checkMaxSQLDate(Date date) {
		if (date == null)
			return null;
		if (date.getTime() > maxMySQLDate_)
			return new Date(maxMySQLDate_);
		else
			return date;
	}

	/**
	 * This method is invoked to persist the object if its state is
	 * <code>NEW_OBJ</code>
	 * 
	 * @param con
	 * @throws Exception
	 */
	protected abstract void insert(Connection con) throws Exception;

	/**
	 * This method is invoked to persist the object if its state is
	 * <code>CHANGED_OBJ</code>
	 * 
	 * @param con
	 * @throws Exception
	 */
	protected abstract void update(Connection con) throws Exception;

	/**
	 * This method is invoked to persist the object if its state is
	 * <code>PRE_DELETED_STORED_OBJ</code>
	 * 
	 * @param con
	 * @throws Exception
	 */
	protected abstract void delete(Connection con) throws Exception;

	/**
	 * @return List of children DAO objects which should be processed according
	 *         to state of this object
	 */
	protected abstract ArrayList getChildren();

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.icard.provider.cardspace.db.IDAO#addListener(org.eclipse.higgins.icard.provider.cardspace.db.IDAOListener)
	 */
	public synchronized void addListener(IDAOListener listener) {
		if (listeners_.contains(listener) == false)
			listeners_.add(listener);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.icard.provider.cardspace.db.IDAO#removeListener(org.eclipse.higgins.icard.provider.cardspace.db.IDAOListener)
	 */
	public void removeListener(IDAOListener listener) {
		listeners_.remove(listener);
	}

}
