/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.icard.provider.cardspace.db.mysql;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Date;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Hashtable;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.InvalidStateException;
import org.eclipse.higgins.icard.provider.cardspace.db.IDAO;
import org.eclipse.higgins.icard.provider.cardspace.db.IDaoInformationCardExtension;
import org.eclipse.higgins.icard.provider.cardspace.db.IDaoPCard;
import org.eclipse.higgins.icard.provider.cardspace.db.IDaoPCardClaim;

public class DaoPCard extends DAO implements IDaoPCard {
	private Log log_ = LogFactory.getLog(DaoPCard.class);

	protected CardContext context_;

	protected String userID_;

	protected String cardID_ = null;

	protected String language_ = null;

	protected String name_ = null;

	protected int version_ = 0;

	protected byte[] image_ = null;

	protected byte[] issuerID_ = null;

	protected String imageType_ = null;

	protected Date timeIssued_ = null;

	protected Date timeExpires_ = null;

	protected Date timeLastUpdated_ = null;

	protected byte[] hashSalt_ = null;

	protected byte[] masterKey_ = null;

	protected byte[] pinDigest_ = null;

	protected byte[] initialImage_ = null;

	protected String initialName_ = null;

	protected String initialImageType_ = null;

	protected byte[] initialMasterKey_ = null;

	protected byte[] initialPinDigest_ = null;

	protected ArrayList claimValues_ = new ArrayList();

	protected Hashtable claimValuesHash_ = new Hashtable();

	protected ArrayList extensionList_ = new ArrayList();

	public DaoPCard(CardContext context, String userID) {
		state_ = NEW_OBJ;
		userID_ = userID;
		context_ = context;
		connectionFactory_ = context.getConnectionFactory();
		addListener(context);
	}

	public DaoPCard(CardContext context, String userID, ResultSet rs, Connection con) throws Exception {
		state_ = STORED_OBJ;
		userID_ = userID;
		context_ = context;
		connectionFactory_ = context.getConnectionFactory();
		addListener(context);
		init(rs, con);
	}

	public void init(ResultSet rs, Connection con) throws Exception {
		id_ = rs.getInt("id_");
		cardID_ = rs.getString("cardID_");
		language_ = rs.getString("language_");
		initialName_ = name_ = rs.getString("name_");
		version_ = rs.getInt("version_");
		initialImage_ = image_ = rs.getBytes("image_");
		initialImageType_ = imageType_ = rs.getString("imageType_");
		timeIssued_ = rs.getDate("timeIssued_");
		timeExpires_ = rs.getDate("timeExpires_");
		timeLastUpdated_ = rs.getDate("timeLastUpdated_");
		hashSalt_ = rs.getBytes("hashSalt_");
		initialMasterKey_ = masterKey_ = rs.getBytes("masterKey_");
		initialPinDigest_ = pinDigest_ = rs.getBytes("pinDigest_");
		issuerID_ = rs.getBytes("issuerID_");
		updateInitialValues();
		initClaimValueList(con);
		initExtensionList(con);
	}

	public int getState() {
		return state_;
	}

	public String getCardID() {
		return cardID_;
	}

	public byte[] getHashSalt() {
		return hashSalt_;
	}

	public int getID() {
		return id_;
	}

	public byte[] getImage() {
		return image_;
	}

	public String getImageType() {
		return imageType_;
	}

	public String getLanguage() {
		return language_;
	}

	public byte[] getMasterKey() {
		return masterKey_;
	}

	public String getName() {
		return name_;
	}

	public Date getTimeExpires() {
		return timeExpires_;
	}

	public Date getTimeIssued() {
		return timeIssued_;
	}

	public Date getTimeLastUpdated() {
		return timeLastUpdated_;
	}

	public byte[] getPinDigest() {
		return pinDigest_;
	}

	public int getVersion() {
		return version_;
	}

	public void setCardID(String cardID) throws InvalidStateException {
		if (id_ != -1)
			throw new InvalidStateException("Unsupported operation for existent card");
		cardID_ = cardID;
	}

	public void setHashSalt(byte[] hashSalt) throws InvalidStateException {
		if (id_ != -1)
			throw new InvalidStateException("Unsupported operation for existent card");
		hashSalt_ = hashSalt;
	}

	public void setImage(byte[] image) throws CardException {
		image_ = image;
		setChanged();
	}

	public void setImageType(String imageType) throws CardException {
		imageType_ = imageType;
		setChanged();
	}

	public void setLanguage(String language) throws InvalidStateException {
		if (id_ != -1)
			throw new InvalidStateException("Unsupported operation for existent card");
		language_ = language;
	}

	public void setMasterKey(byte[] masterKey) throws CardException {
		masterKey_ = masterKey;
		setChanged();
	}

	public void setName(String name) throws CardException {
		name_ = name;
		setChanged();
	}

	public void setTimeExpires(Date timeExpires) throws InvalidStateException {
		if (id_ != -1)
			throw new InvalidStateException("Unsupported operation for existent card");
		timeExpires_ = checkMaxSQLDate(timeExpires);
	}

	public void setTimeIssued(Date timeIssued) throws InvalidStateException {
		if (id_ != -1)
			throw new InvalidStateException("Unsupported operation for existent card");
		timeIssued_ = timeIssued;
	}

	public void setTimeLastUpdated(Date timeLastUpdated) throws InvalidStateException {
		timeLastUpdated_ = timeLastUpdated;
	}

	public void setPinDigest(byte[] pinDigest) throws CardException {
		pinDigest_ = pinDigest;
		setChanged();
	}

	public void setVersion(int version) throws InvalidStateException {
		if (id_ != -1)
			throw new InvalidStateException("Unsupported operation for existent card");
		version_ = version;
	}

	protected void insert(Connection con) throws Exception {
		String query = "INSERT INTO PersonalCard (cardID_, language_, name_, version_, image_, imageType_, timeIssued_, timeExpires_, timeLastUpdated_, hashSalt_, masterKey_, pinDigest_, userID_, issuerID_) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query, Statement.RETURN_GENERATED_KEYS);
			ps.setString(1, cardID_);
			ps.setString(2, language_);
			ps.setString(3, name_);
			ps.setInt(4, version_);
			ps.setBytes(5, image_);
			ps.setString(6, imageType_);
			ps.setDate(7, timeIssued_);
			ps.setDate(8, timeExpires_);
			ps.setDate(9, timeLastUpdated_);
			ps.setBytes(10, hashSalt_);
			ps.setBytes(11, masterKey_);
			ps.setBytes(12, pinDigest_);
			ps.setString(13, userID_);
			ps.setBytes(14, issuerID_);
			ps.execute();
			rs = ps.getGeneratedKeys();
			if (rs.next())
				id_ = rs.getInt(1);
			else
				throw new CardException("Prepared statement did not return primary key for inserted row.");
		} catch (SQLException e) {
			log_.error(e);
			throw new CardException(e);
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	protected void update(Connection con) throws SQLException {
		updateName(con);
		updateImage(con);
		updateImageType(con);
		updatePinDigest(con);
		updateMasterKey(con);
		updateTimeLastUpdated(con);
	}

	protected void updateName(Connection con) throws SQLException {
		if (initialName_ == null && name_ == null)
			return;
		String n1 = (initialName_ == null) ? "" : initialName_;
		String n2 = (name_ == null) ? "" : name_;
		if (n1.equals(n2) == false) {
			String query = "UPDATE PersonalCard SET name_ = ? WHERE id_ = ?";
			PreparedStatement ps = null;
			try {
				ps = con.prepareStatement(query);
				ps.setString(1, name_);
				ps.setInt(2, id_);
				ps.execute();
			} finally {
				try {
					if (ps != null) {
						ps.close();
						ps = null;
					}
				} catch (SQLException e) {
					log_.error(e);
				}
			}
		}
	}

	protected void updateImage(Connection con) throws SQLException {
		if (Arrays.equals(initialImage_, image_))
			return;
		String query = "UPDATE PersonalCard SET image_ = ? WHERE id_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setBytes(1, image_);
			ps.setInt(2, id_);
			ps.execute();
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	protected void updateImageType(Connection con) throws SQLException {
		if (initialImageType_ == null && imageType_ == null)
			return;
		String n1 = (initialImageType_ == null) ? "" : initialImageType_;
		String n2 = (imageType_ == null) ? "" : imageType_;
		if (n1.equals(n2) == false) {
			String query = "UPDATE PersonalCard SET imageType_ = ? WHERE id_ = ?";
			PreparedStatement ps = null;
			try {
				ps = con.prepareStatement(query);
				ps.setString(1, imageType_);
				ps.setInt(2, id_);
				ps.execute();
			} finally {
				try {
					if (ps != null) {
						ps.close();
						ps = null;
					}
				} catch (SQLException e) {
					log_.error(e);
				}
			}
		}
	}

	protected void updateTimeLastUpdated(Connection con) throws SQLException {
		String query = "UPDATE PersonalCard SET timeLastUpdated_ = ? WHERE id_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setDate(1, timeLastUpdated_);
			ps.setInt(2, id_);
			ps.execute();
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	protected void updateMasterKey(Connection con) throws SQLException {
		if (Arrays.equals(masterKey_, initialMasterKey_))
			return;
		String query = "UPDATE PersonalCard SET masterKey_ = ? WHERE id_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setBytes(1, masterKey_);
			ps.setInt(2, id_);
			ps.execute();
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	protected void updatePinDigest(Connection con) throws SQLException {
		if (Arrays.equals(pinDigest_, initialPinDigest_))
			return;
		String query = "UPDATE PersonalCard SET pinDigest_ = ? WHERE id_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setBytes(1, pinDigest_);
			ps.setInt(2, id_);
			ps.execute();
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	protected void delete(Connection con) throws SQLException {
		String query = "DELETE FROM PersonalCard WHERE id_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, id_);
			ps.execute();
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	public ArrayList getChildren() {
		ArrayList children = new ArrayList();
		children.addAll(claimValues_);
		children.addAll(extensionList_);
		return children;
	}

	public void addPersonalCardClaim(IDaoPCardClaim personalCardClaim) throws CardException {
		if (personalCardClaim == null)
			throw new CardException("Parameter \"personalCardClaim\" is null.");
		String type = personalCardClaim.getType();
		if (claimValuesHash_.containsKey(type))
			throw new CardException("There already exists claim value with type = " + type);
		claimValues_.add(personalCardClaim);
		claimValuesHash_.put(type, personalCardClaim);
		setChanged();
	}

	public IDaoPCardClaim createPersonalCardClaim() {
		return new DaoPCardClaim(this);
	}

	public IDaoPCardClaim getPersonalCardClaim(String type) {
		if (type == null)
			return null;
		if (claimValuesHash_.containsKey(type))
			return (IDaoPCardClaim) claimValuesHash_.get(type);
		return null;
	}

	public ArrayList getPersonalCardClaimList() {
		return claimValues_;
	}

	private void initClaimValueList(Connection con) throws SQLException, CardException {
		claimValues_.clear();
		claimValuesHash_.clear();
		String query = "SELECT * FROM PersonalCardClaim WHERE cardID_ = ?";
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, id_);
			rs = ps.executeQuery();
			while (rs.next()) {
				DaoPCardClaim cc = new DaoPCardClaim(this, rs, con);
				claimValues_.add(cc);
				claimValuesHash_.put(cc.getType(), cc);
			}
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	private void initExtensionList(Connection con) throws SQLException {
		extensionList_.clear();
		String query = "SELECT * FROM PCardExtension WHERE cardID_ = ?";
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, id_);
			rs = ps.executeQuery();
			while (rs.next()) {
				DaoInformationCardExtension ts = new DaoInformationCardExtension(this, rs, con);
				extensionList_.add(ts);
			}
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log_.error(e);
			}
		}
	}

	private void removeDeletedExtensions() {
		ArrayList deleted = new ArrayList();
		for (int i = 0, j = extensionList_.size(); i < j; i++) {
			IDAO obj = (IDAO) extensionList_.get(i);
			if (obj.getState() == IDAO.DELETED_OBJ) {
				deleted.add(obj);
			}
		}
		extensionList_.removeAll(deleted);
	}

	private void removeDeletedClaims() {
		ArrayList deletedClaims = new ArrayList();
		for (int i = 0, j = claimValues_.size(); i < j; i++) {
			IDaoPCardClaim claim = (IDaoPCardClaim) claimValues_.get(i);
			if (claim.getState() == IDAO.DELETED_OBJ) {
				deletedClaims.add(claim);
				claimValuesHash_.remove(claim.getType());
			}
		}
		claimValues_.removeAll(deletedClaims);
	}

	private void updateInitialValues() {
		initialName_ = name_;
		initialImage_ = image_;
		initialImageType_ = imageType_;
		initialMasterKey_ = masterKey_;
		initialPinDigest_ = pinDigest_;
	}

	public void commitState() {
		super.commitState();
		removeDeletedClaims();
		removeDeletedExtensions();
		updateInitialValues();
	}

	public String getUserID() {
		return userID_;
	}

	public byte[] getIssuerID() {
		return issuerID_;
	}

	public void setIssuerID(byte[] issuerID) throws CardException {
		if (id_ != -1)
			throw new InvalidStateException("Unsupported operation for existent card");
		issuerID_ = issuerID;
	}

	public void addInformationCardExtension(IDaoInformationCardExtension extension) throws CardException {
		if (extension == null)
			throw new CardException("Parameter \"extension\" is null.");
		extensionList_.add(extension);
		setChanged();
	}

	public IDaoInformationCardExtension createInformationCardExtension() {
		return new DaoInformationCardExtension(this);
	}

	public ArrayList getExtensions() {
		return extensionList_;
	}

}
