/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.icard.provider.cardspace.managed.xml;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;

import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.PasswordCallback;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.AuthenticationException;
import org.eclipse.higgins.icard.CUID;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.ICard;
import org.eclipse.higgins.icard.ICardProvider;
import org.eclipse.higgins.icard.ICardTemplate;
import org.eclipse.higgins.icard.IInformationCard;
import org.eclipse.higgins.icard.IManagedInformationCard;
import org.eclipse.higgins.icard.IPolicy;
import org.eclipse.higgins.icard.auth.ICredential;
import org.eclipse.higgins.icard.provider.cardspace.common.InformationCard;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.CardContext;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.CardCryptography;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.XMLUtils;
import org.eclipse.higgins.registry.IConfiguration;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Implementation of managed CardSpace-interoperable ICard provider stored in
 * xml file form
 */

public class XMLBasedManagedCardProvider implements ICardProvider {

	private Log log = LogFactory.getLog(XMLBasedManagedCardProvider.class);

	// TODO change password initialization
	// private String workDir_ = null;

	protected static final String ID = "org.eclipse.higgins.icard.provider.cardspace.managed.xml";

	private String name_ = "Xml-based CardSpace-compatable managed I-Card Provider";

	private String description_ = name_;

	protected ProviderConfiguration config = new ProviderConfiguration(ID);

	public XMLBasedManagedCardProvider() throws Exception {
		// workDir_ = System.getProperty("user.home") + File.separator +
		// ".higgins";
	}

	private class UserCredentials {
		private String name_;

		private String password_;

		public UserCredentials(String name, String password) {
			password_ = password;
			name_ = name;
		}

		public String getName() {
			return name_;
		}

		public String getPassword() {
			return password_;
		}
	}

	// TODO there may be different users with the same name, so we shouldn't
	// bound users name to file with cards
	private File getStorageFile(String username) {
		if (username == null)
			return null;
		else {
			String workDir = config.getWorkDir();
			File dir = new File(workDir);
			if (dir.exists() == false)
				dir.mkdirs();
			if (dir.isFile() == true) {
				log.error("Can not create the working dirrectory. There already exists a file with such a name : " + workDir);
				return null;
			}
			return new File(workDir + File.separator + username + ".crds");
		}
	}

	/**
	 * @param authHandler
	 * @return
	 * @throws CardException
	 */
	private UserCredentials authenticate(CallbackHandler authHandler) throws CardException {
		NameCallback nc = new NameCallback("User name: ");
		PasswordCallback pc = new PasswordCallback("Password: ", false);
		Callback[] callbacks = new Callback[] { nc, pc };
		try {
			authHandler.handle(callbacks);
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e);
		}
		String userName = nc.getName();
		String password = new String(pc.getPassword());
		pc.clearPassword();
		if (userName == null || userName.trim().length() == 0)
			throw new CardException("Empty user name.");
		if (password == null || password.trim().length() == 0)
			throw new CardException("Empty password.");
		return new UserCredentials(userName, password);
	}

	/**
	 * @param credentials
	 * @return
	 * @throws CardException
	 */
	private Document getCardsStore(UserCredentials credentials) throws CardException {
		DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
		dbf.setIgnoringComments(true);
		dbf.setIgnoringElementContentWhitespace(true);
		dbf.setNamespaceAware(true);
		dbf.setValidating(false);
		DocumentBuilder db;
		try {
			db = dbf.newDocumentBuilder();
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e);
		}
		Document cards;
		File cardsFile = getStorageFile(credentials.getName());
		if (cardsFile.exists()) {
			if (cardsFile.isFile()) {
				FileInputStream fis;
				try {
					fis = new FileInputStream(cardsFile);
				} catch (Exception e) {
					log.error(e);
					throw new CardException(e);
				}
				try {
					cards = CardCryptography.decrypt(fis, credentials.getPassword());
				} catch (Exception e) {
					log.error(e);
					throw new CardException(e);
				}
			} else
				throw new CardException("Unable to open " + cardsFile.getAbsolutePath());
		} else {
			cards = db.newDocument();
			cards.appendChild(cards.createElementNS(CardContext.IC_NS, "RoamingStore"));
		}
		return cards;
	}

	/**
	 * @param cards
	 * @param credentials
	 * @throws Exception
	 */
	private void saveCardsStore(Document cards, UserCredentials credentials) throws Exception {
		FileOutputStream fos = null;
		File tmpFile = null;
		try {
			tmpFile = File.createTempFile("$mc", ".tmp", new File(config.getWorkDir()));
			// tmpFile = File.createTempFile("$mc", ".tmp", new File(workDir_));
			fos = new FileOutputStream(tmpFile);
			CardCryptography.encrypt(cards, fos, credentials.getPassword());
			fos.flush();
			fos.close();
		} catch (Exception e) {
			try {
				if (fos != null)
					fos.close();
				if (tmpFile != null)
					tmpFile.delete();
			} catch (IOException e1) {
			}
			throw e;
		}
		File cardsFile = getStorageFile(credentials.getName());
		if (cardsFile.exists()) {
			File bkpFile = new File(cardsFile.getAbsolutePath() + ".bkp");
			if (bkpFile.exists())
				bkpFile.delete();
			if (cardsFile.renameTo(bkpFile) == false && cardsFile.delete() == false)
				throw new CardException("Couldn't save cards to file " + cardsFile.getAbsolutePath());
		}
		if (tmpFile.renameTo(cardsFile) == false)
			throw new CardException("Couldn't save cards to file " + cardsFile.getAbsolutePath());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.registry.IServiceProvider#setID(java.lang.String)
	 */
	public void setID(String id) throws Exception {
		// id_ = id;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.registry.IServiceProvider#getConfiguration()
	 */
	public IConfiguration getConfiguration() {
		return config;
	}

	/**
	 * @param roamingCard
	 *            RoamingInformationCard <code>Element</code>
	 * @return
	 * @throws CardException
	 */
	public boolean isSelfIssued(Element card) throws CardException {
		if ("RoamingInformationCard".equals(card.getLocalName())) {
			Element metaData = XMLUtils.getChildElement(card, CardContext.IC_NS, "InformationCardMetaData");
			if (metaData == null)
				throw new CardException("Can not get \"InformationCardMetaData\" element from RoamingInformationCard.");
			Element isSelfIssued = XMLUtils.getChildElement(metaData, CardContext.IC_NS, "IsSelfIssued");
			if (isSelfIssued == null)
				throw new CardException("Can not get \"IsSelfIssued\" element from InformationCardMetaData.");
			String isSelf = XMLUtils.getTextContent(isSelfIssued);
			if ("true".equalsIgnoreCase(isSelf))
				return true;
			else if ("false".equalsIgnoreCase(isSelf))
				return false;
			else
				throw new CardException("\"IsSelfIssued\" element contains unexpected value = " + isSelf);
		} else if ("InformationCard".equals(card.getLocalName()))
			return false;
		else
			throw new CardException("Unexpected element " + card.getNodeName());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICardProvider#getICards(javax.security.auth.callback.CallbackHandler,
	 *      org.eclipse.higgins.icard.IPolicy)
	 */
	public Iterator getICards(CallbackHandler authHandler, IPolicy policy) throws CardException {
		List cards = new ArrayList();
		try {
			UserCredentials userCrds = authenticate(authHandler);
			Document doc = getCardsStore(userCrds);
			Element store = doc.getDocumentElement();
			ArrayList roamingCards = XMLUtils.getChildElements(store, CardContext.IC_NS, "RoamingInformationCard");
			int len = roamingCards.size();
			for (int i = 0; i < len; i++) {
				Element roamingCard = (Element) roamingCards.get(i);
				InformationCard card = null;
				if (isSelfIssued(roamingCard))
					card = new XMLBasedPersonalCard(this, roamingCard, userCrds.name_, userCrds.password_);
				else
					card = new XMLBasedManagedCard(this, roamingCard, userCrds.name_, userCrds.password_);
				if (policy == null) {
					cards.add(card);
				} else if (policy.isSatisfiedBy(card)) {
					cards.add(card);
				}
			}
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e);
		}
		return cards.iterator();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICardProvider#getICardByCUID(javax.security.auth.callback.CallbackHandler,
	 *      java.lang.String)
	 */
	public ICard getICardByCUID(CallbackHandler authHandler, String CUID) throws CardException {
		return getICardByCUID(authHandler, new CUID(CUID));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICardProvider#getICardByCUID(javax.security.auth.callback.CallbackHandler,
	 *      org.eclipse.higgins.icard.CUID)
	 */
	public ICard getICardByCUID(CallbackHandler authHandler, CUID cuid) throws CardException {
		if (cuid == null) {
			throw new IllegalArgumentException("UUID parameter can't be null.");
		}
		for (Iterator itr = getICards(authHandler, null); itr.hasNext();) {
			ICard card = (ICard) itr.next();
			if (cuid.equals(card.getCUID())) {
				return card;
			}
		}
		throw new CardException("I-Card with ID=" + cuid + " not found.");
	}

	public ICard getICardByCUID(CallbackHandler authHandler, CUID cuid, ICredential userCredential) throws AuthenticationException, CardException {
		// TODO Need to implement!!!!
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICardProvider#deleteCard(javax.security.auth.callback.CallbackHandler,
	 *      org.eclipse.higgins.icard.ICard)
	 */
	public void deleteCard(CallbackHandler authHandler, ICard card) throws CardException {
		if (card == null)
			throw new IllegalArgumentException("Parameter \"card\" is null");
		UserCredentials userCrds = authenticate(authHandler);
		Document cards = getCardsStore(userCrds);
		if (removeCardFromList(card.getID().toString(), cards))
			try {
				saveCardsStore(cards, userCrds);
			} catch (Exception e) {
				log.error(e);
				throw new CardException(e.getMessage());
			}
		else
			throw new CardException("Unable to delete card with id = " + card.getID().toString() + ".");
	}

	private Element getRoamingCardByID(String cardID, Document cards) throws CardException {
		if (cardID == null)
			throw new CardException("Parameter \"cardID\" is null");
		if (cards == null)
			throw new CardException("Parameter \"cards\" is null");
		Element store = cards.getDocumentElement();
		ArrayList roamingCards = XMLUtils.getChildElements(store, CardContext.IC_NS, "RoamingInformationCard");
		int len = roamingCards.size();
		for (int i = 0; i < len; i++) {
			Element roamingCard = (Element) roamingCards.get(i);
			Element metaData = XMLUtils.getChildElement(roamingCard, CardContext.IC_NS, "InformationCardMetaData");
			if (metaData == null)
				continue;
			Element cardReference = XMLUtils.getChildElement(metaData, CardContext.IC_NS, "InformationCardReference");
			if (cardReference == null)
				continue;
			Element cardId = XMLUtils.getChildElement(cardReference, CardContext.IC_NS, "CardId");
			if (cardId == null)
				continue;
			String id = XMLUtils.getTextContent(cardId);
			if (cardID.equals(id))
				return roamingCard;
		}
		return null;
	}

	/**
	 * @param cardID
	 * @param cards
	 * @return
	 * @throws CardException
	 */
	private boolean removeCardFromList(String cardID, Document cards) throws CardException {
		Element oldCard = getRoamingCardByID(cardID, cards);
		if (oldCard != null) {
			cards.getDocumentElement().removeChild(oldCard);
			return true;
		} else
			return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICardProvider#importCard(javax.security.auth.callback.CallbackHandler,
	 *      java.lang.String)
	 */
	public ICard importCard(CallbackHandler authHandler, String filename) throws CardException {
		if (authHandler == null)
			throw new CardException("Parameter \"authHandler\" is null");
		if (filename == null)
			throw new CardException("Parameter \"filename\" is null");
		FileInputStream fis;
		try {
			fis = new FileInputStream(filename);
		} catch (FileNotFoundException e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
		// TODO need to separate import of different file types
		if (filename.toUpperCase().endsWith(".CRDS")) {
			importFromCrdsFile(authHandler, fis);
			return null;
		} else if (filename.toUpperCase().endsWith(".CRD"))
			return importFromCrdFile(authHandler, fis);
		else
			throw new CardException("Unsupported file extension.");
	}

	/**
	 * @param authHandler
	 * @param is
	 * @return
	 * @throws CardException
	 */
	private ICard importFromCrdFile(CallbackHandler authHandler, InputStream is) throws CardException {
		try {
			Element card = CardCryptography.getCardFromSignedEnvelop(is);
			UserCredentials credentials = authenticate(authHandler);
			XMLBasedManagedCard mic = new XMLBasedManagedCard(this, card, credentials.name_, credentials.password_);
			Document cards = getCardsStore(credentials);
			Element root = cards.getDocumentElement();
			removeCardFromList(mic.getID().toString(), cards);
			Element roamingCard = mic.toXML(cards);
			root.appendChild(roamingCard);
			saveCardsStore(cards, credentials);
			return mic;
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
	}

	/**
	 * @param authHandler
	 * @param is
	 * @throws CardException
	 */
	private void importFromCrdsFile(CallbackHandler authHandler, InputStream is) throws CardException {
		Document cardBackup;
		PasswordCallback pc = new PasswordCallback("Enter password for imported backup file: ", false);
		Callback[] callbacks = new Callback[] { pc };
		try {
			authHandler.handle(callbacks);
			String password = new String(pc.getPassword());
			pc.clearPassword();
			cardBackup = CardCryptography.decrypt(is, password);
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
		List cardsList = new ArrayList();
		NodeList newCards = cardBackup.getDocumentElement().getChildNodes();
		UserCredentials credentials = authenticate(authHandler);
		Document cards = getCardsStore(credentials);
		Element root = cards.getDocumentElement();
		for (int i = 0; i < newCards.getLength(); i++) {
			Node nd = newCards.item(i);
			if (nd.getNodeType() == Node.ELEMENT_NODE) {
				Element roamingCard = (Element) nd;
				InformationCard card = null;
				if (isSelfIssued(roamingCard))
					card = new XMLBasedPersonalCard(this, roamingCard, credentials.name_, credentials.password_);
				else
					card = new XMLBasedManagedCard(this, roamingCard, credentials.name_, credentials.password_);
				cardsList.add(card);
				removeCardFromList(card.getID().toString(), cards);
				Node newCard = cards.importNode(roamingCard, true);
				root.appendChild(newCard);
			}
		}
		try {
			saveCardsStore(cards, credentials);
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICardProvider#getID()
	 */
	public String getID() {
		return ID;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICardProvider#getName()
	 */
	public String getName() {
		return name_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICardProvider#getDescription()
	 */
	public String getDescription() {
		return description_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICardProvider#canCreateCard(javax.security.auth.callback.CallbackHandler,
	 *      java.lang.String, java.util.Properties)
	 */
	public boolean canCreateCard(CallbackHandler authHandler, String id, Properties props) {
		return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICardProvider#createCard(javax.security.auth.callback.CallbackHandler,
	 *      java.lang.String, java.util.Properties)
	 */
	public ICard createCard(CallbackHandler authHandler, String id, Properties props) throws CardException {
		throw new CardException("This provider can't create a card.");
	}

	public void exportCards(CallbackHandler authHandler, Iterator cards, OutputStream out) throws CardException {
		if (authHandler == null)
			throw new CardException("Parameter \"authHandler\" is null");
		PasswordCallback pc = new PasswordCallback("Enter password for exported card(s) backup: ", false);
		Callback[] callbacks = new Callback[] { pc };
		try {
			authHandler.handle(callbacks);
			String password = new String(pc.getPassword());
			pc.clearPassword();
			DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
			dbf.setNamespaceAware(true);
			dbf.setValidating(false);
			DocumentBuilder db = dbf.newDocumentBuilder();
			Document doc = db.newDocument();
			Element root = doc.createElementNS(CardContext.IC_NS, "RoamingStore");
			doc.appendChild(root);
			while (cards.hasNext()) {
				InformationCard card = (InformationCard) cards.next();
				Element roamingCard = card.toXML(doc);
				root.appendChild(roamingCard);
			}
			CardCryptography.encrypt(doc, out, password);
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
	}

	/*
	 * public String getType() { return ICardConstants.I_CARD_TYPE_CARDSPACE; }
	 */

	public Class[] getSupportedTypes() {
		return new Class[] { IManagedInformationCard.class };
	}

	public boolean canImportICard(CallbackHandler authHandler, Element card) {
		if (card == null)
			return false;
		Element crd = null;
		try {
			if ("Signature".equals(card.getLocalName()))
				crd = CardCryptography.getCardFromSignedEnvelop(card);
			else
				crd = card;
			if (isSelfIssued(crd))
				new XMLBasedPersonalCard(this, crd, "nouser", "nopassword");
			else
				new XMLBasedManagedCard(this, crd, "nouser", "nopassword");
			return true;
		} catch (Exception e) {
			log.error(e);
		}
		return false;
	}

	public void exportICard(CallbackHandler authHandler, ICard card, Element root) throws CardException {
		if (card instanceof IInformationCard) {
			IInformationCard icard = (IInformationCard) card;
			Document doc = root.getOwnerDocument();
			Element roamingCard = icard.toXML(doc);
			root.appendChild(roamingCard);
		} else
			throw new CardException("Can not export card. This card is not an instance of IInformationCard.");
	}

	public ICard importICard(CallbackHandler authHandler, Element card) throws CardException {
		try {
			Element crd = null;
			if ("Signature".equals(card.getLocalName()))
				crd = CardCryptography.getCardFromSignedEnvelop(card);
			else
				crd = card;
			UserCredentials credentials = authenticate(authHandler);
			InformationCard iCard = null;
			if (isSelfIssued(crd))
				iCard = new XMLBasedPersonalCard(this, crd, "nouser", "nopassword");
			else
				iCard = new XMLBasedManagedCard(this, crd, "nouser", "nopassword");
			Document cards = getCardsStore(credentials);
			Element root = cards.getDocumentElement();
			removeCardFromList(iCard.getID().toString(), cards);
			Element roamingCard = iCard.toXML(cards);
			root.appendChild(roamingCard);
			saveCardsStore(cards, credentials);
			return iCard;
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
	}

	public void updateCard(XMLBasedManagedCard card) throws CardException {
		try {
			UserCredentials credentials = new UserCredentials(card.user_, card.password_);
			Document cards = getCardsStore(credentials);
			Element root = cards.getDocumentElement();
			removeCardFromList(card.getID().toString(), cards);
			Element roamingCard = card.toXML(cards);
			root.appendChild(roamingCard);
			saveCardsStore(cards, credentials);
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
	}

	public void updateCard(XMLBasedPersonalCard card) throws CardException {
		try {
			UserCredentials credentials = new UserCredentials(card.user_, card.password_);
			Document cards = getCardsStore(credentials);
			Element root = cards.getDocumentElement();
			removeCardFromList(card.getID().toString(), cards);
			Element roamingCard = card.toXML(cards);
			root.appendChild(roamingCard);
			saveCardsStore(cards, credentials);
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
	}

	public ICard createCard(CallbackHandler authHandler, ICardTemplate template) throws CardException {
		throw new CardException("ICard provider " + name_ + " can not create a card.");
	}

	public ICardTemplate[] getCardCreationTemplates(CallbackHandler authHandler) {
		return null;
	}

}
