/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.icard.provider.cardspace.managed;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;

import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.PasswordCallback;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.CUID;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.ICard;
import org.eclipse.higgins.icard.ICardProvider;
import org.eclipse.higgins.icard.ICardTemplate;
import org.eclipse.higgins.icard.IInformationCard;
import org.eclipse.higgins.icard.IPolicy;
import org.eclipse.higgins.icard.common.utils.IdASContext;
import org.eclipse.higgins.icard.io.IElement;
import org.eclipse.higgins.icard.io.IElementFormat;
import org.eclipse.higgins.icard.provider.cardspace.common.ManagedCardProvider;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.CardCryptography;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.IFilter;
import org.eclipse.higgins.idas.api.IFilterEntityTypeAssertion;
import org.eclipse.higgins.idas.common.AuthNNamePasswordMaterials;
import org.eclipse.higgins.idas.registry.IdASRegistry;
import org.eclipse.higgins.idas.registry.discovery.FileDiscovery;
import org.eclipse.higgins.registry.IConfiguration;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Implementation of managed CardSpace-interoperable ICard provider based on
 * IdAS context
 */
public class IdASBasedManagedCardProvider extends ManagedCardProvider implements ICardProvider {
	private Log log = LogFactory.getLog(IdASBasedManagedCardProvider.class);

	private static final String CONTEXT_ID = "context.id";
	private static final String DISCOVERY_FILE = "discovery.filename";

//	private static final String CONTEXT_FACTORY = "context.factory";

	// private static final String SCHEMA_URL = "schema.url";

	// private static final String SCHEMA_FILE = "schema.file";

//	public static final String CONTEXT_FACTORY_ID = "org.eclipse.higgins.idas.cp.jena2.db";

	private ProviderConfiguration config_ = null;

	private String id_ = "org.eclipse.higgins.icard.provider.cardspace.managed";

	private String name_ = "CardSpace-compatable managed I-Card Provider Plug-in";

	private class UserCredentials {
		private String name_;

		private String password_;

		public UserCredentials(String name, String password) {
			password_ = password;
			name_ = name;
		}

		public String getName() {
			return name_;
		}

		public String getPassword() {
			return password_;
		}
	}

	public IdASBasedManagedCardProvider() throws Exception {
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.registry.IServiceProvider#setID(java.lang.String)
	 */
	public void setID(String id) throws Exception {
		if (config_ == null) {
			id_ = id;
		} else {
			throw new Exception("Provider's id can't be modified.");
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.registry.IServiceProvider#getConfiguration()
	 */
	public IConfiguration getConfiguration() {
		if (config_ == null) {
			config_ = new ProviderConfiguration(getID());
		}
		return config_;
	}

	/**
	 * 
	 */
	private IContext getContext(UserCredentials credentials) throws CardException {
		String disc = getConfiguration().getProperty(DISCOVERY_FILE);
		FileDiscovery factories = new FileDiscovery(new File(disc));
		IdASRegistry registry = IdASRegistry.getInstance();
		registry.setDiscovery(factories);
		IContext ctx = null;
		if (config_ != null) {
			String contextID = getConfiguration().getProperty(CONTEXT_ID);
			if (contextID == null)
				throw new CardException("Can not get \"" + CONTEXT_ID + "\" configuration property.");
			try {
				ctx = registry.createContext(contextID);
				AuthNNamePasswordMaterials credential = new AuthNNamePasswordMaterials(ctx, credentials.getName(), credentials.getPassword());
				ctx.open(credential);
				return ctx;
			} catch (Exception e) {
				log.error(e);
				throw new CardException(e);
			}
		}
		return null;
	}

	public Iterator getICards(CallbackHandler authHandler) throws CardException {
		if (authHandler == null)
			throw new CardException("Parameter \"authHandler\" is null");
		List cards = new ArrayList();
		try {
			UserCredentials userCrds = authenticate(authHandler);
			IContext ctx = getContext(userCrds);
			if (ctx != null) {
				try {
					IFilter filter = ctx.buildFilter();
					filter.setOperator(IFilter.OP_AND);
					IFilterEntityTypeAssertion typeAssertion = ctx.buildEntityTypeAssertion();
					typeAssertion.setAssertionValue(IdASContext.ICARD_ManagedInformationCard);
					filter.addFilter(typeAssertion);
					Iterator subjList = ctx.getEntities(filter);
					while (subjList.hasNext()) {
						IEntity subj = (IEntity) subjList.next();
						if (IdASContext.ICARD_ManagedInformationCard.equals(subj.getEntityType())) {
							IdASBasedManagedCard card = new IdASBasedManagedCard(this, subj);
//							if (policy == null) {
								cards.add(card);
//							} else if (policy.isSatisfiedBy(card)) {
//								cards.add(card);
//							}
						}
					}
				}
				finally {
/*					try {
						ctx.close();
					}
					catch(IdASException e) {
						log.error(e);
					}
*/				}
			}
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e);
		}
		return cards.iterator();
	}

	public ICard importCard(CallbackHandler authHandler, String filename) throws CardException {
		if (authHandler == null)
			throw new CardException("Parameter \"authHandler\" is null");
		if (filename == null)
			throw new CardException("Parameter \"filename\" is null");
		FileInputStream fis;
		try {
			fis = new FileInputStream(filename);
		} catch (FileNotFoundException e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
		// TODO need to separate import of different file types
		if (filename.toUpperCase().endsWith(".CRDS")) {
			importFromCrdsFile(authHandler, fis);
			return null;
		} else if (filename.toUpperCase().endsWith(".CRD"))
			return importFromCrdFile(authHandler, fis);
		else
			throw new CardException("Unsupported file extension.");
	}

	private ICard importFromCrdFile(CallbackHandler authHandler, InputStream is) throws CardException {
		try {
			Element card = CardCryptography.getCardFromSignedEnvelop(is);
			UserCredentials credentials = authenticate(authHandler);
			IContext ctx = getContext(credentials);
			if (ctx != null) {
				try {
					IdASBasedManagedCard pic = new IdASBasedManagedCard(this, ctx, card);
					return pic;
				}
				finally {
/*					try {
						ctx.close();
					}
					catch(IdASException e) {
						log.error(e);
					}
*/				}
				
			} else
				return null;
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
	}

	private void importFromCrdsFile(CallbackHandler authHandler, InputStream is) throws CardException {
		Document cardBackup;
		PasswordCallback pc = new PasswordCallback("Enter password for imported backup file: ", false);
		Callback[] callbacks = new Callback[] { pc };
		try {
			authHandler.handle(callbacks);
			String password = new String(pc.getPassword());
			pc.clearPassword();
			cardBackup = CardCryptography.decrypt(is, password);
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
		List cardsList = new ArrayList();
		NodeList newCards = cardBackup.getDocumentElement().getChildNodes();
		UserCredentials credentials = authenticate(authHandler);
		IContext ctx = getContext(credentials);
		if (ctx != null) {
			try {
				for (int i = 0; i < newCards.getLength(); i++) {
					Node nd = newCards.item(i);
					if (nd.getNodeType() == Node.ELEMENT_NODE) {
						Element elm = (Element) nd;
						try {
							IdASBasedManagedCard crd = new IdASBasedManagedCard(this, ctx, elm);
							cardsList.add(crd);
						} catch (Exception e) {
							log.error(e);
						}
					}
				}
			}
			finally {
/*				try {
					ctx.close();
				}
				catch(IdASException e) {
					log.error(e);
				}
*/			}
		}
	}

	public void deleteCard(CallbackHandler authHandler, ICard card) throws CardException {
		if (authHandler == null)
			throw new CardException("Parameter \"authHandler\" is null");
		if (card == null)
			throw new CardException("Parameter \"card\" is null");
		UserCredentials userCrds = authenticate(authHandler);
		IContext ctx = getContext(userCrds);
		if (ctx != null) {
			try {
				if (card instanceof IdASBasedManagedCard) {
					try {
						IdASBasedManagedCard crd = (IdASBasedManagedCard) card;
						String subjId = crd.getCardSubject().getEntityID();
						IEntity subj = ctx.getEntity(subjId);
						subj.remove();
						ctx.applyUpdates();
					} catch (Exception e) {
						log.error(e);
						try {
							ctx.cancelUpdates();
						} catch (Exception e1) {
							log.error(e1);
							throw new CardException(e1);
						}
						throw new CardException(e);
					}
				}
			}
			finally {
/*				try {
					ctx.close();
				}
				catch(IdASException e) {
					log.error(e);
				}
*/			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICardProvider#getID()
	 */
	public String getID() {
		return id_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICardProvider#getName()
	 */
	public String getName() {
		return name_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICardProvider#canCreateCard(javax.security.auth.callback.CallbackHandler,
	 *      java.lang.String, java.util.Properties)
	 */
	public boolean canCreateCard(CallbackHandler authHandler, String id, Properties props) {
		return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICardProvider#createCard(javax.security.auth.callback.CallbackHandler,
	 *      java.lang.String, java.util.Properties)
	 */
	public ICard createCard(CallbackHandler authHandler, String id, Properties props) throws CardException {
		throw new CardException("This provider can't create a card.");
	}


	public String getDescription() {
		return "IdAS-based CardSpace-interoperable managed I-Card provider";
	}

	/*
	public boolean canImportICard(CallbackHandler authHandler, IElement element) {
		if (element == null)
			return false;
		try {
			Element card = (Element) element.getAs(Element.class);
			if (card == null) {
				return false;
			}
			Element crd = null;
			if ("Signature".equals(card.getLocalName()))
				crd = CardCryptography.getCardFromSignedEnvelop(card);
			else
				crd = card;
			IdASBasedManagedCard c = new IdASBasedManagedCard();
			c.initFromXML(crd);
			return true;
		} catch (Exception e) {
			log.error(e);
		}
		return false;
	}
	*/

//	public IElement[] exportICards(CallbackHandler authHandler, IElementFormat format, CUID[] cards) throws CardException {
//		Document doc = cards.getOwnerDocument();
//		if (format instanceof IInformationCard) {
//			IInformationCard icard = (IInformationCard) format;
//			Element roamingCard = icard.toXML(doc);
//			cards.appendChild(roamingCard);
//		} else
//			throw new CardException("Can not export card. This card is not an instance of IInformationCard.");
//	}

	public ICard importICard(CallbackHandler authHandler, IElement element) throws CardException {
		try {
			Element card = (Element) element.getAs(Element.class);
			Element crd = null;
			if ("Signature".equals(card.getLocalName()))
				crd = CardCryptography.getCardFromSignedEnvelop(card);
			else
				crd = card;
			UserCredentials credentials = authenticate(authHandler);
			IContext ctx = getContext(credentials);
			if (ctx != null) {
				try {
					IdASBasedManagedCard mic = new IdASBasedManagedCard(this, ctx, crd);
					return mic;
				}
				finally {
/*					try {
						ctx.close();
					}
					catch(IdASException e) {
						log.error(e);
					}
*/				}
			}
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
		return null;
	}

	protected ICard getICardByID(CallbackHandler authHandler, String CardID) throws CardException {
		try {
			UserCredentials userCrds = authenticate(authHandler);
			IContext ctx = getContext(userCrds);
			if (ctx != null) {
				try {
					IEntity subj = ctx.getEntity(CardID);
					IdASBasedManagedCard card = new IdASBasedManagedCard(this, subj);
					return card;
				}
				finally {
/*					try {
						ctx.close();
					}
					catch(IdASException e) {
						log.error(e);
					}
*/				}
			} else
				return null;
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e);
		}
	}

	public ICard getICardByCUID(CallbackHandler authHandler, CUID cuid) throws CardException {
		if (this.getID().equals(cuid.getProviderID()) == false) {
			log.debug("ProviderID of requested CUID (" + cuid.getProviderID() + ") is not equal to this providerID (" + this.getID() + ").");
			return null;
		}
		ICard crd = getICardByID(authHandler, cuid.getCardID());
		if (crd == null) {
			log.debug("Can not find any card with ID = " + cuid.getCardID());
			return null;
		} else {
			String issuer = cuid.getIssuer();
			if (issuer != null && issuer.equals(crd.getIssuer()))
				return crd;
			else {
				log.debug("Wrong issuer for card with ID = " + cuid.getCardID() + ". Requested issuer = " + issuer + ", real issuer = " + crd.getIssuer());
				return null;
			}
		}
	}

	private UserCredentials authenticate(CallbackHandler authHandler) throws CardException {
		NameCallback nc = new NameCallback("User name: ");
		PasswordCallback pc = new PasswordCallback("Password: ", false);
		Callback[] callbacks = new Callback[] { nc, pc };
		try {
			authHandler.handle(callbacks);
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e);
		}
		String userName = nc.getName();
		String password = new String(pc.getPassword());
		pc.clearPassword();
		if (userName == null || userName.trim().length() == 0)
			throw new CardException("Empty user name.");
		if (password == null || password.trim().length() == 0)
			throw new CardException("Empty password.");
		return new UserCredentials(userName, password);
	}
	
	public ICard createCard(CallbackHandler authHandler, ICardTemplate template) throws CardException {
			throw new CardException("ICard provider " + name_ + " can not create a card.");
	}

}