/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.icard.provider.cardspace.personal.db;

import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.List;

import javax.security.auth.callback.Callback;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.AuthenticationException;
import org.eclipse.higgins.icard.AuthenticationRequiredException;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.ICardConstants;
import org.eclipse.higgins.icard.IClaim;
import org.eclipse.higgins.icard.IClaimType;
import org.eclipse.higgins.icard.IInformationCardExtension;
import org.eclipse.higgins.icard.ISimpleClaim;
import org.eclipse.higgins.icard.ISimpleClaimType;
import org.eclipse.higgins.icard.InvalidClaimException;
import org.eclipse.higgins.icard.InvalidStateException;
import org.eclipse.higgins.icard.InvalidTypeException;
import org.eclipse.higgins.icard.ReadOnlyObjectException;
import org.eclipse.higgins.icard.auth.ICredential;
import org.eclipse.higgins.icard.auth.IPinCodeCredential;
import org.eclipse.higgins.icard.common.ClaimType;
import org.eclipse.higgins.icard.common.ClaimValue;
import org.eclipse.higgins.icard.common.auth.callback.PinCodeCallback;
import org.eclipse.higgins.icard.io.IElement;
import org.eclipse.higgins.icard.provider.cardspace.common.PersonalCard;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.CardCryptography;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.EncryptedMasterKey;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.SelfIssuedCardClaims;
import org.eclipse.higgins.icard.provider.cardspace.db.ICardContext;
import org.eclipse.higgins.icard.provider.cardspace.db.IDaoPCard;
import org.eclipse.higgins.icard.provider.cardspace.db.IDaoPCardClaim;
import org.w3c.dom.Element;

/**
 * Implementation of personal (self-issued) CardSpace-interoperable ICard stored
 * within database
 */
public class JDBCBasedPersonalCard extends PersonalCard {

	private Log log = LogFactory.getLog(JDBCBasedPersonalCard.class);

	private IDaoPCard card_ = null;

	private boolean editMode = false;

	private EncryptedMasterKey currentMasterKey_ = null;

	public JDBCBasedPersonalCard() {
	}

	public JDBCBasedPersonalCard(JDBCBasedPersonalCardProvider provider, ICardContext context, String userID, URI cardId, String cardName,
			HashMap dirtyClaims, byte[] image, String imageMimeType) throws Exception {
		if (provider == null)
			throw new CardException("Parameter \"provider\" is null");
		if (cardId == null)
			throw new CardException("Parameter \"cardId\" is null");
		initPCardConstants();
		provider_ = provider;
		id_ = cardId;
		version_ = "1";
		timeIssued_ = new Date();
		name_ = cardName != null ? cardName : "";
		masterKey_ = CardCryptography.getRandomBytes(32);
		hashSalt_ = CardCryptography.getRandomBytes(16);
		if (image != null && imageMimeType != null) {
			image_ = image;
			imageMimeType_ = imageMimeType;
		}
		dirtyClaimValueMap_ = dirtyClaims;
		super.setClaims(getAllClaimsFromDirtyMap(dirtyClaimValueMap_));
		card_ = context.getPCard(id_.toString(), userID);
		if (card_ == null) {
			card_ = context.createPCard(userID);
		}
		saveCard();
		card_.store();
	}

	public JDBCBasedPersonalCard(JDBCBasedPersonalCardProvider provider, IDaoPCard card) throws Exception {
		if (provider == null)
			throw new CardException("Parameter \"provider\" is null");
		if (card == null)
			throw new CardException("Parameter \"card\" is null");
		initPCardConstants();
		provider_ = provider;
		card_ = card;
		try {
			initFromDAO();
			if (pinDigest_ == null) {
				super.setClaims(getAllClaimsFromDirtyMap(dirtyClaimValueMap_));
			}
		}
		catch(Exception e) {
			log.error(e, e);
			throw new CardException(e);
		}
	}

	public JDBCBasedPersonalCard(JDBCBasedPersonalCardProvider provider, ICardContext context, String userID, Element card) throws Exception {
		if (provider == null)
			throw new CardException("Parameter \"provider\" is null");
		if (card == null)
			throw new CardException("Parameter \"card\" is null");
		initPCardConstants();
		provider_ = provider;
		initFromXML(card);
		IDaoPCard oldCard = context.getPCard(id_.toString(), userID);
		if (oldCard != null) {
			oldCard.setDeleteState();
		}
		card_ = context.createPCard(userID);
		if (pinDigest_ == null) {
			super.setClaims(getAllClaimsFromDirtyMap(dirtyClaimValueMap_));
		}
		saveCard();
		if (oldCard != null)
			oldCard.store();
		card_.store();
	}

	public void initPCardConstants() {
		isSelfIssued_ = true;
		description_ = ICardConstants.I_CARD_TYPE_CARDSPACE;
		issuerName_ = "Self";
		issuer_ = "http://schemas.xmlsoap.org/ws/2005/05/identity/issuer/self";
		//
		supportedTokenTypes_ = new ArrayList();
		supportedTokenTypes_.add(URI.create("urn:oasis:names:tc:SAML:1.0:assertion"));
		supportedTokenTypes_.add(URI.create("http://docs.oasis-open.org/wss/oasis-wss-saml-token-profile-1.1#SAMLV1.1"));
		//
		claimTypes_ = SelfIssuedCardClaims.getSupportedClaimTypeList();
	}

	private ArrayList getAllClaimsFromDirtyMap(HashMap dirtyClaimValueMap) throws CardException {
		ArrayList claimValueList = new ArrayList();
		ArrayList suppportedTypes = SelfIssuedCardClaims.getSupportedClaimTypeList();
		int size = suppportedTypes.size();
		for (int i = 0; i < size; i++) {
			Object obj = suppportedTypes.get(i);
			if (obj instanceof ISimpleClaimType == false)
				throw new CardException("Instance of " + ISimpleClaimType.class.getName() + " expected instead of " + obj.getClass().getName());
			ISimpleClaimType cType = (ISimpleClaimType) obj;
			String type = cType.getType();
			String value = (dirtyClaimValueMap.containsKey(type)) ? (String)dirtyClaimValueMap.get(type) : "";
			if (SelfIssuedCardClaims.GENDER_TYPE.equals(type)) {
				value = SelfIssuedCardClaims.getGenderClaimValue_TEST(value);
			}
			ClaimValue cValue = new ClaimValue(cType, value, this);
			claimValueList.add(cValue);
		}
		return claimValueList;
	}

	/**
	 * @param card
	 * @param claimList
	 * @throws Exception
	 */
	public void initFromDAO() throws Exception {
		initLanguage();
		initCardId();
		initCardVersion();
		initCardName();
		initCardImage();
		initCardImageType();
		initIssuerID();
		initTimeIssued();
		initTimeExpires();
		initHashSalt();
		initTimeLastUpdated();
		initMasterKey();
		initPinDigest();
		initDirtyClaims();
	}

	private void initLanguage() {
		language_ = card_.getLanguage();
	}

	private void initCardId() throws CardException {
		String id = card_.getCardID();
		try {
			id_ = new URI(id);
		} catch (URISyntaxException e) {
			log.error(e);
			throw new CardException(e);
		}
	}

	private void initCardVersion() {
		version_ = String.valueOf(card_.getVersion());
	}

	private void initCardName() {
		name_ = card_.getName();
	}

	private void initCardImage() {
		image_ = card_.getImage();
	}

	private void initCardImageType() {
		imageMimeType_ = card_.getImageType();
	}
	private void initIssuerID() {
		issuerID_ = card_.getIssuerID();
	}

	private void initTimeIssued() {
		timeIssued_ = card_.getTimeIssued();
	}

	private void initTimeExpires() {
		timeExpires_ = card_.getTimeExpires();
	}

	private void initDirtyClaims() throws CardException {
		dirtyClaimValueMap_ = new HashMap();
		ArrayList suppportedTypes = SelfIssuedCardClaims.getSupportedClaimTypeList();
		int size = suppportedTypes.size();
		try {
			for (int i = 0; i < size; i++) {
				Object obj = suppportedTypes.get(i);
				if (obj instanceof ClaimType == false)
					throw new CardException("Instance of " + ClaimType.class.getName() + " expected instead of " + obj.getClass().getName());
				ClaimType cType = (ClaimType) obj;
				String type = cType.getType();
				IDaoPCardClaim daoClaim = card_.getPersonalCardClaim(type);
				String value = (daoClaim != null) ? daoClaim.getValue() : "";
				dirtyClaimValueMap_.put(type, value);
			}
		} catch (CardException e) {
			log.error(e, e);
			throw new CardException(e);
		}
	}

	private void initHashSalt() {
		hashSalt_ = card_.getHashSalt();
	}

	private void initTimeLastUpdated() {
		timeLastUpdated_ = card_.getTimeLastUpdated();
	}

	private void initMasterKey() {
		masterKey_ = card_.getMasterKey();
	}

	private void initPinDigest() {
		pinDigest_ = card_.getPinDigest();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#setName(java.lang.String)
	 */
	public void setName(String name) throws CardException {
		if (!editMode)
			throw new InvalidStateException("Not in edit mode.");
		String oldName = name_;
		try {
			name_ = (name != null) ? name.trim() : "";
			saveCardName();
			saveTimeLastUpdated();
		} catch (Exception e) {
			name_ = oldName;
			log.error(e);
			throw new CardException(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#setImage(byte[], java.lang.String)
	 */
	public void setImage(byte[] image, String imageMimeType) throws CardException {
		if (!editMode)
			throw new InvalidStateException("Not in edit mode.");
		byte[] oldImage = image_;
		String oldImageType = imageMimeType_;
		image_ = image;
		imageMimeType_ = imageMimeType;
		if (image_ == null) {
			image_ =  ((JDBCBasedPersonalCardProvider)provider_).getDefaultImage();
			imageMimeType_ = ((JDBCBasedPersonalCardProvider)provider_).getDefaultPictureMIMEType();
		}
		try {
			saveCardImage();
			saveCardImageType();
			saveTimeLastUpdated();
		} catch (Exception e) {
			image_ = oldImage;
			imageMimeType_ = oldImageType;
			log.error(e);
			throw new CardException(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#setExpiredTime(java.util.Date)
	 */
	public void setTimeExpires(Date date) throws CardException {
		if (!editMode)
			throw new InvalidStateException("Not in edit mode.");
		Date oldDate_ = timeExpires_;
		try {
			timeExpires_ = date;
			saveTimeExpires();
			saveTimeLastUpdated();
		} catch (Exception e) {
			timeExpires_ = oldDate_;
			log.error(e);
			throw new CardException(e);
		}
	}

	/**
	 * Save the card. Must be used only when create
	 * 
	 * @throws Exception
	 */
	private void saveCard() throws CardException {
		saveLanguage();
		saveCardId();
		saveCardVersion();
		saveCardName();
		saveCardImage();
		saveCardImageType();
		saveTimeIssued();
		saveTimeExpires();
		saveClaims();
		saveHashSalt();
		saveTimeLastUpdated();
		saveMasterKey();
		savePinDigest();
		saveIssuerID();
	}

	private void saveLanguage() throws CardException {
		card_.setLanguage(language_);
	}

	private void saveCardId() throws CardException {
		card_.setCardID(id_.toString());
	}

	private void saveCardVersion() throws CardException {
		card_.setVersion(Integer.parseInt(version_));
	}

	private void saveCardName() throws CardException {
		card_.setName(name_);
	}

	private void saveCardImage() throws CardException {
		card_.setImage(image_);
	}

	private void saveCardImageType() throws CardException {
		card_.setImageType(imageMimeType_);
	}

	private void saveTimeIssued() throws CardException {
		java.sql.Date dt = (timeIssued_ != null) ? new java.sql.Date(timeIssued_.getTime()) : null;
		card_.setTimeIssued(dt);
	}

	private void saveTimeExpires() throws CardException {
		java.sql.Date dt = (timeExpires_ != null) ? new java.sql.Date(timeExpires_.getTime()) : null;
		card_.setTimeExpires(dt);
	}

	/**
	 * @throws IdASException
	 */
	private void saveHashSalt() throws CardException {
		card_.setHashSalt(hashSalt_);
	}

	/**
	 * @throws IdASException
	 */
	private void saveTimeLastUpdated() throws CardException {
		timeLastUpdated_ = new Date();
		card_.setTimeLastUpdated(new java.sql.Date(timeLastUpdated_.getTime()));
	}

	/**
	 * @throws IdASException
	 */
	private void saveMasterKey() throws CardException {
		card_.setMasterKey(masterKey_);
	}

	private void savePinDigest() throws CardException {
		card_.setPinDigest(pinDigest_);
	}

	private void saveIssuerID() throws CardException {
		card_.setIssuerID(issuerID_);
	}

	/**
	 * @throws IdASException
	 */
	private void saveClaims() throws CardException {
		Hashtable claims = SelfIssuedCardClaims.getSupportedTypesHash();
		Enumeration keys = claims.keys();
		while (keys.hasMoreElements()) {
			String type = (String)keys.nextElement();
			if (SelfIssuedCardClaims.PPID_TYPE.equals(type))
				continue;
			String value = (String)dirtyClaimValueMap_.get(type);
			if (value == null || value.trim().length() == 0) {
				IDaoPCardClaim cc = card_.getPersonalCardClaim(type);
				if (cc != null)
					cc.setDeleteState();
			}
			else {
				IDaoPCardClaim cc = card_.getPersonalCardClaim(type);
				if (cc == null) {
					cc = card_.createPersonalCardClaim();
					cc.setType(type);
					card_.addPersonalCardClaim(cc);
				}
				cc.setValue(value);
			}
		}
	}

/*
	private void saveClaims() throws CardException {
		Hashtable claims = SelfIssuedCardClaims.getSupportedTypesHash();
		Iterator values = tmpClaimValueList_.iterator();
		while (values.hasNext()) {
			ClaimValue claimValue = (ClaimValue) values.next();
			String val = claimValue.getValue();
			String type = claimValue.getType().getType();
			if (type == null)
				throw new CardException("Claim type is null.");
			if (claims.containsKey(type)) {
				if (val != null)
					claims.put(type, val);
			} else
				throw new CardException("Unsupported claim type : " + type);
		}
		Enumeration keys = claims.keys();
		while (keys.hasMoreElements()) {
			String type = (String) keys.nextElement();
			if (SelfIssuedCardClaims.PPID_TYPE.equals(type))
				continue;
			String value = (String) claims.get(type);
			IDaoPCardClaim cc = card_.getPersonalCardClaim(type);
			if (cc == null)
				cc = card_.createPersonalCardClaim();
			cc.setType(type);
			cc.setValue(value);
			if (cc.getState() == IDAO.NEW_OBJ)
				card_.addPersonalCardClaim(cc);
		}
	}
*/
	
	private byte[] convertUTF8ToUTF16(byte[] utf8) throws CardException {
		byte[] utf16 = null;
		if (utf8 != null) {
			try {
				String strUTF8 = new String(utf8, "UTF-8");
				utf16 = strUTF8.getBytes("UTF-16LE");
			} catch (Exception e) {
				log.error(e);
				throw new CardException(e);
			}
		}
		return utf16;
	}

	public void setPinCode(IPinCodeCredential pinCodeCredential) throws CardException {
		if (!editMode)
			throw new InvalidStateException("Not in edit mode.");
		byte[] oldPinDigest = null;
		byte[] oldMasterKey = null;
		EncryptedMasterKey oldEmk = null;
		byte[] newPinCodeUTF8 = (pinCodeCredential != null) ? pinCodeCredential.getPinCode() : null;
		byte[] newPinCodeUTF16 = convertUTF8ToUTF16(newPinCodeUTF8);
		oldPinDigest = pinDigest_;
		byte[] newSalt, newInitVector, decryptedMasterKey;
		if (oldPinDigest == null) {
			if (newPinCodeUTF8 == null) {
				log.debug("Attempt to reset pin code for unprotected card. Nothing to do.");
				return;
			}
			newSalt = CardCryptography.getRandomBytes(16);
			newInitVector = CardCryptography.getRandomBytes(16);
			oldMasterKey = decryptedMasterKey = masterKey_;
		} else {
			if (isClaimsRetrieved() == false)
				throw new CardException("Claims for this pin-protected card should be retrieved to get pin code.");
			oldEmk = currentMasterKey_;
			newSalt = oldEmk.getSalt();
			newInitVector = oldEmk.getInitVector();
			decryptedMasterKey = oldEmk.getDecryptedMasterKey();
			oldMasterKey = oldEmk.getEncryptedMasterKey();
		}
		try {
			pinDigest_ = (newPinCodeUTF16 != null) ? CardCryptography.getPinDigest(newPinCodeUTF16) : null;
			savePinDigest();
			if (newPinCodeUTF8 != null) {
				currentMasterKey_ = new EncryptedMasterKey(decryptedMasterKey, newSalt, newInitVector, newPinCodeUTF8);
				masterKey_ = currentMasterKey_.getEncryptedMasterKey();
			} else {
				currentMasterKey_ = null;
				masterKey_ = decryptedMasterKey;
			}
			saveMasterKey();
			Hashtable supportedClaimTypes = SelfIssuedCardClaims.getSupportedTypesHash();
			Enumeration claimTypes = supportedClaimTypes.keys();
			while (claimTypes.hasMoreElements()) {
				String claimType = (String) claimTypes.nextElement();
				String value = getClaimValue(claimType, oldEmk);
				setClaimValue(claimType, value, currentMasterKey_);
			}
			saveTimeLastUpdated();
		} catch (Exception e) {
			masterKey_ = oldMasterKey;
			pinDigest_ = oldPinDigest;
			currentMasterKey_ = oldEmk;
			log.error(e, e);
			throw new CardException(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.provider.cardspace.impl.InformationCard#getMasterKey()
	 */
	public byte[] getMasterKey() throws CardException {
		if (pinDigest_ != null) {
			if (isClaimsRetrieved() == false)
				throw new CardException("Claims for this pin-protected card should be retrieved to get pin code.");
			if (currentMasterKey_ != null)
				return currentMasterKey_.getDecryptedMasterKey();
			else
				throw new CardException("Claims for this pin-protected card were retrieved but currentMasterKey is null.");
		} else {
			return masterKey_;
		}
	}

	public void applyUpdates() throws InvalidStateException, CardException {
		if (!editMode)
			throw new InvalidStateException("Not in edit mode.");
		else {
			card_.store();
			editMode = false;
		}
	}

	public void beginUpdates() throws InvalidStateException {
		if (editMode) {
			throw new InvalidStateException("Already in edit mode.");
		} else {
			editMode = true;
		}
	}

	public void cancelUpdates() throws InvalidStateException {
		if (!editMode) {
			throw new InvalidStateException("Not in edit mode.");
		} else {
			reinitCard();
			editMode = false;
		}
	}

	private void reinitCard() throws InvalidStateException {
		try {
			initFromDAO();
			initClaims();
		} catch (Exception e) {
			log.error(e);
			new InvalidStateException(e);
		}
	}

	public boolean isEditMode() {
		return editMode;
	}

	public boolean isEditable() {
		return true;
	}

	public IClaim createClaim(String type) throws InvalidTypeException, ReadOnlyObjectException {
		throw new UnsupportedOperationException();
	}

	public IClaim setClaim(IClaim copyFrom) throws CardException {
		if (!editMode)
			throw new InvalidStateException("Not in edit mode.");
		if (copyFrom == null)
			throw new InvalidClaimException("Passed claim is null.");
		if (copyFrom instanceof ISimpleClaim == false)
			throw new InvalidClaimException("Provider \"" + getProvider().getName()
			        + "\" supports only claims with type " + ISimpleClaim.class.getName());
		ISimpleClaim simpleClaim = (ISimpleClaim) copyFrom;
		List lst = simpleClaim.getValues();
		if (lst == null)
			throw new InvalidClaimException("Passed claim does not contain the values list.");
		Object obj = (lst.size() != 0) ? (String) lst.get(0) : null;
		String value = (obj != null) ? obj.toString() : "";
		IClaimType cType = simpleClaim.getType();
		if (cType == null)
			throw new InvalidTypeException("Claim type is null.");
		String type = cType.getType();
		ClaimType newCT = SelfIssuedCardClaims.getClaimType(type);
		if (newCT == null)
			throw new CardException("Personal card does not support claim type :" + type);
		if (SelfIssuedCardClaims.PPID_TYPE.equals(type)) {
			// returns empty PPID claim
			return new ClaimValue(newCT, "", this);
		}
		SelfIssuedCardClaims.validateClaimValue_TEST(type, value);
		byte[] pinDigest = getPinDigest();
		if (pinDigest != null) {
			if (isClaimsRetrieved() == false)
				throw new CardException("Claims for this pin-protected card should be retrieved to get pin code.");
		}
		ClaimValue cv = new ClaimValue(newCT, value, this);
		setClaimValue(type, value, currentMasterKey_);
		ArrayList claims = new ArrayList();
		claims.add(cv);
		super.setClaims(claims);
		return cv;
	}

	private void setClaimValue(String type, String value, EncryptedMasterKey emk) throws CardException {
		if (SelfIssuedCardClaims.DATE_OF_BIRTH_TYPE.equals(type)) {
			value = SelfIssuedCardClaims.convertICMDate_TEST(value);
		}
		else if (SelfIssuedCardClaims.GENDER_TYPE.equals(type)) {
			value = SelfIssuedCardClaims.getGenderCode_TEST(value);
		}
		if (SelfIssuedCardClaims.PPID_TYPE.equals(type))
			return;
		String val = null;
		if (value != null && value.length() != 0) {
			if (emk != null) {
				byte[] newValue;
				try {
					newValue = CardCryptography.encryptPersonalCardField(value.getBytes("UTF-16LE"), emk.getSalt(), emk
					        .getInitVector(), emk.getPinCode());
				} catch (UnsupportedEncodingException e) {
					log.error(e, e);
					throw new CardException(e);
				}
				val = CardCryptography.encodeBase64(newValue, 1);
			} else
				val = value;
		}
		IDaoPCardClaim cc = card_.getPersonalCardClaim(type);
		if (val == null) {
			if (cc != null)
				cc.setDeleteState();
		}
		else {
			if (cc == null) {
				cc = card_.createPersonalCardClaim();
				cc.setType(type);
				card_.addPersonalCardClaim(cc);
			}
			cc.setValue(val);
		}
		dirtyClaimValueMap_.put(type, val);
		saveTimeLastUpdated();
	}

	/**
	 * @param type
	 * @param emk
	 * @return
	 * @throws CardException
	 */
	private String getClaimValue(String type, EncryptedMasterKey emk) throws CardException {
		// TODO PPID claim value can not be returned
		if (SelfIssuedCardClaims.PPID_TYPE.equals(type))
			return "";
		String value = null;
		IDaoPCardClaim pc = card_.getPersonalCardClaim(type);
		String dirtyVal = (pc != null) ? pc.getValue() : null;
		if (emk != null) {
			if (dirtyVal == null || dirtyVal.trim().length() == 0) {
				value = "";
			} else {
				byte[] encryptedData = CardCryptography.decodeBase64(dirtyVal);
				byte[] decryptedData = CardCryptography.decryptPersonalCardField(encryptedData, emk);
				try {
					value = new String(decryptedData, "UTF-16LE");
				} catch (UnsupportedEncodingException e) {
					log.error(e, e);
					throw new CardException(e);
				}
			}
		} else
			value = dirtyVal;
		if (SelfIssuedCardClaims.GENDER_TYPE.equals(type)) {
			value = SelfIssuedCardClaims.getGenderClaimValue_TEST(value);
		}
		return value;
	}

	protected void retrieveClaims(ICredential credential) throws AuthenticationRequiredException, AuthenticationException, CardException {
		currentMasterKey_ = getEncryptedMasterKey(credential);
		initClaims();
	}

	private void initClaims() throws CardException {
		ArrayList claimValueList = new ArrayList();
		ArrayList suppportedTypes = SelfIssuedCardClaims.getSupportedClaimTypeList();
		int size = suppportedTypes.size();
		try {
			for (int i = 0; i < size; i++) {
				Object obj = suppportedTypes.get(i);
				if (obj instanceof ClaimType == false)
					throw new CardException("Instance of " + ClaimType.class.getName() + " expected instead of " + obj.getClass().getName());
				ClaimType cType = (ClaimType) obj;
				String type = cType.getType();
				String val = getClaimValue(type, currentMasterKey_);
				ClaimValue cValue = new ClaimValue(cType, val, this);
				claimValueList.add(cValue);
			}
		} catch (CardException e) {
			log.error(e, e);
			throw new CardException(e);
		}
		super.setClaims(claimValueList);
	}

	private EncryptedMasterKey getEncryptedMasterKey(ICredential credential) throws AuthenticationException {
		EncryptedMasterKey emk = null;
		if (pinDigest_ != null) {
			if (credential == null)
				throw new AuthenticationException("Parameter \"credential\" is required for pin protected card.");
			Callback[] cbacks = credential.getCallbacks();
			int size = cbacks.length;
			PinCodeCallback pcc = null;
			for (int i = 0; i < size; i++) {
				Callback cback = cbacks[i];
				if (cback instanceof PinCodeCallback) {
					pcc = (PinCodeCallback) cback;
					break;
				}
			}
			if (pcc != null) {
				try {
					byte[] pin = pcc.getPin();
					byte[] digest = CardCryptography.getPinDigest(convertUTF8ToUTF16(pin));
					if (Arrays.equals(pinDigest_, digest))
						emk = new EncryptedMasterKey(masterKey_, pin);
					else
						throw new AuthenticationException("Wrong pin code.");
				} catch (CardException e) {
					log.error(e);
					throw new AuthenticationException(e);
				}
			} else {
				throw new AuthenticationException("Passed credential does not contain pin code callback ("
				        + PinCodeCallback.class.getName() + ")");
			}
		}
		return emk;
	}

	public void setIssuerName(String name) throws CardException {
	    // TODO Current implementation does not support isser name changing
    }

	public void lock(IPinCodeCredential credential) throws Exception {
		// TODO Auto-generated method stub
		
	}

	public void unlock(IPinCodeCredential credential) throws Exception {
		// TODO Auto-generated method stub
		
	}

	public boolean validatePINCode(String pinCode) throws Exception {
		// TODO Auto-generated method stub
		return false;
	}

	public IInformationCardExtension addExtension(IElement extension) throws CardException {
		//TODO implement this method
		throw new CardException ("Not implemented");
	}

	public IInformationCardExtension[] getExtensions() {
		//TODO implement this method
		return null;
	}

	public boolean hasExtensions() {
		//TODO implement this method
		return false;
	}

	public void removeExtension(IInformationCardExtension extension) throws CardException {
		//TODO implement this method
		throw new CardException ("Not implemented");
	}

}
