/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.icard.provider.cardspace.personal.db;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.PasswordCallback;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.CUID;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.ICard;
import org.eclipse.higgins.icard.ICardProvider;
import org.eclipse.higgins.icard.ICardTemplate;
import org.eclipse.higgins.icard.ITemplateValue;
import org.eclipse.higgins.icard.io.IElement;
import org.eclipse.higgins.icard.provider.cardspace.common.PersonalCardProvider;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.CardCryptography;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.PersonalCardTemplateHelper;
import org.eclipse.higgins.icard.provider.cardspace.db.ICardContext;
import org.eclipse.higgins.icard.provider.cardspace.db.IDaoPCard;
import org.eclipse.higgins.registry.IConfiguration;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class JDBCBasedPersonalCardProvider extends PersonalCardProvider implements ICardProvider {
	private Log log = LogFactory.getLog(JDBCBasedPersonalCardProvider.class);

	private static byte[] defaultImage = null;

	private static final String DEFAULT_IMAGE_FILE = "defaultImage.file";

	private static final String DEFAULT_IMAGE_TYPE = "defaultImage.mimeType";

	private ProviderConfiguration config_ = null;

	private String id_ = "org.eclipse.higgins.icard.provider.cardspace.personal.db";

	private String name_ = "CardSpace-compatable JDBC-based personal I-Card Provider Plug-in";

	private ICardContext cc_ = null;
	
	
/*	private List getCardsFromDb(UserCredentials userCreds) throws CardException {
		IDaoUserAccount userAccount = getCardContext().getUserAccount(userCreds.getUserID());
		ArrayList lst = new ArrayList();
		Iterator it = userAccount.getPCards().iterator();
		while (it.hasNext())
            try {
	            lst.add(new JDBCBasedPersonalCard(this, (IDaoPCard) it.next()));
            } catch (Exception e) {
            	log.error(e);
            	throw new CardException(e);
            }
		return lst;
	}
*/	
	
	private class UserCredentials {
		private String userID_;
		public UserCredentials(String userID) {
			userID_ = userID;
		}

		public String getUserID() {
			return userID_;
		}
	}

	private Hashtable getConfigurationProperties() {
		Hashtable ht = new Hashtable();
		IConfiguration cfg = getConfiguration();
		Iterator itr = cfg.getPropertyNames();
		while (itr.hasNext()) {
			String key = itr.next().toString();
			Object value = cfg.getProperty(key);
			ht.put(key, value);
		}
		return ht;
	}

	public void init() throws CardException {
		String cardContextClassName = getConfiguration().getProperty("cardcontext.classname");
		if (cardContextClassName == null)
			throw new CardException("Can not find \"cardcontext.classname\" configuration property");
		Class cls;
		try {
			cls = Class.forName(cardContextClassName);
			cc_ = (ICardContext) cls.newInstance();
		} catch (ClassNotFoundException e) {
			log.error(e);
			throw new CardException(e);
		} catch (InstantiationException e) {
			log.error(e);
			throw new CardException(e);
		} catch (IllegalAccessException e) {
			log.error(e);
			throw new CardException(e);
		}
		Hashtable cfg = getConfigurationProperties();
		cc_.init(cfg);
	}

	private ICardContext getCardContext() throws CardException {
		if (cc_ == null)
			init();
		return cc_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.registry.IServiceProvider#setID(java.lang.String)
	 */
	public void setID(String id) throws Exception {
		if (config_ == null) {
			id_ = id;
		} else {
			throw new Exception("Provider's id can't be modified.");
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.registry.IServiceProvider#getConfiguration()
	 */
	public IConfiguration getConfiguration() {
		if (config_ == null) {
			config_ = new ProviderConfiguration(getID());
		}
		return config_;
	}

//	public Iterator getICards(CallbackHandler authHandler, IPolicy policy) throws CardException {
	public Iterator getICards(CallbackHandler authHandler) throws CardException {
		if (authHandler == null)
			throw new CardException("Parameter \"authHandler\" is null");
		UserCredentials userCrds = authenticate(authHandler);
		ICardContext cc = getCardContext();
		String userID = userCrds.getUserID();
		ArrayList lst = new ArrayList();
		Iterator it = cc.getPCards(userID).iterator();
		while (it.hasNext())
			try {
				lst.add(new JDBCBasedPersonalCard(this, (IDaoPCard) it.next()));
			} catch (Exception e) {
				log.error(e);
				throw new CardException(e);
			}
			return lst.iterator();
	}

	public ICard importCard(CallbackHandler authHandler, String filename) throws CardException {
		if (authHandler == null)
			throw new CardException("Parameter \"authHandler\" is null");
		if (filename == null)
			throw new CardException("Parameter \"filename\" is null");
		FileInputStream fis;
		try {
			fis = new FileInputStream(filename);
		} catch (FileNotFoundException e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
		// TODO need to separate import of different file types
		if (filename.toUpperCase().endsWith(".CRDS")) {
			importFromCrdsFile(authHandler, fis);
			return null;
		} else if (filename.toUpperCase().endsWith(".CRD"))
			return importFromCrdFile(authHandler, fis);
		else
			throw new CardException("Unsupported file extension.");
	}

	// TODO pcard can not be imported from crd file. Need to remove this method
	private ICard importFromCrdFile(CallbackHandler authHandler, InputStream is) throws CardException {
		try {
			Element card = CardCryptography.getCardFromSignedEnvelop(is);
			UserCredentials userCrds = authenticate(authHandler);
			ICardContext cc = getCardContext();
			String userID = userCrds.getUserID();
			JDBCBasedPersonalCard crd = new JDBCBasedPersonalCard(this, cc, userID, card);
			/*//update cache
			ensureInCache(userCrds);
			updateCache(crd, userCrds.getUserID());*/
			return crd;
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
	}

	private void importFromCrdsFile(CallbackHandler authHandler, InputStream is) throws CardException {
		Document cardBackup;
		PasswordCallback pc = new PasswordCallback("Enter password for imported backup file: ", false);
		Callback[] callbacks = new Callback[] { pc };
		try {
			authHandler.handle(callbacks);
			String password = new String(pc.getPassword());
			pc.clearPassword();
			cardBackup = CardCryptography.decrypt(is, password);
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
		List cardsList = new ArrayList();
		NodeList newCards = cardBackup.getDocumentElement().getChildNodes();
		UserCredentials userCrds = authenticate(authHandler);
		ICardContext cc = getCardContext();
		String userID = userCrds.getUserID();
		for (int i = 0, j = newCards.getLength(); i < j; i++) {
			Node nd = newCards.item(i);
			if (nd.getNodeType() == Node.ELEMENT_NODE) {
				Element elm = (Element) nd;
				try {
					JDBCBasedPersonalCard crd = new JDBCBasedPersonalCard(this, cc, userID, elm);
					cardsList.add(crd);
				} catch (Exception e) {
					log.error(e);
				}
			}
		}
	}

	public void deleteCard(CallbackHandler authHandler, ICard card) throws CardException {
		if (authHandler == null)
			throw new CardException("Parameter \"authHandler\" is null");
		if (card == null)
			throw new CardException("Parameter \"card\" is null");
		UserCredentials userCrds = authenticate(authHandler);
		ICardContext cc = getCardContext();
		String userID = userCrds.getUserID();
		IDaoPCard crd = cc.getPCard(card.getID(), userID);
		if (crd != null) {
			crd.setDeleteState();
			crd.store();
		} else
			throw new CardException("Card not found.");
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICardProvider#getID()
	 */
	public String getID() {
		return id_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICardProvider#getName()
	 */
	public String getName() {
		return name_;
	}

	public String getDescription() {
		return "JDBC-based CardSpace-interoperable personal I-Card provider";
	}

	/*
	public boolean canImportICard(CallbackHandler authHandler, IElement element) {
		try {
			Element card = (Element) element.getAs(Element.class);
			JDBCBasedPersonalCard c = new JDBCBasedPersonalCard();
			c.initFromXML(card);
			return true;
		} catch (Exception e) {
			log.trace(e);
			return false;
		}
	}
	*/

//	public IElement[] exportICards(CallbackHandler authHandler, IElementFormat format, CUID[] cards) throws CardException {
//		Document doc = cards.getOwnerDocument();
//		if (format instanceof IInformationCard) {
//			IInformationCard icard = (IInformationCard) format;
//			Element roamingCard = icard.toXML(doc);
//			cards.appendChild(roamingCard);
//		} else
//			throw new CardException("Can not export card. This card is not an instance of IInformationCard.");
//	}

	public ICard importICard(CallbackHandler authHandler, IElement element) throws CardException {
		JDBCBasedPersonalCard pc = null;
		try {
			Element card = (Element) element.getAs(Element.class);
			Element crd = null;
			if ("Signature".equals(card.getLocalName()))
				crd = CardCryptography.getCardFromSignedEnvelop(card, true, true);
			else
				crd = card;
			UserCredentials userCrds = authenticate(authHandler);
			ICardContext cc = getCardContext();
			String userID = userCrds.getUserID();
			pc = new JDBCBasedPersonalCard(this, cc, userID, crd);
        } catch (Exception e) {
        	log.error(e);
        	throw new CardException(e);
        }
		return pc;
	}

	protected ICard getICardByID(CallbackHandler authHandler, String cardID) throws CardException {
		UserCredentials userCrds = authenticate(authHandler);
		String userID = userCrds.getUserID();
		ICardContext cc = getCardContext();
		IDaoPCard crd = cc.getPCard(cardID, userID);
		if (crd != null)
	        try {
	            return new JDBCBasedPersonalCard(this, crd);
            } catch (Exception e) {
            	log.error(e);
            	throw new CardException(e);
            }
        else
			return null;
	}

	public ICard getICardByCUID(CallbackHandler authHandler, CUID cuid) throws CardException {
		if (this.getID().equals(cuid.getProviderID()) == false) {
			log.debug("ProviderID of requested CUID (" + cuid.getProviderID() + ") is not equal to this providerID ("
			        + this.getID() + ").");
			return null;
		}
		ICard crd = getICardByID(authHandler, cuid.getCardID());
		if (crd == null) {
			log.debug("Can not find any card with ID = " + cuid.getCardID());
			return null;
		} else {
			String issuer = cuid.getIssuer();
			if (issuer != null && issuer.equals(crd.getIssuer()))
				return crd;
			else {
				log.debug("Wrong issuer for card with ID = " + cuid.getCardID() + ". Requested issuer = " + issuer
				        + ", real issuer = " + crd.getIssuer());
				return null;
			}
		}
	}

	private UserCredentials authenticate(CallbackHandler authHandler) throws CardException {
		NameCallback nc = new NameCallback("User name: ");
		Callback[] callbacks = new Callback[] { nc };
		try {
			authHandler.handle(callbacks);
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e);
		}
		String userName = nc.getName();

		if (userName == null || userName.trim().length() == 0)
			throw new CardException("Empty user name.");
		return new UserCredentials(userName);
	}

	public boolean canCreateCard(CallbackHandler authHandler, String id, Properties props) {
		if (authHandler == null) {
			log.info("Can not create card. Parameter \"authHandler\" is null.");
			return false;
		}
		if (id == null) {
			log.info("Can not create card. Parameter \"id\" is null.");
			return false;
		}
		try {
			UserCredentials userCrds = authenticate(authHandler);
			String userID = userCrds.getUserID();
			ICardContext cc = getCardContext();
			IDaoPCard crd = cc.getPCard(id, userID);
			if (crd != null) {
				log.info("Can not create card. Card with ID = " + id + " already exists.");
				return false;
			} else
				return true;
		} catch (Exception e) {
			log.error(e);
			return false;
		}
	}

	public ICard createCard(CallbackHandler authHandler, String id, Properties props) throws CardException {
		if (authHandler == null)
			throw new CardException("Parameter \"authHandler\" is null");
		if (id == null)
			throw new CardException("Parameter \"id\" is null");
		URI cardID = null;
		try {
			cardID = new URI(id);
		} catch (URISyntaxException e) {
			log.error(e);
			throw new CardException(e);
		}
		UserCredentials userCrds = authenticate(authHandler);
		String userID = userCrds.getUserID();
		ICardContext cc = getCardContext();
		try {
			JDBCBasedPersonalCard card = new JDBCBasedPersonalCard(this, cc, userID, cardID, null, null, null, null);
	        return card;
        } catch (Exception e) {
        	log.error(e);
        	throw new CardException(e);
        }
	}

	public String getPictureMIMEType(byte[] picure) {
		return "image/jpeg";
	}

	public String getDefaultPictureMIMEType() {
		return getConfiguration().getProperty(DEFAULT_IMAGE_TYPE);
	}

	public byte[] getDefaultImage() {
		if (defaultImage == null) {
			return readDefaultImage();
		}
		return defaultImage;
	}
	
	private synchronized byte[] readDefaultImage() {
		if (defaultImage != null) {
			return defaultImage;
		}
		String fileName = getConfiguration().getProperty(DEFAULT_IMAGE_FILE);
		if (fileName != null) {
			File file = new File(fileName);
			if (file.exists() && file.isFile()) {
				int size;
				long fileLength = file.length();
				if (fileLength <= Integer.MAX_VALUE)
					size = (int) fileLength;
				else {
					log.error("IdASBasedPersonalCardProvider.getDefaultImage - too long default image file size.");
					return null;
				}
				int offset = 0;
				int len = size;
				FileInputStream fis = null;
				try {
					fis = new FileInputStream(file);
					int count = 0;
					byte[] image = new byte[size];
					do {
						count = fis.read(image, offset, len);
						if (count != len) {
							offset += count;
							len -= count;
						}
					} while (count != -1);
					defaultImage = image;
				} catch (IOException e) {
					log.error(e);
				} finally {
					try {
						if (fis != null)
							fis.close();
					} catch (IOException e) {
						log.error(e);
					}
				}
			} else {
				log.error("IdASBasedPersonalCardProvider.getDefaultImage - default image file " + fileName
				        + " does not exist.");
			}
		} else {
			log.error("IdASBasedPersonalCardProvider.getDefaultImage - can not get the name of default image.");
		}
		return defaultImage;
	}

	public ICard createCard(CallbackHandler authHandler, ICardTemplate template) throws CardException {
		if (authHandler == null)
			throw new CardException("Parameter \"authHandler\" is null");
		String cardName = null;
		ITemplateValue nameTV = template.getTemplateValueByID(PersonalCardTemplateHelper.CARD_NAME);
		if (nameTV != null)
			cardName = nameTV.getValue();
		byte[] picture = null;
		String pictureType = null;
		ITemplateValue pictureTV = template.getTemplateValueByID(PersonalCardTemplateHelper.CARD_PICTURE);
		if (pictureTV != null) {
			String picStr = pictureTV.getValue();
			if (picStr != null) {
				picture = CardCryptography.decodeBase64(picStr);
				pictureType = getPictureMIMEType(picture);
			}
		}
		if (picture == null) {
			picture = getDefaultImage();
			pictureType = getDefaultPictureMIMEType();
		}
		HashMap dirtyClaims = PersonalCardTemplateHelper.getDirtyClaimsFromCardTemplate(template);
		UserCredentials userCrds = authenticate(authHandler);
		String userID = userCrds.getUserID();
		ICardContext cc = getCardContext();
		try {
			JDBCBasedPersonalCard card = new JDBCBasedPersonalCard(this, cc, userID, URI.create("urn:"
	                + org.apache.axiom.om.util.UUIDGenerator.getUUID()), cardName, dirtyClaims, picture, pictureType);
			/*//update cache !!!!
			if (!isNoCache) {
				ensureInCache(userCrds);
				updateCache(card, userCrds.getUserID());
			} */
	        return card;
        } catch (Exception e) {
        	log.error(e);
        	throw new CardException(e);
        }
	}

	public ICardTemplate[] getCardCreationTemplates(CallbackHandler authHandler) {
		return new ICardTemplate[] { PersonalCardTemplateHelper.getCardTemplate(this) };
	}

}
