/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.icard.provider.cardspace.personal;

import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;

import javax.security.auth.callback.Callback;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.AuthenticationException;
import org.eclipse.higgins.icard.AuthenticationRequiredException;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.ICardConstants;
import org.eclipse.higgins.icard.ICardProvider;
import org.eclipse.higgins.icard.IClaim;
import org.eclipse.higgins.icard.IClaimType;
import org.eclipse.higgins.icard.ISimpleClaim;
import org.eclipse.higgins.icard.InvalidClaimException;
import org.eclipse.higgins.icard.InvalidStateException;
import org.eclipse.higgins.icard.InvalidTypeException;
import org.eclipse.higgins.icard.ReadOnlyObjectException;
import org.eclipse.higgins.icard.auth.ICredential;
import org.eclipse.higgins.icard.auth.IPinCodeCredential;
import org.eclipse.higgins.icard.common.ClaimType;
import org.eclipse.higgins.icard.common.ClaimValue;
import org.eclipse.higgins.icard.common.auth.callback.PinCodeCallback;
import org.eclipse.higgins.icard.provider.cardspace.common.PersonalCard;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.CardCryptography;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.DateConvertor;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.EncryptedMasterKey;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.IdASContext;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.SelfIssuedCardClaims;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IAttributeValue;
import org.eclipse.higgins.idas.api.IComplexAttrValue;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.ISimpleAttrValue;
import org.eclipse.higgins.idas.api.ISingleValuedAttribute;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.api.NoSuchEntityException;
import org.w3c.dom.Element;

/**
 * Implementation of personal (self-issued) CardSpace-interoperable ICard stored
 * within IdAS context
 */
public class IdASBasedPersonalCard extends PersonalCard {

	private Log log = LogFactory.getLog(IdASBasedPersonalCard.class);

	private IEntity card_ = null;

	private IEntity claimList_ = null;

	private boolean editMode = false;

	private EncryptedMasterKey currentMasterKey_ = null;
	
	private ArrayList tmpClaimValueList_ = null;

	public IdASBasedPersonalCard() {
	}

	/**
	 * @param provider
	 * @param ctx
	 * @param cardId
	 * @param cardName
	 * @param claims
	 * @param image
	 * @param imageMimeType
	 * @throws CardException
	 */
	public IdASBasedPersonalCard(IdASBasedPersonalCardProvider provider, IContext ctx, URI cardId, String cardName,
			ArrayList claims, byte[] image, String imageMimeType) throws CardException {
		if (provider == null)
			throw new CardException("Parameter \"provider\" is null");
		if (cardId == null)
			throw new CardException("Parameter \"cardId\" is null");
		provider_ = provider;
		id_ = cardId;
		description_ = ICardConstants.I_CARD_TYPE_CARDSPACE;
		isSelfIssued_ = true;
		issuer_ = "http://schemas.xmlsoap.org/ws/2005/05/identity/issuer/self";
		issuerName_ = "Self";
		version_ = "1";
		timeIssued_ = new Date();
		supportedTokenTypes_ = new ArrayList();
		supportedTokenTypes_.add(URI.create("urn:oasis:names:tc:SAML:1.0:assertion"));
		supportedTokenTypes_.add(URI.create("http://docs.oasis-open.org/wss/oasis-wss-saml-token-profile-1.1#SAMLV1.1"));
		name_ = cardName != null ? cardName : "Selfish";
		masterKey_ = SecureRandom.getSeed(32);
		hashSalt_ = SecureRandom.getSeed(32);
		if (image != null && imageMimeType != null) {
			image_ = image;
			imageMimeType_ = imageMimeType;
		}
		claimTypes_ = SelfIssuedCardClaims.getSupportedClaimTypeList();
		if (claims != null)
			tmpClaimValueList_ = claims;
		else
			tmpClaimValueList_ = new ArrayList();
		try {
			try {
				card_ = ctx.getEntity(id_.toString());
				throw new CardException("Couldn't create card. Card with id = " + cardId + " already exists.");
			} catch (NoSuchEntityException e) {
				card_ = ctx.addEntity(IdASContext.ICARD_PersonalInformationCard, id_.toString());
				//saveClaimTypes();
			}
			initClaimListSubject(true);
			saveCard();
			ctx.applyUpdates();
		} catch (Exception e) {
			log.error(e);
			try {
				ctx.cancelUpdates();
			} catch (IdASException e1) {
				log.error(e1);
				throw new CardException(e1);
			}
			throw new CardException(e);
		}
	}

	/**
	 * @param provider
	 * @param card
	 * @throws Exception
	 */
	public IdASBasedPersonalCard(IdASBasedPersonalCardProvider provider, IEntity card) throws Exception {
		if (provider == null)
			throw new CardException("Parameter \"provider\" is null");
		if (card == null)
			throw new CardException("Parameter \"card\" is null");
		provider_ = provider;
		card_ = card;
		description_ = ICardConstants.I_CARD_TYPE_CARDSPACE;
		issuerName_ = "Self";
		issuer_ = "http://schemas.xmlsoap.org/ws/2005/05/identity/issuer/self";
		initClaimListSubject(false);
		initFromDS(card_, claimList_);
	}

	/**
	 * @param provider
	 * @param card
	 * @throws Exception
	 */
	public IdASBasedPersonalCard(IdASBasedPersonalCardProvider provider, IContext ctx, Element card) throws Exception {
		if (provider == null)
			throw new CardException("Parameter \"provider\" is null");
		if (card == null)
			throw new CardException("Parameter \"card\" is null");
		provider_ = provider;
		description_ = ICardConstants.I_CARD_TYPE_CARDSPACE;
		issuerName_ = "Self";
		issuer_ = "http://schemas.xmlsoap.org/ws/2005/05/identity/issuer/self";
		initFromXML(card);
		try {
			try {
				card_ = ctx.getEntity(id_.toString());
			} catch (NoSuchEntityException e) {
				card_ = ctx.addEntity(IdASContext.ICARD_PersonalInformationCard, id_.toString());
				//saveClaimTypes();
			}
			initClaimListSubject(true);
			saveCard();
			ctx.applyUpdates();
		} catch (Exception e) {
			log.error(e);
			try {
				ctx.cancelUpdates();
			} catch (Exception e1) {
				log.error(e1);
				throw new CardException(e1.getMessage());
			}
			throw new CardException(e.getMessage());
		}
	}

	/**
	 * @return
	 * @throws IdASException
	 * @throws CardException
	 */
	private IEntity createClaimListSubject() throws IdASException, CardException {
		if (card_ == null)
			throw new CardException("Variable \"card_\" is null.");
		IContext ctx = card_.getContext();

		// / TODO add subject ID generation
		String clamListSubjectID = String.valueOf(System.currentTimeMillis());
		// /
		IEntity claimList = ctx.addEntity(IdASContext.ICARD_ClaimList, clamListSubjectID);
		IComplexAttrValue cv = claimList.getSingleValuedAttribute(IdASContext.ICARD_cardReference).addComplexValue(null);
		cv.getSingleValuedAttribute(IdASContext.ICARD_subjectCardReference).addSimpleValue(null, card_.getEntityID());
		cv.getSingleValuedAttribute(IdASContext.ICARD_contextCardReference).addSimpleValue(null, ctx.getContextID());
		return claimList;
	}

	/**
	 * @param createNew
	 *            if true and card does not contain a reference to the node
	 *            with claim values than new claim list subject will be
	 *            created
	 * @throws IdASException
	 * @throws CardException
	 */
	private void initClaimListSubject(boolean createNew) throws IdASException, CardException {
		ISingleValuedAttribute attr = card_.getSingleValuedAttribute(IdASContext.ICARD_claimValueListRef);
		IAttributeValue val = attr.getValue();
		if (val != null) {
			if (val.isSimple() == false) {
				String subjID = null;
				IComplexAttrValue listRef = (IComplexAttrValue) val;
				Object subj = getSimpleValueData(listRef, IdASContext.ICARD_claimValueListSubject);
				if (subj != null)
					subjID = subj.toString();
				// String contextID = null;
				// Object context = getSimpleValueData(listRef,
				// IdASContext.ICARD_claimValueListContext);
				// if (context != null)
				// contextID = context.toString();
				IEntity claimList = card_.getContext().getEntity(subjID);
				if (claimList == null)
					throw new CardException("Couldn't get the claim list subject reference for card with ID = " + card_.getEntityID() + " .");
				claimList_ = claimList;
			} else
				throw new CardException("Attribute " + IdASContext.ICARD_claimValueListRef + " should be complex");
		} else {
			if (createNew == false)
				throw new CardException("Couldn't get the claim list reference for card with ID = " + card_.getEntityID()
						+ ". This card doesn't have any value with type" + IdASContext.ICARD_claimValueListRef + ".");
			claimList_ = createClaimListSubject();
			IComplexAttrValue cv = attr.addComplexValue(null);
			cv.getSingleValuedAttribute(IdASContext.ICARD_claimValueListSubject).addSimpleValue(null, claimList_.getEntityID());
			cv.getSingleValuedAttribute(IdASContext.ICARD_claimValueListContext).addSimpleValue(null, claimList_.getContext().getContextID().toString());
		}
	}

	/**
	 * @param card
	 * @param claimList
	 * @throws Exception
	 */
	public void initFromDS(IEntity card, IEntity claimList) throws Exception {
		initIsSelfIssued(card);
		initLanguage(card);
		initCardId(card);
		initCardVersion(card);
		initCardName(card);
		initCardImage(card);
		initCardImageType(card);
		initTimeIssued(card);
		initTimeExpires(card);
		initSupportedTokenTypes(card);
		initClaimTypes(card);
		initHashSalt(card);
		initTimeLastUpdated(card);
		initMasterKey(card);
		initPinDigest(card);
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 * @throws ClassCastException
	 */
	private void initIsSelfIssued(IEntity card) throws IdASException, CardException, ClassCastException {
		isSelfIssued_ = true;
		Boolean issi = (Boolean) getSimpleValueData(card, IdASContext.ICARD_isSelfIssued);
		if (issi != null && issi.booleanValue() == false)
			throw new CardException("Can not init managed card with Entity storing data of self-issued (Personal) card");
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 * @throws ClassCastException
	 */
	private void initLanguage(IEntity card) throws IdASException, CardException, ClassCastException {
		language_ = (String) getSimpleValueData(card, IdASContext.ICARD_language);
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 * @throws ClassCastException
	 */
	private void initCardId(IEntity card) throws IdASException, CardException, ClassCastException {
		String id = (String) getSimpleValueData(card, IdASContext.HIGGINS_uniqueIdentifier);
		try {
			id_ = new URI(id);
		} catch (URISyntaxException e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 * @throws ClassCastException
	 */
	private void initCardVersion(IEntity card) throws IdASException, CardException, ClassCastException {
		version_ = (String) getSimpleValueData(card, IdASContext.ICARD_version);
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 * @throws ClassCastException
	 */
	private void initCardName(IEntity card) throws IdASException, CardException, ClassCastException {
		name_ = (String) getSimpleValueData(card, IdASContext.ICARD_name);
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 * @throws ClassCastException
	 */
	private void initCardImage(IEntity card) throws IdASException, CardException, ClassCastException {
		image_ = (byte[]) getSimpleValueData(card, IdASContext.ICARD_image);
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 * @throws ClassCastException
	 */
	private void initCardImageType(IEntity card) throws IdASException, CardException, ClassCastException {
		imageMimeType_ = (String) getSimpleValueData(card, IdASContext.ICARD_imageMimeType);
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 */
	private void initTimeIssued(IEntity card) throws IdASException, CardException {
		timeIssued_ = null;
		Object obj = getSimpleValueData(card, IdASContext.ICARD_timeIssued);
		if (obj != null) {
			if (obj instanceof Date)
				timeIssued_ = (Date) obj;
			else
				timeIssued_ = DateConvertor.parse(obj.toString());
		}
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 */
	private void initTimeExpires(IEntity card) throws IdASException, CardException {
		timeExpires_ = null;
		Object obj = getSimpleValueData(card, IdASContext.ICARD_expiredTime);
		if (obj != null) {
			if (obj instanceof Date)
				timeExpires_ = (Date) obj;
			else
				timeExpires_ = DateConvertor.parse(obj.toString());
		}
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 */
	private void initSupportedTokenTypes(IEntity card) throws IdASException, CardException {
		supportedTokenTypes_ = new ArrayList();
		IAttribute attr = card.getAttribute(IdASContext.ICARD_supportedTokenType);
		Iterator itr = attr.getValues();
		while (itr.hasNext()) {
			IAttributeValue val = (IAttributeValue) itr.next();
			if (val.isSimple()) {
				ISimpleAttrValue sv = (ISimpleAttrValue) val;
				if (sv != null) {
					Object obj = sv.getData();
					if (obj != null) {
						if (obj instanceof URI)
							supportedTokenTypes_.add(obj);
						else {
							try {
								supportedTokenTypes_.add(new URI(obj.toString()));
							} catch (URISyntaxException e) {
								log.error(e);
								throw new CardException(e);
							}
						}
					}
				}
			} else
				throw new CardException("Attribute " + IdASContext.ICARD_supportedTokenType + " contains non simple value");
		}
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 */
	private void initClaimTypes(IEntity card) throws IdASException, CardException {
		claimTypes_ = SelfIssuedCardClaims.getSupportedClaimTypeList();
/*
		IAttribute attr = card.getAttribute(IdASContext.ICARD_supportedClaimType);
		Iterator itr = attr.getValues();
		while (itr.hasNext()) {
			IAttributeValue val = (IAttributeValue) itr.next();
			if (val.isSimple() == false) {
				IComplexAttrValue cv = (IComplexAttrValue) val;
				if (cv != null) {
					URI type = (URI) getSimpleValueData(cv, IdASContext.ICARD_claimTypeType);
					String name = (String) getSimpleValueData(cv, IdASContext.ICARD_claimTypeName);
					String description = (String) getSimpleValueData(cv, IdASContext.ICARD_claimTypeDescription);
					ClaimType ct = new ClaimType(type, name, description);
					claimTypes_.add(ct);
				}
			}
		}
*/	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 * @throws ClassCastException
	 */
	private void initHashSalt(IEntity card) throws IdASException, CardException, ClassCastException {
		hashSalt_ = (byte[]) getSimpleValueData(card, IdASContext.ICARD_hashSalt);
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 */
	private void initTimeLastUpdated(IEntity card) throws IdASException, CardException {
		timeLastUpdated_ = null;
		Object obj = getSimpleValueData(card, IdASContext.ICARD_timeLastUpdated);
		if (obj != null) {
			if (obj instanceof Date)
				timeLastUpdated_ = (Date) obj;
			else
				timeLastUpdated_ = DateConvertor.parse(obj.toString());
		}
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 * @throws ClassCastException
	 */
	private void initMasterKey(IEntity card) throws IdASException, CardException, ClassCastException {
		masterKey_ = (byte[]) getSimpleValueData(card, IdASContext.ICARD_masterKey);
	}

	/**
	 * @param card
	 * @throws IdASException
	 * @throws CardException
	 * @throws ClassCastException
	 */
	private void initPinDigest(IEntity card) throws IdASException, CardException, ClassCastException {
		pinDigest_ = (byte[]) getSimpleValueData(card, IdASContext.ICARD_pinDigest);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#setName(java.lang.String)
	 */
	public void setName(String name) throws CardException {
		if (!editMode)
			throw new InvalidStateException("Not in edit mode.");
		if (name == null || name.trim().length() == 0)
			throw new CardException("Couldn't set empty card name.");
		String oldName = name_;
		try {
			name_ = name.trim();
			saveCardName();
			saveTimeLastUpdated();
		} catch (Exception e) {
			name_ = oldName;
			log.error(e);
			throw new CardException(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#setImage(byte[], java.lang.String)
	 */
	public void setImage(byte[] image, String imageMimeType) throws CardException {
		if (!editMode)
			throw new InvalidStateException("Not in edit mode.");
		if (image == null)
			throw new CardException("Couldn't set null card image.");
		if (imageMimeType == null)
			throw new CardException("Couldn't set null card image MIME type.");
		byte[] oldImage = image_;
		String oldImageType = imageMimeType_;
		image_ = image;
		imageMimeType_ = imageMimeType;
		try {
			saveCardImage();
			saveCardImageType();
			saveTimeLastUpdated();
		} catch (Exception e) {
			image_ = oldImage;
			imageMimeType_ = oldImageType;
			log.error(e);
			throw new CardException(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#setIssuerName(java.lang.String)
	 */
	public void setIssuerName(String name) throws CardException {
		if (!editMode)
			throw new InvalidStateException("Not in edit mode.");
		if (name == null || name.trim().length() == 0)
			throw new CardException("Couldn't set empty card issuer name.");
		String oldName = issuerName_;
		try {
			issuerName_ = name.trim();
			saveIssuerName();
			saveTimeLastUpdated();
		} catch (Exception e) {
			issuerName_ = oldName;
			throw new CardException(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#setExpiredTime(java.util.Date)
	 */
	public void setTimeExpires(Date date) throws CardException {
		if (!editMode)
			throw new InvalidStateException("Not in edit mode.");
		Date oldDate_ = timeExpires_; 
		try {
			timeExpires_ = date;
			saveTimeExpires();
			saveTimeLastUpdated();
		} catch (Exception e) {
			timeExpires_ = oldDate_;
			throw new CardException(e);
		}
	}

	/**
	 * Save the card. Must be used only when create
	 * 
	 * @throws Exception
	 */
	private void saveCard() throws IdASException, CardException {
		saveIsSelfIssued();
		saveLanguage();
		saveCardId();
		saveCardVersion();
		saveCardName();
		saveCardImage();
		saveCardImageType();
		saveIssuer();
		saveIssuerName();
		saveTimeIssued();
		saveTimeExpires();
		saveSupportedTokenTypes();
		//saveClaimTypes();
		saveClaims();
		saveHashSalt();
		saveTimeLastUpdated();
		saveMasterKey();
		savePinDigest();
	}

	/**
	 * @throws IdASException
	 */
	private void saveIsSelfIssued() throws IdASException {
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.ICARD_isSelfIssued);
		attr.addSimpleValue(null, Boolean.valueOf(isSelfIssued_));
	}

	/**
	 * @throws IdASException
	 */
	private void saveLanguage() throws IdASException {
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.ICARD_language);
		attr.addSimpleValue(null, language_);
	}

	/**
	 * @throws IdASException
	 */
	private void saveCardId() throws IdASException {
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.HIGGINS_uniqueIdentifier);
		attr.addSimpleValue(null, id_);
	}

	/**
	 * @throws IdASException
	 */
	private void saveCardVersion() throws IdASException {
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.ICARD_version);
		attr.addSimpleValue(null, version_);
	}

	/**
	 * @throws IdASException
	 */
	private void saveCardName() throws IdASException {
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.ICARD_name);
		attr.addSimpleValue(null, name_);
	}

	/**
	 * @throws IdASException
	 */
	private void saveCardImage() throws IdASException {
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.ICARD_image);
		attr.addSimpleValue(null, image_);
	}

	/**
	 * @throws IdASException
	 */
	private void saveCardImageType() throws IdASException {
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.ICARD_imageMimeType);
		attr.addSimpleValue(null, imageMimeType_);
	}

	/**
	 * @throws IdASException
	 */
	private void saveIssuer() throws IdASException {
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.ICARD_issuer);
		attr.addSimpleValue(null, issuer_);
	}

	/**
	 * @throws IdASException
	 */
	private void saveIssuerName() throws IdASException {
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.ICARD_issuerName);
		attr.addSimpleValue(null, issuerName_);
	}

	/**
	 * @throws IdASException
	 */
	private void saveTimeIssued() throws IdASException {
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.ICARD_timeIssued);
		attr.addSimpleValue(null, timeIssued_);
	}

	/**
	 * @throws IdASException
	 */
	private void saveTimeExpires() throws IdASException {
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.ICARD_expiredTime);
		attr.addSimpleValue(null, timeExpires_);
	}

	/**
	 * @throws IdASException
	 */
	private void saveSupportedTokenTypes() throws IdASException {
		IAttribute attr = card_.getAttribute(IdASContext.ICARD_supportedTokenType);
		Iterator olds = attr.getValues();
		while (olds.hasNext()) {
			IAttributeValue old = (IAttributeValue) olds.next();
			old.remove();
		}
		Iterator it = supportedTokenTypes_.iterator();
		while (it.hasNext()) {
			URI uri = (URI) it.next();
			attr.addSimpleValue(null, uri);
		}
	}

	/**
	 * @throws IdASException
	 */
/*	private void saveClaimTypes() throws IdASException {
		IAttribute attr = card_.getAttribute(IdASContext.ICARD_supportedClaimType);
		Iterator olds = attr.getValues();
		while (olds.hasNext()) {
			IAttributeValue old = (IAttributeValue) olds.next();
			old.remove();
		}
		Iterator it = claimTypes_.iterator();
		while (it.hasNext()) {
			IClaimType claimType = (IClaimType) it.next();
			IComplexAttrValue val = attr.addComplexValue(null);
			val.getSingleValuedAttribute(IdASContext.ICARD_claimTypeType).addSimpleValue(null, claimType.getType());
			val.getSingleValuedAttribute(IdASContext.ICARD_claimTypeDescription).addSimpleValue(null, claimType.getDescription());
			val.getSingleValuedAttribute(IdASContext.ICARD_claimTypeName).addSimpleValue(null, claimType.getDisplayName());
		}
	}
*/
	/**
	 * @throws IdASException
	 */
	private void saveHashSalt() throws IdASException {
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.ICARD_hashSalt);
		attr.addSimpleValue(null, hashSalt_);
	}

	/**
	 * @throws IdASException
	 */
	private void saveTimeLastUpdated() throws IdASException {
		timeLastUpdated_ = new Date();
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.ICARD_timeLastUpdated);
		attr.addSimpleValue(null, timeLastUpdated_);
	}

	/**
	 * @throws IdASException
	 */
	private void saveMasterKey() throws IdASException {
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.ICARD_masterKey);
		attr.addSimpleValue(null, masterKey_);
	}

	/**
	 * @throws IdASException
	 */
	private void savePinDigest() throws IdASException {
		IAttribute attr = card_.getSingleValuedAttribute(IdASContext.ICARD_pinDigest);
		attr.addSimpleValue(null, pinDigest_);
	}

	/**
	 * @throws IdASException
	 */
	private void saveClaims() throws IdASException, CardException {
		Hashtable claims = SelfIssuedCardClaims.getSupportedTypesHash();
		// allTypes.remove();
		Iterator values = tmpClaimValueList_.iterator();
		while (values.hasNext()) {
			ClaimValue claimValue = (ClaimValue) values.next();
			String val = claimValue.getValue();
			String type = claimValue.getType().getType();
			if (type == null)
				throw new CardException("Claim type is null.");
			if (claims.containsKey(type)) {
				if (val != null)
					claims.put(type, val);
			} else
				throw new CardException("Unsupported claim type : " + type);
		}
		Enumeration keys = claims.keys();
		while (keys.hasMoreElements()) {
			String type = (String) keys.nextElement();
			if (SelfIssuedCardClaims.PPID_TYPE.equals(type))
				continue;
			String value = (String) claims.get(type);
			URI typeURI = null;
			try {
				typeURI = new URI(type);
			} catch (Exception e) {
				log.error(e);
				throw new CardException("Wrong claim type URI : \n" + e.getMessage());
			}
			ISingleValuedAttribute sa = claimList_.getSingleValuedAttribute(typeURI);
			if (value.trim().length() == 0)
				sa.remove();
			else
				sa.addSimpleValue(null, value.trim());
		}
	}

	/**
	 * @return
	 * @throws CardException
	 */
	public IEntity getCardSubject() throws CardException {
		return card_;
	}

	/**
	 * @return
	 * @throws CardException
	 */
	public IEntity getClaimListSubject() throws CardException {
		return claimList_;
	}

	/**
	 * @param hint
	 * @param pinDigest
	 * @return
	 * @throws CardException
	 */
/*
	private byte[] askPinCode(String hint, byte[] pinDigest) throws CardException {
		PasswordCallback pc = new PasswordCallback(hint, false);
		Callback[] callbacks = new Callback[] { pc };
		try {
			callbackHandler_.handle(callbacks);
			char[] pass = pc.getPassword();
			pc.clearPassword();
			if (pass == null || pass.length == 0)
				throw new CardException("User did not enter pin code.");
			String pinCode = new String(pass);
			byte[] digest = CardCryptography.getPinDigest(pinCode);
			if (Arrays.equals(pinDigest, digest))
				return pinCode.getBytes("UTF-8");
			else
				throw new CardException("User provided the wrong pin code.");
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
	}
*/
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.provider.cardspace.impl.PersonalCard#getClaims()
	 */
//	public Iterator getClaims() throws CardException {
//		ArrayList claimValueList = new ArrayList();
//		if (claimList_ == null)
//			return null;
//		byte[] pin = null;
//		EncryptedMasterKey emk = null;
//		byte[] pinDigest = (byte[]) getSimpleValueData(card_, IdASContext.ICARD_pinDigest);
//		if (pinDigest != null) {
//			pin = askPinCode("Enter pin code: ", pinDigest);
//			emk = new EncryptedMasterKey(masterKey_, pin);
//		}
//		ArrayList suppportedTypes = SelfIssuedCardClaims.getSupportedClaimTypeList();
//		int size = suppportedTypes.size();
//		for (int i = 0; i < size; i++) {
//			Object obj = suppportedTypes.get(i);
//			if (obj instanceof ClaimType == false)
//				throw new CardException("Instance of " + ClaimType.class.getName() + " expected instead of " + obj.getClass().getName());
//			ClaimType cType = (ClaimType)obj;
//			String type = cType.getType();
//			String val = getClaimValue(type, emk);
//			ClaimValue cValue = new ClaimValue(cType, val, this);
//			claimValueList.add(cValue);
//		}
//		return claimValueList.iterator();
//	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.provider.cardspace.impl.InformationCard#getProvider()
	 */
	public ICardProvider getProvider() {
		return provider_;
	}

	/**
	 * @param val
	 * @param attribure
	 * @return
	 * @throws CardException
	 */
	private Object getSimpleValueData(IComplexAttrValue val, URI attribure) throws CardException {
		try {
			ISingleValuedAttribute attr = val.getSingleValuedAttribute(attribure);
			return getSimpleValueData(attr);
		} catch (IdASException e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
	}

	/**
	 * @param subj
	 * @param attribure
	 * @return
	 * @throws CardException
	 */
	private Object getSimpleValueData(IEntity subj, URI attribure) throws CardException {
		try {
			ISingleValuedAttribute attr = subj.getSingleValuedAttribute(attribure);
			return getSimpleValueData(attr);
		} catch (IdASException e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
	}

	/**
	 * @param attr
	 * @return
	 * @throws IdASException
	 * @throws CardException
	 */
	private Object getSimpleValueData(ISingleValuedAttribute attr) throws IdASException, CardException {
		if (attr != null) {
			IAttributeValue v = attr.getValue();
			if (v != null) {
				if (v.isSimple()) {
					ISimpleAttrValue sv = (ISimpleAttrValue) v;
					return sv.getData();
				} else
					throw new CardException("Attribute " + (attr.getAttrID() != null ? attr.getAttrID().toString() : "") + " contains non simple value");
			}
		}
		return null;
	}

	private byte[] convertUTF8ToUTF16(byte[] utf8) throws CardException {
		byte[] utf16 = null;
		if (utf8 != null) {
			try {
				String strUTF8 = new String(utf8, "UTF-8");
				utf16 = strUTF8.getBytes("UTF-16LE");
			}
			catch (Exception e) {
				log.error(e);
				throw new CardException(e.getMessage());
			}
		}
		return utf16;
	}

	/**
	 * @param authHandler
	 * @param pinCodeCredential
	 *            Pin code used to protect the card. If pinCode is null,
	 *            protection will be reset.
	 * @throws CardException
	 */
	public void setPinCode(IPinCodeCredential pinCodeCredential) throws CardException {
		if (!editMode)
			throw new InvalidStateException("Not in edit mode.");
		byte[] oldPinDigest = null;
		byte[] oldMasterKey = null;
		EncryptedMasterKey oldEmk = null;
		byte[] newPinCodeUTF8 = (pinCodeCredential != null) ? pinCodeCredential.getPinCode() : null;
		byte[] newPinCodeUTF16 = convertUTF8ToUTF16(newPinCodeUTF8);
		oldPinDigest = pinDigest_;
		byte[] newSalt, newInitVector, decryptedMasterKey;
		if (oldPinDigest == null) {
			if (newPinCodeUTF8 == null) {
				log.debug("Attempt to reset pin code for unprotected card. Nothing to do.");
				return;
			}
			newSalt = CardCryptography.getRandomBytes(16);
			newInitVector = CardCryptography.getRandomBytes(16);
			oldMasterKey = decryptedMasterKey = masterKey_;
		}
		else {
			if (isClaimsRetrieved() == false)
				throw new CardException("Claims for this pin-protected card should be retrieved to get pin code.");
			oldEmk = currentMasterKey_;
			newSalt = oldEmk.getSalt();
			newInitVector = oldEmk.getInitVector();
			decryptedMasterKey = oldEmk.getDecryptedMasterKey();
			oldMasterKey = oldEmk.getEncryptedMasterKey();
		}
		try {
			pinDigest_ = (newPinCodeUTF16 != null) ? CardCryptography.getPinDigest(newPinCodeUTF16) : null;
			savePinDigest();
			if (newPinCodeUTF8 != null) {
				currentMasterKey_ = new EncryptedMasterKey(decryptedMasterKey, newSalt, newInitVector, newPinCodeUTF8);
				masterKey_ = currentMasterKey_.getEncryptedMasterKey(); 
			}
			else {
				currentMasterKey_ = null;
				masterKey_ = decryptedMasterKey;
			}
			saveMasterKey();
			Hashtable supportedClaimTypes = SelfIssuedCardClaims.getSupportedTypesHash();
			Enumeration claimTypes = supportedClaimTypes.keys();
			while (claimTypes.hasMoreElements()) {
				String claimType = (String)claimTypes.nextElement();
				String value = getClaimValue(claimType, oldEmk);
				setClaimValue(claimType, value, currentMasterKey_);
			}
			saveTimeLastUpdated();
		} catch (Exception e) {
			masterKey_ = oldMasterKey;
			pinDigest_ = oldPinDigest;
			currentMasterKey_ = oldEmk;
			log.error(e);
			throw new CardException(e.getMessage());
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.provider.cardspace.impl.InformationCard#getMasterKey()
	 */
	public byte[] getMasterKey() throws CardException {
		if (pinDigest_ != null) {
			if (isClaimsRetrieved() == false)
				throw new CardException("Claims for this pin-protected card should be retrieved to get pin code.");
			if(currentMasterKey_ != null)
				return currentMasterKey_.getDecryptedMasterKey();
			else
				throw new CardException("Claims for this pin-protected card were retrieved but currentMasterKey is null.");
		}
		else {
			return masterKey_;
		}
	}

	public void applyUpdates() throws InvalidStateException, CardException {
		if (!editMode) {
			throw new InvalidStateException("Not in edit mode.");
		} else {
			try {
				card_.getContext().applyUpdates();
				editMode = false;
			} catch (IdASException e) {
				log.error(e);
				throw new CardException(e.getMessage());
			}
		}
	}

	public void beginUpdates() throws InvalidStateException {
		if (editMode) {
			throw new InvalidStateException("Already in edit mode.");
		} else {
			editMode = true;
		}
	}

	private void reinitCard() throws InvalidStateException {
		try {
			initFromDS(card_, claimList_);
			initClaims();
		}
		catch(Exception e) {
			new InvalidStateException(e.getMessage());
		}
	}

	public void cancelUpdates() throws InvalidStateException {
		if (!editMode) {
			throw new InvalidStateException("Not in edit mode.");
		} else {
			try {
				card_.getContext().cancelUpdates();
				reinitCard();
				editMode = false;
			} catch (IdASException e) {
				log.error(e);
				throw new InvalidStateException(e.getMessage());
			}
		}
	}

	public boolean isEditMode() {
		return editMode;
	}

	public boolean isEditable() {
		return true;
	}

	public IClaim createClaim(String type) throws InvalidTypeException, ReadOnlyObjectException {
		throw new UnsupportedOperationException();
	}

	public IClaim setClaim(IClaim copyFrom) throws InvalidClaimException, InvalidTypeException, ReadOnlyObjectException, CardException {
		if (!editMode)
			throw new InvalidStateException("Not in edit mode.");
		if (copyFrom == null)
			throw new InvalidClaimException("Passed claim is null.");
		if (copyFrom instanceof ISimpleClaim == false)
			throw new InvalidClaimException("Provider \"" + getProvider().getName() + "\" supports only claims with type " + ISimpleClaim.class.getName());
		ISimpleClaim simpleClaim = (ISimpleClaim) copyFrom;
		List lst = simpleClaim.getValues();
		if (lst == null)
			throw new InvalidClaimException("Passed claim does not contain the values list.");
		Object obj = (lst.size() != 0) ? (String) lst.get(0) : null;
		String value = (obj != null) ? obj.toString() : "";
		IClaimType cType = simpleClaim.getType();
		if (cType == null)
			throw new InvalidTypeException("Claim type is null.");
		String type = cType.getType();
		ClaimType newCT = SelfIssuedCardClaims.getClaimType(type);
		if (newCT == null)
			throw new CardException("Personal card does not support claim type :" + type);
		if (SelfIssuedCardClaims.PPID_TYPE.equals(type)) {
			//returns empty PPID claim
			return new ClaimValue(newCT, "", this);
		}
		byte[] pinDigest = getPinDigest();
		if (pinDigest != null) {
			if (isClaimsRetrieved() == false)
				throw new CardException("Claims for this pin-protected card should be retrieved to get pin code.");
		}
		ClaimValue cv = new ClaimValue(newCT, value, this);
		try {
			setClaimValue(type, value, currentMasterKey_);
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e);
		}
		return cv;
	}

	private void setClaimValue(String type, String value, EncryptedMasterKey emk) throws CardException, UnsupportedEncodingException, IdASException {
		//TODO PPID claim value can not be stored
		if (SelfIssuedCardClaims.PPID_TYPE.equals(type))
			return;
		String val = null;
		if (value != null && value.length() != 0) {
			if (emk != null) {
				byte[] newValue = CardCryptography.encryptPersonalCardField(value.getBytes("UTF-16LE"), emk.getSalt(), emk.getInitVector(), emk.getPinCode());
				val = CardCryptography.encodeBase64(newValue, 1);
			} else
				val = value;
		}
		URI typeURI = null;
		try {
			typeURI = new URI(type);
		} catch (URISyntaxException e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
		claimList_.getSingleValuedAttribute(typeURI).addSimpleValue(null, val);
		saveTimeLastUpdated();
	}

	/**
	 * @param type
	 * @param emk
	 * @return
	 * @throws CardException
	 */
	private String getClaimValue(String type, EncryptedMasterKey emk) throws CardException {
		//TODO PPID claim value can not be returned 
		if (SelfIssuedCardClaims.PPID_TYPE.equals(type))
			return ""; 
		URI typeURI = null;
		try {
			typeURI = new URI(type);
		} catch (URISyntaxException e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
		String value = null;
		String dirtyVal = (String) getSimpleValueData(claimList_, typeURI);
		if (emk != null) {
			if (dirtyVal == null || dirtyVal.trim().length() == 0) {
				value = "";
			} else {
				byte[] encryptedData = CardCryptography.decodeBase64(dirtyVal);
				byte[] decryptedData = CardCryptography.decryptPersonalCardField(encryptedData, emk);
				try {
					value = new String(decryptedData, "UTF-16LE");
				} catch (UnsupportedEncodingException e) {
					log.error(e);
					throw new CardException(e.getMessage());
				}
			}
		} else
			value = dirtyVal;
		return value;
	}

	protected void retrieveClaims(ICredential credential) throws AuthenticationRequiredException, AuthenticationException, CardException {
		if (claimList_ == null)
			return;
		currentMasterKey_ = getEncryptedMasterKey(credential);
		initClaims();
	}

	private void initClaims() throws CardException {
		ArrayList claimValueList = new ArrayList();
		ArrayList suppportedTypes = SelfIssuedCardClaims.getSupportedClaimTypeList();
		int size = suppportedTypes.size();
		for (int i = 0; i < size; i++) {
			Object obj = suppportedTypes.get(i);
			if (obj instanceof ClaimType == false)
				throw new CardException("Instance of " + ClaimType.class.getName() + " expected instead of " + obj.getClass().getName());
			ClaimType cType = (ClaimType)obj;
			String type = cType.getType();
			String val = getClaimValue(type, currentMasterKey_);
			ClaimValue cValue = new ClaimValue(cType, val, this);
			claimValueList.add(cValue);
		}
		setClaims(claimValueList);
	}

	private EncryptedMasterKey getEncryptedMasterKey(ICredential credential) throws AuthenticationException {
		EncryptedMasterKey emk = null;
		if (pinDigest_ != null) {
			if (credential == null)
				throw new AuthenticationException("Parameter \"credential\" is required for pin protected card.");
			Callback[] cbacks = credential.getCallbacks();
			int size = cbacks.length;
			PinCodeCallback pcc = null;
			for (int i = 0; i < size; i++) {
				Callback cback = cbacks[i];
				if (cback instanceof PinCodeCallback) {
					pcc = (PinCodeCallback)cback;
					break;
				}
			}
			if (pcc != null) {
				try {
					byte[] pin = pcc.getPin();
					byte[] digest = CardCryptography.getPinDigest(convertUTF8ToUTF16(pin));
					if (Arrays.equals(pinDigest_, digest))
						emk = new EncryptedMasterKey(masterKey_, pin);
					else
						throw new AuthenticationException("Wrong pin code.");
				} catch (CardException e) {
					log.error(e);
					throw new AuthenticationException(e);
				}
			} else {
				throw new AuthenticationException("Passed credential does not contain pin code callback (" + PinCodeCallback.class.getName() + ")");
			}
		}
		return emk;
	}

}
