/*******************************************************************************
 * Copyright (c) 2006 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Anthony Bussani - Initial API and implementation
 *     Thomas Gross - Initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.icard.provider.securestorage;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.PasswordCallback;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.axiom.om.OMElement;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.cardstore.CardStoreBuilderFactory;
import org.eclipse.higgins.cardstore.ICardStoreBuilder;
import org.eclipse.higgins.cardstore.exceptions.DuplicateCardIdException;
import org.eclipse.higgins.cardstore.exceptions.ExpectedObjectNotPresent;
import org.eclipse.higgins.cardstore.exceptions.StoreDecryptionException;
import org.eclipse.higgins.cardstore.exceptions.StoreEncryptionException;
import org.eclipse.higgins.cardstore.exceptions.UnsupportedObjectModel;
import org.eclipse.higgins.cardstore.schemas._2005._05.identity.IEncryptedStore;
import org.eclipse.higgins.cardstore.schemas._2005._05.identity.IRoamingInformationCard;
import org.eclipse.higgins.cardstore.schemas._2005._05.identity.IRoamingStore;
import org.eclipse.higgins.cardstore.schemas._2005._05.identity.impl.RoamingStore;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.CardStoreException;
import org.eclipse.higgins.icard.CardStoreStrategy;
import org.eclipse.higgins.icard.ICard;
import org.eclipse.higgins.icard.ICardProvider;
import org.eclipse.higgins.icard.provider.cardspace.common.ManagedCard;
import org.eclipse.higgins.icard.provider.cardspace.common.PersonalCard;
import org.eclipse.higgins.registry.IConfiguration;
import org.eclipse.higgins.sts.utilities.XMLHelper;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * ICardStoreStrategy to store ICards in a secure storage based upon an
 * encrypted CRDS file.
 * 
 * @author Anthony Bussani
 * 
 */
public class SecureStorageStrategy implements CardStoreStrategy {
    private static SecureRandom _randomGenerator = null;

    private boolean _initialized = false;

    private Log log = LogFactory.getLog(SecureStorageStrategy.class);

    Element roamingDump;

    ICardStoreBuilder builder = null;

    PasswordCallback passwordCallback = null;

    String filename = null;

    private ICardProvider cardProvider;

    public SecureStorageStrategy() {
	log.debug("init SecureStorageStrategy");
	builder = CardStoreBuilderFactory.newCardStoreBuilder();
    }

    public void initialize(CallbackHandler authHandler,
	    ICardProvider cardProvider, String filename,
	    PasswordCallback passwordCallBack, IConfiguration configuration)
	    throws CardStoreException {
	this.cardProvider = cardProvider;
	this.filename = filename;
	this.passwordCallback = passwordCallBack;
	this._initialized = true;
    }

    /**
         * Stores the changes from the memory-based Icard map of the
         * ICardProvider to the secure CardStore.
         * 
         * @see org.eclipse.higgins.icard.CardStoreStrategy#synchFromMap(java.util.Map)
         * 
         * @author Thomas Gross
         */
    public void synchFromMap(CallbackHandler authHandler, Map icards)
	    throws CardStoreException {
	IEncryptedStore es = getCurrentEncryptedStore();
	IRoamingStore roamingStore = getCurrentRoamingStore(es);

	// search for card on the roaming store, not anymore in the map
	Iterator icardIterator;
	IRoamingInformationCard[] roamingCards = roamingStore
		.getRoamingInformationCards();
	String roamingId;
	for (int i = 0; i < roamingCards.length; i++) {
	    IRoamingInformationCard roamingCard = roamingCards[i];
	    roamingId = roamingCard.getInformationCardMetaData()
		    .getInformationCardReference().getCardId();
	    icardIterator = icards.values().iterator();
	    boolean found = false;
	    while (icardIterator.hasNext()) {
		ICard card = (ICard) icardIterator.next();
		if (card.getID().equals(roamingId)) {
		    found = true;
		    break;
		}
	    }
	    if (!found) {
		// delete this card, not in the cardstore anymore
		log.info("deleting card from roaming store" + roamingId);
		roamingStore.removeRoamingInformationCardByCardId(roamingId);
	    }
	}
	// search for card in the map not in the store
	icardIterator = icards.values().iterator();
	while (icardIterator.hasNext()) {
	    ICard card = (ICard) icardIterator.next();
	    boolean found = false;
	    for (int i = 0; i < roamingCards.length; i++) {
		IRoamingInformationCard roamingCard = roamingCards[i];
		roamingId = roamingCard.getInformationCardMetaData()
			.getInformationCardReference().getCardId();
		if (card.getID().equals(roamingId)) {
		    found = true;
		    break;
		}
	    }
	    if (!found) {
		// add this card as it was not found in the cardstore
		log.info("adding card from roaming store" + card.getID());
		IRoamingInformationCard newCard = builder
			.createRoamingInformationCard();
		try {
		    // OMElement omElement = null;
		    // We will need later a way to differentiate
		    // InformationCard used to encapsulate Idemix card
		    // And normal Infocard: card.getType() ?
		    // if (card instanceof InformationCard)
		    // omElement = ((InformationCard) card)
		    // .toRoamingInfoCardAxiom();
		    // else if (card instanceof IdemixCard)
		    // omElement = ((IdemixCard) card)
		    // .nfoCardAxiom();
		    Element element = null;
		    DocumentBuilderFactory dbf = DocumentBuilderFactory
			    .newInstance();
		    dbf.setNamespaceAware(true);
		    dbf.setValidating(false);
		    DocumentBuilder db = dbf.newDocumentBuilder();
		    Document doc = db.newDocument();
		    if (card instanceof ManagedCard) {
			element = ((ManagedCard) card).toXML(doc);
		    } else if (card instanceof PersonalCard) {
			element = ((PersonalCard) card).toXML(doc);
		    }
		    // Element element = XMLHelper.toDOM(omElement);
		    log.info("element=[" + element.getNodeName() + "]");
		    log.info("element.nms=[" + element.getNamespaceURI() + "]");
		    // log.info("onElement=[\n" +
		    // XMLHelper.toString(omElement)
		    // + "\n]");
		    newCard.fromXml(element);
		    log.info("newCard.informationCardMetaDada=["
			    + newCard.getInformationCardMetaData() + "]");
		} catch (Exception e) {
		    log
			    .error(
				    "Convertion from InformationCard to Roaming.informationCardMetaData failed",
				    e);
		}
		try {
		    roamingStore.addRoamingInformationCard(newCard);
		} catch (DuplicateCardIdException e) {
		    log.error("Error when add card("
			    + newCard.getInformationCardMetaData()
				    .getInformationCardReference().getCardId()
			    + ")");
		}
		log.info("added card to roaming store ["
			+ newCard.getInformationCardMetaData()
				.getInformationCardReference().getCardId()
			+ "]");
	    }
	}
	try {
	    saveRoamingStore(roamingStore, es);
	} catch (StoreEncryptionException e) {
	    throw new CardStoreException(
		    "Error when Encrypting the roaming store", e);
	} catch (IOException e) {
	    throw new CardStoreException("Error when saving the roaming store",
		    e);
	}
    }

    /**
         * @see org.eclipse.higgins.icard.provider.securestorage.CardStoreStrategy#synchFromStore(java.util.Map)
         * 
         * @author Anthony Bussani
         */
    public void synchFromStore(CallbackHandler authHandler, Map icards)
	    throws CardStoreException {
	if (!_initialized)
	    throw new CardStoreException("Not initialized.");

	ICardStoreBuilder builder = CardStoreBuilderFactory
		.newCardStoreBuilder();

	log.info("Reading SecureStore [" + filename + "]");
	File crdsFile = new File(filename);
	byte[] crdsBytes = new byte[(int) crdsFile.length()];
	try {
	    FileInputStream fis = new FileInputStream(crdsFile);
	    fis.read(crdsBytes);
	    fis.close();
	} catch (FileNotFoundException fe) {
	    throw new CardStoreException(fe);
	} catch (IOException ioe) {
	    throw new CardStoreException(ioe);
	}

	IEncryptedStore es;
	try {
	    es = builder.createEncryptedStore(crdsBytes);
	} catch (UnsupportedObjectModel e) {
	    throw new CardStoreException(e);
	} catch (ExpectedObjectNotPresent e) {
	    throw new CardStoreException(e);
	}

	IRoamingStore rs;
	try {
	    log.info("Decrypting SecureStorage [" + filename + ":"
		    + String.valueOf(passwordCallback.getPassword())
		    + " passwordCallback=" + passwordCallback + "]");
	    rs = es.getRoamingStore(passwordCallback);
	    log.info("Decrypted SecureStorage [" + filename + ":"
		    + String.valueOf(passwordCallback.getPassword())
		    + " passwordCallback=" + passwordCallback + "]");
	    this.roamingDump = (Element) rs.toXml();
	} catch (StoreDecryptionException e) {
	    log.error("Error when Decrypting SecureStorage [" + filename + ":"
		    + String.valueOf(passwordCallback.getPassword())
		    + " passwordCallback=" + passwordCallback + "]");
	    throw new CardStoreException(e);
	}
	syncFromStore(rs, icards);
    }

    /**
         * @throws CardStoreException
         * @throws IOException
         */
    public void newStorage() throws CardStoreException, IOException {
	log.debug("create new store [" + filename + "]");
	IEncryptedStore es;
	es = builder.createEncryptedStore();

	IRoamingStore roamingStore = new RoamingStore();
	try {
	    saveRoamingStore(roamingStore, es);
	} catch (Exception e) {
	    log.error(e);
	    throw new CardStoreException(e);
	}
	log.debug("new store created [" + filename + "]");
    }

    protected IEncryptedStore getCurrentEncryptedStore()
	    throws CardStoreException {
	ICardStoreBuilder builder = CardStoreBuilderFactory
		.newCardStoreBuilder();
	IEncryptedStore es = null;
	log.info("Reading SecureStore [" + filename + "]");
	File crdsFile = new File(filename);

	byte[] crdsBytes = new byte[(int) crdsFile.length()];
	try {
	    if (!crdsFile.exists())
		crdsFile.createNewFile();
	    FileInputStream fis = new FileInputStream(crdsFile);
	    fis.read(crdsBytes);
	    fis.close();
	} catch (FileNotFoundException fe) {
	    throw new CardStoreException(fe);
	} catch (IOException ioe) {
	    throw new CardStoreException(ioe);
	}

	if (crdsBytes == null)
	    crdsBytes = new byte[] {};

	try {
	    es = builder.createEncryptedStore(crdsBytes);
	} catch (Exception e) {
	    throw new CardStoreException(e);
	}
	return es;
    }

    protected IRoamingStore getCurrentRoamingStore(IEncryptedStore es)
	    throws CardStoreException {
	IRoamingStore roamingStore = null;
	try {
	    roamingStore = es.getRoamingStore(passwordCallback);
	} catch (Exception e) {
	    throw new CardStoreException(e);
	}
	return roamingStore;
    }

    /**
         * @param roamingStore
         * @param es
         * @throws StoreEncryptionException
         * @throws IOException
         */
    private void saveRoamingStore(IRoamingStore roamingStore, IEncryptedStore es)
	    throws StoreEncryptionException, IOException {
	this.roamingDump = (Element) roamingStore.toXml();
	es.setRoamingStore(roamingStore);
	byte[] blob = es.toXml(passwordCallback);
	FileOutputStream fos = new FileOutputStream(filename);
	fos.write(blob);
	fos.close();
	log.info("saving cardstore in [" + filename + "]");
    }

    /**
         * Just backups the CardStore by renaming it and creating a new file for
         * writing the CardStore to disk.
         * 
         * @throws ICardStoreException
         *                 if the cardstore could not be written or the backup
         *                 file not removed.
         * 
         * @author Thomas Gross
         */
    protected void backupCRDS() throws CardStoreException {
	log.trace("backupCRDS(" + filename + ")");
	File cardStore = new File(filename);
	File backupStore = new File(filename + ".bak");

	if (!cardStore.canRead())
	    throw new CardStoreException("Cannot read CardStore. Filename="
		    + filename);

	if (backupStore.exists()) {
	    if (!backupStore.delete())
		throw new CardStoreException(
			"Could not delete CardStoreBackup file.");
	}
	cardStore.renameTo(backupStore);
    }

    /**
         * This method will generate a random 16 byte salt.
         * 
         * @param the
         *                array to store; the array should have been initialized
         *                already (i.e., shouldn't be null)
         * @return random 16 byte salt
         * @throws NoSuchAlgorithmException
         */
    private static void generateRandomBytes(byte[] b)
	    throws NoSuchAlgorithmException {
	if (b == null) {
	    return;
	}

	if (_randomGenerator == null) {
	    _randomGenerator = SecureRandom.getInstance("SHA1PRNG");
	}

	_randomGenerator.nextBytes(b);
    }

//    /**
//         * @param out
//         * @param element
//         * @throws CardStoreException
//         */
//    protected void serializeElement(OutputStream out, Element element)
//	    throws CardStoreException {
//	XMLSerializer serializer = new XMLSerializer();
//	serializer.setOutputByteStream(out);
//	try {
//	    serializer.serialize(element);
//	} catch (IOException e) {
//	    throw new CardStoreException(
//		    "XML Serizlization could not be written to OutputStream.",
//		    e);
//	}
//    }

    protected void syncFromStore(IRoamingStore rs, Map icards)
	    throws CardStoreException {
	log.info("SyncFromStore: filename=" + this.filename);
	IRoamingInformationCard[] cards = rs.getRoamingInformationCards();
	icards.clear();
	if (cards == null)
	    throw new CardStoreException(
		    "getRoamingInformationCards() returned null");
	for (int i = 0; i < cards.length; i++) {
	    try {
		IRoamingInformationCard storageCard = cards[i];
//		String informationCardSz = XMLHelper
//			.toString((Element) storageCard
//				.getInformationCardMetaData().toXml());
//		String privateDataSz = XMLHelper.toString((Element) storageCard
//			.getInformationCardPrivateData().toXml());
//		if (informationCardSz == null)
//		    throw new CardStoreException(
//			    "RoamingInformationCard InformationCardMetaData could not be parsed. Returned null");
//		if (privateDataSz == null)
//		    throw new CardStoreException(
//			    "RoamingInformationCard InformationCardPrivateData could not be parsed. Returned null");
		ICard translatedCard = null;
//		 TODO in future, support for idemix card encapsulated within an informationcard
//		ITokenType[] supportedTokenTypeList = storageCard
//			.getInformationCardMetaData()
//			.getSupportedTokenTypeList();
//		if (supportedTokenTypeList != null
//			&& supportedTokenTypeList.length > 0
//			&& supportedTokenTypeList[0]
//				.getUri()
//				.equals(
//
//				"http://www.identity-mixer.com/identity-mixer-issuer#IDEMIX_TOKEN")) {
//		    translatedCard = IdemixCard.fromRoamingCard(
//			    informationCardSz, privateDataSz);
//
//		} else {
//		    translatedCard = InformationCard.parseRoamingCard(
//			    informationCardSz, privateDataSz);
//		}
		boolean selfIssued = storageCard.getInformationCardMetaData()
			.isIsSelfIssued();
		log
			.info("Card["
				+ storageCard.getInformationCardMetaData()
					.getCardName() + "] isselfIssued:"
				+ selfIssued);
		if( log.isTraceEnabled()) {
			String informationCardSz = XMLHelper
			    .toString((Element) storageCard
				    .getInformationCardMetaData().toXml());
			log.debug("informationCardSz=[\n" + informationCardSz + "\n]");		    
		}
		Element cardElement = (Element) storageCard.toXml();
		if (selfIssued) {
		    translatedCard = new CardStorePersonalCard(cardProvider,
			    cardElement);

		} else {
		    translatedCard = new CardStoreManagedCard(cardProvider,
			    cardElement);
		}
		// translatedCard.in
		if (translatedCard != null) {
		    log.trace("adding card:" + i);
		    icards.put(translatedCard.getID(), translatedCard);
		}
	    } catch (Exception e) {
		log
			.error(
				"Error when transforming secure cardspace into higgins card",
				e);
		throw new CardStoreException(e);
	    }

	}
    }

    /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.higgins.icard.provider.securestorage.CardStoreStrategy#exportCards(javax.security.auth.callback.CallbackHandler,
         *      java.util.Iterator, java.io.OutputStream)
         * 
         * @author Anthony Bussani
         */
    public void exportCards(CallbackHandler authHandler, Iterator cards,
	    OutputStream out) throws CardException {
	throw new CardException("Operation not supported");
    }

    public boolean isPasswordProtected() {
	return true;
    }

    public void changePassword(char[] oldPassword, char[] newPassword)
	    throws CardStoreException {
	String oldPasswd = String.valueOf(oldPassword);
	String oldPasswd1 = String.valueOf(oldPassword);
	if (!oldPasswd.equals(oldPasswd1)) {
	    throw new CardStoreException("Provided old password not correct");
	}
	IEncryptedStore es = getCurrentEncryptedStore();
	IRoamingStore roamingStore = getCurrentRoamingStore(es);
	try {
	    this.passwordCallback.setPassword(newPassword);
	    saveRoamingStore(roamingStore, es);
	} catch (StoreEncryptionException e) {
	    throw new CardStoreException(
		    "Error when Encrypting the roaming store", e);
	} catch (IOException e) {
	    throw new CardStoreException("Error when saving the roaming store",
		    e);
	}
    }

    public Element getRoamingDump() {
	return roamingDump;
    }

    /**
         * This should be part of a future CardManager interface This allow to
         * import a ASCII/XML defined store (so replacing all cards define in
         * the current store
         */
    public Map importStore(CallbackHandler authHandler, String asciiStore)
	    throws CardStoreException {
	log.debug("create new store [" + filename + "]");
	IRoamingStore roamingStore = new RoamingStore();
	OMElement om0;
	Element element;
	try {
	    om0 = XMLHelper.toOM(asciiStore);
	    element = XMLHelper.toDOM(om0);
	    roamingStore.fromXml(element);
	    Map icards = new HashMap();
	    syncFromStore(roamingStore, icards);
	    return icards;
	} catch (Exception e) {
	    log.error("Error when importing an ASCII Store", e);
	    throw new CardStoreException("Error when importing an ASCII Store",
		    e);
	}
    }

    public void setFilename(String filename) {
	this.filename = filename;
    }

    public void setPasswordCallback(PasswordCallback passwordCallback) {
	this.passwordCallback = passwordCallback;
    }

}