/*******************************************************************************
 * Copyright (c) 2006 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Anthony Bussani - Initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.icard.provider.securestorage.test;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Iterator;
import java.util.Properties;

import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.PasswordCallback;
import javax.security.auth.callback.UnsupportedCallbackException;

import junit.framework.TestCase;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.ICard;
import org.eclipse.higgins.icard.provider.cardspace.common.InformationCard;
import org.eclipse.higgins.icard.provider.securestorage.PasswordHandler;
import org.eclipse.higgins.icard.provider.securestorage.ProviderConfiguration;
import org.eclipse.higgins.icard.provider.securestorage.SecureStorageICardProvider;
import org.eclipse.higgins.icard.provider.securestorage.SecureStorageStrategy;

public class testSecureStorageProvider extends TestCase {
    private Log log = LogFactory.getLog(testSecureStorageProvider.class);

    public final boolean CONSOLE_LOG = true;

    public final boolean FILE_LOG = false;

    public final String CARD_STORAGE_ORI = "config/test/3cards_ori.crds";

    public final String CARD_STORAGE = "config/test/3cards.crds";

    public final String CRD_FILE = "config/test/abhi0-card.crd";

    public final String PASSWD = "passw0rd";

    /**
         * @param src -
         *                source file
         * @param dest -
         *                destination file
         * @throws IOException
         */
    void copyFile(String src, String dest) throws IOException {
	FileInputStream fis = new FileInputStream(new File(src));
	FileOutputStream fos = new FileOutputStream(new File(dest));
	byte[] buf = new byte[1024];
	int i = 0;
	while ((i = fis.read(buf)) != -1) {
	    fos.write(buf, 0, i);
	}
	fis.close();
	fos.close();
    }

    private final Properties default_logging;

    public testSecureStorageProvider() throws Exception {
	default_logging = new Properties();

	default_logging.put("log4j.rootLogger", "error");

	if (CONSOLE_LOG) {
	    default_logging.put("log4j.appender.stdout",
		    "org.apache.log4j.ConsoleAppender");
	    default_logging.put("log4j.appender.stdout.layout",
		    "org.apache.log4j.PatternLayout");
	    default_logging.put(
		    "log4j.appender.stdout.layout.ConversionPattern",
		    "%5p [%t] (%F:%L) - %m%n");
	    default_logging.put("log4j.rootLogger", "trace, stdout");
	}

	if (FILE_LOG) {
	    default_logging.put("log4j.appender.TestFileLog",
		    "org.apache.log4j.FileAppender");
	    default_logging.put("log4j.appender.TestFileLog.file",
		    "testSecureStorageProvider.log");
	    default_logging.put("log4j.appender.TestFileLog.layout",
		    "org.apache.log4j.PatternLayout");
	    default_logging.put(
		    "log4j.appender.TestFileLog.layout.ConversionPattern",
		    "%5p [%t] (%F:%L) - %m%n");
	    default_logging.put("log4j.rootLogger", "trace, TestFileLog");
	}

	if (CONSOLE_LOG && FILE_LOG) {
	    default_logging.put("log4j.rootLogger",
		    "trace, stdout, TestFileLog");
	}
    }

    protected void setUp() throws Exception {
	super.setUp();
	copyFile(CARD_STORAGE_ORI, CARD_STORAGE);
    }

    CallbackHandler handler = new CallbackHandler() {
	public void handle(Callback[] callbacks) throws IOException,
		UnsupportedCallbackException {
	    for (int i = 0; i < callbacks.length; i++) {
		if (callbacks[i] instanceof PasswordCallback) {
		    PasswordCallback p = (PasswordCallback) callbacks[i];
		    p.setPassword(PASSWD.toCharArray());
		} else {
		    throw new UnsupportedCallbackException(callbacks[i]);
		}
	    }
	}
    };

    /**
         * @throws Exception
         */
    public void testReadStorage() throws Exception {
	SecureStorageStrategy secureStorageStrategy = new SecureStorageStrategy();
	ProviderConfiguration provider = new ProviderConfiguration(
		"Testcase Configuration");
	Properties prop = new Properties();
	provider.setProperties(prop);
	PasswordHandler pwdH = new PasswordHandler(PASSWD);
	SecureStorageICardProvider cardProvider = new SecureStorageICardProvider();
	secureStorageStrategy.initialize(handler, cardProvider, CARD_STORAGE,
		pwdH, provider);
	cardProvider.setStrategy(secureStorageStrategy);
	// Verify that the cards are present
	Iterator iter = cardProvider.getICards(null, null);
	while (iter.hasNext()) {
	    Object o = iter.next();
	    InformationCard card = (InformationCard) o;
	    log.info("return card ID:" + card.getID() + "\n");
	    log.info(o);
	}
    }

    /**
         * @throws Exception
         */
    public void testImportCard() throws Exception {
	SecureStorageStrategy secureStorageStrategy = new SecureStorageStrategy();
	ProviderConfiguration provider = new ProviderConfiguration(
		"Testcase Configuration");
	Properties prop = new Properties();
	provider.setProperties(prop);

	PasswordHandler pwdH = new PasswordHandler(PASSWD);
	SecureStorageICardProvider cardProvider = new SecureStorageICardProvider();
	secureStorageStrategy.initialize(handler, cardProvider, CARD_STORAGE,
		pwdH, provider);
	cardProvider.setStrategy(secureStorageStrategy);
	Iterator iter = cardProvider.getICards(null, null);
	int old_count = 0;
	while (iter.hasNext()) {
	    Object o = iter.next();
	    log.info("return card ID:"
		    + ((InformationCard) o).getID() + "\n");
	    old_count++;
	}

	// import
	// ////////////////////////////////////////////////////////////////////////////////////////////////////
	cardProvider.importCard(null, CRD_FILE);

	iter = cardProvider.getICards(null, null);
	int new_count = 0;
	while (iter.hasNext()) {
	    Object o = iter.next();
	    log.info("return card ID:"
		    + ((InformationCard) o).getID() + "\n");
	    new_count++;
	}
	assertEquals(old_count + 1, new_count);
	// need to verifiy the output by import again cards
    }

    /**
         * @throws Exception
         */
    public void testRemoveCard() throws Exception {
	SecureStorageStrategy secureStorageStrategy = new SecureStorageStrategy();
	ProviderConfiguration provider = new ProviderConfiguration(
		"Testcase Configuration");
	Properties prop = new Properties();
	provider.setProperties(prop);

	PasswordHandler pwdH = new PasswordHandler(PASSWD);
	SecureStorageICardProvider cardProvider = new SecureStorageICardProvider();
	secureStorageStrategy.initialize(handler, cardProvider, CARD_STORAGE,
		pwdH, provider);
	cardProvider.setStrategy(secureStorageStrategy);
	Iterator iter = cardProvider.getICards(null, null);
	ICard card2delete = null;
	int old_count = 0;
	while (iter.hasNext()) {
	    Object o = iter.next();
	    card2delete = (ICard) o;
	    log.info("return card ID:"
		    + ((InformationCard) card2delete).getID() + "\n");
	    old_count++;
	}

	// import
	// ////////////////////////////////////////////////////////////////////////////////////////////////////
	log.info("deleting card [" + card2delete.getCUID() + "]");
	cardProvider.deleteCard(null, card2delete);

	iter = cardProvider.getICards(null, null);
	int new_count = 0;
	while (iter.hasNext()) {
	    Object o = iter.next();
	    ICard card = (ICard) o;
	    assertFalse(card.getID().equals(card2delete.getID()));
	    assertFalse(card.getCUID().equals(card2delete.getCUID()));
	    log.info("return card ID:"
		    + ((InformationCard) o).getID() + "\n");
	    new_count++;
	}
	assertEquals(old_count - 1, new_count);
    }

}
