/*******************************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Patrick R Wardrop (IBM Corporation)
 *******************************************************************************/ 
package org.eclipse.higgins.cardstore.manager;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.List;

import javax.security.auth.callback.PasswordCallback;

import org.eclipse.higgins.cardstore.CardStoreBuilderFactory;
import org.eclipse.higgins.cardstore.Constants;
import org.eclipse.higgins.cardstore.ICardStoreBuilder;
import org.eclipse.higgins.cardstore.exceptions.ExpectedObjectNotPresent;
import org.eclipse.higgins.cardstore.exceptions.UnsupportedObjectModel;
import org.eclipse.higgins.cardstore.logging.Log;
import org.eclipse.higgins.cardstore.logging.LogHelper;
import org.eclipse.higgins.cardstore.resources.CardStoreMessages;
import org.eclipse.higgins.cardstore.schemas._2005._05.identity.IEncryptedStore;
import org.eclipse.higgins.cardstore.schemas._2005._05.identity.IRoamingInformationCard;
import org.eclipse.higgins.cardstore.schemas._2005._05.identity.IRoamingStore;
import org.eclipse.higgins.cardstore.utils.MessageHelper;

/**
 * This class implements the UI for managing ICards that are stored in
 * CRDS (EncryptedStore) files.
 *
 */
public class ICardManager {
	public static String copyright = Constants.FULL_COPYRIGHT;

	
	private static final String[] LIST_REQUIRED_OPTIONS = new String[]{Option.CARDSTORE};
	
	public static final int SUCCESS = 0;
	public static final int FAILURE = 1;
	
	private static Log _log = LogHelper.getLogger(ICardManager.class);
	
	ICardStoreBuilder builder = null;
	
	public ICardManager()
	{
		builder = CardStoreBuilderFactory.newCardStoreBuilder();
	}
	
	public static void main(String[] args)
	{
		String method = "main";
		_log.enterMethod(method, args);
		
		int result = 0;
		
		ICardManager icm = new ICardManager();
		
		result = icm.invoke(args);
		
		if (result != SUCCESS)
		{
			usage();
		}
		
		_log.exitMethod(method, new Integer(result));
		System.exit(result);
	}
	
	public int invoke(String[] args)
	{
		String method = "invoke";
		_log.enterMethod(method, args);
		
		int result = 0;
		
		Operation operation = retrieveAction(args);
		
		if (operation != null)
		{
			Option[] options = retrieveOptions(args);
			
			if (operation == Operation.USAGE)
			{
				usage();
				result = SUCCESS;
			} else if (operation == Operation.LIST)
			{
				result = runListOperation(options);
			}
			
		} else {
			result = FAILURE;
		}
		
		_log.exitMethod(method, Integer.toString(result));
		return result;
	}

	private int runListOperation(Option[] options) {
		int rc = 1;
		
		if (options != null && options.length > 0)
		{
			if (isRequiredOptionsPresent(options, LIST_REQUIRED_OPTIONS))
			{
				if (isOptionPresent(options, Option.CARDID))
				{
					rc = runListDisplaySpecificCard(options);
				} else {
					rc = runListCards(options);
				}
			}
		} else {
			System.err.println(MessageHelper.getMessage(CardStoreMessages.no_supported_options_given, new Object[]{Operation.LIST.getLiterals()[0]}));
		}
		
		return rc;
	}

	private int runListCards(Option[] options) {
		int rc = 0;
		
		String cardStoreFilename = retrieveOptionValue(options, Option.CARDSTORE);
		String cardStorePassword = retrieveOptionValue(options, Option.CARDSTOREPASSWD);
		char[] cardStorePasswordChars = null;
		boolean isVerbose = isOptionPresent(options, Option.VERBOSE);

		if (cardStoreFilename == null)
		{
			System.err.println(MessageHelper.getMessage(CardStoreMessages.required_option_value_not_given, new Object[]{Option.CARDSTORE}));
			rc = FAILURE;
			return rc;
		}

		if (cardStorePassword == null)
		{
			cardStorePasswordChars = promptForStorePassword();
		} else {
			cardStorePasswordChars = cardStorePassword.toCharArray();
		}
				
		if (cardStorePasswordChars == null || cardStorePasswordChars.length < 1)
		{
			System.err.println(MessageHelper.getMessage(CardStoreMessages.required_option_value_not_given, new Object[]{Option.CARDSTOREPASSWD}));
			rc = FAILURE;
			return rc;			
		}

		IRoamingStore roamingStore = retrieveRoamingStore(cardStoreFilename, cardStorePasswordChars);
		
		if (roamingStore == null)
		{
			return FAILURE;
		}
		
		IRoamingInformationCard[] cards = roamingStore.getRoamingInformationCards();
		
		displayNumberOfCards(cards);
		
		return rc;
	}
	
	private void displayNumberOfCards(IRoamingInformationCard[] cards)
	{
		System.out.println(MessageHelper.getMessage(CardStoreMessages.card_manager_num_of_cards_in_store, new Object[]{new Integer(cards.length)}));
	}

	private IRoamingStore retrieveRoamingStore(String cardStoreFilename, char[] cardStorePasswordChars) {
		byte[] crdsBytes = readInCrdsFile(cardStoreFilename);
		
		IRoamingStore rs = null; 
		if (crdsBytes != null)
		{
			try {
				IEncryptedStore ec = builder.createEncryptedStore(crdsBytes);
				PasswordCallback pc = new PasswordCallback("password", true);
				pc.setPassword(cardStorePasswordChars);
				rs = ec.getRoamingStore(pc);
			} catch (Exception e) {
				_log.error(null, e);
				rs = null;
			}			
		}
		return rs;
	}

	private byte[] readInCrdsFile(String cardStoreFilename) {
		File crdsFile = new File(cardStoreFilename);
		byte[] crdsBytes = new byte[(int)crdsFile.length()];		
		FileInputStream fis;
		try {
			fis = new FileInputStream(crdsFile);
			fis.read(crdsBytes);	
			fis.close();			
		} catch (Exception e) {
			_log.error(null, e);
			crdsBytes = null;
		}

		return crdsBytes;
	}

	private int runListDisplaySpecificCard(Option[] options) {
		int rc = 0;
		
		String cardStoreFilename = retrieveOptionValue(options, Option.CARDSTORE);
		String cardStorePassword = retrieveOptionValue(options, Option.CARDSTOREPASSWD);
		char[] cardStorePasswordChars = null;
		String cardId = retrieveOptionValue(options, Option.CARDID);;
		boolean isVerbose = isOptionPresent(options, Option.VERBOSE);
		
		if (cardStoreFilename == null)
		{
			System.err.println(MessageHelper.getMessage(CardStoreMessages.required_option_value_not_given, new Object[]{Option.CARDSTORE}));
			rc = FAILURE;
			return rc;
		}
		
		if (cardId == null)
		{
			System.err.println(MessageHelper.getMessage(CardStoreMessages.required_option_value_not_given, new Object[]{Option.CARDID}));
			rc = FAILURE;
			return rc;
		}
		
		if (cardStorePassword == null)
		{
			cardStorePasswordChars = promptForStorePassword();
		}
		
		if (cardStorePasswordChars == null || cardStorePasswordChars.length < 1)
		{
			System.err.println(MessageHelper.getMessage(CardStoreMessages.required_option_value_not_given, new Object[]{Option.CARDSTOREPASSWD}));
			rc = FAILURE;
			return rc;			
		}
		
		return rc;
	}
	
	private char[] promptForStorePassword() {
		try {
			return PasswordPrompt.getPassword(System.in, MessageHelper.getMessage(CardStoreMessages.cardstore_password_prompt, null));
		} catch (Exception e) {
			_log.error(null,e);
		}
		
		return null;
	}

	private Option[] retrieveOptions(String[] args) {
		List list = new ArrayList();
		
		if (args != null && args.length > 1)
		{
			for(int i = 1; i < args.length; i++)
			{
				if (args[i] != null && args[i].startsWith("-"))
				{
					String optionName = args[i];
					String optionValue = null;
					
					if ((i+1) < args.length)
					{
						if (args[i+1] != null && !args[i+1].startsWith("-"))
						{
							optionValue = args[i+1];
						}
					}
					
					list.add(new Option(optionName, optionValue));
				}
			}
		}
		
		return (Option[])list.toArray(new Option[list.size()]);
	}

	private Operation retrieveAction(String[] args) {
		Operation oper = null;
		
		if (args == null || args.length < 1)
		{
			System.err.println(MessageHelper.getMessage(CardStoreMessages.no_operation_given, null));
			usage();
			
			oper = null;	
			
			return oper;
		}
		
		if (args != null && args.length > 0)
		{
			String operation = args[0];
			
			if (operation != null)
			{
				if (!operation.startsWith("-"))
				{
					System.err.println(MessageHelper.getMessage(CardStoreMessages.no_operation_given, null));
					usage();
					
					oper = null;
					
					return oper;
				}
				
				oper = Operation.fromString(operation);
				
				if (oper == null)
				{
					System.err.println(MessageHelper.getMessage(CardStoreMessages.no_supported_operation_given, null));
					usage();					
				}
			}
		}
		
		

		return oper;
	}
	
	private boolean isOptionPresent(Option[] options, String optionCheck)
	{
		if (options == null || optionCheck == null)
		{
			return true;
		}
		
		for (int i = 0; i < options.length; i++)
		{
			if (optionCheck.equals(options[i].getName()))
			{
				return true;
			}
		}
		
		return false;
	}
	
	private String retrieveOptionValue(Option[] options, String optionName)
	{
		if (options == null || optionName == null)
		{
			return null;
		}
		
		for (int i = 0; i < options.length; i++)
		{
			if (optionName.equals(options[i].getName()))
			{
				return options[i].getValue();
			}
		}
		
		return null;
		
	}
	
	private boolean isRequiredOptionsPresent(Option[] givenOptions, String[] requiredOptions)
	{
		boolean result = false;
		
		if (requiredOptions == null || requiredOptions.length < 1)
		{
			result = true;
			return result;
		}
		
		for(int i = 0; i < requiredOptions.length; i++)
		{
			if (!isOptionPresent(givenOptions, requiredOptions[i]))
			{
				result = false;
				break;
			} else {
				result = true;
			}
		}
		
		return result;		
	}
	
	private static void usage()
	{
		System.out.println(MessageHelper.getMessage(CardStoreMessages.card_manager_usage, new Object[]{ICardManager.class.getName()}));
	}
}
