/*******************************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Patrick R Wardrop (IBM Corporation)
 *    Bruce Rich (IBM Corporation)
 *******************************************************************************/ 
package org.eclipse.higgins.cardstore.schemas._2005._05.identity.impl;

import java.util.Date;
import java.util.Locale;

import org.eclipse.higgins.cardstore.Constants;
import org.eclipse.higgins.cardstore.LocalElementNames;
import org.eclipse.higgins.cardstore.exceptions.ExpectedObjectNotPresent;
import org.eclipse.higgins.cardstore.exceptions.UnsupportedObjectModel;
import org.eclipse.higgins.cardstore.resources.CardStoreMessages;
import org.eclipse.higgins.cardstore.schemas._2005._02.trust.ITokenType;
import org.eclipse.higgins.cardstore.schemas._2005._02.trust.impl.TokenType;
import org.eclipse.higgins.cardstore.schemas._2005._05.identity.ICardImage;
import org.eclipse.higgins.cardstore.schemas._2005._05.identity.IInformationCardMetaData;
import org.eclipse.higgins.cardstore.schemas._2005._05.identity.IInformationCardReference;
import org.eclipse.higgins.cardstore.schemas._2005._05.identity.IPrivacyNoticeLocation;
import org.eclipse.higgins.cardstore.schemas._2005._05.identity.IRequireAppliesTo;
import org.eclipse.higgins.cardstore.schemas._2005._05.identity.ISupportedClaimType;
import org.eclipse.higgins.cardstore.schemas._2005._05.identity.ITokenService;
import org.eclipse.higgins.cardstore.utils.Base64Utility;
import org.eclipse.higgins.cardstore.utils.DateUtil;
import org.eclipse.higgins.cardstore.utils.MessageHelper;
import org.eclipse.higgins.cardstore.utils.XmlUtils;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class InformationCardMetaData implements IInformationCardMetaData {
	public static String copyright = Constants.FULL_COPYRIGHT;
	
	private IInformationCardReference _infoCardRef = null;
	
	private Element _infoCardMetaDataElement = null;
	
	private String _cardName = null;
	
	private ICardImage _cardImage = null;
	
	private String _issuer = null;
	
	private Date _timeIssued = null;
	
	private Date _timeExpires = null;
	
	private ITokenType[] _supportedTokenTypes = null;
	
	private ISupportedClaimType[] _supportedClaimTypes = null;
	
	private ITokenService[] _tokenServices = null;
	
	private boolean _isSelfIssue = false;
	
	private boolean _isSetSelfIssue = false;
	
	private byte[] _hashSalt = null;
	
	private Date _timeLastUpdated = null;
	
	private byte[] _issuerId = null;
	
	private String _issuerName = null;
	
	private int _backGroundColor = 0;
	
	private boolean _isBackgroundColorSet = false;
	
	private byte[] _pinDigest = null;
	
	private IRequireAppliesTo _requireAppliesTo = null;
	
	private IPrivacyNoticeLocation _privacyNotice = null;
	
	private Locale _lang = null;
	
	Object _parentNode = null;
	
	public Object getParent()
	{
		return _parentNode;
	}
	
	public void setParent(Object parentNode)
	{
		_parentNode = parentNode;
	}	
	
	
	public int getBackgroundColor() {
		return _backGroundColor;
	}

	public byte[] getHashSalt() {
		return _hashSalt;
	}

	public byte[] getIssuerId() {
		return _issuerId;
	}

	public String getIssuerName() {
		return _issuerName;
	}

	public byte[] getPinDigest() {
		return _pinDigest;
	}

	public Date getTimeLastUpdated() {
		return _timeLastUpdated;
	}

	public boolean isIsSelfIssued() {
		return _isSelfIssue;
	}

	public boolean isSetBackgroundColor() {
		return _isBackgroundColorSet;
	}

	public boolean isSetIsSelfIssued() {
		return _isSetSelfIssue;
	}

	public void setBackgroundColor(int value) {
		_backGroundColor = value;
		_isBackgroundColorSet = true;
	}

	public void setHashSalt(byte[] value) {
		_hashSalt = value;
	}

	public void setIsSelfIssued(boolean value) {
		_isSelfIssue = value;
		_isSetSelfIssue = true;
	}

	public void setIssuerId(byte[] value) {
		_issuerId = value;
	}

	public void setIssuerName(String value) {
		_issuerName = value;
	}

	public void setPinDigest(byte[] value) {
		_pinDigest = value;
	}

	public void setTimeLastUpdated(Date timeLastUpdated) {
		_timeLastUpdated = timeLastUpdated;
	}

	public void unsetBackgroundColor() {
		_backGroundColor = 0;
		_isBackgroundColorSet = false;
	}

	public void unsetIsSelfIssued() {
		_isSelfIssue = false;
		_isSetSelfIssue = false;
	}

	public ICardImage getCardImage() {
		return _cardImage;
	}

	public String getCardName() {
		return _cardName;
	}
	
	public IInformationCardReference getInformationCardReference() {
		return _infoCardRef;
	}

	public String getIssuer() {
		return _issuer;
	}

	public Locale getLang() {
		return _lang;
	}

	public IPrivacyNoticeLocation getPrivacyNotice() {
		return _privacyNotice;
	}

	public IRequireAppliesTo getRequireAppliesTo() {
		return _requireAppliesTo;
	}

	public ISupportedClaimType[] getSupportedClaimTypeList() {
		return _supportedClaimTypes;
	}

	public ITokenType[] getSupportedTokenTypeList() {
		return _supportedTokenTypes;
	}

	public Date getTimeExpires() {
		return _timeExpires;
	}

	public Date getTimeIssued() {
		return _timeIssued;
	}

	public ITokenService[] getTokenServiceList() {
		return _tokenServices;
	}

	public void setCardImage(ICardImage value) {
		_cardImage = value;
	}

	public void setCardName(String value) {
		_cardName = value;
	} 

	public void setInformationCardReference(IInformationCardReference value) {
		_infoCardRef = value;
	}

	public void setIssuer(String value) {
		_issuer = value;
	}

	public void setLang(Locale locale) {
		_lang = locale;
	}

	public void setPrivacyNotice(IPrivacyNoticeLocation value) {
		_privacyNotice = value;
	}

	public void setRequireAppliesTo(IRequireAppliesTo value) {
		_requireAppliesTo = value;
	}

	public void setSupportedClaimTypeList(ISupportedClaimType[] value) {
		_supportedClaimTypes = value;
	}

	public void setSupportedTokenTypeList(ITokenType[] value) {
		_supportedTokenTypes = value;
	}

	public void setTimeExpires(Date timeExpires) {
		_timeExpires = timeExpires;
	}

	public void setTimeIssued(Date timeIssued) {
		_timeIssued = timeIssued;
	}

	public void setTokenServiceList(ITokenService[] tokenServices) {
		_tokenServices = tokenServices;
	}

	public void fromXml(Object obj) throws UnsupportedObjectModel,
			ExpectedObjectNotPresent {
		
		_infoCardMetaDataElement = XmlUtils.retrieveElementFromObject(obj);
		
		if (!XmlUtils.isElementType(
				_infoCardMetaDataElement, 
				LocalElementNames.IDENTITY_INFORMATION_CARD_META_DATA, 
				Constants.NSURI_IDENTITY_2005_05) || _infoCardMetaDataElement == null)
		{
			_infoCardMetaDataElement = null;			
			
			throw new ExpectedObjectNotPresent(
					MessageHelper.getMessage(
							CardStoreMessages.expected_object_not_present, 
							new Object[]{Constants.NSURI_IDENTITY_2005_05 + ":" + LocalElementNames.IDENTITY_INFORMATION_CARD_META_DATA}));

		}
		
		String lang = _infoCardMetaDataElement.getAttributeNS(Constants.NSURI_XML, "lang");
		
		if (lang != null && lang.length() > 0)
		{
			try {
				_lang = buildLocale(lang);
			} catch (Exception e) {
				// ignore
			}
		}
		
		_infoCardRef = new InformationCardReference();
		XmlUtils.buildObjectFromXml(
				this,
				_infoCardRef, 
				_infoCardMetaDataElement, 
				LocalElementNames.IDENTITY_INFORMATION_CARD_REFERENCE, 
				Constants.NSURI_IDENTITY_2005_05);
		
		Element cardNameElement = XmlUtils.retrieveFirstChildMatchOfType(
				_infoCardMetaDataElement, 
				LocalElementNames.IDENTITY_CARD_NAME, 
				Constants.NSURI_IDENTITY_2005_05);
		
		if (cardNameElement != null)
		{
			String s = XmlUtils.retrieveAllTextFromChildTextNodes(cardNameElement);
			
			if (s != null && s.length() > 0)
			{
				setCardName(s);
			}
		}
		
		_cardImage = new CardImage();
		XmlUtils.buildObjectFromXml(
				this,
				_cardImage, 
				_infoCardMetaDataElement, 
				LocalElementNames.IDENTITY_CARD_IMAGE, 
				Constants.NSURI_IDENTITY_2005_05);
		
		if (lang == null || lang.length() <= 0) {
			_infoCardMetaDataElement.setAttributeNS(Constants.NSURI_XML, "xml:lang", "en-us");
		}
		
		Element issuerElement = XmlUtils.retrieveFirstChildMatchOfType(
				_infoCardMetaDataElement, 
				LocalElementNames.IDENTITY_ISSUER, 
				Constants.NSURI_IDENTITY_2005_05);
		
		if (issuerElement != null)
		{
			String s = XmlUtils.retrieveAllTextFromChildTextNodes(issuerElement);
			
			if (s != null && s.length() > 0)
			{
				setIssuer(s);
			}
		}

		
		Element timeIssuedElement = XmlUtils.retrieveFirstChildMatchOfType(
				_infoCardMetaDataElement, 
				LocalElementNames.IDENTITY_TIME_ISSUED, 
				Constants.NSURI_IDENTITY_2005_05);
		
		if (timeIssuedElement != null)
		{
			String s = XmlUtils.retrieveAllTextFromChildTextNodes(timeIssuedElement);
			
			if (s != null && s.length() > 0)
			{
				setTimeIssued(DateUtil.getDateFromIso8601(s));
			}
		}
		
		Element timeExpiredElement = XmlUtils.retrieveFirstChildMatchOfType(
				_infoCardMetaDataElement, 
				LocalElementNames.IDENTITY_TIME_EXPIRES, 
				Constants.NSURI_IDENTITY_2005_05);
		
		if (timeExpiredElement != null)
		{
			String s = XmlUtils.retrieveAllTextFromChildTextNodes(timeExpiredElement);
			
			if (s != null && s.length() > 0)
			{
				setTimeExpires(DateUtil.getDateFromIso8601(s));
			}
		}
		

		Element supportedTokenTypeList = XmlUtils.retrieveFirstChildMatchOfType(
				_infoCardMetaDataElement, 
				LocalElementNames.IDENTITY_SUPPORTED_TOKEN_TYPE_LIST, 
				Constants.NSURI_IDENTITY_2005_05);
		
		if (supportedTokenTypeList != null)
		{
			_supportedTokenTypes = 
				new TokenType[XmlUtils.numberOfChildElementsOfType(
						supportedTokenTypeList, LocalElementNames.TRUST_TOKEN_TYPE, Constants.NSURI_TRUST_2005_02)];
			
			NodeList nl = supportedTokenTypeList.getChildNodes();
			int numOfTokenTypes = 0;
			for(int i = 0; i < nl.getLength(); i++)
			{
				Node n = nl.item(i);
				if (n != null 
						&& n.getNodeType() == Node.ELEMENT_NODE
						&& XmlUtils.isElementType((Element)n, LocalElementNames.TRUST_TOKEN_TYPE, Constants.NSURI_TRUST_2005_02))
				{
					_supportedTokenTypes[numOfTokenTypes] = new TokenType();
					_supportedTokenTypes[numOfTokenTypes].fromXml(n);
					_supportedTokenTypes[numOfTokenTypes].setParent(this);
					numOfTokenTypes++;
				}					
			}
		}
		
		Element supportedClaimTypeList = XmlUtils.retrieveFirstChildMatchOfType(
				_infoCardMetaDataElement, 
				LocalElementNames.IDENTITY_SUPPORTED_CLAIM_TYPE_LIST, 
				Constants.NSURI_IDENTITY_2005_05);
		
		if (supportedClaimTypeList != null)
		{
			_supportedClaimTypes = 
				new SupportedClaimType[XmlUtils.numberOfChildElementsOfType(
						supportedClaimTypeList, LocalElementNames.IDENTITY_SUPPORTED_CLAIM_TYPE, Constants.NSURI_IDENTITY_2005_05)];
			
			NodeList nl = supportedClaimTypeList.getChildNodes();
			int numOfSupportClaimTypes = 0;
			for(int i = 0; i < nl.getLength(); i++)
			{
				Node n = nl.item(i);
				if (n != null 
						&& n.getNodeType() == Node.ELEMENT_NODE
						&& XmlUtils.isElementType((Element)n, LocalElementNames.IDENTITY_SUPPORTED_CLAIM_TYPE, Constants.NSURI_IDENTITY_2005_05))
				{
					_supportedClaimTypes[numOfSupportClaimTypes] = new SupportedClaimType();
					_supportedClaimTypes[numOfSupportClaimTypes].fromXml(n);
					numOfSupportClaimTypes++;
				}					
			}
		}
		
		Element tokenServicesList = XmlUtils.retrieveFirstChildMatchOfType(
				_infoCardMetaDataElement, 
				LocalElementNames.IDENTITY_TOKEN_SERVICE_LIST, 
				Constants.NSURI_IDENTITY_2005_05);
		
		if (tokenServicesList != null)
		{
			_tokenServices = 
				new TokenService[XmlUtils.numberOfChildElementsOfType(
						tokenServicesList, LocalElementNames.IDENTITY_TOKEN_SERVICE, Constants.NSURI_IDENTITY_2005_05)];
			
			NodeList nl = tokenServicesList.getChildNodes();
			int numOfTokenServices = 0;
			for(int i = 0; i < nl.getLength(); i++)
			{
				Node n = nl.item(i);
				if (n != null 
						&& n.getNodeType() == Node.ELEMENT_NODE
						&& XmlUtils.isElementType((Element)n, LocalElementNames.IDENTITY_TOKEN_SERVICE, Constants.NSURI_IDENTITY_2005_05))
				{
					_tokenServices[numOfTokenServices] = new TokenService();
					_tokenServices[numOfTokenServices].fromXml(n);
					numOfTokenServices++;
				}					
			}
		}		
		
		Element isSelfIssuedElement = XmlUtils.retrieveFirstChildMatchOfType(
				_infoCardMetaDataElement, 
				LocalElementNames.IDENTITY_IS_SELF_ISSUED, 
				Constants.NSURI_IDENTITY_2005_05);
		
		if (isSelfIssuedElement != null)
		{
			String s = XmlUtils.retrieveAllTextFromChildTextNodes(isSelfIssuedElement);
			
			if (s != null && s.length() > 0)
			{
				setIsSelfIssued(Boolean.valueOf(s).booleanValue());
			}
		}
		
		Element hashSaltElement = XmlUtils.retrieveFirstChildMatchOfType(
				_infoCardMetaDataElement, 
				LocalElementNames.IDENTITY_HASH_SALT, 
				Constants.NSURI_IDENTITY_2005_05);
		
		if (hashSaltElement != null)
		{
			String s = XmlUtils.retrieveAllTextFromChildTextNodes(hashSaltElement);
			
			if (s != null && s.length() > 0)
			{
				setHashSalt(Base64Utility.decode(s));
			}
			if (this._hashSalt.length > 16) {
				byte[] temp = new byte[16];
				System.arraycopy(this._hashSalt, 1, temp, 0, temp.length);
				setHashSalt(temp);
				// also need to fix up input
				org.w3c.dom.Document doc = hashSaltElement.getOwnerDocument();
				org.w3c.dom.Text value = doc.createTextNode(Base64Utility.encode(temp));
				Element hsE = (Element) hashSaltElement.cloneNode(false);
				hsE.appendChild(value);
				_infoCardMetaDataElement.replaceChild(hsE, hashSaltElement);
			}
		}
		
		Element timeLastUpdatedElement = XmlUtils.retrieveFirstChildMatchOfType(
				_infoCardMetaDataElement, 
				LocalElementNames.IDENTITY_TIME_LAST_UPDATED, 
				Constants.NSURI_IDENTITY_2005_05);
		
		if (timeLastUpdatedElement != null)
		{
			String s = XmlUtils.retrieveAllTextFromChildTextNodes(timeLastUpdatedElement);
			
			if (s != null && s.length() > 0)
			{
				setTimeLastUpdated(DateUtil.getDateFromIso8601(s));
			}
		}		

		Element issuerIdElement = XmlUtils.retrieveFirstChildMatchOfType(
				_infoCardMetaDataElement, 
				LocalElementNames.IDENTITY_ISSUER_ID, 
				Constants.NSURI_IDENTITY_2005_05);
		
		String s1 = null;
		if (issuerIdElement != null)
		{
			s1 = XmlUtils.retrieveAllTextFromChildTextNodes(issuerIdElement); //BAR
			
			if (s1 != null && s1.length() > 0)
			{
				setIssuerId((Base64Utility.decode(s1)));
			}
		}
		if  ((issuerIdElement == null) ||
			 (s1 == null || s1.length() == 0)) {
			byte[] bval = new java.security.SecureRandom().generateSeed(32);
			setIssuerId(bval);
			org.w3c.dom.Document root = _infoCardMetaDataElement.getOwnerDocument();
			org.w3c.dom.Text value = root.createTextNode(Base64Utility.encode(bval));
			Element newIssuerIdElement = null;
			if (issuerIdElement == null) {
				newIssuerIdElement = root.createElementNS(Constants.NSURI_IDENTITY_2005_05, LocalElementNames.IDENTITY_ISSUER_ID);
				issuerIdElement = newIssuerIdElement;
			} else {
				// already had one, but no value
				NodeList nl = issuerIdElement.getChildNodes();
				for (int i=0; i<nl.getLength(); i++)
					issuerIdElement.removeChild(nl.item(i));
			}
			issuerIdElement.appendChild(value);
			if (newIssuerIdElement != null)
				_infoCardMetaDataElement.appendChild(issuerIdElement); // should really go right after timeLastUpdated
		}
		s1 = null;		
		
		Element issuerNameElement = XmlUtils.retrieveFirstChildMatchOfType(
				_infoCardMetaDataElement, 
				LocalElementNames.IDENTITY_ISSUER_NAME, 
				Constants.NSURI_IDENTITY_2005_05);
		
		if (issuerNameElement != null)
		{
			String s = XmlUtils.retrieveAllTextFromChildTextNodes(issuerNameElement);
			
			if (s != null && s.length() > 0)
			{
				setIssuerName(s);
			}
		}

		Element backGroundColorElement = XmlUtils.retrieveFirstChildMatchOfType(
				_infoCardMetaDataElement, 
				LocalElementNames.IDENTITY_BACK_GROUND_COLOR, 
				Constants.NSURI_IDENTITY_2005_05);
		
		if (backGroundColorElement != null)
		{
			String s = XmlUtils.retrieveAllTextFromChildTextNodes(backGroundColorElement);
			
			if (s != null && s.length() > 0)
			{
				try {
					int bc = Integer.parseInt(s, 10);
					setBackgroundColor(bc);
				} catch (Exception e) {
					// ignore;
				}
			}
		}
		
		Element pinDigestElement = XmlUtils.retrieveFirstChildMatchOfType(
				_infoCardMetaDataElement, 
				LocalElementNames.IDENTITY_PIN_DIGEST, 
				Constants.NSURI_IDENTITY_2005_05);
		
		if (pinDigestElement != null)
		{
			String s = XmlUtils.retrieveAllTextFromChildTextNodes(pinDigestElement);
			
			if (s != null && s.length() > 0)
			{
				setPinDigest((Base64Utility.decode(s)));
			}
		}		
		
		Element requireAppliesToElement = XmlUtils.retrieveFirstChildMatchOfType(
				_infoCardMetaDataElement, 
				LocalElementNames.IDENTITY_REQUIRE_APPLIES_TO, 
				Constants.NSURI_IDENTITY_2005_05);
		
		if (requireAppliesToElement != null)
		{
			_requireAppliesTo = new RequireAppliesTo();
			_requireAppliesTo.fromXml(requireAppliesToElement);
		}		
		
		Element privacyNoticeElement = XmlUtils.retrieveFirstChildMatchOfType(
				_infoCardMetaDataElement, 
				LocalElementNames.IDENTITY_PRIVACY_NOTICE, 
				Constants.NSURI_IDENTITY_2005_05);
		
		if (privacyNoticeElement != null)
		{
			_privacyNotice = new PrivacyNoticeLocation();
			_privacyNotice.fromXml(privacyNoticeElement);
		}				
		
	}

	private Locale buildLocale(String lang) {
		Locale result = null;
		
		if (lang != null && (lang.indexOf('-') != -1 || lang.indexOf('_') != -1))
		{
			if (lang.indexOf('-') != -1)
			{
				String split[] = lang.split("-");
				
				if (split.length == 2)
				{
					String l = split[0].toLowerCase();
					String country = split[1].toUpperCase();
					
					result = new Locale(l, country);
				} else {
					result = new Locale(lang.toLowerCase());
				}
			} else if (lang.indexOf('_') != -1)
			{
				String split[] = lang.split("_");
				
				if (split.length == 2)
				{
					String l = split[0].toLowerCase();
					String country = split[1].toUpperCase();
					
					result = new Locale(l, country);
				} else {
					result = new Locale(lang.toLowerCase());
				}
				
			} else {
				result = new Locale(lang.toLowerCase());
			}
		} else if (lang != null)
		{
			result = new Locale(lang.toLowerCase());
		}
		
		return result;
	}

	public Object toXml() {
	    return this._infoCardMetaDataElement;
	}

}
