/*******************************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Patrick R Wardrop (IBM Corporation)
 *******************************************************************************/ 
package org.eclipse.higgins.cardstore.schemas._2005._05.identity.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.higgins.cardstore.Constants;
import org.eclipse.higgins.cardstore.LocalElementNames;
import org.eclipse.higgins.cardstore.exceptions.DuplicateCardIdException;
import org.eclipse.higgins.cardstore.exceptions.ExpectedObjectNotPresent;
import org.eclipse.higgins.cardstore.exceptions.UnsupportedObjectModel;
import org.eclipse.higgins.cardstore.logging.Log;
import org.eclipse.higgins.cardstore.logging.LogHelper;
import org.eclipse.higgins.cardstore.resources.CardStoreMessages;
import org.eclipse.higgins.cardstore.schemas._2005._05.identity.IRoamingInformationCard;
import org.eclipse.higgins.cardstore.schemas._2005._05.identity.IRoamingStore;
import org.eclipse.higgins.cardstore.utils.MessageHelper;
import org.eclipse.higgins.cardstore.utils.XmlUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class RoamingStore implements IRoamingStore{
	public static String copyright = Constants.FULL_COPYRIGHT;
	
	private static Log _log = LogHelper.getLogger(RoamingStore.class);
	
	private Element _roamingStoreElement = null;
	
	private List _roamingInfomationCardsList = new ArrayList();
	
	// this map uses the cardid as the key for the hash map.
	private Map _mapOfInfoCards = new HashMap();
	
	Object _parentNode = null;
	
	boolean _hasChanges = false;
	
	public Object getParent()
	{
		return _parentNode;
	}
	
	public void setParent(Object parentNode)
	{
		_parentNode = parentNode;
	}	
	
	public IRoamingInformationCard getRoamingInformationCardByCardId(String id) {
		if (_mapOfInfoCards != null)
		{
			return (IRoamingInformationCard)_mapOfInfoCards.get(id);
		}
		
		return null;
	}

	public IRoamingInformationCard[] getRoamingInformationCards() {
		IRoamingInformationCard[] cards = null;
		
		if (_roamingInfomationCardsList != null)
		{
			cards = new IRoamingInformationCard[_roamingInfomationCardsList.size()];
			_roamingInfomationCardsList.toArray(cards);
		} else {
			cards = new IRoamingInformationCard[0];
		}

		return cards;
	}

	public void fromXml(Object obj) throws UnsupportedObjectModel, ExpectedObjectNotPresent {		
		_roamingStoreElement = XmlUtils.retrieveElementFromObject(obj);
		
		if (!XmlUtils.isElementType(_roamingStoreElement, LocalElementNames.IDENTITY_ROAMING_STORE, Constants.NSURI_IDENTITY_2005_05))
		{
			_roamingStoreElement = null;
			throw new ExpectedObjectNotPresent(
					MessageHelper.getMessage(
							CardStoreMessages.expected_object_not_present, 
							new Object[]{Constants.NSURI_IDENTITY_2005_05 + ":" + LocalElementNames.IDENTITY_ROAMING_STORE}));

		}

		_roamingInfomationCardsList = new ArrayList();
		
		NodeList nl = _roamingStoreElement.getChildNodes();
		for(int i = 0; i < nl.getLength(); i++)
		{
			Node n = nl.item(i);
			
			if (n.getNodeType() == Node.ELEMENT_NODE)
			{
				if (XmlUtils.isElementType((Element)n, LocalElementNames.IDENTITY_ROAMING_INFORMATION_CARD, Constants.NSURI_IDENTITY_2005_05))
				{
					RoamingInformationCard card = new RoamingInformationCard();
					card.fromXml((Element)n);
					card.setParent(this);
					try {
						addRoamingInformationCard(card);	
					} catch (Exception e) {
						_log.error(null, e);
					}					
				}
			}
		}
		
		_hasChanges = false;
	}

	private void addInfoCardToMap(IRoamingInformationCard card) {
		
		
		if ((card != null)
				&& card.getInformationCardMetaData() != null
				&& card.getInformationCardMetaData().getInformationCardReference() != null
				&& card.getInformationCardMetaData().getInformationCardReference().getCardId() != null)
		{
			_mapOfInfoCards.put(card.getInformationCardMetaData().getInformationCardReference().getCardId(), card);
		}
		
	}

	public Object toXml() {
		
		if (_hasChanges || _roamingStoreElement == null)
		{
			Document doc = XmlUtils.getDocumentRootFromParent(this);
			try {
			    if( doc == null) doc  = XmlUtils.newDocument();
			} catch(Exception e) {
			    throw new Error("Error when creating a new RoamingStore DOC", e);
			}
			
			Element roamingStoreElement = doc.createElementNS(Constants.NSURI_IDENTITY_2005_05, LocalElementNames.IDENTITY_ROAMING_STORE);
			roamingStoreElement.setAttribute("xmlns", Constants.NSURI_IDENTITY_2005_05);
			
			if (_roamingInfomationCardsList != null)
			{
				for(int i = 0; i < _roamingInfomationCardsList.size(); i++)
				{
					try {
						IRoamingInformationCard card = (IRoamingInformationCard)_roamingInfomationCardsList.get(i);
						if (card != null)
						{
							Element cardElement = (Element)card.toXml();
							if (cardElement != null)
							{
								cardElement = (Element)doc.importNode(cardElement, true);
								roamingStoreElement.appendChild(cardElement);
							}
						}
					} catch (Exception e) {
						_log.error(null, e);
					}
				}
			}
			
			_roamingStoreElement = roamingStoreElement;
		}
		
		return _roamingStoreElement;
	}

	public void addRoamingInformationCard(IRoamingInformationCard card) throws DuplicateCardIdException {
		
		if (card != null
				&& card.getInformationCardMetaData() != null
				&& card.getInformationCardMetaData().getInformationCardReference() != null
				&& card.getInformationCardMetaData().getInformationCardReference().getCardId() != null)
		{
			String id = card.getInformationCardMetaData().getInformationCardReference().getCardId();
			
			if (id != null && _mapOfInfoCards.get(id) == null)
			{
				// we only add the card if the cardid is unique.
				_roamingInfomationCardsList.add(card);							
				addInfoCardToMap(card);													
			} else {
				throw new DuplicateCardIdException(MessageHelper.getMessage(CardStoreMessages.dup_cardid_could_not_be_added, new Object[]{id}));
			}
		}
		
		_hasChanges = true;
	}

	public IRoamingInformationCard removeRoamingInformationCardByCardId(String id) {
		IRoamingInformationCard result = getRoamingInformationCardByCardId(id);
		
		if (result != null)
		{
			//
			// find the object in the list and remove
			//
			int index = _roamingInfomationCardsList.indexOf(result);
			result = (IRoamingInformationCard)_roamingInfomationCardsList.remove(index);
			
			if (result != null)
			{
				_hasChanges = true;
			}
		}
		
		return result;
	}

}
