/*******************************************************************************
 * Copyright (c) 2006-2007 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Valery Kokhan - Initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.icard.registry;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Iterator;

import javax.security.auth.callback.CallbackHandler;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.eclipse.higgins.icard.CUID;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.ICard;
import org.eclipse.higgins.icard.ICardProvider;
import org.eclipse.higgins.icard.registry.utils.DataUtils;
import org.eclipse.higgins.icard.registry.utils.XMLUtils;
import org.eclipse.higgins.registry.HigginsRegistry;
import org.eclipse.higgins.registry.IIORegistryExtension;
import org.eclipse.higgins.registry.IRegistryExtension;
import org.eclipse.higgins.registry.SecurityRegistryExtension;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

public class ICardRegistry extends HigginsRegistry {
	
	protected static ICardRegistry instance = null;
	
	/**
	 * Set up the valid service provider categories and automatically register
	 * all available service providers.
	 * 
	 * <p>
	 * The constructor is protected in order to prevent creation of additional
	 * instances.
	 */
	protected ICardRegistry() {
		super(ICardProvider.class);
		initialize();
	}

	/**
	 * @param configPath
	 */
	public ICardRegistry(String configPath) {
		super(ICardProvider.class, configPath);
		initialize();
	}
	
	private void initialize() {
		addExtension(new IIORegistryExtension(ICardProvider.class));
		addExtension(new SecurityRegistryExtension(ICardProvider.class));
		try {
			Class c = Class.forName("org.eclipse.higgins.icard.registry.plugin.ICardRegistryExtension");
			IRegistryExtension e = (IRegistryExtension) c.newInstance();
			addExtension(e);
		} catch (ClassNotFoundException e) {
		} catch (Throwable e) {
			e.printStackTrace();
		}
		loadProviders();
	}

	public Iterator getICardProviders() {
		return getServiceProviders();
		/*
		List<ICardProvider> list = new ArrayList<ICardProvider>();
		for (ICardServiceProvider s : getServiceProviders()) {
			list.add(s.getICardProvider());
		}
		return list;
		*/
	}

	public ICardProvider getICardProvider(String extID) {
		if (extID == null) {
			throw new IllegalArgumentException("id == null!");
		}
		return (ICardProvider) getServiceProvider(extID);
		/*
		ICardServiceProvider provider = getServiceProvider(extID);
		if (provider == null) {
			return null;
		}
		return provider.getICardProvider();
		*/
	}
	
	public ICard getICardByCUID(CallbackHandler authHandler, String CUID) throws CardException {
		return getICardByCUID(authHandler, new CUID(CUID));
	}
	
	public ICard getICardByCUID(CallbackHandler authHandler, CUID cuid) throws CardException {
		/*
		// Split UUID onto provider ID and i-card ID
		int idx = UUID.indexOf("#");
		if (idx == -1) {
			throw new CardException("Can't identify i-card provider using UUID=\"" + UUID + "\"");
		}
		String providerID = UUID.substring(0, idx);
		String cardID = UUID.substring(idx + 1);
		*/
		ICardProvider p = getICardProvider(cuid.getProviderID());
		if (p == null) {
			throw new CardException("Can't find i-card provider with ID=\"" + cuid.getProviderID() + "\"");
		}
		return p.getICardByCUID(authHandler, cuid);
	}
	
	public void deleteICard(CallbackHandler authHandler, String CUID) throws CardException {
		deleteICard(authHandler, new CUID(CUID));
	}
	
	public void deleteICard(CallbackHandler authHandler, CUID cuid) throws CardException {
		ICardProvider p = getICardProvider(cuid.getProviderID());
		if (p == null) {
			throw new CardException("Can't find i-card provider with ID=\"" + cuid.getProviderID() + "\"");
		}
		ICard c = p.getICardByCUID(authHandler, cuid);
		if (c == null) {
			throw new CardException("Can't find i-card with CUID=\"" + cuid + "\"");
		}
		p.deleteCard(authHandler, c);
	}
	
	public void importICards(CallbackHandler authHandler, InputStream in) throws CardException {
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		ByteArrayInputStream tmp = null;
		try {
			DataUtils.copy(in, out);
			tmp = new ByteArrayInputStream(out.toByteArray());
			DocumentBuilderFactory f = DocumentBuilderFactory.newInstance();
			f.setNamespaceAware(true);
			DocumentBuilder builder = f.newDocumentBuilder();
			Document doc = builder.parse(tmp);
			Element root = doc.getDocumentElement();
			String name = root.getNodeName();
			CardException lastErr = null;
			if (name.equals("EncryptedStore")) {
				// TODO process import from .crds format
			} else {
				// try to import
				for (Iterator itr = getICardProviders(); itr.hasNext(); ) {
					ICardProvider p = (ICardProvider) itr.next();
					if (p.canImportICard(authHandler, root)) {
						try {
							ICard c = p.importICard(authHandler, root);
							if (c != null) {
								return;
							}
						} catch (CardException e) {
							lastErr = e;
						}
					}
				}
				if (lastErr != null) {
					throw lastErr;
				} else {
					throw new CardException("Can't find suitable provider to import i-card \n[\n" + XMLUtils.toString(root) + "\n]");
				}
			}
		} catch (ParserConfigurationException e) {
			throw new CardException("Can't initialize xml parser", e);
		} catch (SAXException e) {
			// For now we just throw an exception if input is not valid xml file
			// TODO Consider different import formats
			throw new CardException("Input is not valid xml", e);
		} catch (IOException e) {
			throw new CardException("Can't read import data", e);
		} finally {
			try {
				in.close();
			} catch (Exception e) {
			}
		}
	}

	protected String getConfigFolder() {
		return ".icard";
	}

	public synchronized static ICardRegistry getInstance() {
		if (instance == null) {
			instance = new ICardRegistry();
		}
		return instance;
	}
}
