/*******************************************************************************
 * Copyright (c) 2006-2007 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Valery Kokhan - Initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.icard.registry;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import javax.security.auth.callback.CallbackHandler;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.CUID;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.ICard;
import org.eclipse.higgins.icard.ICardProvider;
import org.eclipse.higgins.icard.auth.ICredential;
import org.eclipse.higgins.icard.io.IElement;
import org.eclipse.higgins.icard.io.IElementFormat;
import org.eclipse.higgins.icard.io.IFormatDescriptor;
import org.eclipse.higgins.icard.io.IFormatProvider;
import org.eclipse.higgins.icard.io.IICardElement;
import org.eclipse.higgins.icard.io.IInputProcessor;
import org.eclipse.higgins.icard.io.IOutputProcessor;
import org.eclipse.higgins.icard.registry.utils.DataUtils;
import org.eclipse.higgins.icard.userprofile.IUserProfileService;
import org.eclipse.higgins.registry.HigginsRegistry;
import org.eclipse.higgins.registry.IIORegistryExtension;
import org.eclipse.higgins.registry.IRegistryExtension;
import org.eclipse.higgins.registry.SecurityRegistryExtension;

public class ICardRegistry extends HigginsRegistry {
	private Log log = LogFactory.getLog(ICardRegistry.class);
	
	protected static ICardRegistry instance = null;
	
	/**
	 * Set up the valid service provider categories and automatically register
	 * all available service providers.
	 * 
	 * <p>
	 * The constructor is protected in order to prevent creation of additional
	 * instances.
	 */
	protected ICardRegistry() {
		super(ICardProvider.class);
		initialize();
	}

	/**
	 * @param configPath
	 */
	public ICardRegistry(String configPath) {
		super(ICardProvider.class, configPath);
		initialize();
	}
	
	private void initialize() {
		addExtension(new IIORegistryExtension(ICardProvider.class));
		addExtension(new SecurityRegistryExtension(ICardProvider.class));
		try {
			Class c = Class.forName("org.eclipse.higgins.icard.registry.plugin.ICardRegistryExtension");
			IRegistryExtension e = (IRegistryExtension) c.newInstance();
			addExtension(e);
		} catch (ClassNotFoundException e) {
		} catch (Throwable e) {
			e.printStackTrace();
		}
		loadProviders();
	}

	public Iterator getICardProviders() {
		return getServiceProviders();
		/*
		List<ICardProvider> list = new ArrayList<ICardProvider>();
		for (ICardServiceProvider s : getServiceProviders()) {
			list.add(s.getICardProvider());
		}
		return list;
		*/
	}

	public ICardProvider getICardProvider(String extID) {
		if (extID == null) {
			throw new IllegalArgumentException("id == null!");
		}
		return (ICardProvider) getServiceProvider(extID);
		/*
		ICardServiceProvider provider = getServiceProvider(extID);
		if (provider == null) {
			return null;
		}
		return provider.getICardProvider();
		*/
	}
	
	public ICard getICardByCUID(CallbackHandler authHandler, String CUID) throws CardException {
		return getICardByCUID(authHandler, new CUID(CUID));
	}
	
	public ICard getICardByCUID(CallbackHandler authHandler, CUID cuid) throws CardException {
		/*
		// Split UUID onto provider ID and i-card ID
		int idx = UUID.indexOf("#");
		if (idx == -1) {
			throw new CardException("Can't identify i-card provider using UUID=\"" + UUID + "\"");
		}
		String providerID = UUID.substring(0, idx);
		String cardID = UUID.substring(idx + 1);
		*/
		ICardProvider p = getICardProvider(cuid.getProviderID());
		if (p == null) {
			throw new CardException("Can't find i-card provider with ID=\"" + cuid.getProviderID() + "\"");
		}
		return p.getICardByCUID(authHandler, cuid);
	}
	
	public void deleteICard(CallbackHandler authHandler, String CUID) throws CardException {
		deleteICard(authHandler, new CUID(CUID));
	}
	
	public void deleteICard(CallbackHandler authHandler, CUID cuid) throws CardException {
		ICardProvider p = getICardProvider(cuid.getProviderID());
		if (p == null) {
			throw new CardException("Can't find i-card provider with ID=\"" + cuid.getProviderID() + "\"");
		}
		ICard c = p.getICardByCUID(authHandler, cuid);
		if (c == null) {
			throw new CardException("Can't find i-card with CUID=\"" + cuid + "\"");
		}
		p.deleteCard(authHandler, c);
	}

	/*
	public Iterator importICards(CallbackHandler authHandler, InputStream in) throws CardException {
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		ByteArrayInputStream tmp = null;
		ArrayList cards = new ArrayList();
		try {
			DataUtils.copy(in, out);
			tmp = new ByteArrayInputStream(out.toByteArray());
			DocumentBuilderFactory f = DocumentBuilderFactory.newInstance();
			f.setNamespaceAware(true);
			DocumentBuilder builder = f.newDocumentBuilder();
			Document doc = builder.parse(tmp);
			Element root = doc.getDocumentElement();
			String name = root.getNodeName();
			CardException lastErr = null;
			if (name.equals("EncryptedStore")) {
				// TODO process import from .crds format
			} else {
				log.debug("Import .crd file.");
				for (Iterator itr = getICardProviders(); itr.hasNext(); ) {
					ICardProvider p = (ICardProvider) itr.next();
					boolean canImport = p.canImportICard(authHandler, root);
					log.debug("ICard provider " + p.getID() + " can " + ((canImport) ? "" : "not ") + "import card");
					if (canImport) {
						try {
							lastErr = null;
							ICard c = p.importICard(authHandler, root);
							if (c != null) {
								cards.add(c);
								break;
							}
						} catch (CardException e) {
							lastErr = e;
						}
					}
				}
				if (lastErr != null)
					throw lastErr;
				if (cards.size() == 0)
					throw new CardException("Can't find suitable provider to import i-card \n[\n" + XMLUtils.toString(root) + "\n]");
			}
		} catch (ParserConfigurationException e) {
			throw new CardException("Can't initialize xml parser", e);
		} catch (SAXException e) {
			// For now we just throw an exception if input is not valid xml file
			// TODO Consider different import formats
			throw new CardException("Input is not valid xml", e);
		} catch (IOException e) {
			throw new CardException("Can't read import data", e);
		} finally {
			try {
				in.close();
			} catch (Exception e) {
			}
		}
		return cards.iterator();
	}
	*/
	
	/**
	 * Checks whether this <code>ICardRegistry</code> can recognize data
	 * format of the provided input stream. Some formats require authentication
	 * information in order to proccess input data. This method should be used
	 * prior to actual attempt to import data from the input stream in order to
	 * retrieve an information about data format and required authentication
	 * data to proccess the data successfully.
	 * 
	 * @return the information about input stream data format if it was
	 *         recognized successfully and <code>null</code> otherwise.
	 */
	public IFormatDescriptor checkInput(InputStream in) {
		IFormatDescriptor format = null;
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		ByteArrayInputStream copy = null;
		
		try {
			DataUtils.copy(in, out);
		} catch (IOException e) {
			log.trace(e);
			return format;
		} finally {
			try {
				in.close();
			} catch (Exception e) {
			}
		}
			
		copy = new ByteArrayInputStream(out.toByteArray());

		IOFormatRegistry formatRegistry = IOFormatRegistry.getInstance();
			
		for (Iterator itr = formatRegistry.getFormatProviders(); itr.hasNext(); ) {
			IFormatProvider p = (IFormatProvider) itr.next();
			IInputProcessor ip = p.getInputProcessor();
			if (ip != null) {
				copy.reset();
				IFormatDescriptor fd = ip.checkInput(copy);
				if (fd != null) {
					format = fd;
					break;
				}
			}
		}
		return format;
	}

	/**
	 * Imports cards from the provided input stream.
	 * 
	 * @param authHandler
	 *            the callback handler used to provide user identity
	 * @param in
	 *            the input stream to import from
	 * @param formatID
	 *            the data format of the input stream if available. If
	 *            <code>null</code> is used all available data formats will be
	 *            checked in order to import.
	 * @param credential
	 *            the authentication data to proccess the input stream if
	 *            required
	 * @return
	 * @throws CardException
	 */
	public Iterator importICards(CallbackHandler authHandler, InputStream in, String formatID, ICredential credential) throws CardException {
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		ByteArrayInputStream copy = null;
		
		try {
			DataUtils.copy(in, out);
		} catch (IOException e) {
			log.trace(e);
			throw new CardException("Can't read import data", e);
		} finally {
			try {
				in.close();
			} catch (Exception e) {
			}
		}
			
		copy = new ByteArrayInputStream(out.toByteArray());

		IElement[] elements = null;
		IOFormatRegistry formatRegistry = IOFormatRegistry.getInstance();
		
		IFormatProvider fp = null;
		IInputProcessor ip = null;
		
		if (formatID != null) {
			fp = formatRegistry.getFormatProvider(formatID);
			if (fp != null) {
				ip = fp.getInputProcessor();
			}
		}
		
		if (ip == null) {
			for (Iterator itr = formatRegistry.getFormatProviders(); itr.hasNext(); ) {
				fp = (IFormatProvider) itr.next();
				IInputProcessor ipp = fp.getInputProcessor();
				if (ipp != null) {
					copy.reset();
					if (ipp.checkInput(copy) != null) {
						ip = ipp;
						break;
					}
				}
			}
		}
		
		if (ip != null) {
			copy.reset();
			elements = ip.process(copy, credential);
		}

		if (elements == null) {
			throw new CardException("Can't find suitable format provider to process import data");
		}
		
		ArrayList cards = new ArrayList();

		for (int i = 0; i < elements.length; i++) {
			ICardProvider p = null;
			IICardElement iio = null;
			CardException lastErr = null;
			String pID = null;
			
			if (elements[i] instanceof IICardElement) {
				iio = (IICardElement) elements[i];
				CUID cuid = iio.getCUID();
				if (cuid != null) {
					pID = cuid.getProviderID();
				}
			}
			
			if (pID != null) {
				p = getICardProvider(pID);
			}
			
			ICard card = null;
			
			if (p != null) {
				if (p.canImportICard(authHandler, elements[i])) {
					try {
						card = p.importICard(authHandler, elements[i]);
					} catch (CardException e) {
						log.debug("Can't import card", e);
						card = null;
						lastErr = e;
					}
				}
			}
			
			if (card == null) {
				for (Iterator itr = getICardProviders(); itr.hasNext(); ) {
					p = (ICardProvider) itr.next();
					if (p.canImportICard(authHandler, elements[i])) {
						try {
							card = p.importICard(authHandler, elements[i]);
						} catch (CardException e) {
							card = null;
							log.debug("Can't import card", e);
							lastErr = e;
						}
					}
					if (card != null) {
						break;
					}
				}
			}
			
			if (card == null) {
				if (lastErr == null) {
					log.trace("Can't find suitable provider to import card: " + elements[i].toString());
				} else {
					log.trace("Can't import card: " + elements[i].toString(), lastErr);
				}
			} else {
				cards.add(card);
			}
		}
			
		return cards.iterator();
	}
	
	public IFormatProvider getFormatProvider(String formatID) {
		return IOFormatRegistry.getInstance().getFormatProvider(formatID);
	}
	
	/**
	 * Returns an array of <code>IFormatDesctiptor</code>s which provide an
	 * information about all output formats available.
	 */
	public IFormatDescriptor[] getOutputFormats() {
		IFormatDescriptor[] res = null;
		IOFormatRegistry formatRegistry = IOFormatRegistry.getInstance();
		List formats = new ArrayList();
		for (Iterator itr = formatRegistry.getFormatProviders(); itr.hasNext(); ) {
			IFormatProvider p = (IFormatProvider) itr.next();
			IOutputProcessor ip = p.getOutputProcessor();
			if (ip != null) {
				IFormatDescriptor fd = p.getFormatDescriptor();
				if (fd != null) {
					formats.add(fd);
				}
			}
		}
		res = (IFormatDescriptor[]) formats.toArray(new IFormatDescriptor[formats.size()]);
		return res;
	}
	
	public Iterator getICardsByFormat(CallbackHandler authHandler, String formatID) throws CardException {
		List res = new ArrayList();
		IFormatProvider fp = getFormatProvider(formatID);
		if (fp == null) {
			throw new CardException("Can't find format provider with id=\"" + formatID + "\"");
		}
		
		IOutputProcessor op = fp.getOutputProcessor();
		if (op == null) {
			throw new CardException("Format provider with id=\"" + formatID + "\" doesn't support output operations.");
		}
		
		IFormatDescriptor fd = fp.getFormatDescriptor();
		if (fd == null) {
			throw new CardException("Format provider with id=\"" + formatID + "\" doesn't have valid format descriptor.");
		}
		
		IElementFormat ed = fd.getElementDescriptor();
		if (ed == null) {
			throw new CardException("Format provider with id=\"" + formatID + "\" doesn't have valid element descriptor.");
		}
		
		for (Iterator itr = getICardProviders(); itr.hasNext(); ) {
			ICardProvider cp = (ICardProvider) itr.next();
			for (Iterator itrc = cp.getICardsByFormat(authHandler, ed); itrc.hasNext(); ) {
				res.add(itrc.next());
			}
		}
		
		return res.iterator();
	}
	
	public void exportICards(CallbackHandler authHandler, OutputStream out, String formatID, CUID[] cards, ICredential credential) throws CardException {
		IFormatProvider fp = getFormatProvider(formatID);
		if (fp == null) {
			throw new CardException("Can't find format provider with id=\"" + formatID + "\"");
		}
		
		IOutputProcessor op = fp.getOutputProcessor();
		if (op == null) {
			throw new CardException("Format provider with id=\"" + formatID + "\" doesn't support output operations.");
		}
		
		IFormatDescriptor fd = fp.getFormatDescriptor();
		if (fd == null) {
			throw new CardException("Format provider with id=\"" + formatID + "\" doesn't have valid format descriptor.");
		}
		
		IElementFormat ed = fd.getElementDescriptor();
		if (ed == null) {
			throw new CardException("Format provider with id=\"" + formatID + "\" doesn't have valid element descriptor.");
		}
		
		HashMap map = new HashMap();
		for (int i = 0; i < cards.length; i++) {
			String pid = cards[i].getProviderID();
			if (pid == null) {
				log.error("Can't find provider for card with CUID=\"" + cards[i] + "\"");
			} else {
				List l = (List) map.get(pid);
				if (l == null) {
					l = new ArrayList();
					map.put(pid, l);
				}
				l.add(cards[i]);
			}
		}
		
		List elements = new ArrayList();
		
		for (Iterator itr = map.keySet().iterator(); itr.hasNext(); ) {
			String pid = (String) itr.next();
			ICardProvider cp = getICardProvider(pid);
			if (cp == null) {
				throw new CardException("Can't find provider with id=\"" + pid + "\"");
			}
			List l = (List) map.get(pid);
			elements.addAll(Arrays.asList(cp.exportICards(authHandler, ed, (CUID[]) l.toArray(new CUID[l.size()]))));
		}
		
		op.process((IElement[]) elements.toArray(new IElement[elements.size()]), out, credential);
	}

	protected String getConfigFolder() {
		return ".icard";
	}

	public synchronized static ICardRegistry getInstance() {
		if (instance == null) {
			instance = new ICardRegistry();
		}
		return instance;
	}
	
	public IUserProfileService getUserProfileService() {
		Iterator it = UserProfileRegistry.getInstance().getServiceProviders();
		//there is always only user profile service provider
		if (it.hasNext()) {
			return (IUserProfileService) it.next();
		}
		return null;
	}
	
}
