/*******************************************************************************
 * Copyright (c) 2006-2007 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Valery Kokhan - Initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.icard;

import java.util.Iterator;

/**
 * The <code>IClaimContainer</code> represents an interface of container
 * object for {@link IClaim} objects. Defines methods to create and update some
 * individual {@link IClaim} objects as well as to retrieve them.
 */
public interface IClaimContainer extends IClaimTypeContainer {
	/**
	 * Return all claims of this container object.
	 * 
	 * @return A List of {@link IClaim}s.
	 * @throws AuthenticationRequiredException when authentication is required to perform this request
	 * @throws CardException
	 */
	public Iterator getClaims() throws AuthenticationRequiredException, CardException;

	/**
	 * Returns the specified claim of this object.
	 * 
	 * @param type
	 *            the identifier of the claim to be returned.
	 * @return an {@link IClaim} or {@code null} if none exits.
	 * @throws AuthenticationRequiredException when authentication is required to perform this request
	 * @throws InvalidTypeException
	 *             if this object can't contains a claim with the specified
	 *             type.
	 * @throws CardException
	 *             in case of some unexpected error. 
	 */
	public IClaim getClaim(String type) throws AuthenticationRequiredException, InvalidTypeException, CardException;

	/**
	 * Creates a new {@link IClaim} object for this container of claims with the
	 * type specified by <code>copyFrom.getType()</code> if corresponding
	 * {@link IClaim} object doesn't exist yet in this container. Performs deep
	 * copy of the data from the provided object to the contained one.
	 * 
	 * @param copyFrom
	 *            an {@link IClaim} object from which data is copied to the
	 *            contained {@link IClaim} object.
	 * @return contained {@link IClaim} object.
	 * @throws InvalidClaimException
	 *             if provided claim object is invalid for this container.
	 * @throws InvalidTypeException
	 *             if this object can't contains a claim with the specified
	 *             type.
	 * @throws ReadOnlyObjectException
	 *             if implicit subject doesn't support modifications of its
	 *             claims.
	 * @throws CardException
	 *             in case of some unexpected error. 
	 */
	public IClaim setClaim(IClaim copyFrom) throws InvalidClaimException,
			InvalidTypeException, ReadOnlyObjectException, CardException;

	/**
	 * Creates a new {@link IClaim} object for this container of claims and adds
	 * it to the container. This call is typically followed by one or more calls
	 * to on the returned {@link IClaim} object to set its value(s).
	 * 
	 * @param type
	 *            the String specifying the type of the IClaim being created.
	 * @return a newly created IClaim object.
	 * @throws InvalidTypeException
	 *             if this object can't contains a claim with the specified
	 *             type.
	 * @throws ReadOnlyObjectException
	 *             if implicit subject doesn't support modifications of its
	 *             claims.
	 * @throws CardException
	 *             in case of some unexpected error. 
	 */
	public IClaim createClaim(String type) throws InvalidTypeException,
			ReadOnlyObjectException, CardException;
}
