/** 
 * Copyright (c) 2007 Parity Communications, Inc.  
 * All rights reserved. This program and the accompanying materials  
 * are made available under the terms of the Eclipse Public License v1.0  
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 *  
 * Contributors: 
 *		Yuriy Pilipenko - API and implementation 
 */
package org.eclipse.higgins.icard.userprofile;

import java.net.URI;
import java.util.List;

import javax.security.auth.callback.CallbackHandler;

import org.eclipse.higgins.icard.CUID;

/**
 * @author PilipenkoYN
 *
 */
public interface ICardUsageManager {
	
	/**
	 * Adds info about card usage event
	 * @param cardUsage
	 * @return newly added CardUsage object
	 */
	public CardUsage addCardHistory(CUID cuid, URI uri) throws UserProfileException;
		
	/**
	 * Returns the card usage history
	 * @param cuid
	 * @return Array List of CardUsage objects
	 */
	public List getCardHistory(CUID cuid) throws UserProfileException;
	
	/**
	 * Returns the card usage history in a paging manner
	 * @param cuid
	 * @param startFrom
	 * @param size
	 * @param orderBy - string constants from CardUsage class, by what field to order output
	 * @param desc
	 * @return
	 * @throws UserProfileException
	 */
	public List getCardHistory(CUID cuid, int startFrom, int size, String orderBy, boolean desc) throws UserProfileException;
	
	/**
	 * Returns the total size of history for paging output
	 * @return
	 * @throws UserProfileException
	 */
	public int getHistoryTotalSize(CUID cuid) throws UserProfileException;
	
	/**
	 * Deletes all the usage events of the given card on all URIs unless 
	 * it is checked as "Use Always".
	 * @param cuid
	 */
	//TODO Do we need such method???
	public void clearCardHistory(CUID cuid) throws UserProfileException;
	
	/**
	 * Remembers this card for this site
	 * @param cuid
	 * @param URI TODO
	 * @return CardUsage object with just added card-URI relation
	 * @throws UserProfileException 
	 */
	public CardUsage setUsedAlways(CUID cuid, URI uri) throws UserProfileException;
	
	/**
	 * Deletes the "Use Always" relation between this card and this site
	 * @param cuid
	 * @param URI
	 * @return CardInformation object with all related card information, or null if card info not found
	 * @throws UserProfileException
	 */
	public CardUsage deleteUsedAlways(CUID cuid, URI uri) throws UserProfileException;
	
	/**
	 * Returns the List of CardUsage objects of all the cards, which ever been logged in to this site
	 * 
	 * @param URI
	 * @return
	 * @throws UserProfileException
	 */
	public List getUsedCards(URI uri) throws UserProfileException;	
	
	/**
	 * Returns the List of CardUsage objects with URIs for which the given card is remembered
	 * 
	 * @param cuid
	 * @return
	 * @throws UserProfileException
	 */
	public List getUsedAlwaysURLs(CUID cuid) throws UserProfileException;
	
	/**
	 * Returns the card remebered for the given site
	 * @param URI
	 * @return
	 * @throws UserProfileException
	 */
	public CUID getUsedAlwaysCard(URI uri) throws UserProfileException;
	
	/**
	 * Deletes state "Use Always" on all URIs for the given card 
	 * @param cuid
	 * @throws UserProfileException
	 */
	public void clearUsedAlwaysURLs(CUID cuid) throws UserProfileException;
	
	/**
	 * Returns card relations to URI with last used dates and state remembered it on this site or not
	 * @param handler
	 * @param cuid
	 * @return
	 * @throws UserProfileException
	 */
	public List getCardUsageSummary(CUID cuid) throws UserProfileException;
		
	/**
	 * Returns card relations to URI with last used dates and state remembered it on this site or not
	 * @param handler
	 * @param cuid
	 * @param startFrom
	 * @param size
	 * @param orderBy - string constants from CardUsage class, by what field to order output
	 * @param desc
	 * @return
	 * @throws UserProfileException
	 */
	public List getCardUsageSummary(CUID cuid, int startFrom, int size, String orderBy, boolean desc) throws UserProfileException;
	
	/**
	 * 
	 * @param handler
	 * @param cuid
	 * @return
	 * @throws UserProfileException
	 */
	public int getSummaryTotalSize(CUID cuid) throws UserProfileException; 		
	
	/**
	 * Finishes all the work done, releases contexts, resourses etc.
	 * @throws UserProfileException
	 */
	public void close()  throws UserProfileException;
	
}
