/*******************************************************************************
 * Copyright (c) 2006-2007 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Paul Trevithick - Initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.icard;

import java.util.Date;
import java.util.List;

import org.eclipse.higgins.icard.auth.ICredentialDescriptor;
import org.eclipse.higgins.icard.io.CardIOException;
import org.eclipse.higgins.icard.io.IElement;
import org.eclipse.higgins.icard.io.IElementFormat;
import org.eclipse.higgins.icard.io.UnsupportedElementFormatException;

public interface ICard extends IClaimTypeContainer, IClaimContainer {

	/**
	 * Returns the type of this i-card (e.g. "uri", "cardspace", "idemix",
	 * "openid", etc.)
	 */
	public String getType();

	/**
	 * Gets the ID of the i-card. Useful in subsequent import operations, so
	 * that cards can be updated or overwritten.
	 * <p>
	 * The ID of the i-card must be unique per i-card provider but different
	 * i-card providers could contains i-cards with the same ID however.
	 */
	public String getID();

	/**
	 * Gets the unique identifier of the i-card in the i-card registry.
	 * <p>
	 * Because different i-card providers could contains i-cards with the same
	 * ID, it could be difficult to retrieve the same i-card from the i-card
	 * registry again using i-card ID.
	 * <p>
	 * The UUID must be unique over all i-card providers in the i-card registry
	 * to allow retrieve the same i-card from the i-card registry multiple
	 * times.
	 * <p>
	 * Current implementation of i-card registry assumes that ICard.UUID must
	 * equals to ICardProvider.ID + "#" + ICard.ID
	 */
	public CUID getCUID();

	/**
	 * Return the version of the card. Useful in subsequent import operations,
	 * so that cards can be updated or overwritten.
	 */
	public String getVersion();

	/**
	 * Return the human friendly name of the card. The only thing that will be
	 * possible to modify after the import.
	 */
	public String getName();

	/**
	 * Returns byte representation of a background image of the card.
	 */
	public byte[] getImage();

	/**
	 * Return the mime type of the background image (JPEG or GIF).
	 */
	public String getImageType();

	/**
	 * Name of the issuer of the card. Used to match the required issuer, if a
	 * relying party specifies an issuer in the policy.
	 */
	public String getIssuer();

	/**
	 * Returns the human friendly name of the card issuer, originator, creator.
	 */
	public String getIssuerName();

	/**
	 * Returns when the card was first issued, created, orginated.
	 */
	public Date getTimeIssued();

	/**
	 * Returns (optionally) the time after which the card should be considered
	 * expired, invalid. Otherwise returns null.
	 */
	public Date getTimeExpires();

	/**
	 * Return a list of all possible types of claims (IClaimType) that are
	 * supported.
	 * 
	 * @see IClaimType
	 */
	// public Iterator getSupportedClaimTypes();

	/**
	 * Return a list of all possible types of claims (as String) that are
	 * supported.
	 */
	public List getSupportedClaimTypesUris();

	/**
	 * Return a list of claims provided by this ICard.
	 * @throws CardException 
	 * 
	 * @see IClaim
	 */
	// public Iterator getClaims() throws CardException;
	
	/**
	 * @param type
	 * @return
	 */
	// public IClaim getClaim(String type);

	/**
	 * @param typeLocalName
	 * @return
	 * @throws AuthenticationRequiredException TODO
	 * @throws InvalidTypeException TODO
	 * @throws CardException TODO
	 */
	public IClaim getClaimByLocalName(String typeLocalName) throws AuthenticationRequiredException, InvalidTypeException, CardException;

	/*
	 * /** Given a relying party policy Policy, return true if this card can
	 * provide the claims required/desired by the relying party else return
	 * false
	 */
	// public boolean isMatch(IPolicy policy);
	/**
	 * Should it be here? /
	 */
	public ICardProvider getProvider();

	/**
	 * 
	 */
	public String getDescription();

	/**
	 * Used to change the name of the card
	 * 
	 * @param newName
	 * @throws CardException
	 */
	public void setName(String newName) throws CardException;

	/**
	 * Used to change the picture associated with this card
	 * 
	 * @param newImage
	 * @param newImageType
	 * @throws CardException
	 */
	public void setImage(byte[] newImage, String newImageType) throws CardException;

	/**
	 * Sets the card's human friendly name of the card issuer
	 * 
	 * @param name
	 * @throws CardNotOpenException
	 */
	void setIssuerName(String name) throws CardException;

	/**
	 * @param date
	 * @throws CardNotOpenException
	 */
	void setTimeExpires(Date date) throws CardException;

	/**
	 * @return list of token types which can be provided for this card
	 */
	public List getSupportedTokenTypes();

	/**
	 * Indicates that this card has been issued by user (claim values can be
	 * editable)
	 */
	public boolean isSelfIssued();

	/**
	 * Set the expiration date and time
	 * 
	 * @return date when the card was last updated
	 * @throws CardException
	 */
	public Date getTimeLastUpdated();

	// DigitalIdentity requestDigitalIdentity(Policy policy, DigitalIdentity
	// credential);

	/**
	 * Determines whether this <code>ICard</code> could be edited by the user.
	 * General idea is that managed cards (<code>isSelfIssued() == false</code>)
	 * can not be edited by the user but some types of managed cards could also
	 * allow the user to edit their data.
	 */
	public boolean isEditable();

	/**
	 * Puts this <code>ICard</code> object into edit mode when an update
	 * operations on card or on any contained elements are allowed.
	 * <p>
	 * Note that any update operation on this <code>ICard</code> prior to this
	 * method call and after either {@link #applyUpdates()} or
	 * {@link #cancelUpdates()} throws the {@link InvalidStateException}.
	 * 
	 * @throws InvalidStateException
	 *             if this <code>ICard</code> object is in edit mode already.
	 */
	public void beginUpdates() throws InvalidStateException;

	/**
	 * Cancels any updates made to this <code>ICard</code> object or any
	 * contained elements and terminate edit mode. Reloads card's data from
	 * backing data store.
	 * 
	 * @throws InvalidStateException
	 *             if this <code>ICard</code> object is not in edit mode.
	 */
	public void cancelUpdates() throws InvalidStateException;

	/**
	 * Applies an updates made to this <code>ICard</code> object to the
	 * backing data store.
	 * <p>
	 * On success terminate edit mode of this <code>ICard</code> object.
	 * <p>
	 * On error this <code>ICard</code> object remains in edit mode.
	 * 
	 * @throws InvalidStateException
	 *             if this <code>ICard</code> object is not in edit mode.
	 * @throws CardException
	 *             if an error occurs at update backing data store.
	 */
	public void applyUpdates() throws InvalidStateException, CardException;

	/**
	 * Determines wheter this <code>ICard</code> object is in edit mode.
	 */
	public boolean isEditMode();

	/**
	 * Checks whether this card already has claims retrieved or it is necessary
	 * to make additional call to
	 * {@link ICardProvider#getICardByCUID(javax.security.auth.callback.CallbackHandler, CUID, org.eclipse.higgins.icard.auth.ICredential)}
	 * to retrieve them.
	 */
	public boolean isClaimsRetrieved();
	
	/**
	 * Returns an array of credential descriptors which should be used to
	 * request protected clam's information. For some types of cards it is
	 * possible to request such information using different service endpoints
	 * each of which could requere different credentials. Each element in
	 * returned array describe all information sufficient to request such
	 * information on single endpoint. Returned array could be empty or null if
	 * authentiation is not required to request claim's information.
	 * 
	 * @see ICardProvider#getICardByCUID(javax.security.auth.callback.CallbackHandler,
	 *      CUID, org.eclipse.higgins.icard.auth.ICredential)
	 */
	public ICredentialDescriptor[] getRequiredCredentials();
	
	public boolean isFormatSupported(IElementFormat format);
	
	public IElement toElement(IElementFormat format) throws CardException, CardIOException, UnsupportedElementFormatException;
}
